// This is a simple sample application, which creates a window with menu, enabling to
// open and close a camera, start and stop acquisition. The acquired image buffers
// processing is done in the main thread, using the Windows timer.
// NOTE: This sample uses the SynView plain C API, but the code of the sample
// is written in C++.

#define _CRT_SECURE_NO_DEPRECATE // for Visual Studio 8.0 and later

#include <windows.h>
#include <string>
#include <vector>
#include <sv.synview.h>

//=============================================================================
// The CCamera class implements access to one camera. In this simple sample
// it is not necessary; we could use global variables and functions instead.
// However, the CCamera class becomes practical when working with multiple
// cameras (not demonstrated in this sample).

#define NUMBER_OF_BUFFERS %%Buffers%%

class CCamera
{
public:
    CCamera();
    ~CCamera();
    void OpenCamera(HWND hDisplayWnd, LvHSystem hSystem);
    void CloseCamera();
    void StartAcquisition();
    void StopAcquisition();
    --- %%IF IncSimpleImgProcess=1 -----------
    void SetProcessing(bool bDoProcessing);
    --- %%ENDIF ------------------------------
    bool IsOpen();
    bool IsAcquiring();
    void ProcessBuffersInOutputQueue();
    --- %%IF IncUniImgPreprocess=1 -----------
    void SetOptimalUniPixelFormat();
    --- %%ENDIF ------------------------------
    
private:
    LvHSystem    m_hSystem;
    LvHInterface m_hInterface;
    LvHDevice    m_hDevice;
    LvHStream    m_hStream;
    LvHBuffer    m_hBuffers[NUMBER_OF_BUFFERS];
    --- %%IF IncImgDisplay=1 -----------------
    LvHRenderer  m_hRenderer;
    HWND         m_hDisplayWnd;
    --- %%ENDIF ------------------------------
    LvHEvent     m_hEvent;
    --- %%IF IncSimpleImgProcess=1 -----------
    bool         m_bDoProcessing;
    --- %%ENDIF ------------------------------
    --- %%IF IncChunkData=1 ------------------
    --- %%ENDIF ------------------------------
};

// global variables
LvHSystem    g_hSystem         = 0;
CCamera*     g_pCamera         = NULL;
--- %%IF IncSimpleImgProcess=1 -----------
bool         g_bProcessImage   = false;
--- %%ENDIF ------------------------------
HWND         g_hMainWnd        = NULL;
char         g_szWindowText[1024];
UINT_PTR     g_iTimer          = 0;

//-----------------------------------------------------------------------------
// Error check with a message box displayed, if it is error

bool ErrorOccurred(LvStatus ErrorStatus)
{
    if (ErrorStatus == LVSTATUS_OK) return false;
    char szMessage[1024];
    LvGetLastErrorMessage(szMessage, sizeof(szMessage));
    MessageBox(GetActiveWindow(), szMessage, "Error", MB_OK | MB_ICONEXCLAMATION);
    return true;
}

//-----------------------------------------------------------------------------
// In case we would need to set the window title bar text from other thread
// (which is not the case in this application), then using directly the Win32
// API SetWindowText() function can cause a deadlock, because it is internally
// converted to SendMessage(WM_SETTEXT) and SendMessage() requires the message
// loop in the window is not blocked. So safer is to pass the SetWindowText()
// to the main thread, using the WM_APP message.

void SetWindowTitle(HWND hWindow, const char* pszText)
{
    strcpy(g_szWindowText, pszText);
    PostMessage(hWindow, WM_APP, 0, 0);
}

//-----------------------------------------------------------------------------
// CCamera constructor

CCamera::CCamera()
{
    m_hSystem       = 0;
    m_hInterface    = 0;
    m_hDevice       = 0;
    m_hStream       = 0;
    --- %%IF IncImgDisplay=1 ----------
    m_hRenderer     = 0;
    m_hDisplayWnd   = NULL;
    --- %%ENDIF -----------------------
    m_hEvent        = 0;
    --- %%IF IncSimpleImgProcess=1 ----
    m_bDoProcessing = false;
    --- %%ENDIF -----------------------
    --- %%IF IncChunkData=1 ------------------
    --- %%ENDIF ------------------------------
    memset(m_hBuffers, 0, sizeof(m_hBuffers));
}

//-----------------------------------------------------------------------------
// CCamera destructor

CCamera::~CCamera()
{
    if (m_hDevice != 0) CloseCamera();
}

//-----------------------------------------------------------------------------

void CCamera::OpenCamera(HWND hDisplayWnd, LvHSystem hSystem)
{
    if (m_hDevice != 0) CloseCamera();
    m_hSystem = hSystem;
    --- %%IF IncImgDisplay=1 -----------------
    m_hDisplayWnd = hDisplayWnd;
    --- %%ENDIF ------------------------------

    %%REM The SynViewStatus is used instead of ErrorStatus, because the SynViewGen inserts ErrorStatus in CameraRemoteFeaturesCode
    LvStatus SynViewStatus;
    LvHInterface hInterface;
    LvHDevice hDevice;
    LvSystemUpdateInterfaceList(hSystem);
    --- %%IF OpenInterface=ByInterfaceId -----
    SynViewStatus = LvInterfaceOpen(hSystem, "%%INTERFACE_ID%%", &hInterface);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF OpenInterface=ByInterfaceTlType -
    char InterfaceId[256];
    SynViewStatus = LvSystemFindInterface(hSystem, LvFindBy_TLType, "%%INTERFACE_TLTYPE%%", InterfaceId, sizeof(InterfaceId));
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvInterfaceOpen(hSystem, InterfaceId, &hInterface);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------

    LvInterfaceUpdateDeviceList(hInterface);
    --- %%IF OpenDevice=ByDeviceId -----------
    SynViewStatus = LvDeviceOpen(hInterface, "%%DEVICE_ID%%", &hDevice, LvDeviceAccess_Control);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF OpenDevice=ByDeviceModelName ----
    char DeviceId[256];
    SynViewStatus = LvInterfaceFindDevice(hInterface, LvFindBy_ModelName, "%%DEVICE_MODELNAME%%", DeviceId, sizeof(DeviceId));
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvDeviceOpen(hInterface, DeviceId, &hDevice, LvDeviceAccess_Control);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF OpenDevice=ByDeviceIpAddress ----
    char DeviceId[256];
    SynViewStatus = LvInterfaceFindDevice(hInterface, LvFindBy_GevIPAddress, "%%DEVICE_IPADDRESS%%", DeviceId, sizeof(DeviceId));
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvDeviceOpen(hInterface, DeviceId, &hDevice, LvDeviceAccess_Control);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF OpenDevice=ByDeviceUserId -------
    char DeviceId[256];
    SynViewStatus = LvInterfaceFindDevice(hInterface, LvFindBy_UserID, "%%DEVICE_USERID%%", DeviceId, sizeof(DeviceId));
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvDeviceOpen(hInterface, DeviceId, &hDevice, LvDeviceAccess_Control);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF Sys.AutoLoadedCfg=1 -------------
    // Note: When the code was generated, in SynView Explorer was switched ON the option
    // to autoload the camera settings after opening the camera. For this reason the following
    // line loads the camera settings from the file to assure the same starting status.
    --- %%ENDIF ------------------------------
    --- %%IF Sys.SnippetFile=1 ---------------
    SynViewStatus = LvDeviceLoadSettings(hDevice, "", "%%Sys.CameraCfgFile%%", LvSaveFlag_All);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------

    --- %%IF Snippet=All ---------------------
    // The #error line below is intentionally inserted to the code in case you
    // generate the code from streamable or all writable features.
    #error Review the feature settings code and remove the unnecessary items!
    // Before removing this line from the code, go carefully through all the feature
    // settings below and leave there only those, which really need to be set.

    --- %%ENDIF ------------------------------
    --- %%IF Snippet=Streamable --------------
    // The #error line below is intentionally inserted to the code in case you
    // generate the code from streamable or all writable features.
    #error Review the feature settings code and remove the unnecessary items!
    // Before removing this line from the code, go carefully through all the feature
    // settings below and leave there only those, which really need to be set.

    --- %%ENDIF ------------------------------
    %%CameraRemoteFeaturesCode%%

    m_hInterface = hInterface;
    m_hDevice = hDevice;

    --- %%IF IncChunkData=1 ------------------
    --- %%IF FEATURE ChunkModeActive ---
    SynViewStatus = LvSetBool(m_hDevice, LvDevice_ChunkModeActive, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%IF CHUNK_ENABLED OffsetX ---
    SynViewStatus = LvSetEnum (m_hDevice, LvDevice_ChunkSelector, LvChunkSelector_OffsetX);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvSetBool(m_hDevice, LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED OffsetY ---
    SynViewStatus = LvSetEnum (m_hDevice, LvDevice_ChunkSelector, LvChunkSelector_OffsetY);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvSetBool(m_hDevice, LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED Width ---
    SynViewStatus = LvSetEnum (m_hDevice, LvDevice_ChunkSelector, LvChunkSelector_Width);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvSetBool(m_hDevice, LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED Height ---
    SynViewStatus = LvSetEnum (m_hDevice, LvDevice_ChunkSelector, LvChunkSelector_Height);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvSetBool(m_hDevice, LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED PixelFormat ---
    SynViewStatus = LvSetEnum (m_hDevice, LvDevice_ChunkSelector, LvChunkSelector_PixelFormat);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvSetBool(m_hDevice, LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED LinePitch ---
    SynViewStatus = LvSetEnum (m_hDevice, LvDevice_ChunkSelector, LvChunkSelector_LinePitch);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvSetBool(m_hDevice, LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED FrameID ---
    SynViewStatus = LvSetEnum (m_hDevice, LvDevice_ChunkSelector, LvChunkSelector_FrameID);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvSetBool(m_hDevice, LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED Timestamp ---
    SynViewStatus = LvSetEnum (m_hDevice, LvDevice_ChunkSelector, LvChunkSelector_Timestamp);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvSetBool(m_hDevice, LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED ExposureTime ---
    SynViewStatus = LvSetEnum (m_hDevice, LvDevice_ChunkSelector, LvChunkSelector_ExposureTime);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvSetBool(m_hDevice, LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED Gain ---
    SynViewStatus = LvSetEnum (m_hDevice, LvDevice_ChunkSelector, LvChunkSelector_Gain);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvSetBool(m_hDevice, LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED LineStatusAll ---
    SynViewStatus = LvSetEnum (m_hDevice, LvDevice_ChunkSelector, LvChunkSelector_LineStatusAll);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvSetBool(m_hDevice, LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED LvExternalADCValue ---
    SynViewStatus = LvSetEnum (m_hDevice, LvDevice_ChunkSelector, LvChunkSelector_LvExternalADCValue);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvSetBool(m_hDevice, LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%ENDIF ------------------------------
    --- %%ENDIF ------------------------------

    --- %%IF IncUniImgPreprocess=1 -----------
    SynViewStatus = LvSetEnum(m_hDevice, LvDevice_LvUniProcessMode, LvUniProcessMode_Auto);
    if (ErrorOccurred(SynViewStatus)) return;
    SetOptimalUniPixelFormat();
    --- %%ENDIF ------------------------------
    SynViewStatus = LvStreamOpen(m_hDevice, "", &m_hStream);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvEventOpen(m_hStream, LvEventType_NewBuffer, &m_hEvent);
    if (ErrorOccurred(SynViewStatus)) return;
    for (int i=0; i<NUMBER_OF_BUFFERS; i++)
    {
        SynViewStatus = LvBufferOpen(m_hStream, NULL, 0, NULL, 0, &m_hBuffers[i]);
        if (ErrorOccurred(SynViewStatus)) return;
        SynViewStatus = LvBufferQueue(m_hBuffers[i]);
        if (ErrorOccurred(SynViewStatus)) return;
    }
    --- %%IF IncImgDisplay=1 -----------------
    SynViewStatus = LvRendererOpen(m_hStream, &m_hRenderer);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvRendererSetWindow(m_hRenderer, m_hDisplayWnd);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
}

//-----------------------------------------------------------------------------
// Starts acquisition

void CCamera::StartAcquisition()
{
    if (m_hDevice == 0) return;
    LvStatus SynViewStatus;
    SynViewStatus = LvDeviceAcquisitionStart(m_hDevice);
    if (ErrorOccurred(SynViewStatus)) return;
}

//-----------------------------------------------------------------------------
// Stops acquisition

void CCamera::StopAcquisition()
{
    if (!IsAcquiring()) return;
    LvStatus SynViewStatus;
    SynViewStatus = LvDeviceAcquisitionStop(m_hDevice);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvStreamFlushQueue(m_hStream, LvQueueOperation_AllToInput);
    if (ErrorOccurred(SynViewStatus)) return;
}

//-----------------------------------------------------------------------------
// Closes the cameras

void CCamera::CloseCamera()
{
    if (m_hDevice == 0) return;
    if (IsAcquiring()) StopAcquisition();
    LvStatus SynViewStatus;
    SynViewStatus = LvEventClose(&m_hEvent);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%IF IncImgDisplay=1 -----------------
    SynViewStatus = LvRendererClose(&m_hRenderer);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    SynViewStatus = LvStreamFlushQueue(m_hStream, LvQueueOperation_AllDiscard);
    if (ErrorOccurred(SynViewStatus)) return;
    for (int i=0; i<NUMBER_OF_BUFFERS; i++)
        if (m_hBuffers[i] != 0)
        {
            SynViewStatus = LvBufferClose(&m_hBuffers[i]);
            if (ErrorOccurred(SynViewStatus)) return;
        }
    SynViewStatus = LvStreamClose(&m_hStream);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvDeviceClose(&m_hDevice);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvInterfaceClose(&m_hInterface);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%IF IncImgDisplay=1 -----------------
    InvalidateRect(m_hDisplayWnd, NULL, TRUE);
    --- %%ENDIF ------------------------------
}

//-----------------------------------------------------------------------------
// Utility function for enabling/disabling menu items

bool CCamera::IsOpen()
{
    return m_hDevice != 0;
}

//-----------------------------------------------------------------------------
// Utility function for enabling/disabling menu items

bool CCamera::IsAcquiring()
{
    if (m_hDevice == 0) return false;
    int32_t iIsAcquiring;
    LvGetInt32(m_hDevice, LvDevice_LvDeviceIsAcquiring, &iIsAcquiring);
    return iIsAcquiring != 0;
}

--- %%IF IncSimpleImgProcess=1 -----------
//-----------------------------------------------------------------------------
// Switches ON/OFF image processing done in the ProcessBuffersInOutputQueue()

void CCamera::SetProcessing(bool bDoProcessing)
{
    m_bDoProcessing = bDoProcessing;
}

--- %%ENDIF ------------------------------
//-----------------------------------------------------------------------------
// Called from the timer handler to process images in the output queue

void CCamera::ProcessBuffersInOutputQueue()
{
    for (;;)
    {
        int32_t AwaitDelivery;
        if (LvGetInt32(m_hStream, LvStream_LvNumAwaitDelivery, &AwaitDelivery) != LVSTATUS_OK) return;
        if (AwaitDelivery == 0) return;
        LvHBuffer hBuffer;
        if (LvEventWaitAndGetNewBuffer(m_hEvent, &hBuffer, 100) != LVSTATUS_OK) return;

        --- %%IF IncSimpleImgProcess=1 -----------
        // get the pointer to the image data
        void* pData = NULL;
        int32_t iImageOffset = 0;
        --- %%IF IncUniImgPreprocess=1 ------------------
        LvGetPtr (hBuffer, LvBuffer_UniBase, &pData);
        LvGetInt32 (hBuffer, LvBuffer_UniImageOffset, &iImageOffset);
        --- %%ELSE --------------------------------------
        LvGetPtr (hBuffer, LvBuffer_Base, &pData);
        LvGetInt32 (hBuffer, LvBuffer_ImageOffset, &iImageOffset);
        --- %%ENDIF -------------------------------------
        pData = (uint8_t*)pData + iImageOffset;
        if (m_bDoProcessing && pData != NULL)
        {
            // we will do some easy processing - invert the pixel values in an area
            int32_t iWidth;
            int32_t iHeight;
            int32_t iLinePitch;
            LvEnum iPixelFormat;    // LvPixelFormat enumeration value
            LvGetInt32(m_hDevice, LvDevice_Width, &iWidth);
            LvGetInt32(m_hDevice, LvDevice_Height, &iHeight);
            --- %%IF IncUniImgPreprocess=1 -----------
            LvGetEnum(m_hDevice, LvDevice_LvUniPixelFormat, &iPixelFormat);
            LvGetInt32(m_hDevice, LvDevice_LvUniLinePitch, &iLinePitch);
            int32_t iBytesPerPixel = ((iPixelFormat & LV_PIX_EFFECTIVE_PIXEL_SIZE_MASK) >> LV_PIX_EFFECTIVE_PIXEL_SIZE_SHIFT) / 8;
            --- %%ELSE -------------------------------
            LvGetEnum(m_hDevice, LvDevice_PixelFormat, &iPixelFormat);
            int iBitsPerPixel = (iPixelFormat & LV_PIX_EFFECTIVE_PIXEL_SIZE_MASK) >> LV_PIX_EFFECTIVE_PIXEL_SIZE_SHIFT;
            iLinePitch = ((iWidth * iBitsPerPixel)+7)/8;
            int32_t iBytesPerPixel = iBitsPerPixel / 8;
            --- %%ENDIF ------------------------------

            for (int32_t j=0; j<(iHeight/2); j++)
            {
                uint8_t* pPixel = ((uint8_t*)pData) + (iHeight/4 + j)*iLinePitch + (iWidth/4)*iBytesPerPixel;
                for (int32_t i=0; i<(iWidth/2); i++)
                {
                    for (int32_t k=0; k<iBytesPerPixel; k++)
                    {
                        *pPixel = ~*pPixel;
                        pPixel++;
                    }
                }
            }
        }
        --- %%ELSE -------------------------------
        // no image processing demonstrated - switch it ON in the code generation Wizard if you need it
        --- %%ENDIF ------------------------------

        --- %%IF IncChunkData=1 ------------------
        --- %%IF FEATURE ChunkModeActive ---
        // before the chunk data can be read for this buffer, they must be parsed.
        LvBufferParseChunkData(hBuffer);
        int64_t ChunkValue;
        char szChunkLine[256];
        szChunkLine[0] = 0;
        char szValue[128];
        --- %%IF CHUNK_ENABLED OffsetX ---
        LvGetInt64 (m_hDevice, LvDevice_ChunkOffsetX, &ChunkValue);
        sprintf(szValue, "X=%lld | ", ChunkValue);
        strcat(szChunkLine, szValue);
        --- %%ENDIF ------------------------------
        --- %%IF CHUNK_ENABLED OffsetY ---
        LvGetInt64 (m_hDevice, LvDevice_ChunkOffsetY, &ChunkValue);
        sprintf(szValue, "Y=%lld | ", ChunkValue);
        strcat(szChunkLine, szValue);
        --- %%ENDIF ------------------------------
        --- %%IF CHUNK_ENABLED Width ---
        LvGetInt64 (m_hDevice, LvDevice_ChunkWidth, &ChunkValue);
        sprintf(szValue, "W=%lld | ", ChunkValue);
        strcat(szChunkLine, szValue);
        --- %%ENDIF ------------------------------
        --- %%IF CHUNK_ENABLED Height ---
        LvGetInt64 (m_hDevice, LvDevice_ChunkHeight, &ChunkValue);
        sprintf(szValue, "H=%lld | ", ChunkValue);
        strcat(szChunkLine, szValue);
        --- %%ENDIF ------------------------------
        --- %%IF CHUNK_ENABLED PixelFormat ---
        LvGetInt64 (m_hDevice, LvDevice_ChunkPixelFormat, &ChunkValue);
        sprintf(szValue, "PF=0x%llx | ", ChunkValue);
        strcat(szChunkLine, szValue);
        --- %%ENDIF ------------------------------
        --- %%IF CHUNK_ENABLED LinePitch ---
        LvGetInt64 (m_hDevice, LvDevice_ChunkLinePitch, &ChunkValue);
        sprintf(szValue, "LPitch=%lld | ", ChunkValue);
        strcat(szChunkLine, szValue);
        --- %%ENDIF ------------------------------
        --- %%IF CHUNK_ENABLED FrameID ---
        LvGetInt64 (m_hDevice, LvDevice_ChunkFrameID, &ChunkValue);
        sprintf(szValue, "FrameID=%lld | ", ChunkValue);
        strcat(szChunkLine, szValue);
        --- %%ENDIF ------------------------------
        --- %%IF CHUNK_ENABLED Timestamp ---
        LvGetInt64 (m_hDevice, LvDevice_ChunkTimestamp, &ChunkValue);
        sprintf(szValue, "Time=%lld | ", ChunkValue);
        strcat(szChunkLine, szValue);
        --- %%ENDIF ------------------------------
        --- %%IF CHUNK_ENABLED ExposureTime ---
        LvGetInt64 (m_hDevice, LvDevice_ChunkExposureTime, &ChunkValue);
        sprintf(szValue, "Exp=%lld | ", ChunkValue);
        strcat(szChunkLine, szValue);
        --- %%ENDIF ------------------------------
        --- %%IF CHUNK_ENABLED Gain ---
        // the LvIsAvailableEnumEntry() is time consuming, better if you move it outside this function
        if (LvIsAvailableEnumEntry(m_hDevice, LvDevice_ChunkGainSelector, LvChunkGainSelector_AnalogAll) != 0)
        {
            LvSetEnum (m_hDevice, LvDevice_ChunkGainSelector, LvChunkGainSelector_AnalogAll);
            LvGetInt64 (m_hDevice, LvDevice_ChunkGain, &ChunkValue);
            sprintf(szValue, "GainA=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
        }
        if (LvIsAvailableEnumEntry(m_hDevice, LvDevice_ChunkGainSelector, LvChunkGainSelector_DigitalAll) != 0)
        {
            LvSetEnum (m_hDevice, LvDevice_ChunkGainSelector, LvChunkGainSelector_DigitalAll);
            LvGetInt64 (m_hDevice, LvDevice_ChunkGain, &ChunkValue);
            sprintf(szValue, "GainD=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
        }
        --- %%ENDIF ------------------------------
        --- %%IF CHUNK_ENABLED LineStatusAll ---
        LvGetInt64 (m_hDevice, LvDevice_ChunkLineStatusAll, &ChunkValue);
        sprintf(szValue, "Line=0x%llx | ", ChunkValue);
        strcat(szChunkLine, szValue);
        --- %%ENDIF ------------------------------
        --- %%IF CHUNK_ENABLED LvExternalADCValue ---
        // the LvIsAvailableEnumEntry() is time consuming, better if you move it outside this function
        if (LvIsAvailableEnumEntry(m_hDevice, LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC1) != 0)
        {
            LvSetEnum (m_hDevice, LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC1);
            LvGetInt64 (m_hDevice, LvDevice_ChunkLvExternalADCValue, &ChunkValue);
            sprintf(szValue, "ADC1=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
        }
        if (LvIsAvailableEnumEntry(m_hDevice, LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC2) != 0)
        {
            LvSetEnum (m_hDevice, LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC2);
            LvGetInt64 (m_hDevice, LvDevice_ChunkLvExternalADCValue, &ChunkValue);
            sprintf(szValue, "ADC2=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
        }
        if (LvIsAvailableEnumEntry(m_hDevice, LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC3) != 0)
        {
            LvSetEnum (m_hDevice, LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC3);
            LvGetInt64 (m_hDevice, LvDevice_ChunkLvExternalADCValue, &ChunkValue);
            sprintf(szValue, "ADC3=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
        }
        if (LvIsAvailableEnumEntry(m_hDevice, LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC4) != 0)
        {
            LvSetEnum (m_hDevice, LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC4);
            LvGetInt64 (m_hDevice, LvDevice_ChunkLvExternalADCValue, &ChunkValue);
            sprintf(szValue, "ADC4=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
        }
        --- %%ENDIF ------------------------------
        SetWindowTitle(g_hMainWnd, szChunkLine);
        --- %%ENDIF ------------------------------
        --- %%ENDIF ------------------------------

        --- %%IF IncImgDisplay=1 -----------------
        LvRendererDisplayImage(m_hRenderer, hBuffer);
        --- %%ELSE -------------------------------
        // no image display demonstrated - switch it ON in the code generation Wizard if you need it
        --- %%ENDIF ------------------------------
        LvBufferQueue(hBuffer);
    }
}

--- %%IF IncUniImgPreprocess=1 -----------
//-----------------------------------------------------------------------------
// Determines optimal pixel format for the uni-process

void CCamera::SetOptimalUniPixelFormat()
{
    LvEnum PixelFormat;
    LvGetEnum(m_hDevice, LvDevice_PixelFormat, &PixelFormat);
    LvEnum UniPixelFormat = PixelFormat;
    switch (PixelFormat)
    {
        case LvPixelFormat_Mono8:
        case LvPixelFormat_Mono10:
        case LvPixelFormat_Mono12:
        case LvPixelFormat_Mono16:
            UniPixelFormat = LvPixelFormat_Mono8;
            break;

        case LvPixelFormat_BayerGR8:
        case LvPixelFormat_BayerRG8:
        case LvPixelFormat_BayerGB8:
        case LvPixelFormat_BayerBG8:
        case LvPixelFormat_BayerGR10:
        case LvPixelFormat_BayerRG10:
        case LvPixelFormat_BayerGB10:
        case LvPixelFormat_BayerBG10:
        case LvPixelFormat_BayerGR12:
        case LvPixelFormat_BayerRG12:
        case LvPixelFormat_BayerGB12:
        case LvPixelFormat_BayerBG12:
        case LvPixelFormat_RGB8Packed:
        case LvPixelFormat_RGBA8Packed:
            UniPixelFormat = LvPixelFormat_BGRA8Packed;
            break;
    }
    LvSetEnum(m_hDevice, LvDevice_LvUniPixelFormat, UniPixelFormat);
}

--- %%ENDIF ------------------------------
//=============================================================================
//=== Windows application skeleton ============================================
//=============================================================================

HINSTANCE    g_hInstance;              
TCHAR*       g_pszTitle        = "SynView Sample Skeleton: Sample 01";
TCHAR*       g_pszMainWndClass = "SynViewSampleMainWnd";
HMENU        g_hMenu           = NULL;    // Main window menu

// Menu definitions         
#define IDM_OPEN_CAMERA           101
#define IDM_CLOSE_CAMERA          102
#define IDM_ACQUISITION_START     103
#define IDM_ACQUISITION_STOP      104
--- %%IF IncSimpleImgProcess=1 -----------
#define IDM_PROCESS_IMAGE         105
--- %%ENDIF ------------------------------
#define IDM_EXIT                  199

#define CHECK_MENU(id,check)   CheckMenuItem (g_hMenu, id, ((check) ? MF_CHECKED : MF_UNCHECKED) | MF_BYCOMMAND)
#define ENABLE_MENU(id,enable) EnableMenuItem(g_hMenu, id, ((enable) ?  MF_ENABLED : MF_GRAYED) | MF_BYCOMMAND)

//-----------------------------------------------------------------------------

void UpdateControls()
{
    ENABLE_MENU(IDM_OPEN_CAMERA      ,  !g_pCamera->IsOpen());
    ENABLE_MENU(IDM_CLOSE_CAMERA     ,  g_pCamera->IsOpen() && !g_pCamera->IsAcquiring());
    ENABLE_MENU(IDM_ACQUISITION_START,  g_pCamera->IsOpen() && !g_pCamera->IsAcquiring());
    ENABLE_MENU(IDM_ACQUISITION_STOP ,  g_pCamera->IsOpen() && g_pCamera->IsAcquiring());
    --- %%IF IncSimpleImgProcess=1 -----------
    CHECK_MENU(IDM_PROCESS_IMAGE     ,  g_bProcessImage);
    --- %%ENDIF ------------------------------
    char szTitle[256];
    sprintf(szTitle, "Camera %s", g_pCamera->IsOpen() ? "connected" : "not connected");
    if (g_pCamera->IsOpen())
    {
        if (g_pCamera->IsAcquiring())
            strcat(szTitle, ", acquiring.");
        else
            strcat(szTitle, ", not acquiring.");
    }
    SetWindowTitle(g_hMainWnd, szTitle);
}

//-----------------------------------------------------------------------------

long CALLBACK MainWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    switch (message) 
    {
        case WM_APP:
        {
            SetWindowText(hWnd, g_szWindowText);
            break;
        }

        case WM_COMMAND:
        {
            WORD Id = LOWORD(wParam);
            switch (Id) 
            {
                case IDM_OPEN_CAMERA:  
                    g_pCamera->OpenCamera(g_hMainWnd, g_hSystem);
                    UpdateControls();
                    break;
                case IDM_ACQUISITION_START:
                    g_pCamera->StartAcquisition();
                    if (g_iTimer == 0)
                        g_iTimer = SetTimer(hWnd, 100, 50, NULL); // 50 ms
                    UpdateControls();
                    break;
                case IDM_ACQUISITION_STOP: 
                    g_pCamera->StopAcquisition();
                    if (g_iTimer != 0)
                    {
                        KillTimer(hWnd, g_iTimer);
                        g_iTimer = 0;
                    }
                    g_pCamera->ProcessBuffersInOutputQueue();
                    UpdateControls();
                    break;
                --- %%IF IncSimpleImgProcess=1 -----------
                case IDM_PROCESS_IMAGE:
                    g_bProcessImage = !g_bProcessImage;
                    g_pCamera->SetProcessing(g_bProcessImage);
                    UpdateControls();
                    break;
                --- %%ENDIF ------------------------------
                case IDM_CLOSE_CAMERA:
                    g_pCamera->CloseCamera();
                    UpdateControls();
                    break;
                case IDM_EXIT:  
                    DestroyWindow(hWnd);                     
                    break;
                default:
                    return (long) DefWindowProc(hWnd, message, wParam, lParam);
            }
            break;
        }

        case WM_DESTROY:
        {
            if (g_iTimer != 0)
            {
                KillTimer(hWnd, g_iTimer);
                g_iTimer = 0;
            }
            g_pCamera->CloseCamera();
            PostQuitMessage(0);
            break;
        }
    
        case WM_TIMER:
        {
            g_pCamera->ProcessBuffersInOutputQueue();
            break;
        }

        default:
            return (long) DefWindowProc(hWnd, message, wParam, lParam);
    }
    return 0;
}

//-----------------------------------------------------------------------------

BOOL InitApplication(HINSTANCE hInstance)
{
    WNDCLASSEX wcex;
    wcex.cbSize = sizeof(WNDCLASSEX);
  
    wcex.style           = CS_HREDRAW | CS_VREDRAW;
    wcex.lpfnWndProc     = (WNDPROC)MainWndProc;
    wcex.cbClsExtra      = 0;
    wcex.cbWndExtra      = 0;
    wcex.hInstance       = hInstance;
    wcex.hIcon           = LoadIcon(NULL, IDI_APPLICATION);
    wcex.hCursor         = LoadCursor(NULL, IDC_ARROW);
    wcex.hbrBackground   = (HBRUSH)(COLOR_WINDOW+1);
    wcex.lpszMenuName    = NULL;
    wcex.lpszClassName   = g_pszMainWndClass;
    wcex.hIconSm         = LoadIcon(NULL, IDI_APPLICATION);
    RegisterClassEx(&wcex);
  
    return TRUE;
}

//-----------------------------------------------------------------------------

HMENU BuildMenu(void)
{
    HMENU hMenu=CreateMenu();
    HMENU hSubMenu=CreatePopupMenu();
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_EXIT, "Exit");
    AppendMenu(hMenu, MF_POPUP | MF_ENABLED, (UINT_PTR)hSubMenu, "File");
    hSubMenu=CreatePopupMenu();
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_OPEN_CAMERA,       "Connect camera");
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_CLOSE_CAMERA,      "Disconnect camera");
    AppendMenu(hSubMenu, MF_SEPARATOR, -1, "");
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_ACQUISITION_START, "Acquisition Start");  
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_ACQUISITION_STOP,  "Acquisition Stop");  
    --- %%IF IncSimpleImgProcess=1 -----------
    AppendMenu(hSubMenu, MF_SEPARATOR, -1, "");
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_PROCESS_IMAGE,     "Process image");
    --- %%ENDIF ------------------------------
    AppendMenu(hMenu, MF_POPUP | MF_ENABLED, (UINT_PTR)hSubMenu, "Camera");
    return hMenu;
}

//-----------------------------------------------------------------------------

BOOL InitInstance(HINSTANCE hInstance, int nCmdShow)
  {
    // Create the main window menu
    g_hMenu=BuildMenu();
  
    int iScreenWidth  = GetSystemMetrics(SM_CXSCREEN); 
    int iScreenHeight = GetSystemMetrics(SM_CYSCREEN); 
    int iAppWidth = (iScreenWidth * 9)/10; // 90% of width 
    int iAppHeight = (iScreenHeight * 9)/10; // 90% of height
    int iAppXOffset = (iScreenWidth - iAppWidth)/2;
    int iAppYOffset = (iScreenHeight - iAppHeight)/2;
  
    // Create the main window
    g_hInstance = hInstance;
    g_hMainWnd = CreateWindow(
         g_pszMainWndClass, 
         g_pszTitle, 
         WS_OVERLAPPEDWINDOW,
         iAppXOffset, 
         iAppYOffset, 
         iAppWidth, 
         iAppHeight, 
         NULL, 
         g_hMenu, 
         hInstance, NULL);
    if (g_hMainWnd == NULL) return FALSE;

    ShowWindow(g_hMainWnd, nCmdShow);
    UpdateWindow(g_hMainWnd);
    UpdateControls();
    return TRUE;
  }

//-----------------------------------------------------------------------------

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE /*hPrevInstance*/,
                   LPSTR /*lpCmdLine*/, int nCmdShow)
{
    memset(g_szWindowText, 0, sizeof(g_szWindowText));

    if (LvOpenLibrary() != LVSTATUS_OK)
    {
        MessageBox(g_hMainWnd, "Opening the library failed.", "Error", MB_OK | MB_ICONEXCLAMATION);
        return 0;
    }
    if (LvSystemOpen("", &g_hSystem) != LVSTATUS_OK) 
    {
        MessageBox(g_hMainWnd, "Opening the system failed.", "Error", MB_OK | MB_ICONEXCLAMATION);
        return 0;
    }
    // Create one global CCamera class instance
    g_pCamera = new CCamera();
  
    MSG msg;
    if (!InitApplication(hInstance)) 
    {
        MessageBox(g_hMainWnd, "Application init failed.", "Error", MB_OK | MB_ICONEXCLAMATION);
        delete g_pCamera;
        LvSystemClose(&g_hSystem);
        return 0;
    }
  
    if (!InitInstance(hInstance, nCmdShow)) 
    {
        MessageBox(g_hMainWnd, "Instance init failed.", "Error", MB_OK | MB_ICONEXCLAMATION);
        delete g_pCamera;
        LvSystemClose(&g_hSystem);
        return 0;
    }
  
    while (GetMessage(&msg, NULL, NULL, NULL)) 
    {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }
  
    delete g_pCamera;
    LvSystemClose(&g_hSystem);
    LvCloseLibrary();
    return (int) msg.wParam;
}

//-----------------------------------------------------------------------------
