#include "paint.h"

#include <QApplication>
#include <QtGui/QImage>
#ifndef _WIN32
    #include <QX11Info>
#endif

/******************************************************************************

Displaying images in QT.
=======================

QT is very slow in display of images, thus we must utilize the system API to
display images.

WINDOWS:

It is necessary to create a widget (here in the code represented by the
CWidgetPaint, derived from QWidget). This widget will be set as a child widget
of QScrollArea.

Widget Size
-----------

Generally, we need to use either full 1:1 display with scrollbars, or scale-to
fit size, which fits to the window client area. Optionally, the scale-to fit
can be in form of multiple tiles. This requires 2 different behaviors of the
CWidgetPaint - in the first case it should have fixed size - the QScrollArea
then automatically handles scrolling of the visible part of the widget. In the
second case we need that the size of the widget follows the size of the parent
widget and no scroll bars appear.

These 2 behaviors are set in the function CWidgetPaint::SetRenderType(). For
the first case the SizePolicy is set to QSizePolicy::Fixed and for the second
case it is set to QSizePolicy::Ignored. Also the CWidgetPaint::sizeHint() is
defined to give the parent widget possibility to determine possible child
widget size.

In case of fixed size, the widget must be resized to proper size before the
first image is to be painted. This is assured by CWidgetPaint::SetImageParam(),
which is to be called before the acquisition start. This function calls
resize() with image width and height. After this, you do not need to worry
about scrollbars, they are automatically handled by QT, so you only need to
paint the image.

Painting the image
------------------

The paint engine of QT is slow due to many operations involved. We paint the
image using Windows API - these functions are inside the
LvRendererDisplayImage() and LvRendererRepaint() functions, which need a
window handle to paint the image. In Windows you get the handle from QT by the
winId() function. What is important is that you cannot call this paint from
anywhere - the image paint must be called in the main thread in the paint
handler - calling it anywhere else hangs the program totally. So when you want
to paint the image from some other thread or place, save the handle to the
stream and buffer and call the update() function - see the
CWidgetPaint::DisplayImageFromAnotherThread() function. The function
passes a paint event in the queue by creating a QUpdateLaterEvent and it is 
handled by the CWidgetPaint::paintEvent() function. There you can paint 
the image - see CWidgetPaint::OsDisplayImage().

When the paint event happens, QT attepmts to paint the window background. This
would make problems when a live image would be displayed - before each image
the background would be repainted, which would cause flickering. Thus we must
disable painting the background by setting the attribute
Qt::WA_NoSystemBackground. This has a side effect discussed below.

Furthermore, QT by default uses a double buffering technique to combine
in-memory all the paints and place to the window only a result of this
operation. This combining does not know anything about we have painted in the
window already an image and thus it results in overpainting our image by the
window background. To prevent this double buffering, you must override the
CWidgetPaint::paintEngine() function, so that it returns NULL - this switches
off the additional painting. 


No window background
--------------------

Disabling painting the window background brings a problem, that whenever the
window is to be repainted, it simply holds, what was on screen in the window
area before. For this case the Renderer provides a possibility to repaint the
window background (see the LvRenderFlags_RepaintBackground flag).

However, repainting the background is needed only in case the window needs to
be repainted because of change of size or display type or some other overlaying
window was removed. In case a new image is to be painted, the background must
not be repainted, otherwise it would lead to a flickering.

As we paint the image in the paint handler (and the call of this handler can
be a cummulation of multiple update() events), we must do a trick to get info,
whether the paint event was caused by a new image or not. The trick is to post
the QUpdateLaterEvent in case of a new image with the rectangle equal to the
size of the image - see CWidgetPaint::DisplayImageFromAnotherThread().

This rectangle is then passed CWidgetPaint::paintEvent - there we compare it
with the image size and can determine in this way, whether the event comes
from a new image, or is just a repaint event. In case it is a repaint event,
the rectangle will surely have a different size.

Repainting
----------

Another way, how to distinguish the paint and repaint events is to check the
handle of the buffer - if it is the same as during the last paint, we need
only to repaint. The reason to distinguish these 2 cases is only in the tile
mode - the renderer automatically remembers the last buffers used and repaints
all the tiles if possible (see LvRendererRepaint()).

******************************************************************************/

CWidgetPaint::CWidgetPaint(QWidget *parent)
    : QWidget(parent)
{
    m_iSrcWidth = 0;
    m_iSrcHeight = 0;
    m_iPaintWidth = 0;
    m_iPaintHeight = 0;
    m_RenderType = LvRenderType_FullSize;

    m_hDevice = 0;
    m_hStream = 0;
    m_hBuffer = 0;
    m_WinId = 0;
    #ifndef _WIN32
        m_pDisplay = NULL;
    #endif

    m_hLastBuffer = 0;

    // Indicates that the widget has no background, i.e. when the widget receives paint events, 
    // the background is not automatically repainted. Note: Unlike WA_OpaquePaintEvent, newly 
    // exposed areas are never filled with the background (e.g., after showing a window for the 
    // first time the user can see "through" it until the application processes the paint events). 
    // This flag is set or cleared by the widget's author.
    setAttribute(Qt::WA_NoSystemBackground, true);

    // Indicates that the widget wants to draw directly onto the screen. 
    // Widgets with this attribute set do not participate in composition management, 
    // i.e. they cannot be semi-transparent or shine through semi-transparent overlapping widgets. 
    // This flag is only supported on X11 and it disables double buffering.
    setAttribute(Qt::WA_PaintOnScreen, true); 

    // Indicates that the widget paints all its pixels when it receives a paint event. Thus, 
    // it is not required for operations like updating, resizing, scrolling and focus changes 
    // to erase the widget before generating paint events. 
    setAttribute(Qt::WA_OpaquePaintEvent, true);

    m_bBackgroundRepaintNeeded = true;
    m_iLastParentWidth = 0;
    m_iLastParentHeight = 0;

    m_hRenderer = 0;
}

//-----------------------------------------------------------------------------

CWidgetPaint::~CWidgetPaint()
{
}

//-----------------------------------------------------------------------------

void CWidgetPaint::SetRenderer(LvHRenderer hRenderer)
{
    m_hRenderer = hRenderer;
    // here the winId() is not yet valid, so call LvRendererSetWindow() just before the first image
}

//-----------------------------------------------------------------------------

QSize CWidgetPaint::sizeHint() const
{
    int iSrcWidth = 0;
    int iSrcHeight = 0;
    if (m_hDevice != 0)
    {
        LvGetInt32(m_hDevice, LvDevice_Width, &iSrcWidth);
        LvGetInt32(m_hDevice, LvDevice_Height, &iSrcHeight);
    }
    int iFrameWidth = 0;
    int iFrameHeight = 0;
    if (parentWidget() != NULL)
    {
      iFrameWidth  = parentWidget()->width();
      iFrameHeight = parentWidget()->height();
    }
    QSize Size;
    switch (m_RenderType)
    {
        case LvRenderType_FullSize:
        {
            Size.setWidth(iSrcWidth);
            Size.setHeight(iSrcHeight);
            break;
        }
        case LvRenderType_ScaleToFit:
        {
            Size.setWidth(iFrameWidth);
            Size.setHeight(iFrameHeight);
            break;
        }
        case LvRenderType_ScaleToSize:
        {
            Size.setWidth(iFrameWidth);
            Size.setHeight(iFrameHeight);
            break;
        }
        case LvRenderType_ScaleToTiles:
        {
            Size.setWidth(iFrameWidth);
            Size.setHeight(iFrameHeight);
            break;
        }
    }
    if (Size.width() == 0 || Size.height() == 0)
    {
        // set dummy values
        Size.setWidth(640);
        Size.setHeight(480);
    }
    return Size;
}

//-----------------------------------------------------------------------------

void CWidgetPaint::SetRenderType(LvRenderType RenderType)
{
    if (m_RenderType == RenderType) return;
    m_RenderType = RenderType;
    LvSetEnum(m_hRenderer, LvRenderer_LvRenderType, m_RenderType);
    m_bBackgroundRepaintNeeded = true;
    QSizePolicy SizePolicy;
    SizePolicy.setHorizontalPolicy(QSizePolicy::Ignored);
    SizePolicy.setVerticalPolicy(QSizePolicy::Ignored);
    switch (m_RenderType)
    {
        case LvRenderType_FullSize:
        {
            LvGetInt32(m_hDevice, LvDevice_Width, &m_iSrcWidth);
            LvGetInt32(m_hDevice, LvDevice_Height, &m_iSrcHeight);
            SizePolicy.setHorizontalPolicy(QSizePolicy::Fixed);
            SizePolicy.setVerticalPolicy(QSizePolicy::Fixed);
            resize(m_iSrcWidth, m_iSrcHeight); // sets the scroll bars
            break;
        }
        case LvRenderType_ScaleToFit:
        case LvRenderType_ScaleToSize: // not used in this demo
        case LvRenderType_ScaleToTiles:
        {
            if (parentWidget() != NULL)
                resize(parentWidget()->width(), parentWidget()->height());
            break;
        }
    }
    setSizePolicy(SizePolicy);
    update(); // invalidates the paint region
}

//-----------------------------------------------------------------------------

void CWidgetPaint::paintEvent(QPaintEvent * pEvent)
{
    if (pEvent->rect().width() != m_iPaintWidth ||  
        pEvent->rect().height() != m_iPaintHeight)
    {
        // this update event probably does not come from new image, rather from the system
        // that means the window may be needed repainted fully, including unused areas
        m_bBackgroundRepaintNeeded = true;
        m_iPaintWidth  = pEvent->rect().width();
        m_iPaintHeight = pEvent->rect().height();
    }
    OsDisplayImage();
    pEvent->setAccepted(true);
}

//-----------------------------------------------------------------------------

QPaintEngine* CWidgetPaint::paintEngine() const
{
    //disable the paint engine for this widget, we will use direct painting
    return NULL;
}

//-----------------------------------------------------------------------------
// must be called from the main thread - best before the StartAcquisition()

void CWidgetPaint::SetImageParam(LvHDevice hCamera)
{
    // reset history
    m_hLastBuffer = 0;
    m_hDevice = hCamera;
    LvGetInt32(m_hDevice, LvDevice_Width, &m_iSrcWidth);
    LvGetInt32(m_hDevice, LvDevice_Height, &m_iSrcHeight);
    resize(m_iSrcWidth, m_iSrcHeight); // sets the scroll bars
}

//-----------------------------------------------------------------------------

void CWidgetPaint::DisplayImageFromAnotherThread(LvHStream hStream, LvHBuffer hBuffer)
{
    m_hStream = hStream;
    m_hBuffer = hBuffer;
    QRect Rect;
    Rect.setHeight(m_iSrcHeight);
    Rect.setWidth(m_iSrcWidth);
    QApplication::postEvent(this, new QUpdateLaterEvent(Rect));
}

//-----------------------------------------------------------------------------

void CWidgetPaint::OsDisplayImage()
{
    #ifdef _WIN32
        // check if window handle changed
        if (m_WinId != winId())
        {
            m_WinId = winId();
            LvRendererSetWindow(m_hRenderer, m_WinId);
        }
    #else
        if (m_pDisplay != (void*) QX11Info::display() || m_WinId != winId())
        {
            m_pDisplay = (void*) QX11Info::display();
            m_WinId = winId();
            LvRendererSetWindow(m_hRenderer, m_pDisplay, m_WinId);
        }
    #endif
    
    if (m_hBuffer == 0) 
    {
        LvRendererDisplayImage(m_hRenderer, 0, LvRenderFlags_RepaintBackground);
        return; // nothing to display yet
    }
    int iWidth = 0;
    int iHeight = 0;

    if (m_RenderType != LvRenderType_FullSize && parentWidget() != NULL)
    {
        iWidth = parentWidget()->width();
        iHeight = parentWidget()->height();
    }

    if (m_iLastParentWidth != iWidth || m_iLastParentHeight != iHeight)
    {
        m_bBackgroundRepaintNeeded = true;
        m_iLastParentWidth = iWidth;
        m_iLastParentHeight = iHeight;
    }
    uint32_t dwFlags = 0;
    if (m_bBackgroundRepaintNeeded) dwFlags |= LvRenderFlags_RepaintBackground;

    if (m_hLastBuffer != m_hBuffer)
    {
        m_hLastBuffer = m_hBuffer;
        LvRendererDisplayImage(m_hRenderer, m_hBuffer, dwFlags);
    }
    else
    {
        // only repaint is needed
        LvRendererRepaint(m_hRenderer, dwFlags);
    }
    m_bBackgroundRepaintNeeded = false;
}

//-----------------------------------------------------------------------------
// This is needed to satisfy the linker,
// copied from QEvent.cpp, to be checked with new QT version!

QUpdateLaterEvent::QUpdateLaterEvent(const QRegion& paintRegion)
    : QEvent(UpdateLater), m_region(paintRegion)
{
}

QUpdateLaterEvent::~QUpdateLaterEvent()
{
}



