#define _CRT_SECURE_NO_WARNINGS
#define _CRT_SECURE_NO_DEPRECATE

#include <QtGui>
#include "mainwindow.h"

//-----------------------------------------------------------------------------
// This helper class changes the cursor to wait (hourglass) cursor during its exsitence
class CHourGlass
{
public:
    CHourGlass();
    ~CHourGlass();
};
CHourGlass::CHourGlass()  {QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));}
CHourGlass::~CHourGlass() {QApplication::restoreOverrideCursor();}

//-----------------------------------------------------------------------------

CMainWindow::CMainWindow(QApplication* pApplication)
{
    m_pApplication = pApplication;
    setWindowTitle(tr("SynView QT 4.8 Sample"));
    CreateMenuAndToolBar();

    // the main window is splitted to a panel in the left side and image frame in the right side
    m_pPanelImgSplitter = new QSplitter(this);
    setCentralWidget(m_pPanelImgSplitter);
    // the panel in this sample is empty, it is prepared for adding some controls
    m_pPanel = new QWidget(this);
    m_pPanelImgSplitter->addWidget(m_pPanel);
    QWidget* pWidget = new QWidget(this);
    // to the scroll area is placed the paint widget, when the camera is connected
    m_pScrollAreaImage = new QScrollArea;
    // in this lable the image counter, fps and chunk data are displayed
    m_pLabelChunkData = new QLabel(this);
    QVBoxLayout *pLayout = new QVBoxLayout;
    pWidget->setLayout(pLayout);
    pLayout->addWidget(m_pLabelChunkData);
    pLayout->addWidget(m_pScrollAreaImage);
    m_pPanelImgSplitter->addWidget(pWidget);

    // scaling and tiling is currently supported only in Windows
    #ifdef _WIN32
        connect(m_pActionDisplayClipped, SIGNAL(triggered()), this, SLOT(ActionDisplayClipped()));
        connect(m_pActionDisplayScaled , SIGNAL(triggered()), this, SLOT(ActionDisplayScaled()));
        connect(m_pActionDisplayTiled  , SIGNAL(triggered()), this, SLOT(ActionDisplayTiled()));
    #endif
    
    connect(m_pActionAcquisitionStart, SIGNAL(triggered()), this, SLOT(ActionAcquisitionStart()));
    connect(m_pActionAcquisitionStop , SIGNAL(triggered()), this, SLOT(ActionAcquisitionStop()));
    connect(m_pActionConnectCamera   , SIGNAL(triggered()), this, SLOT(ActionConnectCamera()));
    connect(m_pActionDisconnectCamera, SIGNAL(triggered()), this, SLOT(ActionDisconnectCamera()));
    --- %%IF IncSimpleImgProcess=1 -----------
    connect(m_pActionSimpleImageProcessing, SIGNAL(triggered(bool)), this, SLOT(ActionSimpleImageProcessing(bool)));
    --- %%ENDIF ------------------------------
    connect(m_pActionExit            , SIGNAL(triggered()), this, SLOT(close()));

    m_dwLastFrameCount = 0;
    // QT requires the access to the widgets to be done in the main thread, 
    // so we utilize a timer for refreshing the chunk data label
    m_pTimer = new QTimer(this);
    connect(m_pTimer, SIGNAL(timeout()), this, SLOT(OnTimer()));
    m_pTimer->start(200);

    // in this sample we use only one camera
    m_pCamera = NULL;
    CreateCameraClass();
    UpdateControls();
    SetSizes();
}

//-----------------------------------------------------------------------------

void CMainWindow::SetSizes()
{
    // simply derive the application size from the screen size
    QRect ScreenRect = m_pApplication->desktop()->screenGeometry();
    resize((ScreenRect.width()*8)/10, (ScreenRect.height()*8)/10);
    move(ScreenRect.width()/20, ScreenRect.height()/20);
    QList<int> SplitterSizes;
    SplitterSizes.push_back((width()*1)/10);
    SplitterSizes.push_back((width()*9)/10);  
    m_pPanelImgSplitter->setSizes(SplitterSizes);
}

//-----------------------------------------------------------------------------

void CMainWindow::CreateMenuAndToolBar()
{
    m_pToolBar = new QToolBar;
    // we use 32x32 icons, disable icon rescaling
    QSize IconSize(32, 32);
    m_pToolBar->setIconSize(IconSize);

    #ifdef _WIN32
        m_pActionDisplayClipped = new QAction(QIcon(":/Icons/displayclipped.png"), tr("Display in full size"), this);
        m_pActionDisplayScaled  = new QAction(QIcon(":/Icons/displayscaled.png") , tr("Display scaled to fit"), this);
        m_pActionDisplayTiled   = new QAction(QIcon(":/Icons/displaytiles.png")  , tr("Display as tiles"), this);
        m_pActionDisplayClipped->setCheckable(true);
        m_pActionDisplayScaled ->setCheckable(true);
        m_pActionDisplayTiled  ->setCheckable(true);
        m_pActionDisplayClipped->setChecked(true);
        m_pActionGroupDisplayMode = new QActionGroup(this);
        m_pActionGroupDisplayMode->addAction(m_pActionDisplayClipped);
        m_pActionGroupDisplayMode->addAction(m_pActionDisplayScaled );
        m_pActionGroupDisplayMode->addAction(m_pActionDisplayTiled  );
    #endif

    m_pActionAcquisitionStart = new QAction(QIcon(":/Icons/acquisitionstart.png"), tr("Start acquisition"), this);
    m_pActionAcquisitionStop  = new QAction(QIcon(":/Icons/acquisitionstop.png"), tr("Stop acquisition"), this);
    m_pActionAcquisitionStop->setEnabled(false);

    m_pActionConnectCamera    = new QAction(QIcon(":/Icons/opencamera.png"), tr("Connect camera"), this);
    m_pActionDisconnectCamera = new QAction(QIcon(":/Icons/closecamera.png"), tr("Disconnect camera"), this);

    --- %%IF IncSimpleImgProcess=1 -----------
    m_pActionSimpleImageProcessing  = new QAction(QIcon(":/Icons/imageprocessing.png"), tr("Simple image processing"), this);
    m_pActionSimpleImageProcessing->setCheckable(true);
    m_pActionSimpleImageProcessing->setChecked(false);
    --- %%ENDIF ------------------------------
    m_pActionExit = new QAction(QIcon(":/Icons/exit.png"), tr("Exit"), this);

    m_pMenuFile = new QMenu(tr("File"), this);
    m_pMenuFile->addAction(m_pActionExit);
    
    m_pMenuAcquisition = new QMenu(tr("Acquisition"), this);
    m_pMenuAcquisition->addAction(m_pActionAcquisitionStart);
    m_pMenuAcquisition->addAction(m_pActionAcquisitionStop);

    m_pMenuCamera = new QMenu(tr("Camera"), this);
    m_pMenuCamera->addAction(m_pActionConnectCamera);
    m_pMenuCamera->addAction(m_pActionDisconnectCamera);
    --- %%IF IncSimpleImgProcess=1 -----------
    m_pMenuCamera->addSeparator();
    m_pMenuCamera->addAction(m_pActionSimpleImageProcessing);
    --- %%ENDIF ------------------------------

    #ifdef _WIN32
        m_pMenuDisplay = new QMenu(tr("Display"), this);
        m_pMenuDisplay->addAction(m_pActionDisplayClipped);
        m_pMenuDisplay->addAction(m_pActionDisplayScaled );
        m_pMenuDisplay->addAction(m_pActionDisplayTiled  );
    #endif

    menuBar()->addMenu(m_pMenuFile);
    menuBar()->addMenu(m_pMenuCamera);
    menuBar()->addMenu(m_pMenuAcquisition);
    #ifdef _WIN32
        menuBar()->addMenu(m_pMenuDisplay);
    #endif

    m_pToolBar->addAction(m_pActionConnectCamera);
    m_pToolBar->addAction(m_pActionDisconnectCamera);
    m_pToolBar->addSeparator();
    m_pToolBar->addAction(m_pActionAcquisitionStart);
    m_pToolBar->addAction(m_pActionAcquisitionStop);
    #ifdef _WIN32
        m_pToolBar->addSeparator();
        m_pToolBar->addAction(m_pActionDisplayClipped);
        m_pToolBar->addAction(m_pActionDisplayScaled );
        m_pToolBar->addAction(m_pActionDisplayTiled  );
    #endif
    --- %%IF IncSimpleImgProcess=1 -----------
    m_pToolBar->addSeparator();
    m_pToolBar->addAction(m_pActionSimpleImageProcessing);
    --- %%ENDIF ------------------------------
    m_pToolBar->setMovable(false);

    addToolBar(Qt::TopToolBarArea, m_pToolBar);
}

//-----------------------------------------------------------------------------
// This enables/disables the menu items and toolbar buttons according to the current status 

void CMainWindow::UpdateControls()
{
    m_pActionAcquisitionStart->setEnabled(m_pCamera->IsConnected() && !m_pCamera->IsAcquiring());
    m_pActionAcquisitionStop->setEnabled(m_pCamera->IsAcquiring());
    m_pActionConnectCamera->setEnabled(!m_pCamera->IsConnected());
    m_pActionDisconnectCamera->setEnabled(m_pCamera->IsConnected());
}

//-----------------------------------------------------------------------------

void CMainWindow::ActionDisplayClipped()
{
    // this sets the fixed size of the widget, so scrollbars become active
    m_pScrollAreaImage->setWidgetResizable(false);
    m_pCamera->SetRenderType(LvRenderType_FullSize);
    UpdateControls();
}

//-----------------------------------------------------------------------------

void CMainWindow::ActionDisplayScaled()
{
    // this enables to resize the paint widget
    m_pScrollAreaImage->setWidgetResizable(true);
    m_pCamera->SetRenderType(LvRenderType_ScaleToFit);
    UpdateControls();
}

//-----------------------------------------------------------------------------

void CMainWindow::ActionDisplayTiled()
{
    // this enables to resize the paint widget
    m_pScrollAreaImage->setWidgetResizable(true);
    m_pCamera->SetRenderType(LvRenderType_ScaleToTiles);
    UpdateControls();
}

//-----------------------------------------------------------------------------

void CMainWindow::ActionConnectCamera()
{
    CHourGlass HourGlass;
    m_pCamera->ConnectCamera(m_pScrollAreaImage);
    UpdateControls();
}

//-----------------------------------------------------------------------------

void CMainWindow::ActionAcquisitionStart()
{
    m_pCamera->StartAcquisition();
    UpdateControls();
}

//-----------------------------------------------------------------------------

void CMainWindow::ActionAcquisitionStop()
{
    m_pCamera->StopAcquisition();
    UpdateControls();
}

//-----------------------------------------------------------------------------

void CMainWindow::ActionDisconnectCamera()
{
    CHourGlass HourGlass;
    m_pCamera->DisconnectCamera();
    UpdateControls();
}

--- %%IF IncSimpleImgProcess=1 -----------
//-----------------------------------------------------------------------------
void CMainWindow::ActionSimpleImageProcessing(bool bChecked)
{
    m_pCamera->SetProcessing(bChecked);
    UpdateControls();
}
--- %%ENDIF ------------------------------
//-----------------------------------------------------------------------------
// Here we update the label, because it cannot be updated from the callback, 
// which is called from another thread

void CMainWindow::OnTimer()
{
    char szText[512];
    szText[0]=0;
    double dFps = m_pCamera->CalculateFramesPerSecond();
    if (m_dwLastFrameCount != m_pCamera->GetFrameCount())
    {
        m_dwLastFrameCount = m_pCamera->GetFrameCount();
        sprintf(szText, "Image %04d | %0.1f f/sec", m_dwLastFrameCount, dFps);
        --- %%IF IncChunkData=1 ------------------
        --- %%IF FEATURE ChunkModeActive ---
        strcat(szText, " | ");
        strcat(szText, m_pCamera->GetChunkDataString());
        --- %%ENDIF ------------------------------
        --- %%ENDIF ------------------------------
        m_pLabelChunkData->setText(szText);
    }
}

//-----------------------------------------------------------------------------

void CMainWindow::closeEvent(QCloseEvent *event)
{
    m_pTimer->stop();
    DeleteCameraClass();
    event->accept();
}

//-----------------------------------------------------------------------------

void CMainWindow::CreateCameraClass()
{
    if (LvOpenLibrary() != LVSTATUS_OK) 
    {
        QMessageBox::critical(this, tr("Error"), tr("Opening the library failed."));
        return;
    }
    m_pCamera = new CCamera(this);
}

//-----------------------------------------------------------------------------

void CMainWindow::DeleteCameraClass()
{
    delete m_pCamera;
    LvCloseLibrary();
}

//-----------------------------------------------------------------------------

