// This is a simple sample application, which creates a window,
// enabling to open and close a camera, start and stop acquisition.
// The acquired images are processed with the OpenCV library - on a ROI
// of each image is applied EITHER the Canny operator - in case the
// image is acquired or converted to 8-bit pixel format, OR the Smooth
// operator, in case the image is in different that 8-bit pixel format.

// The acquired images are processed using OpenCV library
// To run the sample, the OpenCV library (2.3.1 or newer) must be installed
// The Makefile expects the library in /usr/local. For other locations,
// the Makefile has to be adjusted

// All the parts where the OpenCV is used are bracketed by
// the #ifdef USE_OPENCV_PROCESSING macro.

// NOTE: This sample uses the SynView plain C API, but the code of the sample
// is written in C++.

// Important note: If the 10-bit or 12-bit monochrome pixel format is
// processed by OpenCV, the resulting pixel values can be outside the permitted
// pixel range, as OpenCV treats such images as 16-bit. If you copy such image
// back to SynView buffer and apply on such image the LUT from NET Image
// Processing library, you can get a crash caused by the fact that the pixel
// value points outside the LUT.

#include <stdio.h>
#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>
#include <X11/Intrinsic.h>
#include <X11/Xproto.h>
#include <X11/XKBlib.h>

#include <sv.synview.h>

#define USE_OPENCV_PROCESSING

#ifdef USE_OPENCV_PROCESSING
    #undef True
    #undef False
    #include <opencv2/imgproc/imgproc_c.h>
    #include <opencv2/imgproc/imgproc.hpp>
    #include <opencv2/highgui/highgui.hpp>
    // add more OpenCV modules if needed ...
    #define True 1
    #define False 0
#endif

//=============================================================================
// The CCamera class implements access to one camera. In this simple sample
// it is not necessary; we could use global variables and functions instead.
// However, the CCamera class becomes practical when working with multiple
// cameras (not demonstrated in this sample).

#define NUMBER_OF_BUFFERS %%Buffers%%

class CCamera
{
public:
    CCamera();
    ~CCamera();
    void OpenCamera(Display *pDisplay, Window hWindow, LvHSystem hSystem);
    void CloseCamera();
    void StartAcquisition();
    void StopAcquisition();
    void SetProcessing(bool bDoProcessing);
    bool IsOpen();
    bool IsAcquiring();
    void CallbackNewBuffer(LvHBuffer hBuffer, void* pBuffer);
    --- %%IF IncUniImgPreprocess=1 ------------------
    void SetOptimalUniPixelFormat();
    --- %%ENDIF -------------------------------------

private:
    #ifdef USE_OPENCV_PROCESSING
        void OpenCvProcessingOpen();
        void OpenCvCreateImages();
        void OpenCvProcessImage(void* pSrcData);
        void OpenCvProcessingClose();

        IplImage* m_pSrcImg;
        IplImage* m_pDstImg;
        CvFont    m_TextFont;
        CvScalar  m_DrawColorBlack;
        CvScalar  m_DrawColorWhite;
        int       m_SizeROI;    // initial size of ROI in percentage of window size
    #endif

    LvHSystem    m_hSystem;
    LvHInterface m_hInterface;
    LvHDevice    m_hDevice;
    LvHStream    m_hStream;
    LvHBuffer    m_hBuffers[NUMBER_OF_BUFFERS];
    --- %%IF IncImgDisplay=1 -----------------
    LvHRenderer  m_hRenderer;
    Display*     m_pDisplay;
    Window       m_hWindow;
    --- %%ENDIF ------------------------------
    LvHEvent     m_hEvent;
    bool         m_bDoProcessing;
};

// global variables
LvHSystem    g_hSystem         = 0;
CCamera*     g_pCamera         = NULL;
bool         g_bProcessImage   = true;
Display*     g_pDisplay        = NULL;
Window       g_hWindow         = 0;
char         g_ErrMsg[1024]    = {0};

// Forward declarations
static void SetWindowTitle (const char *szTitle);

//-----------------------------------------------------------------------------
// Error check with a message displayed, if it is error
void DisplayError (const char *Message)
{
    printf ("Error: %s\n", Message);
    fflush (stdout);

    snprintf (g_ErrMsg, sizeof(g_ErrMsg), "[Error] %s", Message);
    XEvent expose_event;
    expose_event.type = Expose;
    XSendEvent (g_pDisplay, g_hWindow, 0, 0, &expose_event);
}

void ClearError ()
{
    sprintf (g_ErrMsg, "");
    XEvent expose_event;
    expose_event.type = Expose;
    XSendEvent (g_pDisplay, g_hWindow, 0, 0, &expose_event);
}

bool ErrorOccurred(LvStatus ErrorStatus)
{
    if (ErrorStatus == LVSTATUS_OK) return false;
    char szMessage[1024];
    LvGetLastErrorMessage(szMessage, sizeof(szMessage));
    DisplayError (szMessage);
    return true;
}

void UpdateStatus ()
{
    if (!g_pCamera) return;

    char szTitle[256];
    sprintf(szTitle, "Camera %s", g_pCamera->IsOpen() ? "open" : "not open");
    if (g_pCamera->IsOpen())
    {
        if (g_pCamera->IsAcquiring())
            strcat(szTitle, ", acquiring.");
        else
            strcat(szTitle, ", not acquiring.");
    }
    SetWindowTitle (szTitle);
}

//-----------------------------------------------------------------------------
// Callback function for the delivered image. Cannot be a method of a class,
// thus we use the pUserParam to set the pointer to the CCamera class instance
// when registering the callback, so that we can then pass the callback to the
// CallbackNewBuffer() method of CCamera class.

void LV_STDC CallbackNewBufferFunction(LvHBuffer hBuffer,
                                       void* pUserPointer,
                                       void* pUserParam)
{
    CCamera* pCamera = (CCamera*) pUserParam;
    pCamera->CallbackNewBuffer(hBuffer, pUserPointer);
}

//-----------------------------------------------------------------------------
// CCamera constructor

CCamera::CCamera()
{
    m_hSystem       = 0;
    m_hInterface    = 0;
    m_hDevice       = 0;
    m_hStream       = 0;
    --- %%IF IncImgDisplay=1 ----------
    m_hRenderer     = 0;
    m_pDisplay      = NULL;
    m_hWindow       = 0;
    --- %%ENDIF -----------------------
    m_hEvent        = 0;
    m_bDoProcessing = g_bProcessImage;
    memset(m_hBuffers, 0, sizeof(m_hBuffers));
}

//-----------------------------------------------------------------------------
// CCamera destructor

CCamera::~CCamera()
{
    if (m_hDevice != 0) CloseCamera();
}

//-----------------------------------------------------------------------------

void CCamera::OpenCamera(Display *pDisplay, Window hWindow, LvHSystem hSystem)
{
    if (m_hDevice != 0) CloseCamera();

    #ifdef USE_OPENCV_PROCESSING
        OpenCvProcessingOpen();
    #endif

    m_hSystem = hSystem;
    --- %%IF IncImgDisplay=1 -----------------
    m_pDisplay = pDisplay;
    m_hWindow = hWindow;
    --- %%ENDIF ------------------------------

    LvStatus SynViewStatus;
    LvHInterface hInterface;
    LvHDevice hDevice;
    LvSystemUpdateInterfaceList(hSystem);
    --- %%IF OpenInterface=ByInterfaceId -----
    SynViewStatus = LvInterfaceOpen(hSystem, "%%INTERFACE_ID%%", &hInterface);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF OpenInterface=ByInterfaceTlType -
    char InterfaceId[256];
    SynViewStatus = LvSystemFindInterface(hSystem, LvFindBy_TLType, "%%INTERFACE_TLTYPE%%", InterfaceId, sizeof(InterfaceId));
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvInterfaceOpen(hSystem, InterfaceId, &hInterface);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------

    LvInterfaceUpdateDeviceList(hInterface);
    --- %%IF OpenDevice=ByDeviceId -----------
    SynViewStatus = LvDeviceOpen(hInterface, "%%DEVICE_ID%%", &hDevice, LvDeviceAccess_Control);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF OpenDevice=ByDeviceModelName ----
    char DeviceId[256];
    SynViewStatus = LvInterfaceFindDevice(hInterface, LvFindBy_ModelName, "%%DEVICE_MODELNAME%%", DeviceId, sizeof(DeviceId));
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvDeviceOpen(hInterface, DeviceId, &hDevice, LvDeviceAccess_Control);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF OpenDevice=ByDeviceIpAddress ----
    char DeviceId[256];
    SynViewStatus = LvInterfaceFindDevice(hInterface, LvFindBy_GevIPAddress, "%%DEVICE_IPADDRESS%%", DeviceId, sizeof(DeviceId));
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvDeviceOpen(hInterface, DeviceId, &hDevice, LvDeviceAccess_Control);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF OpenDevice=ByDeviceUserId -------
    char DeviceId[256];
    SynViewStatus = LvInterfaceFindDevice(hInterface, LvFindBy_UserID, "%%DEVICE_USERID%%", DeviceId, sizeof(DeviceId));
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvDeviceOpen(hInterface, DeviceId, &hDevice, LvDeviceAccess_Control);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF Sys.AutoLoadedCfg=1 -------------
    // Note: When the code was generated, in SynView Explorer was switched ON the option
    // to autoload the camera settings after opening the camera. For this reason the following
    // line loads the camera settings from the file to assure the same starting status.
    --- %%ENDIF ------------------------------
    --- %%IF Sys.SnippetFile=1 ---------------
    SynViewStatus = LvDeviceLoadSettings(hDevice, "", "%%Sys.CameraCfgFile%%", LvSaveFlag_All);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------

    --- %%IF Snippet=All ---------------------
    // The #error line below is intentionally inserted to the code in case you
    // generate the code from streamable or all writable features.
    #error Review the feature settings code and remove the unnecessary items!
    // Before removing this line from the code, go carefully through all the feature
    // settings below and leave there only those, which really need to be set.

    --- %%ENDIF ------------------------------
    --- %%IF Snippet=Streamable --------------
    // The #error line below is intentionally inserted to the code in case you
    // generate the code from streamable or all writable features.
    #error Review the feature settings code and remove the unnecessary items!
    // Before removing this line from the code, go carefully through all the feature
    // settings below and leave there only those, which really need to be set.

    --- %%ENDIF ------------------------------
    %%CameraRemoteFeaturesCode%%

    m_hInterface = hInterface;
    m_hDevice = hDevice;

    --- %%IF IncUniImgPreprocess=1 ------------------
    SynViewStatus = LvSetEnum(m_hDevice, LvDevice_LvUniProcessMode, LvUniProcessMode_Auto);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%IF ForceUniConvertTo8bit=1 ----------------
    // OpenCV enables to use the Canny operator only on 8-bit monochrome image
    // so we set UniProcessing to this pixel format.
    SynViewStatus = LvSetEnum(m_hDevice, LvDevice_LvUniPixelFormat, LvPixelFormat_Mono8);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ELSE  --------------------------------
    SetOptimalUniPixelFormat();
    --- %%ENDIF --------------------------------
    --- %%ENDIF --------------------------------

    --- %%IF SwTriggered=1 ------------------
    // Set the camera to software triggered mode
    SynViewStatus = LvSetEnum(hDevice, LvDevice_TriggerSelector, LvTriggerSelector_FrameStart);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvSetEnum(hDevice, LvDevice_TriggerMode, LvTriggerMode_On);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvSetEnum(hDevice, LvDevice_TriggerSource, LvTriggerSource_Software);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------

    SynViewStatus = LvStreamOpen(m_hDevice, "", &m_hStream);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvEventOpen(m_hStream, LvEventType_NewBuffer, &m_hEvent);
    if (ErrorOccurred(SynViewStatus)) return;
    for (int i=0; i<NUMBER_OF_BUFFERS; i++)
    {
        SynViewStatus = LvBufferOpen(m_hStream, NULL, 0, NULL, 0, &m_hBuffers[i]);
        if (ErrorOccurred(SynViewStatus)) return;
        SynViewStatus = LvBufferQueue(m_hBuffers[i]);
        if (ErrorOccurred(SynViewStatus)) return;
    }
    --- %%IF IncImgDisplay=1 -----------------
    SynViewStatus = LvRendererOpen(m_hStream, &m_hRenderer);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvRendererSetWindow(m_hRenderer, m_pDisplay, m_hWindow);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    SynViewStatus = LvEventSetCallbackNewBuffer(m_hEvent, CallbackNewBufferFunction, this);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvEventStartThread(m_hEvent);
    if (ErrorOccurred(SynViewStatus)) return;
    #ifdef USE_OPENCV_PROCESSING
        // Create an image descriptor for source image with full sensor resolution
        // and the destination image buffer
        OpenCvCreateImages();
    #endif
}

//-----------------------------------------------------------------------------
// Starts acquisition

void CCamera::StartAcquisition()
{
    if (m_hDevice == 0) return;
    LvStatus SynViewStatus;
    SynViewStatus = LvDeviceAcquisitionStart(m_hDevice);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%IF SwTriggered=1 -------------------
    // and issue the first software trigger
    SynViewStatus = LvCmdExecute(m_hDevice, LvDevice_TriggerSoftware);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
}

//-----------------------------------------------------------------------------
// Stops acquisition

void CCamera::StopAcquisition()
{
    if (!IsAcquiring()) return;
    LvStatus SynViewStatus;
    SynViewStatus = LvDeviceAcquisitionStop(m_hDevice);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvStreamFlushQueue(m_hStream, LvQueueOperation_AllToInput);
    if (ErrorOccurred(SynViewStatus)) return;
}

//-----------------------------------------------------------------------------
// Closes the cameras

void CCamera::CloseCamera()
{
    if (m_hDevice == 0) return;
    if (IsAcquiring()) StopAcquisition();
    LvStatus SynViewStatus;
    SynViewStatus = LvEventStopThread(m_hEvent);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvEventClose(&m_hEvent);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%IF IncImgDisplay=1 -----------------
    SynViewStatus = LvRendererClose(&m_hRenderer);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    SynViewStatus = LvStreamFlushQueue(m_hStream, LvQueueOperation_AllDiscard);
    if (ErrorOccurred(SynViewStatus)) return;
    for (int i=0; i<NUMBER_OF_BUFFERS; i++)
        if (m_hBuffers[i] != 0)
        {
            SynViewStatus = LvBufferClose(&m_hBuffers[i]);
            if (ErrorOccurred(SynViewStatus)) return;
        }
    SynViewStatus = LvStreamClose(&m_hStream);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvDeviceClose(&m_hDevice);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = LvInterfaceClose(&m_hInterface);
    if (ErrorOccurred(SynViewStatus)) return;
    #ifdef USE_OPENCV_PROCESSING
        OpenCvProcessingClose();
    #endif
}

//-----------------------------------------------------------------------------
// Utility function for enabling/disabling menu items

bool CCamera::IsOpen()
{
    return m_hDevice != 0;
}

//-----------------------------------------------------------------------------
// Utility function for enabling/disabling menu items

bool CCamera::IsAcquiring()
{
    if (m_hDevice == 0) return false;
    int32_t iIsAcquiring;
    LvGetInt32(m_hDevice, LvDevice_LvDeviceIsAcquiring, &iIsAcquiring);
    return iIsAcquiring != 0;
}

//-----------------------------------------------------------------------------
// Switches ON/OFF image processing done in the callback

void CCamera::SetProcessing(bool bDoProcessing)
{
    m_bDoProcessing = bDoProcessing;
}

//-----------------------------------------------------------------------------
// Callback function called for each delivered image

void CCamera::CallbackNewBuffer(LvHBuffer hBuffer, void* pUserPointer)
{
    // get the pointer to the image data
    void* pData = NULL;
    int32_t iImageOffset = 0;
    --- %%IF IncUniImgPreprocess=1 ------------------
    LvGetPtr (hBuffer, LvBuffer_UniBase, &pData);
    LvGetInt32 (hBuffer, LvBuffer_UniImageOffset, &iImageOffset);
    --- %%ELSE --------------------------------------
    LvGetPtr (hBuffer, LvBuffer_Base, &pData);
    LvGetInt32 (hBuffer, LvBuffer_ImageOffset, &iImageOffset);
    --- %%ENDIF -------------------------------------
    pData = (uint8_t*)pData + iImageOffset;
    if (m_bDoProcessing && pData != NULL)
    {
        #ifdef USE_OPENCV_PROCESSING
            OpenCvProcessImage(pData);
        #else
            // we will do some easy processing - invert the pixel values in an area
            int32_t iWidth;
            int32_t iHeight;
            int32_t iLinePitch;
            LvEnum Val_PixelFormat;    // LvPixelFormat enumeration value
            LvGetInt32(m_hDevice, LvDevice_Width, &iWidth);
            LvGetInt32(m_hDevice, LvDevice_Height, &iHeight);
            --- %%IF IncUniImgPreprocess=1 ------------------
            LvGetEnum(m_hDevice, LvDevice_LvUniPixelFormat, &Val_PixelFormat);
            --- %%ELSE --------------------------------------
            LvGetEnum(m_hDevice, LvDevice_PixelFormat, &Val_PixelFormat);
            --- %%ENDIF -------------------------------------
            int iBitsPerPixel = (Val_PixelFormat & LV_PIX_EFFECTIVE_PIXEL_SIZE_MASK) >> LV_PIX_EFFECTIVE_PIXEL_SIZE_SHIFT;
            iLinePitch = ((iWidth * iBitsPerPixel)+7)/8;
            int32_t iBytesPerPixel = iBitsPerPixel / 8;

            for (int32_t j=0; j<(iHeight/2); j++)
            {
                uint8_t* pPixel = ((uint8_t*)pData) + (iHeight/4 + j)*iLinePitch + (iWidth/4)*iBytesPerPixel;
                for (int32_t i=0; i<(iWidth/2); i++)
                {
                    for (int32_t k=0; k<iBytesPerPixel; k++)
                    {
                        *pPixel = ~*pPixel;
                        pPixel++;
                    }
                }
            }
        #endif
    }

    --- %%IF IncImgDisplay=1 -----------------
    if (g_hWindow) LvRendererDisplayImage(m_hRenderer, hBuffer);
    --- %%ELSE -------------------------------
    // no image display demonstrated - switch it ON in the code generation Wizard if you need it
    --- %%ENDIF ------------------------------
    LvBufferQueue(hBuffer);
    --- %%IF SwTriggered=1 -------------------
    // finished, trigger another image acquisition
    if (IsAcquiring())
        LvCmdExecute(m_hDevice, LvDevice_TriggerSoftware);
    --- %%ENDIF ------------------------------
}

--- %%IF IncUniImgPreprocess=1 ------------------
//-----------------------------------------------------------------------------
// Determines optimal pixel format for the uni-process

void CCamera::SetOptimalUniPixelFormat()
{
    LvEnum PixelFormat;
    LvGetEnum(m_hDevice, LvDevice_PixelFormat, &PixelFormat);
    LvEnum UniPixelFormat = PixelFormat;
    switch (PixelFormat)
    {
        case LvPixelFormat_Mono8:
        case LvPixelFormat_Mono10:
        case LvPixelFormat_Mono12:
        case LvPixelFormat_Mono16:
            UniPixelFormat = LvPixelFormat_Mono8;
            break;

        case LvPixelFormat_BayerGR8:
        case LvPixelFormat_BayerRG8:
        case LvPixelFormat_BayerGB8:
        case LvPixelFormat_BayerBG8:
        case LvPixelFormat_BayerGR10:
        case LvPixelFormat_BayerRG10:
        case LvPixelFormat_BayerGB10:
        case LvPixelFormat_BayerBG10:
        case LvPixelFormat_BayerGR12:
        case LvPixelFormat_BayerRG12:
        case LvPixelFormat_BayerGB12:
        case LvPixelFormat_BayerBG12:
        case LvPixelFormat_RGB8Packed:
        case LvPixelFormat_RGBA8Packed:
            UniPixelFormat = LvPixelFormat_BGRA8Packed;
            break;
    }
    LvSetEnum(m_hDevice, LvDevice_LvUniPixelFormat, UniPixelFormat);
}

--- %%ENDIF ------------------------------
//-----------------------------------------------------------------------------

#ifdef USE_OPENCV_PROCESSING

    void CCamera::OpenCvProcessingOpen()
    {
        m_pSrcImg = NULL;
        m_pDstImg = NULL;
        m_DrawColorBlack = CV_RGB (0, 0, 0);
        m_DrawColorWhite = CV_RGB (0xFF, 0xFF, 0xFF);
        cvInitFont (&m_TextFont, CV_FONT_HERSHEY_SIMPLEX, 0.6, 0.6);
        m_SizeROI = 50; // initial size of ROI in percentage of window size
    }

//----------------------------------------------------------------------------
// Create an OpenCV image header for full sensor resolution

    void CCamera::OpenCvCreateImages()
    {
        m_pSrcImg = NULL;
        int32_t Val_Width;
        int32_t Val_Height;
        LvEnum  Val_PixelFormat;

        LvGetInt32(m_hDevice, LvDevice_Width , &Val_Width );
        LvGetInt32(m_hDevice, LvDevice_Height, &Val_Height);
        --- %%IF IncUniImgPreprocess=1 ------------------
        LvGetEnum(m_hDevice, LvDevice_LvUniPixelFormat, &Val_PixelFormat );
        --- %%ELSE --------------------------------------
        LvGetEnum(m_hDevice, LvDevice_PixelFormat, &Val_PixelFormat );
        --- %%ENDIF -------------------------------------

        try
        {
            CvSize ImgSize = cvSize (Val_Width, Val_Height);
            m_DrawColorWhite = cvScalar (0xFF, 0xFF, 0xFF, 0xFF);

            switch (Val_PixelFormat)
            {
                case LvPixelFormat_Mono8:
                case LvPixelFormat_BayerGR8:
                case LvPixelFormat_BayerRG8:
                case LvPixelFormat_BayerGB8:
                case LvPixelFormat_BayerBG8:
                    m_pSrcImg = cvCreateImageHeader (ImgSize, IPL_DEPTH_8U, 1);
                    m_pDstImg = cvCreateImage       (ImgSize, IPL_DEPTH_8U, 1);
                    break;

		        case LvPixelFormat_Mono8Signed: // unsupported on NET HW
			        m_pSrcImg = cvCreateImageHeader (ImgSize, IPL_DEPTH_8S, 1);
			        break;

                case LvPixelFormat_Mono10:
		        case LvPixelFormat_Mono12:
		        case LvPixelFormat_Mono14:
		        case LvPixelFormat_Mono16:
                case LvPixelFormat_BayerGR10:
		        case LvPixelFormat_BayerRG10:
		        case LvPixelFormat_BayerGB10:
		        case LvPixelFormat_BayerBG10:
                case LvPixelFormat_BayerGR12:
		        case LvPixelFormat_BayerRG12:
		        case LvPixelFormat_BayerGB12:
		        case LvPixelFormat_BayerBG12:
			        m_pSrcImg = cvCreateImageHeader (ImgSize, IPL_DEPTH_16U, 1);
                    m_pDstImg = cvCreateImage       (ImgSize, IPL_DEPTH_16U, 1);
                    m_DrawColorWhite = CV_RGB (0xFFFF, 0xFFFF, 0xFFFF);
			        break;

		        case LvPixelFormat_BGR8Packed:
		        case LvPixelFormat_RGB8Packed:
			        m_pSrcImg = cvCreateImageHeader (ImgSize, IPL_DEPTH_8U, 3);
                    m_pDstImg = cvCreateImage       (ImgSize, IPL_DEPTH_8U, 3);
			        break;

		        case LvPixelFormat_BGRA8Packed:
		        case LvPixelFormat_RGBA8Packed:
			        m_pSrcImg = cvCreateImageHeader (ImgSize, IPL_DEPTH_8U, 4);
                    m_pDstImg = cvCreateImage       (ImgSize, IPL_DEPTH_8U, 4);
			        break;

                default:
                {
                    char msg[256];
                    sprintf (msg, "Unsupported pixel format 0x%X. Cannot create the OpenCV Image Header", Val_PixelFormat);
                    DisplayError (msg);
                    break;
                }
            }
        }
        catch (cv::Exception e)
        {
            DisplayError (e.what());
        }
    }

//-----------------------------------------------------------------------------

    void CCamera::OpenCvProcessImage(void* pSrcData)
    {
        LvEnum Val_PixelFormat;
        --- %%IF IncUniImgPreprocess=1 ------------------
        LvGetEnum(m_hDevice, LvDevice_LvUniPixelFormat, &Val_PixelFormat);
        --- %%ELSE --------------------------------------
        LvGetEnum(m_hDevice, LvDevice_PixelFormat, &Val_PixelFormat);
        --- %%ENDIF -------------------------------------
        int iBitsPerPixel = (Val_PixelFormat & LV_PIX_EFFECTIVE_PIXEL_SIZE_MASK) >> LV_PIX_EFFECTIVE_PIXEL_SIZE_SHIFT;
        if (m_pSrcImg != NULL && m_pDstImg != NULL)
        {
            try
            {
                char Txt[128];

                // Adjust the address for this buffer
                m_pSrcImg->imageData= (char*) pSrcData;

                //define ROI for image processing
                CvRect rectTmp;
                rectTmp = cvRect (m_pSrcImg->width*(100-m_SizeROI)/2/100, m_pSrcImg->height*(100-m_SizeROI)/2/100,
                                  m_pSrcImg->width*m_SizeROI/100, m_pSrcImg->height*m_SizeROI/100);
                cvSetImageROI(m_pSrcImg, rectTmp);
                cvSetImageROI(m_pDstImg, rectTmp);

                // Call the cv processing function
                if (iBitsPerPixel == 8)
                {
                    cvCanny (m_pSrcImg, m_pDstImg, 50, 150, 3);
                    strcpy(Txt, "Image processed using the OpenCV cvCanny operator");
                }
                else
                {
	                cvSmooth(m_pSrcImg, m_pDstImg, CV_GAUSSIAN, 11);
                    strcpy(Txt, "Image processed using the OpenCV cvSmooth operator");
                }
                // Copy back the result to source image for display
                cvCopy (m_pDstImg, m_pSrcImg);

                // draw a frame around ROI
                cvRectangle (m_pSrcImg, cvPoint(2,2), cvPoint(m_pSrcImg->roi->width-3,
                             m_pSrcImg->roi->height-3), m_DrawColorWhite);

                cvResetImageROI (m_pSrcImg);
                cvResetImageROI (m_pDstImg);

                // Draw message to image.
                cvPutText(m_pSrcImg, Txt, cvPoint(49,49), &m_TextFont, m_DrawColorBlack);
                cvPutText(m_pSrcImg, Txt, cvPoint(49,51), &m_TextFont, m_DrawColorBlack);
                cvPutText(m_pSrcImg, Txt, cvPoint(51,49), &m_TextFont, m_DrawColorBlack);
                cvPutText(m_pSrcImg, Txt, cvPoint(51,51), &m_TextFont, m_DrawColorBlack);
                cvPutText(m_pSrcImg, Txt, cvPoint(50,50), &m_TextFont, m_DrawColorWhite);
            }
            catch (cv::Exception e)
            {
                // do nothing, we are in the callback, so no MsgBox possible
            }
        }
    }

//-----------------------------------------------------------------------------

    void CCamera::OpenCvProcessingClose()
    {
        try
        {
            if (m_pSrcImg != NULL)
            {
                cvReleaseImageHeader (&m_pSrcImg);
                m_pSrcImg = NULL;
            }
            if (m_pDstImg != NULL)
            {
                cvReleaseImage (&m_pDstImg);
                m_pDstImg = NULL;
            }
        }
        catch (cv::Exception e)
        {
            DisplayError (e.what());
        }
    }

#endif


//=============================================================================
//=== Xlib application skeleton ===============================================
//=============================================================================

const char* g_pszTitle = "SynView Sample Using OpenCV";
const char *g_apszMenu[] = {
    "------------------------------------",
    "Exit: X",
    "------------------------------------",
    "Open camera: O",
    "Close camera: C",
    "------------------------------------",
    "Start acquisition: Left mouse button",
    "Stop acquisition: Right mouse button",
    "------------------------------------",
    "Toggle image processing: P",
    "------------------------------------",
    0
    };

static void SetWindowTitle (const char *szTitle)
{
    if (!g_hWindow) return;

    XTextProperty win_title;
    XStringListToTextProperty ((char**)&szTitle, 1, &win_title);
    XSetWMName (g_pDisplay, g_hWindow, &win_title);
}

static int ErrorHandler (Display *display, XErrorEvent *event)
{
    if (event->error_code!=BadMatch || event->request_code!=X_PutImage)
    {
        //unknown error, display message and exit
        char error_text[512], message[1024];
        XGetErrorText (display, event->error_code, error_text, 512);
        char msg[256];
        sprintf (msg, "X error occured: %s\n\tMajor opcode: %d\n", error_text, (int) event->request_code);
        DisplayError (msg);
        exit (1);
    }

    // this is our well known bad match caused in XPutImage
    g_pCamera->StopAcquisition();
    DisplayError ("Error displaying the image, unexpected image format, freezing acquisition");
    UpdateStatus ();
    return 0;
}


int main ()
{
    if (LvOpenLibrary() != LVSTATUS_OK)
    {
        DisplayError ("Error: Opening the library failed");
        exit (1);
    }
    if (LvSystemOpen("", &g_hSystem) != LVSTATUS_OK)
    {
        DisplayError ("Error: Opening the system failed");
        LvCloseLibrary();
        exit (1);
    }
    // Create one global CCamera class instance
    g_pCamera = new CCamera();
    XInitThreads ();
    g_pDisplay =  XOpenDisplay (NULL);
    int iScreenWidth  = DisplayWidth(g_pDisplay, DefaultScreen(g_pDisplay)); 
    int iScreenHeight = DisplayHeight(g_pDisplay, DefaultScreen(g_pDisplay)); 
    int iAppWidth = (iScreenWidth * 9)/10; // 90% of width 
    int iAppHeight = (iScreenHeight * 9)/10; // 90% of height
    int iAppXOffset = (iScreenWidth - iAppWidth)/2;
    int iAppYOffset = (iScreenHeight - iAppHeight)/2;
    g_hWindow = XCreateSimpleWindow (g_pDisplay, DefaultRootWindow(g_pDisplay), iAppXOffset, iAppYOffset, iAppWidth, iAppHeight, 0, WhitePixel(g_pDisplay, DefaultScreen(g_pDisplay)), WhitePixel(g_pDisplay, DefaultScreen(g_pDisplay)));
    SetWindowTitle (g_pszTitle);
    XSelectInput (g_pDisplay, g_hWindow, KeyPressMask | ButtonPressMask | ExposureMask | StructureNotifyMask);
    Atom protocols[1];
    protocols[0] = XInternAtom (g_pDisplay, "WM_DELETE_WINDOW", True);
    XSetWMProtocols (g_pDisplay, g_hWindow, protocols, 1);
    XSetErrorHandler (ErrorHandler);
    XMapWindow (g_pDisplay, g_hWindow);
    GC gc = XCreateGC (g_pDisplay, g_hWindow, 0, 0);

    XEvent expose_event;
    expose_event.type = Expose;

    XEvent event;
    //message loop
    while (true)
    {
        XNextEvent (g_pDisplay, &event);
        switch (event.type)
        {
            case KeyPress:
            {
                ClearError ();

                KeySym key = XkbKeycodeToKeysym (g_pDisplay, event.xkey.keycode, 0, 0);
                switch (key)
                {
                    case XK_F1:
                    case XK_Alt_L:
                    case XK_Meta_L:
                    case XK_Alt_R:
                    case XK_Meta_R:
                    {
                        if (g_pCamera) g_pCamera->StopAcquisition();
                        XSendEvent (g_pDisplay, g_hWindow, 0, 0, &expose_event);
                    }
                    break;

                    case XK_x:
                    case XK_X:
                    {
                        if (g_pCamera) 
                        {
                            g_pCamera->StopAcquisition();
                            g_pCamera->CloseCamera();
                            delete g_pCamera;
                            g_pCamera = NULL;
                        }
                        XFreeGC (g_pDisplay, gc);
                        XDestroyWindow (g_pDisplay, g_hWindow);
                        g_hWindow = 0;
                    }
                    break;

                    case XK_o:
                    case XK_O:
                    {
                        g_pCamera->OpenCamera(g_pDisplay, g_hWindow, g_hSystem);
                    }
                    break;

                    case XK_c:
                    case XK_C:
                    {
                        g_pCamera->CloseCamera();
                        XClearWindow (g_pDisplay, g_hWindow);
                    }
                    break;

                    case XK_p:
                    case XK_P:
                    {
                        g_bProcessImage = !g_bProcessImage;
                        g_pCamera->SetProcessing(g_bProcessImage);
                    }
                    break;
                }

                UpdateStatus ();
            }
            break;
            
            case ButtonPress:
            {
                ClearError ();

                switch (event.xbutton.button)
                {
                    case Button1:
                    {
                        g_pCamera->StartAcquisition();
                    }
                    break;

                    case Button2:
                    case Button3:
                    {
                        g_pCamera->StopAcquisition();
                    }
                    break;
                }

                UpdateStatus ();
            }
            break;

            case Expose:
                if (g_hWindow)
                {
                    XClearWindow (g_pDisplay, g_hWindow);
                    {
                        int x = 20, y = 20, item = 0;
                        while (g_apszMenu[item])
                        {
                            XDrawString (g_pDisplay, g_hWindow, gc, x, y, g_apszMenu[item], strlen (g_apszMenu[item]));
                            y += 20;
                            item++;
                        }
                        y += 100;
                        XDrawString (g_pDisplay, g_hWindow, gc, x, y, g_ErrMsg, strlen (g_ErrMsg));
                    }
                    XFlush (g_pDisplay);
                }
            break;

            case ClientMessage:
                if (strcmp (XGetAtomName(g_pDisplay, event.xclient.data.l[0]), "WM_DELETE_WINDOW") == 0)
                {
                    XDestroyWindow (g_pDisplay, g_hWindow);
                }
            break;

            case UnmapNotify:
                if (g_pCamera)
                {
                    g_pCamera->StopAcquisition();
                    g_pCamera->CloseCamera();
                }
            break;

            case DestroyNotify:
                if (g_pCamera) delete g_pCamera;
                LvSystemClose(&g_hSystem);
                LvCloseLibrary();
                XCloseDisplay (g_pDisplay);
                exit (0);
            break;

        }
    }

    // Cleanup is performed in the Unmap/DestroyNotify handlers, see above
         
    return 0;
}

//-----------------------------------------------------------------------------
