
using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Runtime.InteropServices; // for Marshal
using NewElectronicTechnology.SynView;


namespace CsNetSample01
{
    public class CCamera
    {
        private const int NumberOfBuffers = %%Buffers%%;
        private LvSystem     m_pSystem;
        private LvInterface  m_pInterface;
        private LvDevice     m_pDevice;
        private LvStream     m_pStream;
        private LvRenderer   m_pRenderer;
        private LvBuffer[]   m_Buffers;
        private IntPtr       m_hDisplayWnd;
        private LvEvent      m_pEvent;
        --- %%IF IncSimpleImgProcess=1 ------------------
        private bool         m_bDoProcessing;
        --- %%ENDIF -------------------------------------

        //-----------------------------------------------------------------------------
        // CCamera constructor

        public CCamera()
        { 
            m_pSystem     = null;
            m_pInterface  = null;
            m_pDevice     = null;
            m_pStream     = null;
            m_pRenderer   = null;
            m_pEvent      = null;
            m_Buffers     = new LvBuffer[NumberOfBuffers]; 
            for (int i=0; i<NumberOfBuffers; i++)
                m_Buffers[i] = null;

            m_hDisplayWnd = (IntPtr) 0;
            --- %%IF IncSimpleImgProcess=1 ------------------
            m_bDoProcessing = false;
            --- %%ENDIF -------------------------------------
        }

        //-----------------------------------------------------------------------------

        public void OpenCamera(IntPtr hDisplayWnd, LvSystem pSystem)
        {
            try
            {
                if (m_pDevice != null) CloseCamera();

                m_pSystem = pSystem;
                m_hDisplayWnd = hDisplayWnd;

                LvInterface pInterface = null;
                LvDevice pDevice = null;
                pSystem.UpdateInterfaceList();

                --- %%IF OpenInterface=ByInterfaceId -------
                pSystem.OpenInterface("%%INTERFACE_ID%%", ref pInterface);
                --- %%ENDIF --------------------------------
                --- %%IF OpenInterface=ByInterfaceTlType ---
                String InterfaceId = "";
                pSystem.FindInterface(LvFindBy.TLType, "%%INTERFACE_TLTYPE%%", ref InterfaceId);
                pSystem.OpenInterface(InterfaceId, ref pInterface);
                --- %%ENDIF --------------------------------

                pInterface.UpdateDeviceList();
                --- %%IF OpenDevice=ByDeviceId -------------
                pInterface.OpenDevice("%%DEVICE_ID%%", ref pDevice, LvDeviceAccess.Control);
                --- %%ENDIF --------------------------------
                --- %%IF OpenDevice=ByDeviceModelName ------
                String DeviceId = "";
                pInterface.FindDevice(LvFindBy.ModelName, "%%DEVICE_MODELNAME%%", ref DeviceId);
                pInterface.OpenDevice(DeviceId, ref pDevice, LvDeviceAccess.Control);
                --- %%ENDIF --------------------------------
                --- %%IF OpenDevice=ByDeviceIpAddress ------
                String DeviceId = "";
                pInterface.FindDevice(LvFindBy.GevIPAddress, "%%DEVICE_IPADDRESS%%", ref DeviceId);
                pInterface.OpenDevice(DeviceId, ref pDevice, LvDeviceAccess.Control);
                --- %%ENDIF --------------------------------
                --- %%IF OpenDevice=ByDeviceUserId ---------
                String DeviceId = "";
                pInterface.FindDevice(LvFindBy.UserID, "%%DEVICE_USERID%%", ref DeviceId);
                pInterface.OpenDevice(DeviceId, ref pDevice, LvDeviceAccess.Control);
                --- %%ENDIF --------------------------------
                --- %%IF Sys.AutoLoadedCfg=1 -------------
                // Note: When the code was generated, in SynView Explorer was switched ON the option
                // to autoload the camera settings after opening the camera. For this reason the following
                // line loads the camera settings from the file to assure the same starting status.
                --- %%ENDIF ------------------------------
                --- %%IF Sys.SnippetFile=1 -----------------
                pDevice.LoadSettings("", "%%Sys.CameraCfgFile%%", (uint)LvSaveFlag.All);
                --- %%ENDIF --------------------------------

                --- %%IF Snippet=All ---------------------
                // The #error line below is intentionally inserted to the code in case you
                // generate the code from streamable or all writable features.
                #error Review the feature settings code and remove the unnecessary items!
                // Before removing this line from the code, go carefully through all the feature
                // settings below and leave there only those, which really need to be set.

                --- %%ENDIF ------------------------------
                --- %%IF Snippet=Streamable --------------
                // The #error line below is intentionally inserted to the code in case you
                // generate the code from streamable or all writable features.
                #error Review the feature settings code and remove the unnecessary items!
                // Before removing this line from the code, go carefully through all the feature
                // settings below and leave there only those, which really need to be set.

                --- %%ENDIF ------------------------------
                %%CameraRemoteFeaturesCode%%

                m_pInterface = pInterface;
                m_pDevice = pDevice;

                --- %%IF IncUniImgPreprocess=1 -------------
                m_pDevice.SetEnum(LvDeviceFtr.LvUniProcessMode, (int)LvUniProcessMode.Auto);
                SetOptimalUniPixelFormat();

                --- %%ENDIF --------------------------------
                m_pDevice.OpenStream("", ref m_pStream);
                m_pStream.OpenEvent(LvEventType.NewBuffer, ref m_pEvent);
                for (int i = 0; i < NumberOfBuffers; i++)
                    m_pStream.OpenBuffer((IntPtr)0, 0, (IntPtr)0, 0, ref m_Buffers[i]); 
                m_pStream.SetInt32(LvStreamFtr.LvPostponeQueueBuffers, 3);
                m_pStream.OpenRenderer(ref m_pRenderer);                      
                m_pRenderer.SetWindow(m_hDisplayWnd);                     

                m_pEvent.OnEventNewBuffer += new LvEventNewBufferHandler(NewBufferEventHandler); 
                m_pEvent.SetCallbackNewBuffer(true, (IntPtr)0);      
                m_pEvent.StartThread();                                   
            }
            catch (LvException ex)
            {
                MessageBox.Show(ex.Message, "Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
            }
        }



        //-----------------------------------------------------------------------------
        // Starts acquisition

        public void StartAcquisition()
        {
            try
            {
                m_pStream.FlushQueue(LvQueueOperation.AllToInput);
                if (m_pDevice == null) return;
                m_pDevice.AcquisitionStart();                             
            }
            catch (LvException ex)
            {
                MessageBox.Show(ex.Message, "Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
            }
        }

        //-----------------------------------------------------------------------------
        // Stops acquisition

        public void StopAcquisition()
        {
            try
            {
                if (m_pStream == null) return;
                m_pDevice.AcquisitionStop();          
            }
            catch (LvException ex)
            {
                MessageBox.Show(ex.Message, "Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
            }
        }

        //-----------------------------------------------------------------------------
        // Closes the cameras

        public void CloseCamera()
        {
            try
            {
                if (m_pDevice == null) return;
                if (IsAcquiring()) StopAcquisition();
                m_pEvent.StopThread();                
                m_pEvent.SetCallbackNewBuffer(false, (IntPtr)0);      
                m_pStream.CloseEvent(ref m_pEvent);       
                m_pStream.CloseRenderer(ref m_pRenderer);
                m_pStream.FlushQueue(LvQueueOperation.AllDiscard);
                for (int i = 0; i < NumberOfBuffers; i++)
                    if (m_Buffers[i] != null)
                        m_pStream.CloseBuffer(ref m_Buffers[i]);
                m_pDevice.CloseStream(ref m_pStream);     
                m_pInterface.CloseDevice(ref m_pDevice); 
                m_pSystem.CloseInterface(ref m_pInterface); 
            }
            catch (LvException ex)
            {
                MessageBox.Show(ex.Message, "Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
            }
        }

        //-----------------------------------------------------------------------------
        // Utility function for enabling/disabling buttons

        public bool IsOpen()
        {
            return m_pDevice != null;
        }

        //-----------------------------------------------------------------------------
        // Utility function for enabling/disabling buttons

        public bool IsAcquiring()
        {
            Boolean iIsAcquiring = false;
            try
            {
                if (m_pDevice == null) return false;
                m_pDevice.GetBool(LvDeviceFtr.LvDeviceIsAcquiring, ref iIsAcquiring);
            }
            catch (LvException)
            {
                // no message
            }
            return iIsAcquiring;
        }

        --- %%IF IncSimpleImgProcess=1 ------------------
        //-----------------------------------------------------------------------------
        // Switches ON/OFF image processing done in the callback

        public void SetProcessing(bool bDoProcessing)
        {
            m_bDoProcessing = bDoProcessing;
        }

        --- %%ENDIF -------------------------------------
        --- %%IF IncUniImgPreprocess=1 ------------------
        //-----------------------------------------------------------------------------
        // Determines optimal pixel format for the uni-process

        void SetOptimalUniPixelFormat()
        {
            UInt32 PixelFormat = (UInt32)LvPixelFormat.Mono8;
            m_pDevice.GetEnum(LvDeviceFtr.PixelFormat, ref PixelFormat);
            UInt32 UniPixelFormat = PixelFormat;
            switch ((LvPixelFormat)PixelFormat)
            {
                case LvPixelFormat.Mono8:
                case LvPixelFormat.Mono10:
                case LvPixelFormat.Mono12:
                case LvPixelFormat.Mono16:
                    UniPixelFormat = (UInt32)LvPixelFormat.Mono8;
                    break;

                case LvPixelFormat.BayerGR8:
                case LvPixelFormat.BayerRG8:
                case LvPixelFormat.BayerGB8:
                case LvPixelFormat.BayerBG8:
                case LvPixelFormat.BayerGR10:
                case LvPixelFormat.BayerRG10:
                case LvPixelFormat.BayerGB10:
                case LvPixelFormat.BayerBG10:
                case LvPixelFormat.BayerGR12:
                case LvPixelFormat.BayerRG12:
                case LvPixelFormat.BayerGB12:
                case LvPixelFormat.BayerBG12:
                case LvPixelFormat.RGB8Packed:
                case LvPixelFormat.RGBA8Packed:
                    UniPixelFormat = (UInt32)LvPixelFormat.BGRA8Packed;
                    break;
            }
            m_pDevice.SetEnum(LvDeviceFtr.LvUniPixelFormat, UniPixelFormat);
        }

        --- %%ENDIF -------------------------------------
        //-----------------------------------------------------------------------------

        public void Repaint()
        {
            try
            {
                if (m_pRenderer != null)
                    m_pRenderer.Repaint();
            }
            catch (LvException)
            {
                // no message
            }
        }

        //-----------------------------------------------------------------------------

        void NewBufferEventHandler(System.Object sender, LvNewBufferEventArgs e)
        {
            try
            {
                if (e.Buffer == null) return;

                --- %%IF IncSimpleImgProcess=1 ------------------
                IntPtr pData = (IntPtr) 0;
                Int64 iImageOffset = 0;
                --- %%IF IncUniImgPreprocess=1 ------------------
                e.Buffer.GetPtr(LvBufferFtr.UniBase, ref pData);
                e.Buffer.GetInt(LvBufferFtr.UniImageOffset, ref iImageOffset);
                --- %%ELSE --------------------------------------
                e.Buffer.GetPtr(LvBufferFtr.Base, ref pData);
                e.Buffer.GetInt(LvBufferFtr.ImageOffset, ref iImageOffset);
                --- %%ENDIF -------------------------------------
                pData = (IntPtr)(pData.ToInt64() + iImageOffset);
                if (m_bDoProcessing && pData != (IntPtr)0)
                {
                    // we will do some easy processing - invert the pixel values in an area
                    Int32 iWidth = 0;
                    Int32 iHeight = 0;
                    Int32 iLinePitch = 0;
                    UInt32 iPixelFormat = 0;    // LvPixelFormat enumeration value
                    m_pDevice.GetInt32(LvDeviceFtr.Width, ref iWidth);
                    m_pDevice.GetInt32(LvDeviceFtr.Height, ref iHeight);
                    --- %%IF IncUniImgPreprocess=1 ------------------
                    m_pDevice.GetEnum(LvDeviceFtr.LvUniPixelFormat, ref iPixelFormat);
                    m_pDevice.GetInt32(LvDeviceFtr.LvUniLinePitch, ref iLinePitch);
                    Int32 iBytesPerPixel = (Int32)((iPixelFormat & 0x00FF0000) >> 16) / 8;
                    --- %%ELSE --------------------------------------
                    m_pDevice.GetEnum(LvDeviceFtr.PixelFormat, ref iPixelFormat);
                    Int32 iBitsPerPixel = (Int32)((iPixelFormat & 0x00FF0000) >> 16);
                    iLinePitch = ((iWidth * iBitsPerPixel)+7)/8;
                    Int32 iBytesPerPixel = iBitsPerPixel / 8;
                    --- %%ENDIF -------------------------------------
                    for (Int32 j = 0; j < (iHeight / 2); j++)
                    {
                        Int32 BaseOffset = (iHeight / 4 + j) * iLinePitch + (iWidth / 4) * iBytesPerPixel;
                        for (Int32 i = 0; i < (iBytesPerPixel*iWidth / 2); i += iBytesPerPixel)
                        {
                            for (Int32 k = 0; k < iBytesPerPixel; k++)
                            {
                                // In C# we can use the Marshal class to access data pointed 
                                // by unmanaged pointers. However, this way of accessing unmanaged 
                                // data is ineffective (may slow down the acquisition).
                                Byte Pixel = Marshal.ReadByte(pData, BaseOffset + i + k);
                                Pixel = (Byte)~Pixel;
                                Marshal.WriteByte(pData, BaseOffset + i + k, Pixel);
                            }
                        }
                    }
                }
                --- %%ELSE --------------------------------------
                // no image processing demonstrated - switch it ON in the code generation Wizard if you need it
                --- %%ENDIF -------------------------------------

                m_pRenderer.DisplayImage(e.Buffer);
                e.Buffer.Queue();
            }
            catch (LvException)
            {
                // no message
            }
        }

        //-----------------------------------------------------------------------------


    }
}
