// This is a simple sample application, which creates a window with menu, enabling to
// open and close a camera, start and stop acquisition.

#define _CRT_SECURE_NO_DEPRECATE // for Visual Studio 8.0 and later

#include <windows.h>
#include <string>
#include <vector>
#include <sv.synview.class.h>

//=============================================================================
// The CCamera class implements access to one camera. In this simple sample
// it is not necessary; we could use global variables and functions instead.
// However, the CCamera class becomes practical when working with multiple 
// cameras (not demonstrated in this sample).

#define NUMBER_OF_BUFFERS %%Buffers%%

class CCamera
{
public:
    CCamera();
    ~CCamera();
    void OpenCamera(HWND hDisplayWnd, LvSystem* pSystem);
    void CloseCamera();
    void StartAcquisition();
    void StopAcquisition();
    --- %%IF IncSimpleImgProcess=1 ------------------
    void SetProcessing(bool bDoProcessing);
    --- %%ENDIF -------------------------------------
    bool IsOpen();
    bool IsAcquiring();
    void ProcessBuffersInOutputQueue();
    --- %%IF IncUniImgPreprocess=1 ------------------
    void SetOptimalUniPixelFormat();
    --- %%ENDIF -------------------------------------

private:
    LvSystem*    m_pSystem;
    LvInterface* m_pInterface;
    LvDevice*    m_pDevice;
    LvStream*    m_pStream;
    LvBuffer*    m_Buffers[NUMBER_OF_BUFFERS];
    --- %%IF IncImgDisplay=1 ------------------------
    LvRenderer*  m_pRenderer;
    --- %%ENDIF -------------------------------------
    LvEvent*     m_pEvent;
    --- %%IF IncSimpleImgProcess=1 ------------------
    bool         m_bDoProcessing;
    --- %%ENDIF -------------------------------------
    HWND         m_hDisplayWnd;
};

// global variables
LvSystem*    g_pSystem         = 0;
CCamera* g_pCamera     = NULL;
--- %%IF IncSimpleImgProcess=1 ------------------
bool         g_bProcessImage   = false;
--- %%ENDIF -------------------------------------
HWND         g_hMainWnd        = NULL;
char         g_szWindowText[1024];
UINT_PTR     g_iTimer          = 0;


//-----------------------------------------------------------------------------
// Error check with a message box displayed, if it is error

bool ErrorOccurred(LvStatus ErrorStatus)
{
    if (ErrorStatus == LVSTATUS_OK) return false;
    MessageBox(GetActiveWindow(), LvLibrary::GetLastErrorMessage().c_str(), "Error", MB_OK | MB_ICONEXCLAMATION);
    return true;
}

//-----------------------------------------------------------------------------
// In case we would need to set the window title bar text from other thread
// (which is not the case in this application), then using directly the Win32
// API SetWindowText() function can cause a deadlock, because it is internally
// converted to SendMessage(WM_SETTEXT) and SendMessage() requires the message
// loop in the window is not blocked. So safer is to pass the SetWindowText()
// to the main thread, using the WM_APP message.

void SetWindowTitle(HWND hWindow, const char* pszText)
{
    strcpy(g_szWindowText, pszText);
    PostMessage(hWindow, WM_APP, 0, 0);
}

//-----------------------------------------------------------------------------
// CCamera constructor

CCamera::CCamera()
{
    m_pSystem   = NULL;
    m_pInterface= NULL;
    m_pDevice   = NULL;
    m_pStream   = NULL;
    m_hDisplayWnd = NULL;
    --- %%IF IncImgDisplay=1 ------------------
    m_pRenderer = NULL;
    --- %%ENDIF -------------------------------
    m_pEvent    = NULL;
    --- %%IF IncSimpleImgProcess=1 ------------
    m_bDoProcessing = false;
    --- %%ENDIF -------------------------------
    memset(m_Buffers, 0, sizeof(m_Buffers));
}

//-----------------------------------------------------------------------------
// CCamera destructor

CCamera::~CCamera()
{
    if (m_pDevice != NULL) CloseCamera();
}

//-----------------------------------------------------------------------------

void CCamera::OpenCamera(HWND hDisplayWnd, LvSystem* pSystem)
{
    if (m_pDevice != NULL) CloseCamera();
    m_pSystem = pSystem;
    m_hDisplayWnd = hDisplayWnd;

    LvStatus SynViewStatus;
    LvInterface* pInterface = NULL;
    LvDevice* pDevice = NULL;
    pSystem->UpdateInterfaceList();
    --- %%IF OpenInterface=ByInterfaceId -------
    SynViewStatus = pSystem->OpenInterface("%%INTERFACE_ID%%", pInterface);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF --------------------------------
    --- %%IF OpenInterface=ByInterfaceTlType ---
    char InterfaceId[256];
    SynViewStatus = pSystem->FindInterface(LvFindBy_TLType, "%%INTERFACE_TLTYPE%%", InterfaceId, sizeof(InterfaceId));
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = pSystem->OpenInterface(InterfaceId, pInterface);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF --------------------------------

    pInterface->UpdateDeviceList();
    --- %%IF OpenDevice=ByDeviceId -------------
    SynViewStatus = pInterface->OpenDevice("%%DEVICE_ID%%", pDevice, LvDeviceAccess_Control);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF --------------------------------
    --- %%IF OpenDevice=ByDeviceModelName ------
    char DeviceId[256];
    SynViewStatus = pInterface->FindDevice(LvFindBy_ModelName, "%%DEVICE_MODELNAME%%", DeviceId, sizeof(DeviceId));
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = pInterface->OpenDevice(DeviceId, pDevice, LvDeviceAccess_Control);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF --------------------------------
    --- %%IF OpenDevice=ByDeviceIpAddress ------
    char DeviceId[256];
    SynViewStatus = pInterface->FindDevice(LvFindBy_GevIPAddress, "%%DEVICE_IPADDRESS%%", DeviceId, sizeof(DeviceId));
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = pInterface->OpenDevice(DeviceId, pDevice, LvDeviceAccess_Control);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF --------------------------------
    --- %%IF OpenDevice=ByDeviceUserId ---------
    char DeviceId[256];
    SynViewStatus = pInterface->FindDevice(LvFindBy_UserID, "%%DEVICE_USERID%%", DeviceId, sizeof(DeviceId));
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = pInterface->OpenDevice(DeviceId, pDevice, LvDeviceAccess_Control);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF --------------------------------
    --- %%IF Sys.AutoLoadedCfg=1 ---------------
    // Note: When the code was generated, in SynView Explorer was switched ON the option
    // to autoload the camera settings after opening the camera. For this reason the following
    // line loads the camera settings from the file to assure the same starting status.
    --- %%ENDIF --------------------------------
    --- %%IF Sys.SnippetFile=1 -----------------
    SynViewStatus = pDevice->LoadSettings("", "%%Sys.CameraCfgFile%%", LvSaveFlag_All);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF --------------------------------

    --- %%IF Snippet=All ---------------------
    // The #error line below is intentionally inserted to the code in case you
    // generate the code from streamable or all writable features.
    #error Review the feature settings code and remove the unnecessary items!
    // Before removing this line from the code, go carefully through all the feature
    // settings below and leave there only those, which really need to be set.

    --- %%ENDIF ------------------------------
    --- %%IF Snippet=Streamable --------------
    // The #error line below is intentionally inserted to the code in case you
    // generate the code from streamable or all writable features.
    #error Review the feature settings code and remove the unnecessary items!
    // Before removing this line from the code, go carefully through all the feature
    // settings below and leave there only those, which really need to be set.

    --- %%ENDIF ------------------------------
    %%CameraRemoteFeaturesCode%%

    m_pInterface = pInterface;
    m_pDevice = pDevice;

    --- %%IF IncChunkData=1 ------------------
    --- %%IF FEATURE ChunkModeActive ---
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkModeActive, 1);
    if (ErrorOccurred(SynViewStatus)) return;

    --- %%IF CHUNK_ENABLED OffsetX ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_OffsetX);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED OffsetY ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_OffsetY);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED Width ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_Width);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED Height ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_Height);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED PixelFormat ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_PixelFormat);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED LinePitch ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_LinePitch);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED FrameID ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_FrameID);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED Timestamp ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_Timestamp);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED ExposureTime ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_ExposureTime);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED Gain ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_Gain);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED LineStatusAll ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_LineStatusAll);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED LvExternalADCValue ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_LvExternalADCValue);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%ENDIF ------------------------------
    --- %%ENDIF ------------------------------
    --- %%IF IncUniImgPreprocess=1 -------------

    SynViewStatus = m_pDevice->SetEnum(LvDevice_LvUniProcessMode, LvUniProcessMode_Auto);
    if (ErrorOccurred(SynViewStatus)) return;
    SetOptimalUniPixelFormat();
    --- %%ENDIF --------------------------------
    SynViewStatus = m_pDevice->OpenStream("", m_pStream);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pStream->OpenEvent(LvEventType_NewBuffer, m_pEvent);
    if (ErrorOccurred(SynViewStatus)) return;
    for (int i=0; i<NUMBER_OF_BUFFERS; i++)
    {
        SynViewStatus = m_pStream->OpenBuffer(NULL, 0, NULL, 0, m_Buffers[i]);
        if (ErrorOccurred(SynViewStatus)) return;
        SynViewStatus = m_Buffers[i]->Queue();
        if (ErrorOccurred(SynViewStatus)) return;
    }
    --- %%IF IncImgDisplay=1 ------------------
    SynViewStatus = m_pStream->OpenRenderer(m_pRenderer);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pRenderer->SetWindow(m_hDisplayWnd);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF -------------------------------
}

//-----------------------------------------------------------------------------
// Starts acquisition

void CCamera::StartAcquisition()
{
    if (m_pDevice == NULL) return;
    LvStatus SynViewStatus;
    SynViewStatus = m_pDevice->AcquisitionStart();
    if (ErrorOccurred(SynViewStatus)) return;
}

//-----------------------------------------------------------------------------
// Stops acquisition

void CCamera::StopAcquisition()
{
    if (!IsAcquiring()) return;
    LvStatus SynViewStatus;
    SynViewStatus = m_pDevice->AcquisitionStop();
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pStream->FlushQueue(LvQueueOperation_AllToInput);
    if (ErrorOccurred(SynViewStatus)) return;
}

//-----------------------------------------------------------------------------
// Closes the cameras

void CCamera::CloseCamera()
{
    if (m_pDevice == NULL) return;
    if (IsAcquiring()) StopAcquisition();
    LvStatus SynViewStatus;
    SynViewStatus = m_pStream->CloseEvent(m_pEvent);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%IF IncImgDisplay=1 ------------------
    SynViewStatus = m_pStream->CloseRenderer(m_pRenderer);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF -------------------------------
    SynViewStatus = m_pStream->FlushQueue(LvQueueOperation_AllDiscard);
    if (ErrorOccurred(SynViewStatus)) return;
    for (int i=0; i<NUMBER_OF_BUFFERS; i++)
        if (m_Buffers[i] != NULL)
        {
            SynViewStatus = m_pStream->CloseBuffer(m_Buffers[i]);
            if (ErrorOccurred(SynViewStatus)) return;
        }
    SynViewStatus = m_pDevice->CloseStream(m_pStream);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pInterface->CloseDevice(m_pDevice);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pSystem->CloseInterface(m_pInterface);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%IF IncImgDisplay=1 ------------------
    InvalidateRect(m_hDisplayWnd, NULL, TRUE);
    --- %%ENDIF -------------------------------
}

//-----------------------------------------------------------------------------
// Utility function for enabling/disabling menu items

bool CCamera::IsOpen()
{
    return m_pDevice != NULL;
}

//-----------------------------------------------------------------------------
// Utility function for enabling/disabling menu items

bool CCamera::IsAcquiring()
{
    if (m_pDevice == NULL) return false;
    int32_t iIsAcquiring;
    m_pDevice->GetInt32(LvDevice_LvDeviceIsAcquiring, &iIsAcquiring);
    return iIsAcquiring != 0;
}

--- %%IF IncSimpleImgProcess=1 ------------------
//-----------------------------------------------------------------------------
// Switches ON/OFF image processing done in the callback

void CCamera::SetProcessing(bool bDoProcessing)
{
    m_bDoProcessing = bDoProcessing;
}

--- %%ENDIF -------------------------------------
//-----------------------------------------------------------------------------
// Callback function called for each delivered image

void CCamera::ProcessBuffersInOutputQueue()
{
    for (;;)
    {
        int32_t AwaitDelivery;
        if (m_pStream->GetInt32(LvStream_LvNumAwaitDelivery, &AwaitDelivery) != LVSTATUS_OK) return;
        if (AwaitDelivery == 0) return;
        LvBuffer* pBuffer;
        if (m_pEvent->WaitAndGetNewBuffer(pBuffer, 100) != LVSTATUS_OK) return;

        --- %%IF IncSimpleImgProcess=1 ------------------
        // get the pointer to the image data
        void* pData = NULL;
        int32_t iImageOffset = 0;
        --- %%IF IncUniImgPreprocess=1 ------------------
        pBuffer->GetPtr (LvBuffer_UniBase, &pData);
        pBuffer->GetInt32 (LvBuffer_UniImageOffset, &iImageOffset);
        --- %%ELSE --------------------------------------
        pBuffer->GetPtr (LvBuffer_Base, &pData);
        pBuffer->GetInt32 (LvBuffer_ImageOffset, &iImageOffset);
        --- %%ENDIF -------------------------------------
        pData = (uint8_t*)pData + iImageOffset;
        if (m_bDoProcessing && pData != NULL)
        {
            // we will do some easy processing - invert the pixel values in an area
            int32_t iWidth;
            int32_t iHeight;
            int32_t iLinePitch;
            LvEnum iPixelFormat;    // LvPixelFormat enumeration value
            m_pDevice->GetInt32(LvDevice_Width, &iWidth);
            m_pDevice->GetInt32(LvDevice_Height, &iHeight);
            --- %%IF IncUniImgPreprocess=1 ------------------
            m_pDevice->GetEnum(LvDevice_LvUniPixelFormat, &iPixelFormat);
            m_pDevice->GetInt32(LvDevice_LvUniLinePitch, &iLinePitch);
            int32_t iBytesPerPixel = ((iPixelFormat & LV_PIX_EFFECTIVE_PIXEL_SIZE_MASK) >> LV_PIX_EFFECTIVE_PIXEL_SIZE_SHIFT) / 8;
            --- %%ELSE --------------------------------------
            m_pDevice->GetEnum(LvDevice_PixelFormat, &iPixelFormat);
            int iBitsPerPixel = (iPixelFormat & LV_PIX_EFFECTIVE_PIXEL_SIZE_MASK) >> LV_PIX_EFFECTIVE_PIXEL_SIZE_SHIFT;
            iLinePitch = ((iWidth * iBitsPerPixel)+7)/8;
            int32_t iBytesPerPixel = iBitsPerPixel / 8;
            --- %%ENDIF -------------------------------------

            for (int32_t j=0; j<(iHeight/2); j++)
            {
                uint8_t* pPixel = ((uint8_t*)pData) + (iHeight/4 + j)*iLinePitch + (iWidth/4)*iBytesPerPixel;
                for (int32_t i=0; i<(iWidth/2); i++)
                {
                    for (int32_t k=0; k<iBytesPerPixel; k++)
                    {
                        *pPixel = ~*pPixel;
                        pPixel++;
                    }
                }
            }
        }
        --- %%ELSE --------------------------------------
        // no image processing demonstrated - switch it ON in the code generation Wizard if you need it
        --- %%ENDIF -------------------------------------

        --- %%IF IncChunkData=1 ------------------
        --- %%IF FEATURE ChunkModeActive ---
        // before the chunk data can be read for this buffer, they must be parsed.
        pBuffer->ParseChunkData();
        int64_t ChunkValue;
        char szChunkLine[256];
        szChunkLine[0] = 0;
        char szValue[128];
        --- %%IF CHUNK_ENABLED OffsetX ---
        m_pDevice->GetInt64 (LvDevice_ChunkOffsetX, &ChunkValue);
        sprintf(szValue, "X=%lld | ", ChunkValue);
        strcat(szChunkLine, szValue);
        --- %%ENDIF ------------------------------
        --- %%IF CHUNK_ENABLED OffsetY ---
        m_pDevice->GetInt64 (LvDevice_ChunkOffsetY, &ChunkValue);
        sprintf(szValue, "Y=%lld | ", ChunkValue);
        strcat(szChunkLine, szValue);
        --- %%ENDIF ------------------------------
        --- %%IF CHUNK_ENABLED Width ---
        m_pDevice->GetInt64 (LvDevice_ChunkWidth, &ChunkValue);
        sprintf(szValue, "W=%lld | ", ChunkValue);
        strcat(szChunkLine, szValue);
        --- %%ENDIF ------------------------------
        --- %%IF CHUNK_ENABLED Height ---
        m_pDevice->GetInt64 (LvDevice_ChunkHeight, &ChunkValue);
        sprintf(szValue, "H=%lld | ", ChunkValue);
        strcat(szChunkLine, szValue);
        --- %%ENDIF ------------------------------
        --- %%IF CHUNK_ENABLED PixelFormat ---
        m_pDevice->GetInt64 (LvDevice_ChunkPixelFormat, &ChunkValue);
        sprintf(szValue, "PF=0x%llx | ", ChunkValue);
        strcat(szChunkLine, szValue);
        --- %%ENDIF ------------------------------
        --- %%IF CHUNK_ENABLED LinePitch ---
        m_pDevice->GetInt64 (LvDevice_ChunkLinePitch, &ChunkValue);
        sprintf(szValue, "LPitch=%lld | ", ChunkValue);
        strcat(szChunkLine, szValue);
        --- %%ENDIF ------------------------------
        --- %%IF CHUNK_ENABLED FrameID ---
        m_pDevice->GetInt64 (LvDevice_ChunkFrameID, &ChunkValue);
        sprintf(szValue, "FrameID=%lld | ", ChunkValue);
        strcat(szChunkLine, szValue);
        --- %%ENDIF ------------------------------
        --- %%IF CHUNK_ENABLED Timestamp ---
        m_pDevice->GetInt64 (LvDevice_ChunkTimestamp, &ChunkValue);
        sprintf(szValue, "Time=%lld | ", ChunkValue);
        strcat(szChunkLine, szValue);
        --- %%ENDIF ------------------------------
        --- %%IF CHUNK_ENABLED ExposureTime ---
        m_pDevice->GetInt64 (LvDevice_ChunkExposureTime, &ChunkValue);
        sprintf(szValue, "Exp=%lld | ", ChunkValue);
        strcat(szChunkLine, szValue);
        --- %%ENDIF ------------------------------
        --- %%IF CHUNK_ENABLED Gain ---
        // the IsAvailableEnumEntry() is time consuming, better if you move it outside this callback
        if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkGainSelector, LvChunkGainSelector_AnalogAll) != 0)
        {
            m_pDevice->SetEnum (LvDevice_ChunkGainSelector, LvChunkGainSelector_AnalogAll);
            m_pDevice->GetInt64 (LvDevice_ChunkGain, &ChunkValue);
            sprintf(szValue, "GainA=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
        }
        if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkGainSelector, LvChunkGainSelector_DigitalAll) != 0)
        {
            m_pDevice->SetEnum (LvDevice_ChunkGainSelector, LvChunkGainSelector_DigitalAll);
            m_pDevice->GetInt64 (LvDevice_ChunkGain, &ChunkValue);
            sprintf(szValue, "GainD=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
        }
        --- %%ENDIF ------------------------------
        --- %%IF CHUNK_ENABLED LineStatusAll ---
        m_pDevice->GetInt64 (LvDevice_ChunkLineStatusAll, &ChunkValue);
        sprintf(szValue, "Line=0x%llx | ", ChunkValue);
        strcat(szChunkLine, szValue);
        --- %%ENDIF ------------------------------
        --- %%IF CHUNK_ENABLED LvExternalADCValue ---
        // the IsAvailableEnumEntry() is time consuming, better if you move it outside this callback
        if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC1) != 0)
        {
            m_pDevice->SetEnum (LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC1);
            m_pDevice->GetInt64 (LvDevice_ChunkLvExternalADCValue, &ChunkValue);
            sprintf(szValue, "ADC1=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
        }
        if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC2) != 0)
        {
            m_pDevice->SetEnum (LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC2);
            m_pDevice->GetInt64 (LvDevice_ChunkLvExternalADCValue, &ChunkValue);
            sprintf(szValue, "ADC2=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
        }
        if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC3) != 0)
        {
            m_pDevice->SetEnum (LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC3);
            m_pDevice->GetInt64 (LvDevice_ChunkLvExternalADCValue, &ChunkValue);
            sprintf(szValue, "ADC3=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
        }
        if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC4) != 0)
        {
            m_pDevice->SetEnum (LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC4);
            m_pDevice->GetInt64 (LvDevice_ChunkLvExternalADCValue, &ChunkValue);
            sprintf(szValue, "ADC4=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
        }
        --- %%ENDIF ------------------------------
        SetWindowTitle(m_hDisplayWnd, szChunkLine);
        --- %%ENDIF ------------------------------
        --- %%ENDIF ------------------------------

        --- %%IF IncImgDisplay=1 ------------------------
        m_pRenderer->DisplayImage(pBuffer);
        --- %%ELSE --------------------------------------
        // no image display demonstrated - switch it ON in the code generation Wizard if you need it
        --- %%ENDIF -------------------------------------
        pBuffer->Queue();
    }
}

--- %%IF IncUniImgPreprocess=1 ------------------
//-----------------------------------------------------------------------------
// Determines optimal pixel format for the uni-process

void CCamera::SetOptimalUniPixelFormat()
{
    LvEnum PixelFormat;
    m_pDevice->GetEnum(LvDevice_PixelFormat, &PixelFormat);
    LvEnum UniPixelFormat = PixelFormat;
    switch (PixelFormat)
    {
        case LvPixelFormat_Mono8:
        case LvPixelFormat_Mono10:
        case LvPixelFormat_Mono12:
        case LvPixelFormat_Mono16:
            UniPixelFormat = LvPixelFormat_Mono8;
            break;

        case LvPixelFormat_BayerGR8:
        case LvPixelFormat_BayerRG8:
        case LvPixelFormat_BayerGB8:
        case LvPixelFormat_BayerBG8:
        case LvPixelFormat_BayerGR10:
        case LvPixelFormat_BayerRG10:
        case LvPixelFormat_BayerGB10:
        case LvPixelFormat_BayerBG10:
        case LvPixelFormat_BayerGR12:
        case LvPixelFormat_BayerRG12:
        case LvPixelFormat_BayerGB12:
        case LvPixelFormat_BayerBG12:
        case LvPixelFormat_RGB8Packed:
        case LvPixelFormat_RGBA8Packed:
            UniPixelFormat = LvPixelFormat_BGRA8Packed;
            break;
    }
    m_pDevice->SetEnum(LvDevice_LvUniPixelFormat, UniPixelFormat);
}

--- %%ENDIF -------------------------------------
//=============================================================================
//=== Windows application skeleton ============================================
//=============================================================================

HINSTANCE    g_hInstance;              
TCHAR*       g_pszTitle        = "SynView Sample Skeleton: Sample 01";
TCHAR*       g_pszMainWndClass = "SynViewSampleMainWnd";
HMENU        g_hMenu           = NULL;    // Main window menu

// Menu definitions         
#define IDM_OPEN_CAMERA           101
#define IDM_CLOSE_CAMERA          102
#define IDM_ACQUISITION_START     103
#define IDM_ACQUISITION_STOP      104
--- %%IF IncSimpleImgProcess=1 ------------------
#define IDM_PROCESS_IMAGE         105
--- %%ENDIF -------------------------------------
#define IDM_EXIT                  199

#define CHECK_MENU(id,check)   CheckMenuItem (g_hMenu, id, ((check) ? MF_CHECKED : MF_UNCHECKED) | MF_BYCOMMAND)
#define ENABLE_MENU(id,enable) EnableMenuItem(g_hMenu, id, ((enable) ?  MF_ENABLED : MF_GRAYED) | MF_BYCOMMAND)

//-----------------------------------------------------------------------------

void UpdateControls()
{
    ENABLE_MENU(IDM_OPEN_CAMERA      ,  !g_pCamera->IsOpen());
    ENABLE_MENU(IDM_CLOSE_CAMERA     ,  g_pCamera->IsOpen() && !g_pCamera->IsAcquiring());
    ENABLE_MENU(IDM_ACQUISITION_START,  g_pCamera->IsOpen() && !g_pCamera->IsAcquiring());
    ENABLE_MENU(IDM_ACQUISITION_STOP ,  g_pCamera->IsOpen() && g_pCamera->IsAcquiring());
    --- %%IF IncSimpleImgProcess=1 ------------------
    CHECK_MENU(IDM_PROCESS_IMAGE     ,  g_bProcessImage);
    --- %%ENDIF -------------------------------------
    char szTitle[256];
    sprintf(szTitle, "Camera %s", g_pCamera->IsOpen() ? "open" : "not open");
    if (g_pCamera->IsOpen())
    {
        if (g_pCamera->IsAcquiring())
            strcat(szTitle, ", acquiring.");
        else
            strcat(szTitle, ", not acquiring.");
    }
    SetWindowTitle(g_hMainWnd, szTitle);
}

//-----------------------------------------------------------------------------

long CALLBACK MainWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    switch (message) 
    {
        case WM_APP:
        {
            SetWindowText(hWnd, g_szWindowText);
            break;
        }
        
        case WM_COMMAND:
        {
            WORD Id = LOWORD(wParam);
            switch (Id) 
            {
                case IDM_OPEN_CAMERA:  
                    g_pCamera->OpenCamera(g_hMainWnd, g_pSystem);
                    UpdateControls();
                    break;
                case IDM_ACQUISITION_START:
                    g_pCamera->StartAcquisition();
                    if (g_iTimer == 0)
                        g_iTimer = SetTimer(hWnd, 100, 50, NULL); // 50 ms
                    UpdateControls();
                    break;
                case IDM_ACQUISITION_STOP: 
                    g_pCamera->StopAcquisition();
                    if (g_iTimer != 0)
                    {
                        KillTimer(hWnd, g_iTimer);
                        g_iTimer = 0;
                    }
                    g_pCamera->ProcessBuffersInOutputQueue();
                    UpdateControls();
                    break;
                --- %%IF IncSimpleImgProcess=1 ------------------
                case IDM_PROCESS_IMAGE:
                    g_bProcessImage = !g_bProcessImage;
                    g_pCamera->SetProcessing(g_bProcessImage);
                    UpdateControls();
                    break;
                --- %%ENDIF -------------------------------------
                case IDM_CLOSE_CAMERA:
                    g_pCamera->CloseCamera();
                    UpdateControls();
                    break;
                case IDM_EXIT:  
                    DestroyWindow(hWnd);                     
                    break;
                default:
                    return (long) DefWindowProc(hWnd, message, wParam, lParam);
            }
            break;
        }
    
        case WM_DESTROY:
        {
            if (g_iTimer != 0)
            {
                KillTimer(hWnd, g_iTimer);
                g_iTimer = 0;
            }
            g_pCamera->CloseCamera();
            PostQuitMessage(0);
            break;
        }
    
        case WM_TIMER:
        {
            g_pCamera->ProcessBuffersInOutputQueue();
            break;
        }
        
        default:
            return (long) DefWindowProc(hWnd, message, wParam, lParam);
    }
    return 0;
}

//-----------------------------------------------------------------------------

BOOL InitApplication(HINSTANCE hInstance)
{
  	WNDCLASSEX wcex;
  	wcex.cbSize = sizeof(WNDCLASSEX);
  
  	wcex.style			  = CS_HREDRAW | CS_VREDRAW;
  	wcex.lpfnWndProc	  = (WNDPROC)MainWndProc;
  	wcex.cbClsExtra		  = 0;
  	wcex.cbWndExtra		  = 0;
  	wcex.hInstance		  = hInstance;
  	wcex.hIcon			  = LoadIcon(NULL, IDI_APPLICATION);
  	wcex.hCursor		  = LoadCursor(NULL, IDC_ARROW);
  	wcex.hbrBackground	  = (HBRUSH)(COLOR_WINDOW+1);
  	wcex.lpszMenuName     = NULL;
  	wcex.lpszClassName	  = g_pszMainWndClass;
    wcex.hIconSm		  = LoadIcon(NULL, IDI_APPLICATION);
  	RegisterClassEx(&wcex);
  
    return TRUE;
}

//-----------------------------------------------------------------------------

HMENU BuildMenu(void)
{
    HMENU hMenu=CreateMenu();
    HMENU hSubMenu=CreatePopupMenu();
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_EXIT, "Exit");
    AppendMenu(hMenu, MF_POPUP | MF_ENABLED, (UINT_PTR)hSubMenu, "File");
    hSubMenu=CreatePopupMenu();
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_OPEN_CAMERA,       "Open camera");  
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_CLOSE_CAMERA,      "Close camera");  
    AppendMenu(hSubMenu, MF_SEPARATOR, -1, "");
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_ACQUISITION_START, "Acquisition Start");  
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_ACQUISITION_STOP,  "Acquisition Stop");  
    --- %%IF IncSimpleImgProcess=1 ------------------
    AppendMenu(hSubMenu, MF_SEPARATOR, -1, "");
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_PROCESS_IMAGE,     "Process image");  
    --- %%ENDIF -------------------------------------
    AppendMenu(hMenu, MF_POPUP | MF_ENABLED, (UINT_PTR)hSubMenu, "Camera");
    return hMenu;
}

//-----------------------------------------------------------------------------

BOOL InitInstance(HINSTANCE hInstance, int nCmdShow)
  {
    // Create the main window menu
    g_hMenu=BuildMenu();
  
    int iScreenWidth  = GetSystemMetrics(SM_CXSCREEN); 
    int iScreenHeight = GetSystemMetrics(SM_CYSCREEN); 
    int iAppWidth = (iScreenWidth * 9)/10; // 90% of width 
    int iAppHeight = (iScreenHeight * 9)/10; // 90% of height
    int iAppXOffset = (iScreenWidth - iAppWidth)/2;
    int iAppYOffset = (iScreenHeight - iAppHeight)/2;
  
    // Create the main window
    g_hInstance = hInstance;
    g_hMainWnd = CreateWindow(
         g_pszMainWndClass, 
         g_pszTitle, 
         WS_OVERLAPPEDWINDOW,
         iAppXOffset, 
         iAppYOffset, 
         iAppWidth, 
         iAppHeight, 
         NULL, 
         g_hMenu, 
         hInstance, NULL);
    if (g_hMainWnd == NULL) return FALSE;

    ShowWindow(g_hMainWnd, nCmdShow);
    UpdateWindow(g_hMainWnd);
    UpdateControls();
    return TRUE;
  }

//-----------------------------------------------------------------------------

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE /*hPrevInstance*/,
                   LPSTR /*lpCmdLine*/, int nCmdShow)
{
    memset(g_szWindowText, 0, sizeof(g_szWindowText));

    if (LvLibrary::OpenLibrary() != LVSTATUS_OK)
    {
        MessageBox(g_hMainWnd, "Opening the library failed.", "Error", MB_OK | MB_ICONEXCLAMATION);
        return 0;
    }
    if (LvSystem::Open("", g_pSystem) != LVSTATUS_OK) 
    {
        MessageBox(g_hMainWnd, "Opening the system failed.", "Error", MB_OK | MB_ICONEXCLAMATION);
        LvLibrary::CloseLibrary();
        return 0;
    }
    // Create one global CCamera class instance
    g_pCamera = new CCamera();
  
    MSG msg;
    if (!InitApplication(hInstance)) 
    {
        MessageBox(g_hMainWnd, "Application init failed.", "Error", MB_OK | MB_ICONEXCLAMATION);
        delete g_pCamera;
        LvSystem::Close(g_pSystem);
        LvLibrary::CloseLibrary();
        return 0;
    }
  
    if (!InitInstance(hInstance, nCmdShow)) 
    {
        MessageBox(g_hMainWnd, "Instance init failed.", "Error", MB_OK | MB_ICONEXCLAMATION);
        delete g_pCamera;
        LvSystem::Close(g_pSystem);
        LvLibrary::CloseLibrary();
        return 0;
    }
  
    while (GetMessage(&msg, NULL, NULL, NULL)) 
    {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }
  
    delete g_pCamera;
    LvSystem::Close(g_pSystem);
    LvLibrary::CloseLibrary();
    return (int) msg.wParam;
}

//-----------------------------------------------------------------------------
