// This sample belongs to New Electronic Technology SynView library.

#define _CRT_SECURE_NO_DEPRECATE // for Visual Studio 8.0 and later

#include <windows.h>
#include <string>
#include <vector>

#include <sv.synview.class.h>
#include "CppSampleWin02_Camera.h"

#define MAX_APPS 2

typedef std::vector<std::string> CStringVector;

typedef struct
{
    HWND       hWnd;
    HMENU      hMenu;
    CCamera*   pCamera;
    char       szWindowText[1024];
} TApp;


HINSTANCE    g_hInstance;
TCHAR*       g_pszMainWndClass = "SynViewDemoMainWnd";
TApp         g_App[MAX_APPS];
LvSystem*    g_pSystem         = 0;
int          g_iApps           = MAX_APPS; // you can reduce the number here

// Menu definitions
#define IDM_OPEN_CAMERA               100
#define IDM_ACQUISITION_START         101
#define IDM_ACQUISITION_STOP          102
#define IDM_CLOSE_CAMERA              103
--- %%IF IncRenderModes=1 ----------------
#define IDM_DISPLAY_FULL              120
#define IDM_DISPLAY_SCALED            121
#define IDM_DISPLAY_TILES             122
--- %%ENDIF ------------------------------
--- %%IF IncSimpleImgProcess=1 -----------
#define IDM_PROCESS_IMAGE             130
--- %%ENDIF ------------------------------

#define IDM_OPEN_CAMERA_ALL           200
#define IDM_ACQUISITION_START_ALL     201
#define IDM_ACQUISITION_STOP_ALL      202
#define IDM_CLOSE_CAMERA_ALL          203

#define IDM_EXIT                      300

#define CHECK_MENU(iAppIndex,id,check)   CheckMenuItem (g_App[iAppIndex].hMenu, id, ((check) ? MF_CHECKED : MF_UNCHECKED) | MF_BYCOMMAND)
#define ENABLE_MENU(iAppIndex,id,enable) EnableMenuItem(g_App[iAppIndex].hMenu, id, ((enable) ?  MF_ENABLED : MF_GRAYED) | MF_BYCOMMAND)

//-----------------------------------------------------------------------------

void ErrorMsg(const char* pszMsg)
{
    MessageBox(NULL, pszMsg, "Error", MB_OK | MB_ICONEXCLAMATION);
}

//-----------------------------------------------------------------------------

void UpdateControls(int iApp)
{
    ENABLE_MENU(iApp, IDM_ACQUISITION_START, g_App[iApp].pCamera->IsOpen() && !g_App[iApp].pCamera->IsAcquiring());
    ENABLE_MENU(iApp, IDM_ACQUISITION_STOP , g_App[iApp].pCamera->IsAcquiring());
    ENABLE_MENU(iApp, IDM_OPEN_CAMERA      , !g_App[iApp].pCamera->IsOpen());
    ENABLE_MENU(iApp, IDM_CLOSE_CAMERA     , g_App[iApp].pCamera->IsOpen() && !g_App[iApp].pCamera->IsAcquiring());
    --- %%IF IncRenderModes=1 ----------------
    CHECK_MENU (iApp, IDM_DISPLAY_FULL     , g_App[iApp].pCamera->GetRenderType() == LvRenderType_FullSize);
    CHECK_MENU (iApp, IDM_DISPLAY_SCALED   , g_App[iApp].pCamera->GetRenderType() == LvRenderType_ScaleToFit);
    CHECK_MENU (iApp, IDM_DISPLAY_TILES    , g_App[iApp].pCamera->GetRenderType() == LvRenderType_ScaleToTiles);
    --- %%ENDIF ------------------------------
    --- %%IF IncSimpleImgProcess=1 -----------
    CHECK_MENU (iApp, IDM_PROCESS_IMAGE    , g_App[iApp].pCamera->GetProcessing());
    --- %%ENDIF ------------------------------

    bool bAllDisconnected = true;
    bool bAllConnected    = true;
    bool bAllStopped      = true;
    bool bAllStarted      = true;
    for (int i=0; i<g_iApps; i++)
    {
        if (g_App[i].pCamera->IsOpen())
            bAllDisconnected = false;
        else
            bAllConnected = false;
        if (g_App[i].pCamera->IsAcquiring())
            bAllStopped = false;
        else
            bAllStarted = false;
    }
    ENABLE_MENU(0, IDM_ACQUISITION_START_ALL, bAllConnected && bAllStopped);
    ENABLE_MENU(0, IDM_ACQUISITION_STOP_ALL , bAllStarted);
    ENABLE_MENU(0, IDM_OPEN_CAMERA_ALL      , bAllDisconnected);
    ENABLE_MENU(0, IDM_CLOSE_CAMERA_ALL     , bAllStopped && bAllConnected);
}

//-----------------------------------------------------------------------------
// We will need to set the window title bar text from other thread. Using directly
// the Win32 API SetWindowText() function can cause a deadlock - it is internally
// converted to SendMessage(WM_SETTEXT) and SendMessage() requires the message
// loop in the window is not blocked. Safer is to pass the SetWindowText()
// to the main thread, using the WM_APP message.

void SetWindowTitle(int iCamera, const char* pszText)
{
    strcpy(g_App[iCamera].szWindowText, pszText);
    PostMessage(g_App[iCamera].hWnd, WM_APP, 0, 0);
}

//-----------------------------------------------------------------------------
#define LINE_HEIGHT   16

long CALLBACK MainWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    int iApp;
    for (iApp=0; iApp<g_iApps; iApp++)
        if (g_App[iApp].hWnd == hWnd)
            break;

    --- %%IF IncRenderModes=1 -----------------------
    SCROLLINFO si;
    --- %%ENDIF -------------------------------------

    switch (message)
    {
        case WM_APP:
        {
            SetWindowText(hWnd, g_App[iApp].szWindowText);
            break;
        }
        case WM_COMMAND:
        {
            WORD Id = LOWORD(wParam);
            switch (Id)
            {
                case IDM_OPEN_CAMERA:
                    g_App[iApp].pCamera->OpenCamera(iApp, g_App[iApp].hWnd, g_pSystem);
                    InvalidateRect(g_App[iApp].hWnd, NULL, TRUE);
                    UpdateControls(iApp);
                    break;
                case IDM_CLOSE_CAMERA:
                    g_App[iApp].pCamera->CloseCamera();
                    InvalidateRect(g_App[iApp].hWnd, NULL, TRUE);
                    UpdateControls(iApp);
                    break;
                case IDM_ACQUISITION_START:
                    InvalidateRect(g_App[iApp].hWnd, NULL, TRUE);
                    g_App[iApp].pCamera->StartAcquisition();
                    UpdateControls(iApp);
                    break;
                case IDM_ACQUISITION_STOP:
                    g_App[iApp].pCamera->StopAcquisition();
                    UpdateControls(iApp);
                    break;
                --- %%IF IncSimpleImgProcess=1 -----------
                case IDM_PROCESS_IMAGE:
                    g_App[iApp].pCamera->SetProcessing(!g_App[iApp].pCamera->GetProcessing());
                    UpdateControls(iApp);
                    break;
                --- %%ENDIF ------------------------------
                case IDM_OPEN_CAMERA_ALL:
                    for (int i=0; i<g_iApps; i++)
                    {
                        g_App[i].pCamera->OpenCamera(i, g_App[i].hWnd, g_pSystem);
                        InvalidateRect(g_App[i].hWnd, NULL, TRUE);
                        UpdateControls(i);
                    }
                    break;
                case IDM_CLOSE_CAMERA_ALL:
                    for (int i=0; i<g_iApps; i++)
                    {
                        g_App[i].pCamera->CloseCamera();
                        InvalidateRect(g_App[i].hWnd, NULL, TRUE);
                        UpdateControls(i);
                    }
                    break;
                case IDM_ACQUISITION_START_ALL:
                    for (int i=0; i<g_iApps; i++)
                    {
                        InvalidateRect(g_App[i].hWnd, NULL, TRUE);
                        g_App[i].pCamera->StartAcquisition();
                        UpdateControls(i);
                    }
                    break;
                case IDM_ACQUISITION_STOP_ALL:
                    for (int i=0; i<g_iApps; i++)
                    {
                        g_App[i].pCamera->StopAcquisition();
                        UpdateControls(i);
                    }
                    break;
                --- %%IF IncRenderModes=1 ----------------
                case IDM_DISPLAY_FULL:
                    InvalidateRect(g_App[iApp].hWnd, NULL, TRUE); // clear the window contents
                    g_App[iApp].pCamera->SetRenderType(LvRenderType_FullSize);
                    UpdateControls(iApp);
                    break;
                case IDM_DISPLAY_SCALED:
                    InvalidateRect(g_App[iApp].hWnd, NULL, TRUE);
                    g_App[iApp].pCamera->SetRenderType(LvRenderType_ScaleToFit);
                    UpdateControls(iApp);
                    break;
                case IDM_DISPLAY_TILES:
                    InvalidateRect(g_App[iApp].hWnd, NULL, TRUE);
                    g_App[iApp].pCamera->SetRenderType(LvRenderType_ScaleToTiles);
                    UpdateControls(iApp);
                    break;
                --- %%ENDIF ------------------------------
                case IDM_EXIT:
                    DestroyWindow(hWnd);
                    break;
                default:
                    return (long) DefWindowProc(hWnd, message, wParam, lParam);
            }
            break;
        }

        case WM_PAINT:
        {
            PAINTSTRUCT PaintStruct;
            BeginPaint(hWnd, &PaintStruct);
            g_App[iApp].pCamera->Repaint();
            EndPaint(hWnd, &PaintStruct);
            break;
        }

        --- %%IF IncRenderModes=1 ----------------
        case WM_EXITSIZEMOVE:
        {
            g_App[iApp].pCamera->SetScrollbars();
            g_App[iApp].pCamera->Repaint();
            break;
        }

        case WM_VSCROLL:
        {
            // Get all the vertial scroll bar information
            si.cbSize = sizeof (si);
            si.fMask  = SIF_ALL;
            GetScrollInfo (hWnd, SB_VERT, &si);
            // Save the position for comparison later on
            int yPos = si.nPos;

            switch (LOWORD (wParam))
            {
                case SB_TOP: // user clicked the HOME keyboard key
                    si.nPos = si.nMin;
                    break;

                case SB_BOTTOM: // user clicked the END keyboard key
                    si.nPos = si.nMax;
                    break;

                case SB_LINEUP: // user clicked the top arrow
                    si.nPos -= LINE_HEIGHT;
                    break;

                case SB_LINEDOWN: // user clicked the bottom arrow
                    si.nPos += LINE_HEIGHT;
                    break;

                case SB_PAGEUP: // user clicked the shaft above the scroll box
                    si.nPos -= si.nPage;
                    break;

                case SB_PAGEDOWN: // user clicked the shaft below the scroll box
                    si.nPos += si.nPage;
                    break;

                case SB_THUMBTRACK: // user dragged the scroll box
                    si.nPos = si.nTrackPos;
                    break;

                default:
                    break;
            }

            // Set the position and then retrieve it.  Due to adjustments
            //   by Windows it may not be the same as the value set.
            si.fMask = SIF_POS;
            SetScrollInfo (hWnd, SB_VERT, &si, TRUE);
            GetScrollInfo (hWnd, SB_VERT, &si);

            // If the position has changed, scroll window and update it
            if (si.nPos != yPos)
            {
                ScrollWindow(hWnd, 0, yPos - si.nPos, NULL, NULL);
                UpdateWindow (hWnd);
            }
            return 0;
        }

        case WM_HSCROLL:
        {
            // Get all the vertial scroll bar information
            si.cbSize = sizeof (si);
            si.fMask  = SIF_ALL;
            // Save the position for comparison later on
            GetScrollInfo (hWnd, SB_HORZ, &si);
            int xPos = si.nPos;

            switch (LOWORD (wParam))
            {
                case SB_LINELEFT: // user clicked left arrow
                    si.nPos -= LINE_HEIGHT;
                    break;

                case SB_LINERIGHT: // user clicked right arrow
                    si.nPos += LINE_HEIGHT;
                    break;

                case SB_PAGELEFT: // user clicked shaft left of the scroll box
                    si.nPos -= si.nPage;
                    break;

                case SB_PAGERIGHT: // user clicked shaft right of the scroll box
                    si.nPos += si.nPage;
                    break;

                case SB_THUMBTRACK: // user dragged the scroll box
                    si.nPos = si.nTrackPos;
                    break;
                default:
                    break;
            }

            // Set the position and then retrieve it.  Due to adjustments
            // by Windows it may not be the same as the value set.
            si.fMask = SIF_POS;
            SetScrollInfo (hWnd, SB_HORZ, &si, TRUE);
            GetScrollInfo (hWnd, SB_HORZ, &si);

            // If the position has changed, scroll the window
            if (si.nPos != xPos)
            {
                ScrollWindow(hWnd, xPos - si.nPos, 0, NULL, NULL);
                UpdateWindow (hWnd);
            }
            return 0;
        }

        --- %%ENDIF ------------------------------
        case WM_DESTROY:
            g_App[iApp].pCamera->CloseCamera();
            PostQuitMessage(0);
            break;

        default:
            return (long) DefWindowProc(hWnd, message, wParam, lParam);
    }
    return 0;
}

//-----------------------------------------------------------------------------

BOOL InitApplication(HINSTANCE hInstance)
{
    WNDCLASSEX wcex;
  	wcex.cbSize = sizeof(WNDCLASSEX);
  	wcex.style			    = CS_HREDRAW | CS_VREDRAW;
  	wcex.lpfnWndProc	  = (WNDPROC)MainWndProc;
  	wcex.cbClsExtra		  = 0;
  	wcex.cbWndExtra		  = 0;
  	wcex.hInstance		  = hInstance;
  	wcex.hIcon			    = LoadIcon(NULL, IDI_APPLICATION);
  	wcex.hCursor		    = LoadCursor(NULL, IDC_ARROW);
  	wcex.hbrBackground	= (HBRUSH)(COLOR_WINDOW+1);
  	wcex.lpszMenuName  	= NULL;
  	wcex.lpszClassName	= g_pszMainWndClass;
    wcex.hIconSm		    = LoadIcon(NULL, IDI_APPLICATION);
  	RegisterClassEx(&wcex);
    return TRUE;
}

//-----------------------------------------------------------------------------

HMENU BuildMenu(int iApp)
{
    HMENU hMenu=CreateMenu();
    HMENU hSubMenu=CreatePopupMenu();
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_EXIT, "Exit");
    AppendMenu(hMenu, MF_POPUP | MF_ENABLED, (UINT_PTR)hSubMenu, "File");

    if (iApp == 0)
    {
        hSubMenu=CreatePopupMenu();
        AppendMenu(hMenu, MF_POPUP | MF_ENABLED, (UINT_PTR)hSubMenu, "All cameras");
        AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_OPEN_CAMERA_ALL,    "Connect all cameras");
        AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_CLOSE_CAMERA_ALL,   "Disconnect all cameras");
        AppendMenu(hSubMenu, MF_SEPARATOR, -1, "");
        AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_ACQUISITION_START_ALL, "Acquisition start on all cameras");
        AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_ACQUISITION_STOP_ALL, "Acquisition stop on all cameras");
    }

    hSubMenu=CreatePopupMenu();
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_OPEN_CAMERA,    "Connect camera");
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_CLOSE_CAMERA,   "Disconnect camera");
    AppendMenu(hSubMenu, MF_SEPARATOR, -1, "");
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_ACQUISITION_START, "Acquisition start");
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_ACQUISITION_STOP, "Acquisition stop");
    AppendMenu(hMenu, MF_POPUP | MF_ENABLED, (UINT_PTR)hSubMenu, "This camera");

    hSubMenu=CreatePopupMenu();
    --- %%IF IncRenderModes=1 ----------------
    --- %%IF IncSimpleImgProcess=1 -----------
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_PROCESS_IMAGE, "Process image");
    AppendMenu(hSubMenu, MF_SEPARATOR, -1, "");
    --- %%ENDIF ------------------------------
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_DISPLAY_FULL, "Full image size");
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_DISPLAY_SCALED, "Scale image to fit");
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_DISPLAY_TILES, "Tile images");
    --- %%IF IncSimpleImgProcess=1 -----------
    AppendMenu(hMenu, MF_POPUP | MF_ENABLED, (UINT_PTR)hSubMenu, "Process and display");
    --- %%ELSE -------------------------------
    AppendMenu(hMenu, MF_POPUP | MF_ENABLED, (UINT_PTR)hSubMenu, "Display");
    --- %%ENDIF ------------------------------
    --- %%ELSE -------------------------------
    --- %%IF IncSimpleImgProcess=1 -----------
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_PROCESS_IMAGE, "Process image");
    AppendMenu(hMenu, MF_POPUP | MF_ENABLED, (UINT_PTR)hSubMenu, "Process");
    --- %%ENDIF ------------------------------
    --- %%ENDIF ------------------------------

    return hMenu;
}

//-----------------------------------------------------------------------------

BOOL InitInstance(HINSTANCE hInstance, int nCmdShow)
  {
    g_hInstance = hInstance;
    int iScreenWidth  = GetSystemMetrics(SM_CXSCREEN);
    int iScreenHeight = GetSystemMetrics(SM_CYSCREEN);
    int iAppWidth   = ((iScreenWidth * 9)/10)/g_iApps;
    int iAppHeight  = (iScreenHeight * 9)/10;
    int iAppXOffset = iScreenWidth/20;
    int iAppYOffset = iScreenHeight/20;

    for (int iApp=0; iApp<g_iApps; iApp++)
    {
        g_App[iApp].hMenu = BuildMenu(iApp);

        // Create the main window
        char szCaption[200];
        switch (iApp)
        {
            case 0:
                wsprintf(szCaption, "%s: not connected", "%%DEVICE_DISPLAY_NAME_01%%");
                break;
            case 1:
                wsprintf(szCaption, "%s: not connected", "%%DEVICE_DISPLAY_NAME_02%%");
                break;
        }
        g_App[iApp].hWnd = CreateWindow(
             g_pszMainWndClass,
             szCaption,
             WS_OVERLAPPEDWINDOW,
             iAppXOffset + iAppWidth*iApp,
             iAppYOffset,
             iAppWidth,
             iAppHeight,
             NULL,
             g_App[iApp].hMenu,
             hInstance, NULL);
        ShowWindow(g_App[iApp].hWnd, nCmdShow);
        UpdateWindow(g_App[iApp].hWnd);
        UpdateControls(iApp);
    }
    return TRUE;
  }

//-----------------------------------------------------------------------------

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE /*hPrevInstance*/,
                   LPSTR lpCmdLine, int nCmdShow)
{
    if (LvLibrary::OpenLibrary() != LVSTATUS_OK)
    {
        ErrorMsg("Opening the library failed.");
        return 0;
    }
    if (LvSystem::Open("", g_pSystem) != LVSTATUS_OK)
    {
        ErrorMsg("Opening the system failed.");
        LvLibrary::CloseLibrary();
        return 0;
    }

    for (int i=0; i<g_iApps; i++)
    {
        memset(&g_App[i], 0, sizeof(TApp));
        g_App[i].pCamera = new CCamera();
    }

    MSG msg;
    if (!InitApplication(hInstance))
    {
        ErrorMsg("Application init failed.");
        for (int i=0; i<g_iApps; i++)
        {
            delete(g_App[i].pCamera);
        }
        LvSystem::Close(g_pSystem);
        LvLibrary::CloseLibrary();
        return 0;
    }

    if (!InitInstance(hInstance, nCmdShow))
    {
        ErrorMsg("Instance init failed.");
        for (int i=0; i<g_iApps; i++)
        {
            delete(g_App[i].pCamera);
        }
        LvSystem::Close(g_pSystem);
        LvLibrary::CloseLibrary();
        return 0;
    }

    while (GetMessage(&msg, NULL, NULL, NULL))
    {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }

    for (int i=0; i<g_iApps; i++)
    {
        delete(g_App[i].pCamera);
    }
    LvSystem::Close(g_pSystem);
    LvLibrary::CloseLibrary();
    return (int) msg.wParam;
}

//-----------------------------------------------------------------------------
