#define _CRT_SECURE_NO_WARNINGS
#define _CRT_SECURE_NO_DEPRECATE
#include "CppSampleWin02_Camera.h"

//-----------------------------------------------------------------------------
// Helper class for displaying the hourglass cursor, while the camera is
// being opened

class CWaitCursor
{
public:
    CWaitCursor();
    ~CWaitCursor();
private:
    HCURSOR m_hSaveCursor;
};

CWaitCursor::CWaitCursor()
{
    m_hSaveCursor = SetCursor(LoadCursor(NULL, IDC_WAIT));
    ShowCursor(TRUE);
}

CWaitCursor::~CWaitCursor()
{
    SetCursor(m_hSaveCursor);
    ShowCursor(TRUE);
}

void SetWindowTitle(int iCamera, const char* pszText); // defined in the _Main.cpp

//-----------------------------------------------------------------------------
// Error check with a message box displayed, if it is error

bool ErrorOccurred(LvStatus ErrorStatus)
{
    if (ErrorStatus == LVSTATUS_OK) return false;
    MessageBox(GetActiveWindow(), LvLibrary::GetLastErrorMessage().c_str(), "Error", MB_OK | MB_ICONEXCLAMATION);
    return true;
}

--- %%IF IncRenderModes=1 ----------------
//-----------------------------------------------------------------------------

int GetXScrollPos(HWND hWnd)
{
    SCROLLINFO si;
    si.cbSize = sizeof (si);
    si.fMask  = SIF_POS;
    if (GetScrollInfo (hWnd, SB_HORZ, &si))
        return si.nPos;
    else
        return 0;
}

//-----------------------------------------------------------------------------

int GetYScrollPos(HWND hWnd)
{
    SCROLLINFO si;
    si.cbSize = sizeof (si);
    si.fMask  = SIF_POS;
    if (GetScrollInfo (hWnd, SB_VERT, &si))
        return si.nPos;
    else
        return 0;
}

--- %%ENDIF ------------------------------
//-----------------------------------------------------------------------------
// callback function

void WINAPI CallbackNewBufferFunction(LvHBuffer hBuffer,
                                      void* pUserPointer, 
                                      void* pUserParam)
{
    CCamera* pCamera = (CCamera*) pUserParam;
    // in UserPointer we hold pointer to buffer
    pCamera->CallbackNewBuffer((LvBuffer*) pUserPointer);
}

//-----------------------------------------------------------------------------

CCamera::CCamera()
{
    m_pSystem       = NULL;
    m_pInterface    = NULL;
    m_pDevice       = NULL;
    m_pStream       = NULL;
    m_pRenderer     = NULL;
    m_pEvent        = NULL;
    m_hDisplayWnd   = NULL;
    m_iCamera       = 0;
    --- %%IF IncRenderModes=1 -----------------------
    m_RenderType = LvRenderType_FullSize;
    --- %%ENDIF -------------------------------------
    m_bAcquisitionStarted = false;
    --- %%IF CalcFrameRate=1 ------------------------
    m_dwGrabStartTimeForFps = GetTickCount();
    m_dwFrameCounterForFps = 0;
    m_dwFrameCounter = 0;
    m_dFramesPerSec  = 0;
    --- %%ENDIF -------------------------------------

    m_iPayloadSize   = 0;
    memset(m_Buffers, 0, sizeof(m_Buffers));
    --- %%IF IncSimpleImgProcess=1 ------------------
    m_bDoProcessing = false;
    --- %%ENDIF -------------------------------------
}

//-----------------------------------------------------------------------------

CCamera::~CCamera()
{
    if (m_pDevice != NULL) CloseCamera();
}

//-----------------------------------------------------------------------------

void CCamera::OpenCamera(int iCamera,
                         HWND hDisplayWnd, 
                         LvSystem* pSystem)
{
    CWaitCursor WaitCursor;
    if (m_pDevice != NULL) CloseCamera();
    m_pSystem = pSystem;
    m_hDisplayWnd = hDisplayWnd;
    m_iCamera = iCamera;

    LvStatus SynViewStatus;
    LvInterface* pInterface = NULL;
    LvDevice* pDevice = NULL;
    pSystem->UpdateInterfaceList();

    switch (iCamera)
    {
        case 0: // %%DEVICE_DISPLAY_NAME_01%%
        {
            --- %%IF OpenInterface=ByInterfaceId -------
            SynViewStatus = pSystem->OpenInterface("%%INTERFACE_ID_01%%", pInterface);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF --------------------------------
            --- %%IF OpenInterface=ByInterfaceTlType ---
            char InterfaceId[256];
            SynViewStatus = pSystem->FindInterface(LvFindBy_TLType, "%%INTERFACE_TLTYPE_01%%",
                                                   InterfaceId, sizeof(InterfaceId));
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pSystem->OpenInterface(InterfaceId, pInterface);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF --------------------------------

            pInterface->UpdateDeviceList();
            --- %%IF OpenDevice=ByDeviceId -------------
            SynViewStatus = pInterface->OpenDevice("%%DEVICE_ID_01%%", pDevice, LvDeviceAccess_Control);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF --------------------------------
            --- %%IF OpenDevice=ByDeviceModelName ------
            char DeviceId[256];
            SynViewStatus = pInterface->FindDevice(LvFindBy_ModelName, "%%DEVICE_MODELNAME_01%%",
                                                   DeviceId, sizeof(DeviceId));
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pInterface->OpenDevice(DeviceId, pDevice, LvDeviceAccess_Control);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF --------------------------------
            --- %%IF OpenDevice=ByDeviceIpAddress ------
            char DeviceId[256];
            SynViewStatus = pInterface->FindDevice(LvFindBy_GevIPAddress, "%%DEVICE_IPADDRESS_01%%",
                                                   DeviceId, sizeof(DeviceId));
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pInterface->OpenDevice(DeviceId, pDevice, LvDeviceAccess_Control);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF --------------------------------
            --- %%IF OpenDevice=ByDeviceUserId ---------
            char DeviceId[256];
            SynViewStatus = pInterface->FindDevice(LvFindBy_UserID, "%%DEVICE_USERID_01%%",
                                                   DeviceId, sizeof(DeviceId));
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pInterface->OpenDevice(DeviceId, pDevice, LvDeviceAccess_Control);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF --------------------------------
            --- %%IF Sys.AutoLoadedCfg=1 ---------------
            // Note: When the code was generated, in SynView Explorer was switched ON the option
            // to autoload the camera settings after opening the camera. For this reason the following
            // line loads the camera settings from the file to assure the same starting status.
            --- %%ENDIF --------------------------------
            --- %%IF Sys.SnippetFile=1 -----------------
            SynViewStatus = pDevice->LoadSettings("", "%%Sys.CameraCfgFile01%%", LvSaveFlag_All);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF --------------------------------

            --- %%IF Snippet=All ---------------------
            // The #error line below is intentionally inserted to the code in case you
            // generate the code from streamable or all writable features.
            #error Review the feature settings code and remove the unnecessary items!
            // Before removing this line from the code, go carefully through all the feature
            // settings below and leave there only those, which really need to be set.

            --- %%ENDIF ------------------------------
            --- %%IF Snippet=Streamable --------------
            // The #error line below is intentionally inserted to the code in case you
            // generate the code from streamable or all writable features.
            #error Review the feature settings code and remove the unnecessary items!
            // Before removing this line from the code, go carefully through all the feature
            // settings below and leave there only those, which really need to be set.

            --- %%ENDIF ------------------------------
            %%CameraRemoteFeaturesCode01%%
            
            --- %%IF IncChunkData=1 ------------------
            --- %%IF FEATURE ChunkModeActive ---
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkModeActive, 1);
            if (ErrorOccurred(SynViewStatus)) return;

            --- %%IF CHUNK_ENABLED01 OffsetX ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_OffsetX);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED01 OffsetY ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_OffsetY);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED01 Width ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_Width);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED01 Height ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_Height);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED01 PixelFormat ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_PixelFormat);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED01 LinePitch ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_LinePitch);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED01 FrameID ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_FrameID);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED01 Timestamp ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_Timestamp);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED01 ExposureTime ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_ExposureTime);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED01 Gain ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_Gain);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED01 LineStatusAll ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_LineStatusAll);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED01 LvExternalADCValue ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_LvExternalADCValue);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%ENDIF ------------------------------
            --- %%ENDIF ------------------------------
            break;
        }
        case 1: // %%DEVICE_DISPLAY_NAME_02%%
        {
            --- %%IF OpenInterface=ByInterfaceId -------
            SynViewStatus = pSystem->OpenInterface("%%INTERFACE_ID_02%%", pInterface);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF --------------------------------
            --- %%IF OpenInterface=ByInterfaceTlType ---
            char InterfaceId[256];
            SynViewStatus = pSystem->FindInterface(LvFindBy_TLType, "%%INTERFACE_TLTYPE_02%%",
                                                   InterfaceId, sizeof(InterfaceId));
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pSystem->OpenInterface(InterfaceId, pInterface);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF --------------------------------

            pInterface->UpdateDeviceList();
            --- %%IF OpenDevice=ByDeviceId -------------
            SynViewStatus = pInterface->OpenDevice("%%DEVICE_ID_02%%", pDevice, LvDeviceAccess_Control);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF --------------------------------
            --- %%IF OpenDevice=ByDeviceModelName ------
            char DeviceId[256];
            SynViewStatus = pInterface->FindDevice(LvFindBy_ModelName, "%%DEVICE_MODELNAME_02%%",
                                                   DeviceId, sizeof(DeviceId));
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pInterface->OpenDevice(DeviceId, pDevice, LvDeviceAccess_Control);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF --------------------------------
            --- %%IF OpenDevice=ByDeviceIpAddress ------
            char DeviceId[256];
            SynViewStatus = pInterface->FindDevice(LvFindBy_GevIPAddress, "%%DEVICE_IPADDRESS_02%%",
                                                   DeviceId, sizeof(DeviceId));
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pInterface->OpenDevice(DeviceId, pDevice, LvDeviceAccess_Control);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF --------------------------------
            --- %%IF OpenDevice=ByDeviceUserId ---------
            char DeviceId[256];
            SynViewStatus = pInterface->FindDevice(LvFindBy_UserID, "%%DEVICE_USERID_02%%",
                                                   DeviceId, sizeof(DeviceId));
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pInterface->OpenDevice(DeviceId, pDevice, LvDeviceAccess_Control);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF --------------------------------
            --- %%IF Sys.AutoLoadedCfg=1 ---------------
            // Note: When the code was generated, in SynView Explorer was switched ON the option
            // to autoload the camera settings after opening the camera. For this reason the following
            // line loads the camera settings from the file to assure the same starting status.
            --- %%ENDIF --------------------------------
            --- %%IF Sys.SnippetFile=1 -----------------
            SynViewStatus = pDevice->LoadSettings("", "%%Sys.CameraCfgFile02%%", LvSaveFlag_All);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF --------------------------------

            --- %%IF Snippet=All ---------------------
            // The #error line below is intentionally inserted to the code in case you
            // generate the code from streamable or all writable features.
            #error Review the feature settings code and remove the unnecessary items!
            // Before removing this line from the code, go carefully through all the feature
            // settings below and leave there only those, which really need to be set.

            --- %%ENDIF ------------------------------
            --- %%IF Snippet=Streamable --------------
            // The #error line below is intentionally inserted to the code in case you
            // generate the code from streamable or all writable features.
            #error Review the feature settings code and remove the unnecessary items!
            // Before removing this line from the code, go carefully through all the feature
            // settings below and leave there only those, which really need to be set.

            --- %%ENDIF ------------------------------
            %%CameraRemoteFeaturesCode02%%
			
            --- %%IF IncChunkData=1 ------------------
            --- %%IF FEATURE ChunkModeActive ---
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkModeActive, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%IF CHUNK_ENABLED02 OffsetX ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_OffsetX);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED02 OffsetY ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_OffsetY);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED02 Width ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_Width);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED02 Height ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_Height);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED02 PixelFormat ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_PixelFormat);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED02 LinePitch ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_LinePitch);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED02 FrameID ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_FrameID);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED02 Timestamp ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_Timestamp);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED02 ExposureTime ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_ExposureTime);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED02 Gain ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_Gain);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED02 LineStatusAll ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_LineStatusAll);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED02 LvExternalADCValue ---
            SynViewStatus = pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_LvExternalADCValue);
            if (ErrorOccurred(SynViewStatus)) return;
            SynViewStatus = pDevice->SetBool(LvDevice_ChunkEnable, 1);
            if (ErrorOccurred(SynViewStatus)) return;
            --- %%ENDIF ------------------------------
            --- %%ENDIF ------------------------------
            --- %%ENDIF ------------------------------
            break;
        }
    }

    m_pInterface = pInterface;
    m_pDevice = pDevice;
    --- %%IF IncUniImgPreprocess=1 -------------

    SynViewStatus = m_pDevice->SetEnum(LvDevice_LvUniProcessMode, LvUniProcessMode_Auto);
    if (ErrorOccurred(SynViewStatus)) return;
    SetOptimalUniPixelFormat();
    --- %%ENDIF --------------------------------

    SynViewStatus = m_pDevice->OpenStream("", m_pStream);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pStream->OpenEvent(LvEventType_NewBuffer, m_pEvent);
    if (ErrorOccurred(SynViewStatus)) return;
    if (!ReallocateBuffers()) return;
    SynViewStatus = m_pStream->OpenRenderer(m_pRenderer);
    if (ErrorOccurred(SynViewStatus)) return;
    m_pRenderer->SetWindow(m_hDisplayWnd);                              

    SynViewStatus = m_pRenderer->SetEnum(LvRenderer_LvRenderType, LvRenderType_ScaleToTiles);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pRenderer->SetInt32(LvRenderer_LvNumberOfTiles, 6);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pRenderer->SetEnum(LvRenderer_LvRenderType, LvRenderType_FullSize);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pStream->SetInt32(LvStream_LvPostponeQueueBuffers, 6);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pEvent->SetCallbackNewBuffer(CallbackNewBufferFunction, this);
    if (ErrorOccurred(SynViewStatus)) return;

    SynViewStatus = m_pDevice->GetString(LvDevice_LvDeviceDisplayName, m_sDisplayName);
    if (ErrorOccurred(SynViewStatus)) return;
    UpdateInfoInTitleBar();
}

//-----------------------------------------------------------------------------

bool CCamera::ReallocateBuffers()
{
    LvStatus SynViewStatus;
    int iPayloadSize;
    SynViewStatus = m_pStream->GetInt32(LvStream_LvCalcPayloadSize, &iPayloadSize);
    if (ErrorOccurred(SynViewStatus)) return false;
    if (iPayloadSize != m_iPayloadSize)
    {
        SynViewStatus = m_pStream->FlushQueue(LvQueueOperation_AllDiscard);
        if (ErrorOccurred(SynViewStatus)) return false;
        // buffers are either not yet allocated, or something has changed, so there is a need to reallocate
        for (int i=0; i<NUMBER_OF_BUFFERS; i++)
            if (m_Buffers[i] != NULL)
                m_pStream->CloseBuffer(m_Buffers[i]);
        m_iPayloadSize = iPayloadSize;
        for (int i=0; i<NUMBER_OF_BUFFERS; i++) 
        {
            SynViewStatus = m_pStream->OpenBuffer(NULL, 0, NULL, 0, m_Buffers[i]);
            if (ErrorOccurred(SynViewStatus)) return false;
            SynViewStatus = m_Buffers[i]->Queue();
            if (ErrorOccurred(SynViewStatus)) return false;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------

void CCamera::StartAcquisition()
{
    if (m_pDevice == NULL) return;
    LvStatus SynViewStatus;
    --- %%IF IncRenderModes=1 -----------------------
    SetScrollbars();
    --- %%ENDIF -------------------------------------
    if (!ReallocateBuffers()) return;
    SynViewStatus = m_pStream->FlushQueue(LvQueueOperation_AllToInput);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pEvent->StartThread();
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->AcquisitionStart();
    if (ErrorOccurred(SynViewStatus)) return;
    m_bAcquisitionStarted = true;
    --- %%IF CalcFrameRate=1 ------------------------
    m_dwFrameCounterForFps = 0;
    m_dwFrameCounter = 0;
    --- %%ENDIF -------------------------------------
    UpdateInfoInTitleBar();
}

//-----------------------------------------------------------------------------

void CCamera::StopAcquisition()
{
    if (!m_bAcquisitionStarted) return;
    LvStatus SynViewStatus;
    SynViewStatus = m_pDevice->AcquisitionStop();
    if (ErrorOccurred(SynViewStatus)) return;
    m_bAcquisitionStarted = false;
    SynViewStatus = m_pEvent->StopThread();
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pStream->FlushQueue(LvQueueOperation_AllToInput);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%IF CalcFrameRate=1 ------------------------
    m_dwFrameCounterForFps = 0;
    --- %%ENDIF -------------------------------------
    UpdateInfoInTitleBar();
}

//-----------------------------------------------------------------------------

void CCamera::CloseCamera()
{
    CWaitCursor WaitCursor;
    if (m_pDevice == NULL) return;
    LvStatus SynViewStatus;
    StopAcquisition();
    SynViewStatus = m_pStream->CloseEvent(m_pEvent);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pStream->CloseRenderer(m_pRenderer);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pStream->FlushQueue(LvQueueOperation_AllDiscard);
    if (ErrorOccurred(SynViewStatus)) return;
    for (int i=0; i<NUMBER_OF_BUFFERS; i++)
        if (m_Buffers[i] != NULL)
            m_pStream->CloseBuffer(m_Buffers[i]);
    m_iPayloadSize = 0;
    SynViewStatus = m_pDevice->CloseStream(m_pStream);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pInterface->CloseDevice(m_pDevice);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pSystem->CloseInterface(m_pInterface);
    if (ErrorOccurred(SynViewStatus)) return;
    InvalidateRect(m_hDisplayWnd, NULL, TRUE);
    UpdateInfoInTitleBar();
}

//-----------------------------------------------------------------------------

bool CCamera::IsOpen()
{
    return m_pDevice != NULL;
}

//-----------------------------------------------------------------------------

bool CCamera::IsAcquiring()
{
    if (m_pDevice == NULL) return false;
    int32_t iIsAcquiring;
    m_pDevice->GetInt32(LvDevice_LvDeviceIsAcquiring, &iIsAcquiring);
    return iIsAcquiring != 0;
}

--- %%IF IncSimpleImgProcess=1 ------------------
//-----------------------------------------------------------------------------
// Switches ON/OFF image processing done in the callback

void CCamera::SetProcessing(bool bDoProcessing)
{
    m_bDoProcessing = bDoProcessing;
}

//-----------------------------------------------------------------------------

bool CCamera::GetProcessing()
{
    return m_bDoProcessing;
}

--- %%ENDIF -------------------------------------
//-----------------------------------------------------------------------------

void CCamera::CallbackNewBuffer(LvBuffer* pBuffer)
{
    static int LastXPos = 0;
    static int LastYPos = 0;
    --- %%IF CalcFrameRate=1 ------------------------
    m_dwFrameCounter++;
    m_dwFrameCounterForFps++;
    CalculateFramesPerSecond();
    --- %%ENDIF -------------------------------------

    --- %%IF IncSimpleImgProcess=1 ------------------
    // get the pointer to the image data
    void* pData = NULL;
    int32_t iImageOffset = 0;
    --- %%IF IncUniImgPreprocess=1 ------------------
    pBuffer->GetPtr (LvBuffer_UniBase, &pData);
    pBuffer->GetInt32 (LvBuffer_UniImageOffset, &iImageOffset);
    --- %%ELSE --------------------------------------
    pBuffer->GetPtr (LvBuffer_Base, &pData);
    pBuffer->GetInt32 (LvBuffer_ImageOffset, &iImageOffset);
    --- %%ENDIF -------------------------------------
    pData = (uint8_t*)pData + iImageOffset;
    if (m_bDoProcessing && pData != NULL)
    {
        // we will do some easy processing - invert the pixel values in an area
        int32_t iWidth;
        int32_t iHeight;
        int32_t iLinePitch;
        LvEnum iPixelFormat;    // LvPixelFormat enumeration value
        m_pDevice->GetInt32(LvDevice_Width, &iWidth);
        m_pDevice->GetInt32(LvDevice_Height, &iHeight);
        --- %%IF IncUniImgPreprocess=1 ------------------
        m_pDevice->GetEnum(LvDevice_LvUniPixelFormat, &iPixelFormat);
        m_pDevice->GetInt32(LvDevice_LvUniLinePitch, &iLinePitch);
        int32_t iBytesPerPixel = ((iPixelFormat & LV_PIX_EFFECTIVE_PIXEL_SIZE_MASK) >> LV_PIX_EFFECTIVE_PIXEL_SIZE_SHIFT) / 8;
        --- %%ELSE --------------------------------------
        m_pDevice->GetEnum(LvDevice_PixelFormat, &iPixelFormat);
        int iBitsPerPixel = (iPixelFormat & LV_PIX_EFFECTIVE_PIXEL_SIZE_MASK) >> LV_PIX_EFFECTIVE_PIXEL_SIZE_SHIFT;
        iLinePitch = ((iWidth * iBitsPerPixel)+7)/8;
        int32_t iBytesPerPixel = iBitsPerPixel / 8;
        --- %%ENDIF -------------------------------------

        for (int32_t j=0; j<(iHeight/2); j++)
        {
            uint8_t* pPixel = ((uint8_t*)pData) + (iHeight/4 + j)*iLinePitch + (iWidth/4)*iBytesPerPixel;
            for (int32_t i=0; i<(iWidth/2); i++)
            {
                for (int32_t k=0; k<iBytesPerPixel; k++)
                {
                    *pPixel = ~*pPixel;
                    pPixel++;
                }
            }
        }
    }
    --- %%ELSE --------------------------------------
    // no image processing demonstrated - switch it ON in the code generation Wizard if you need it
    --- %%ENDIF -------------------------------------

    --- %%IF IncChunkData=1 ------------------
    --- %%IF FEATURE ChunkModeActive ---
    // before the chunk data can be read for this buffer, they must be parsed.
    pBuffer->ParseChunkData();
    int64_t ChunkValue;
    char szChunkLine[256];
    szChunkLine[0] = 0;
    char szValue[128];

    switch (m_iCamera)
    {
        case 0: // %%DEVICE_DISPLAY_NAME_01%%
        {
            --- %%IF CHUNK_ENABLED01 OffsetX ---
            m_pDevice->GetInt64 (LvDevice_ChunkOffsetX, &ChunkValue);
            sprintf(szValue, "X=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED01 OffsetY ---
            m_pDevice->GetInt64 (LvDevice_ChunkOffsetY, &ChunkValue);
            sprintf(szValue, "Y=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED01 Width ---
            m_pDevice->GetInt64 (LvDevice_ChunkWidth, &ChunkValue);
            sprintf(szValue, "W=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED01 Height ---
            m_pDevice->GetInt64 (LvDevice_ChunkHeight, &ChunkValue);
            sprintf(szValue, "H=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED01 PixelFormat ---
            m_pDevice->GetInt64 (LvDevice_ChunkPixelFormat, &ChunkValue);
            sprintf(szValue, "PF=0x%llx | ", ChunkValue);
            strcat(szChunkLine, szValue);
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED01 LinePitch ---
            m_pDevice->GetInt64 (LvDevice_ChunkLinePitch, &ChunkValue);
            sprintf(szValue, "LPitch=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED01 FrameID ---
            m_pDevice->GetInt64 (LvDevice_ChunkFrameID, &ChunkValue);
            sprintf(szValue, "FrameID=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED01 Timestamp ---
            m_pDevice->GetInt64 (LvDevice_ChunkTimestamp, &ChunkValue);
            sprintf(szValue, "Time=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED01 ExposureTime ---
            m_pDevice->GetInt64 (LvDevice_ChunkExposureTime, &ChunkValue);
            sprintf(szValue, "Exp=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED01 Gain ---
            // the IsAvailableEnumEntry() is time consuming, better if you move it outside this callback
            if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkGainSelector, LvChunkGainSelector_AnalogAll) != 0)
            {
                m_pDevice->SetEnum (LvDevice_ChunkGainSelector, LvChunkGainSelector_AnalogAll);
                m_pDevice->GetInt64 (LvDevice_ChunkGain, &ChunkValue);
                sprintf(szValue, "GainA=%lld | ", ChunkValue);
                strcat(szChunkLine, szValue);
            }
            if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkGainSelector, LvChunkGainSelector_DigitalAll) != 0)
            {
                m_pDevice->SetEnum (LvDevice_ChunkGainSelector, LvChunkGainSelector_DigitalAll);
                m_pDevice->GetInt64 (LvDevice_ChunkGain, &ChunkValue);
                sprintf(szValue, "GainD=%lld | ", ChunkValue);
                strcat(szChunkLine, szValue);
            }
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED01 LineStatusAll ---
            m_pDevice->GetInt64 (LvDevice_ChunkLineStatusAll, &ChunkValue);
            sprintf(szValue, "Line=0x%llx | ", ChunkValue);
            strcat(szChunkLine, szValue);
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED01 LvExternalADCValue ---
            // the IsAvailableEnumEntry() is time consuming, better if you move it outside this callback
            if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC1) != 0)
            {
                m_pDevice->SetEnum (LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC1);
                m_pDevice->GetInt64 (LvDevice_ChunkLvExternalADCValue, &ChunkValue);
                sprintf(szValue, "ADC1=%lld | ", ChunkValue);
                strcat(szChunkLine, szValue);
            }
            if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC2) != 0)
            {
                m_pDevice->SetEnum (LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC2);
                m_pDevice->GetInt64 (LvDevice_ChunkLvExternalADCValue, &ChunkValue);
                sprintf(szValue, "ADC2=%lld | ", ChunkValue);
                strcat(szChunkLine, szValue);
            }
            if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC3) != 0)
            {
                m_pDevice->SetEnum (LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC3);
                m_pDevice->GetInt64 (LvDevice_ChunkLvExternalADCValue, &ChunkValue);
                sprintf(szValue, "ADC3=%lld | ", ChunkValue);
                strcat(szChunkLine, szValue);
            }
            if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC4) != 0)
            {
                m_pDevice->SetEnum (LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC4);
                m_pDevice->GetInt64 (LvDevice_ChunkLvExternalADCValue, &ChunkValue);
                sprintf(szValue, "ADC4=%lld | ", ChunkValue);
                strcat(szChunkLine, szValue);
            }
            --- %%ENDIF ------------------------------
            break;
        }
        case 1: // %%DEVICE_DISPLAY_NAME_02%%
        {
            --- %%IF CHUNK_ENABLED02 OffsetX ---
            m_pDevice->GetInt64 (LvDevice_ChunkOffsetX, &ChunkValue);
            sprintf(szValue, "X=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED02 OffsetY ---
            m_pDevice->GetInt64 (LvDevice_ChunkOffsetY, &ChunkValue);
            sprintf(szValue, "Y=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED02 Width ---
            m_pDevice->GetInt64 (LvDevice_ChunkWidth, &ChunkValue);
            sprintf(szValue, "W=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED02 Height ---
            m_pDevice->GetInt64 (LvDevice_ChunkHeight, &ChunkValue);
            sprintf(szValue, "H=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED02 PixelFormat ---
            m_pDevice->GetInt64 (LvDevice_ChunkPixelFormat, &ChunkValue);
            sprintf(szValue, "PF=0x%llx | ", ChunkValue);
            strcat(szChunkLine, szValue);
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED02 LinePitch ---
            m_pDevice->GetInt64 (LvDevice_ChunkLinePitch, &ChunkValue);
            sprintf(szValue, "LPitch=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED02 FrameID ---
            m_pDevice->GetInt64 (LvDevice_ChunkFrameID, &ChunkValue);
            sprintf(szValue, "FrameID=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED02 Timestamp ---
            m_pDevice->GetInt64 (LvDevice_ChunkTimestamp, &ChunkValue);
            sprintf(szValue, "Time=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED02 ExposureTime ---
            m_pDevice->GetInt64 (LvDevice_ChunkExposureTime, &ChunkValue);
            sprintf(szValue, "Exp=%lld | ", ChunkValue);
            strcat(szChunkLine, szValue);
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED02 Gain ---
            // the IsAvailableEnumEntry() is time consuming, better if you move it outside this callback
            if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkGainSelector, LvChunkGainSelector_AnalogAll) != 0)
            {
                m_pDevice->SetEnum (LvDevice_ChunkGainSelector, LvChunkGainSelector_AnalogAll);
                m_pDevice->GetInt64 (LvDevice_ChunkGain, &ChunkValue);
                sprintf(szValue, "GainA=%lld | ", ChunkValue);
                strcat(szChunkLine, szValue);
            }
            if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkGainSelector, LvChunkGainSelector_DigitalAll) != 0)
            {
                m_pDevice->SetEnum (LvDevice_ChunkGainSelector, LvChunkGainSelector_DigitalAll);
                m_pDevice->GetInt64 (LvDevice_ChunkGain, &ChunkValue);
                sprintf(szValue, "GainD=%lld | ", ChunkValue);
                strcat(szChunkLine, szValue);
            }
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED02 LineStatusAll ---
            m_pDevice->GetInt64 (LvDevice_ChunkLineStatusAll, &ChunkValue);
            sprintf(szValue, "Line=0x%llx | ", ChunkValue);
            strcat(szChunkLine, szValue);
            --- %%ENDIF ------------------------------
            --- %%IF CHUNK_ENABLED02 LvExternalADCValue ---
            // the IsAvailableEnumEntry() is time consuming, better if you move it outside this callback
            if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC1) != 0)
            {
                m_pDevice->SetEnum (LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC1);
                m_pDevice->GetInt64 (LvDevice_ChunkLvExternalADCValue, &ChunkValue);
                sprintf(szValue, "ADC1=%lld | ", ChunkValue);
                strcat(szChunkLine, szValue);
            }
            if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC2) != 0)
            {
                m_pDevice->SetEnum (LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC2);
                m_pDevice->GetInt64 (LvDevice_ChunkLvExternalADCValue, &ChunkValue);
                sprintf(szValue, "ADC2=%lld | ", ChunkValue);
                strcat(szChunkLine, szValue);
            }
            if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC3) != 0)
            {
                m_pDevice->SetEnum (LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC3);
                m_pDevice->GetInt64 (LvDevice_ChunkLvExternalADCValue, &ChunkValue);
                sprintf(szValue, "ADC3=%lld | ", ChunkValue);
                strcat(szChunkLine, szValue);
            }
            if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC4) != 0)
            {
                m_pDevice->SetEnum (LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC4);
                m_pDevice->GetInt64 (LvDevice_ChunkLvExternalADCValue, &ChunkValue);
                sprintf(szValue, "ADC4=%lld | ", ChunkValue);
                strcat(szChunkLine, szValue);
            }
            --- %%ENDIF ------------------------------
            break;
        }
    }
    SetWindowTitle(m_iCamera, szChunkLine);
    --- %%ENDIF ------------------------------
    --- %%ENDIF ------------------------------

    --- %%IF IncRenderModes=1 -----------------------
    int XPos = GetXScrollPos(m_hDisplayWnd);
    int YPos = GetYScrollPos(m_hDisplayWnd);
    if (XPos != LastXPos || YPos != LastYPos)
    {
        LastXPos = XPos;
        LastYPos = YPos;
        m_pRenderer->SetInt32(LvRenderer_LvOffsetX, -XPos);
        m_pRenderer->SetInt32(LvRenderer_LvOffsetY, -YPos);
    }
    --- %%ENDIF -------------------------------------
    m_pRenderer->DisplayImage(pBuffer);
    pBuffer->Queue();
    --- %%IF CalcFrameRate=1 ------------------------
    UpdateInfoInTitleBar();
    --- %%ENDIF -------------------------------------
}

//-----------------------------------------------------------------------------

void CCamera::UpdateInfoInTitleBar()
{
    char szText[512];
    if (IsOpen())
    {
        if (IsAcquiring())
        {
            --- %%IF CalcFrameRate=1 ------------------------
            double dFps = CalculateFramesPerSecond();
            sprintf (szText, "%s: Image %03d | %0.1f fps", m_sDisplayName.c_str(), m_dwFrameCounter, dFps);
            --- %%ELSE --------------------------------------
            sprintf (szText, "%s: acquiring", m_sDisplayName.c_str());
            --- %%ENDIF -------------------------------------
        }
        else
        {
            sprintf (szText, "%s: connected", m_sDisplayName.c_str());
        }
    }
    else
    {
        sprintf (szText, "%s: not connected", m_sDisplayName.c_str());
    }
    SetWindowTitle(m_iCamera, szText);
}

--- %%IF IncRenderModes=1 -----------------------
//-----------------------------------------------------------------------------

void CCamera::SetScrollbars()
{
    if (m_pDevice == NULL) return;
    RECT ClientRect;
    GetClientRect(m_hDisplayWnd, &ClientRect);

    int32_t iWidth;
    int32_t iHeight;
    m_pDevice->GetInt32(LvDevice_Width, &iWidth);
    m_pDevice->GetInt32(LvDevice_Height, &iHeight);

    SCROLLINFO si;
    si.cbSize = sizeof (si);
    si.fMask = SIF_RANGE | SIF_PAGE;
    si.nMin = 0;
    si.nMax = m_RenderType == LvRenderType_FullSize ? iHeight : ClientRect.bottom-1;
    si.nPage = ClientRect.bottom;
    SetScrollInfo (m_hDisplayWnd, SB_VERT, &si, TRUE);

    si.fMask = SIF_RANGE | SIF_PAGE;
    si.nMin = 0;
    si.nMax = m_RenderType == LvRenderType_FullSize ? iWidth : ClientRect.right-1;
    si.nPage = ClientRect.right;
    SetScrollInfo (m_hDisplayWnd, SB_HORZ, &si, TRUE);
}

//-----------------------------------------------------------------------------

void CCamera::SetRenderType(LvRenderType RenderType)
{
    m_RenderType = RenderType;
    if (m_pRenderer != NULL) 
        m_pRenderer->SetEnum(LvRenderer_LvRenderType, m_RenderType);
    SetScrollbars();
    Repaint();
}

//-----------------------------------------------------------------------------

LvRenderType CCamera::GetRenderType()
{
    if (m_pRenderer != NULL) 
    {
        LvEnum RenderType;
        m_pRenderer->GetEnum(LvRenderer_LvRenderType, &RenderType);
        m_RenderType = (LvRenderType) RenderType;
    }
    return m_RenderType;
}

--- %%ENDIF -------------------------------------
//-----------------------------------------------------------------------------

void CCamera::Repaint()
{
    --- %%IF IncRenderModes=1 ----------------
    static int LastXPos = 0;
    static int LastYPos = 0;
    --- %%ENDIF ------------------------------
    if (m_pRenderer != NULL)
    {
        --- %%IF IncRenderModes=1 ----------------
        int XPos = GetXScrollPos(m_hDisplayWnd);
        int YPos = GetYScrollPos(m_hDisplayWnd);
        if (XPos != LastXPos || YPos != LastYPos)
        {
            LastXPos = XPos;
            LastYPos = YPos;
            m_pRenderer->SetInt32(LvRenderer_LvOffsetX, -XPos);
            m_pRenderer->SetInt32(LvRenderer_LvOffsetY, -YPos);
        }
        --- %%ENDIF ------------------------------
        m_pRenderer->Repaint();
    }
}

--- %%IF CalcFrameRate=1 ------------------------
//-----------------------------------------------------------------------------

double CCamera::CalculateFramesPerSecond()
{
    uint32_t dwCurrentTime = GetTickCount();
    if (dwCurrentTime > m_dwGrabStartTimeForFps && 
        (dwCurrentTime >= (m_dwGrabStartTimeForFps + 3000) ||
         m_dwFrameCounterForFps > 200))
    {
        if (m_dwFrameCounterForFps == 0)
            m_dFramesPerSec = 0;
        else
            m_dFramesPerSec = (((double)m_dwFrameCounterForFps + 0.5)*1000) / (double)(dwCurrentTime - m_dwGrabStartTimeForFps);
        m_dwFrameCounterForFps = 0;
        m_dwGrabStartTimeForFps = dwCurrentTime;
    }
    return m_dFramesPerSec;
}

--- %%ENDIF -------------------------------------
--- %%IF IncUniImgPreprocess=1 ------------------
//-----------------------------------------------------------------------------
// Determines optimal pixel format for the uni-process

void CCamera::SetOptimalUniPixelFormat()
{
    LvEnum PixelFormat;
    m_pDevice->GetEnum(LvDevice_PixelFormat, &PixelFormat);
    LvEnum UniPixelFormat = PixelFormat;
    switch (PixelFormat)
    {
        case LvPixelFormat_Mono8:
        case LvPixelFormat_Mono10:
        case LvPixelFormat_Mono12:
        case LvPixelFormat_Mono16:
            UniPixelFormat = LvPixelFormat_Mono8;
            break;

        case LvPixelFormat_BayerGR8:
        case LvPixelFormat_BayerRG8:
        case LvPixelFormat_BayerGB8:
        case LvPixelFormat_BayerBG8:
        case LvPixelFormat_BayerGR10:
        case LvPixelFormat_BayerRG10:
        case LvPixelFormat_BayerGB10:
        case LvPixelFormat_BayerBG10:
        case LvPixelFormat_BayerGR12:
        case LvPixelFormat_BayerRG12:
        case LvPixelFormat_BayerGB12:
        case LvPixelFormat_BayerBG12:
        case LvPixelFormat_RGB8Packed:
        case LvPixelFormat_RGBA8Packed:
            UniPixelFormat = LvPixelFormat_BGRA8Packed;
            break;
    }
    m_pDevice->SetEnum(LvDevice_LvUniPixelFormat, UniPixelFormat);
}

--- %%ENDIF -------------------------------------
//-----------------------------------------------------------------------------


