// This is a simple sample application, which creates a window,
// enabling to open and close a camera, start and stop acquisition.

#include <stdio.h>
#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>
#include <X11/Intrinsic.h>
#include <X11/Xproto.h>

#include <string>
#include <vector>
#include <sv.synview.class.h>

//=============================================================================
// The CCamera class implements access to one camera. In this simple sample
// it is not necessary; we could use global variables and functions instead.
// However, the CCamera class becomes practical when working with multiple 
// cameras (not demonstrated in this sample).

#define NUMBER_OF_BUFFERS %%Buffers%%

class CCamera
{
public:
    CCamera();
    ~CCamera();
    void OpenCamera(Display *pDisplay, Window hWindow, LvSystem* pSystem);
    void CloseCamera();
    void StartAcquisition();
    void StopAcquisition();
    --- %%IF IncSimpleImgProcess=1 ------------------
    void SetProcessing(bool bDoProcessing);
    --- %%ENDIF -------------------------------------
    bool IsOpen();
    bool IsAcquiring();
    void CallbackNewBuffer(LvBuffer* pBuffer);
    --- %%IF IncUniImgPreprocess=1 ------------------
    void SetOptimalUniPixelFormat();
    --- %%ENDIF -------------------------------------

private:
    LvSystem*    m_pSystem;
    LvInterface* m_pInterface;
    LvDevice*    m_pDevice;
    LvStream*    m_pStream;
    LvBuffer*    m_Buffers[NUMBER_OF_BUFFERS];
    --- %%IF IncImgDisplay=1 ------------------------
    LvRenderer*  m_pRenderer;
    Display*     m_pDisplay;
    Window       m_hWindow;
    --- %%ENDIF -------------------------------------
    LvEvent*     m_pEvent;
    --- %%IF IncSimpleImgProcess=1 ------------------
    bool         m_bDoProcessing;
    --- %%ENDIF -------------------------------------
};

// global variables
LvSystem*    g_pSystem         = 0;
CCamera* g_pCamera     = NULL;
--- %%IF IncSimpleImgProcess=1 ------------------
bool         g_bProcessImage   = false;
--- %%ENDIF -------------------------------------
Display*     g_pDisplay        = NULL;
Window       g_hWindow         = 0;
char         g_ErrMsg[1024]    = {0};

// forward declaration
static void SetWindowTitle (const char *szTitle);

//-----------------------------------------------------------------------------
// Error check with a message displayed, if it is error
void DisplayError (const char *Message)
{
    printf ("Error: %s\n", Message);
    fflush (stdout);

    snprintf (g_ErrMsg, sizeof(g_ErrMsg), "[Error] %s", Message);
    XEvent expose_event;
    expose_event.type = Expose;
    XSendEvent (g_pDisplay, g_hWindow, 0, 0, &expose_event);
}

void ClearError ()
{
    sprintf (g_ErrMsg, "");
    XEvent expose_event;
    expose_event.type = Expose;
    XSendEvent (g_pDisplay, g_hWindow, 0, 0, &expose_event);
}

bool ErrorOccurred(LvStatus ErrorStatus)
{
    if (ErrorStatus == LVSTATUS_OK) return false;
    char szMessage[1024];
    LvGetLastErrorMessage(szMessage, sizeof(szMessage));
    DisplayError (szMessage);
    return true;
}

void UpdateStatus ()
{
    if (!g_pCamera) return;

    char szTitle[256];
    sprintf(szTitle, "Camera %s", g_pCamera->IsOpen() ? "open" : "not open");
    if (g_pCamera->IsOpen())
    {
        if (g_pCamera->IsAcquiring())
            strcat(szTitle, ", acquiring.");
        else
            strcat(szTitle, ", not acquiring.");
    }
    SetWindowTitle (szTitle);
}

//-----------------------------------------------------------------------------
// Callback function for the delivered image. Cannot be a method of a class,
// thus we use the pUserParam to set the pointer to the CCamera class instance
// when registering the callback, so that we can then pass the callback to the 
// CallbackNewBuffer() method of CCamera class.

void LV_STDC CallbackNewBufferFunction(LvHBuffer hBuffer,
                                       void* pUserPointer,
                                       void* pUserParam)
{
    CCamera* pCamera = (CCamera*) pUserParam;
    // in UserPointer we hold pointer to buffer
    pCamera->CallbackNewBuffer((LvBuffer*) pUserPointer);
}

//-----------------------------------------------------------------------------
// CCamera constructor

CCamera::CCamera()
{
    m_pSystem   = NULL;
    m_pInterface= NULL;
    m_pDevice   = NULL;
    m_pStream   = NULL;
    --- %%IF IncImgDisplay=1 ------------------
    m_pRenderer = NULL;
    m_pDisplay      = NULL;
    m_hWindow       = 0;
    --- %%ENDIF -------------------------------
    m_pEvent    = NULL;
    --- %%IF IncSimpleImgProcess=1 ------------
    m_bDoProcessing = false;
    --- %%ENDIF -------------------------------
    memset(m_Buffers, 0, sizeof(m_Buffers));
}

//-----------------------------------------------------------------------------
// CCamera destructor

CCamera::~CCamera()
{
    if (m_pDevice != NULL) CloseCamera();
}

//-----------------------------------------------------------------------------

void CCamera::OpenCamera(Display *pDisplay, Window hWindow, LvSystem* pSystem)
{
    if (m_pDevice != NULL) CloseCamera();
    m_pSystem = pSystem;
    --- %%IF IncImgDisplay=1 -------------------
    m_pDisplay = pDisplay;
    m_hWindow = hWindow;
    --- %%ENDIF --------------------------------

    LvStatus SynViewStatus;
    LvInterface* pInterface = NULL;
    LvDevice* pDevice = NULL;
    pSystem->UpdateInterfaceList();
    --- %%IF OpenInterface=ByInterfaceId -------
    SynViewStatus = pSystem->OpenInterface("%%INTERFACE_ID%%", pInterface);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF --------------------------------
    --- %%IF OpenInterface=ByInterfaceTlType ---
    char InterfaceId[256];
    SynViewStatus = pSystem->FindInterface(LvFindBy_TLType, "%%INTERFACE_TLTYPE%%", InterfaceId, sizeof(InterfaceId));
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = pSystem->OpenInterface(InterfaceId, pInterface);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF --------------------------------

    pInterface->UpdateDeviceList();
    --- %%IF OpenDevice=ByDeviceId -------------
    SynViewStatus = pInterface->OpenDevice("%%DEVICE_ID%%", pDevice, LvDeviceAccess_Control);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF --------------------------------
    --- %%IF OpenDevice=ByDeviceModelName ------
    char DeviceId[256];
    SynViewStatus = pInterface->FindDevice(LvFindBy_ModelName, "%%DEVICE_MODELNAME%%", DeviceId, sizeof(DeviceId));
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = pInterface->OpenDevice(DeviceId, pDevice, LvDeviceAccess_Control);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF --------------------------------
    --- %%IF OpenDevice=ByDeviceIpAddress ------
    char DeviceId[256];
    SynViewStatus = pInterface->FindDevice(LvFindBy_GevIPAddress, "%%DEVICE_IPADDRESS%%", DeviceId, sizeof(DeviceId));
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = pInterface->OpenDevice(DeviceId, pDevice, LvDeviceAccess_Control);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF --------------------------------
    --- %%IF OpenDevice=ByDeviceUserId ---------
    char DeviceId[256];
    SynViewStatus = pInterface->FindDevice(LvFindBy_UserID, "%%DEVICE_USERID%%", DeviceId, sizeof(DeviceId));
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = pInterface->OpenDevice(DeviceId, pDevice, LvDeviceAccess_Control);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF --------------------------------
    --- %%IF Sys.AutoLoadedCfg=1 -------------
    // Note: When the code was generated, in SynView Explorer was switched ON the option
    // to autoload the camera settings after opening the camera. For this reason the following
    // line loads the camera settings from the file to assure the same starting status.
    --- %%ENDIF ------------------------------
    --- %%IF Sys.SnippetFile=1 -----------------
    SynViewStatus = pDevice->LoadSettings("", "%%Sys.CameraCfgFile%%", LvSaveFlag_All);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF --------------------------------

    --- %%IF Snippet=All ---------------------
    // The #error line below is intentionally inserted to the code in case you
    // generate the code from streamable or all writable features.
    #error Review the feature settings code and remove the unnecessary items!
    // Before removing this line from the code, go carefully through all the feature
    // settings below and leave there only those, which really need to be set.

    --- %%ENDIF ------------------------------
    --- %%IF Snippet=Streamable --------------
    // The #error line below is intentionally inserted to the code in case you
    // generate the code from streamable or all writable features.
    #error Review the feature settings code and remove the unnecessary items!
    // Before removing this line from the code, go carefully through all the feature
    // settings below and leave there only those, which really need to be set.

    --- %%ENDIF ------------------------------
    %%CameraRemoteFeaturesCode%%

    m_pInterface = pInterface;
    m_pDevice = pDevice;

    --- %%IF IncChunkData=1 ------------------
    --- %%IF FEATURE ChunkModeActive ---
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkModeActive, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%IF CHUNK_ENABLED OffsetX ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_OffsetX);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED OffsetY ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_OffsetY);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED Width ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_Width);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED Height ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_Height);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED PixelFormat ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_PixelFormat);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED LinePitch ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_LinePitch);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED FrameID ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_FrameID);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED Timestamp ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_Timestamp);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED ExposureTime ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_ExposureTime);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED Gain ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_Gain);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED LineStatusAll ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_LineStatusAll);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED LvExternalADCValue ---
    SynViewStatus = m_pDevice->SetEnum (LvDevice_ChunkSelector, LvChunkSelector_LvExternalADCValue);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pDevice->SetBool(LvDevice_ChunkEnable, 1);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
    --- %%ENDIF ------------------------------
    --- %%ENDIF ------------------------------
    --- %%IF IncUniImgPreprocess=1 -------------

    m_pDevice->SetEnum(LvDevice_LvUniProcessMode, LvUniProcessMode_Auto);
    SetOptimalUniPixelFormat();
    --- %%ENDIF --------------------------------
    SynViewStatus = m_pDevice->OpenStream("", m_pStream);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pStream->OpenEvent(LvEventType_NewBuffer, m_pEvent);
    if (ErrorOccurred(SynViewStatus)) return;
    for (int i=0; i<NUMBER_OF_BUFFERS; i++)
    {
        SynViewStatus = m_pStream->OpenBuffer(NULL, 0, NULL, 0, m_Buffers[i]);
        if (ErrorOccurred(SynViewStatus)) return;
        SynViewStatus = m_Buffers[i]->Queue();
        if (ErrorOccurred(SynViewStatus)) return;
    }
    --- %%IF IncImgDisplay=1 ------------------
    SynViewStatus = m_pStream->OpenRenderer(m_pRenderer);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pRenderer->SetWindow(m_pDisplay, m_hWindow);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF -------------------------------
    SynViewStatus = m_pEvent->SetCallbackNewBuffer(CallbackNewBufferFunction, this);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pEvent->StartThread();
    if (ErrorOccurred(SynViewStatus)) return;
}

//-----------------------------------------------------------------------------
// Starts acquisition

void CCamera::StartAcquisition()
{
    if (m_pDevice == NULL) return;
    LvStatus SynViewStatus;
    SynViewStatus = m_pDevice->AcquisitionStart();
    if (ErrorOccurred(SynViewStatus)) return;
}

//-----------------------------------------------------------------------------
// Stops acquisition

void CCamera::StopAcquisition()
{
    if (m_pStream == NULL) return;
    LvStatus SynViewStatus;
    SynViewStatus = m_pDevice->AcquisitionStop();
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pStream->FlushQueue(LvQueueOperation_AllToInput);
    if (ErrorOccurred(SynViewStatus)) return;
}

//-----------------------------------------------------------------------------
// Closes the cameras

void CCamera::CloseCamera()
{
    if (m_pDevice == NULL) return;
    if (IsAcquiring()) StopAcquisition();
    LvStatus SynViewStatus;
    SynViewStatus = m_pEvent->StopThread();
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pStream->CloseEvent(m_pEvent);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%IF IncImgDisplay=1 ------------------
    SynViewStatus = m_pStream->CloseRenderer(m_pRenderer);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF -------------------------------
    SynViewStatus = m_pStream->FlushQueue(LvQueueOperation_AllDiscard);
    if (ErrorOccurred(SynViewStatus)) return;
    for (int i=0; i<NUMBER_OF_BUFFERS; i++)
        if (m_Buffers[i] != NULL)
        {
            SynViewStatus = m_pStream->CloseBuffer(m_Buffers[i]);
            if (ErrorOccurred(SynViewStatus)) return;
        }
    SynViewStatus = m_pDevice->CloseStream(m_pStream);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pInterface->CloseDevice(m_pDevice);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pSystem->CloseInterface(m_pInterface);
    if (ErrorOccurred(SynViewStatus)) return;
}

//-----------------------------------------------------------------------------
// Utility function for enabling/disabling menu items

bool CCamera::IsOpen()
{
    return m_pDevice != NULL;
}

//-----------------------------------------------------------------------------
// Utility function for enabling/disabling menu items

bool CCamera::IsAcquiring()
{
    if (m_pDevice == NULL) return false;
    int32_t iIsAcquiring;
    m_pDevice->GetInt32(LvDevice_LvDeviceIsAcquiring, &iIsAcquiring);
    return iIsAcquiring != 0;
}

--- %%IF IncSimpleImgProcess=1 ------------------
//-----------------------------------------------------------------------------
// Switches ON/OFF image processing done in the callback

void CCamera::SetProcessing(bool bDoProcessing)
{
    m_bDoProcessing = bDoProcessing;
}

--- %%ENDIF -------------------------------------
//-----------------------------------------------------------------------------
// Callback function called for each delivered image

void CCamera::CallbackNewBuffer(LvBuffer* pBuffer)
{
    --- %%IF IncSimpleImgProcess=1 ------------------
    // get the pointer to the image data
    void*   pData = NULL;
    int32_t iImageOffset = 0;
    --- %%IF IncUniImgPreprocess=1 ------------------
    pBuffer->GetPtr (LvBuffer_UniBase, &pData);
    pBuffer->GetInt32 (LvBuffer_UniImageOffset, &iImageOffset);
    --- %%ELSE --------------------------------------
    pBuffer->GetPtr (LvBuffer_Base, &pData);
    pBuffer->GetInt32 (LvBuffer_ImageOffset, &iImageOffset);
    --- %%ENDIF -------------------------------------
    pData = (uint8_t*)pData + iImageOffset;
    if (m_bDoProcessing && pData != NULL)
    {
        // we will do some easy processing - invert the pixel values in an area
        int32_t iWidth;
        int32_t iHeight;
        int32_t iLinePitch;
        LvEnum iPixelFormat;    // LvPixelFormat enumeration value
        m_pDevice->GetInt32(LvDevice_Width, &iWidth);
        m_pDevice->GetInt32(LvDevice_Height, &iHeight);
        --- %%IF IncUniImgPreprocess=1 ------------------
        m_pDevice->GetEnum(LvDevice_LvUniPixelFormat, &iPixelFormat);
        m_pDevice->GetInt32(LvDevice_LvUniLinePitch, &iLinePitch);
        int32_t iBytesPerPixel = ((iPixelFormat & LV_PIX_EFFECTIVE_PIXEL_SIZE_MASK) >> LV_PIX_EFFECTIVE_PIXEL_SIZE_SHIFT) / 8;
        --- %%ELSE --------------------------------------
        m_pDevice->GetEnum(LvDevice_PixelFormat, &iPixelFormat);
        int iBitsPerPixel = (iPixelFormat & LV_PIX_EFFECTIVE_PIXEL_SIZE_MASK) >> LV_PIX_EFFECTIVE_PIXEL_SIZE_SHIFT;
        iLinePitch = ((iWidth * iBitsPerPixel)+7)/8;
        int32_t iBytesPerPixel = iBitsPerPixel / 8;
        --- %%ENDIF -------------------------------------

        for (int32_t j=0; j<(iHeight/2); j++)
        {
            uint8_t* pPixel = ((uint8_t*)pData) + (iHeight/4 + j)*iLinePitch + (iWidth/4)*iBytesPerPixel;
            for (int32_t i=0; i<(iWidth/2); i++)
            {
                for (int32_t k=0; k<iBytesPerPixel; k++)
                {
                    *pPixel = ~*pPixel;
                    pPixel++;
                }
            }
        }
    }
    --- %%ELSE --------------------------------
    // no image processing demonstrated - switch it ON in the code generation Wizard if you need it
    --- %%ENDIF -------------------------------
    --- %%IF IncChunkData=1 ------------------

    // before the chunk data can be read for this buffer, they must be parsed.
    pBuffer->ParseChunkData();
    int64_t ChunkValue;
    char szChunkLine[256];
    szChunkLine[0] = 0;
    char szValue[128];
    --- %%IF CHUNK_ENABLED OffsetX ---
    m_pDevice->GetInt64 (LvDevice_ChunkOffsetX, &ChunkValue);
    sprintf(szValue, "X=%lld | ", ChunkValue);
    strcat(szChunkLine, szValue);
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED OffsetY ---
    m_pDevice->GetInt64 (LvDevice_ChunkOffsetY, &ChunkValue);
    sprintf(szValue, "Y=%lld | ", ChunkValue);
    strcat(szChunkLine, szValue);
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED Width ---
    m_pDevice->GetInt64 (LvDevice_ChunkWidth, &ChunkValue);
    sprintf(szValue, "W=%lld | ", ChunkValue);
    strcat(szChunkLine, szValue);
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED Height ---
    m_pDevice->GetInt64 (LvDevice_ChunkHeight, &ChunkValue);
    sprintf(szValue, "H=%lld | ", ChunkValue);
    strcat(szChunkLine, szValue);
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED PixelFormat ---
    m_pDevice->GetInt64 (LvDevice_ChunkPixelFormat, &ChunkValue);
    sprintf(szValue, "PF=0x%llx | ", ChunkValue);
    strcat(szChunkLine, szValue);
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED LinePitch ---
    m_pDevice->GetInt64 (LvDevice_ChunkLinePitch, &ChunkValue);
    sprintf(szValue, "LPitch=%lld | ", ChunkValue);
    strcat(szChunkLine, szValue);
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED FrameID ---
    m_pDevice->GetInt64 (LvDevice_ChunkFrameID, &ChunkValue);
    sprintf(szValue, "FrameID=%lld | ", ChunkValue);
    strcat(szChunkLine, szValue);
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED Timestamp ---
    m_pDevice->GetInt64 (LvDevice_ChunkTimestamp, &ChunkValue);
    sprintf(szValue, "Time=%lld | ", ChunkValue);
    strcat(szChunkLine, szValue);
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED ExposureTime ---
    m_pDevice->GetInt64 (LvDevice_ChunkExposureTime, &ChunkValue);
    sprintf(szValue, "Exp=%lld | ", ChunkValue);
    strcat(szChunkLine, szValue);
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED Gain ---
    // the IsAvailableEnumEntry() is time consuming, better if you move it outside this callback
    if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkGainSelector, LvChunkGainSelector_AnalogAll) != 0)
    {
        m_pDevice->SetEnum (LvDevice_ChunkGainSelector, LvChunkGainSelector_AnalogAll);
        m_pDevice->GetInt64 (LvDevice_ChunkGain, &ChunkValue);
        sprintf(szValue, "GainA=%lld | ", ChunkValue);
        strcat(szChunkLine, szValue);
    }
    if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkGainSelector, LvChunkGainSelector_DigitalAll) != 0)
    {
        m_pDevice->SetEnum (LvDevice_ChunkGainSelector, LvChunkGainSelector_DigitalAll);
        m_pDevice->GetInt64 (LvDevice_ChunkGain, &ChunkValue);
        sprintf(szValue, "GainD=%lld | ", ChunkValue);
        strcat(szChunkLine, szValue);
    }
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED LineStatusAll ---
    m_pDevice->GetInt64 (LvDevice_ChunkLineStatusAll, &ChunkValue);
    sprintf(szValue, "Line=0x%llx | ", ChunkValue);
    strcat(szChunkLine, szValue);
    --- %%ENDIF ------------------------------
    --- %%IF CHUNK_ENABLED LvExternalADCValue ---
    // the IsAvailableEnumEntry() is time consuming, better if you move it outside this callback
    if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC1) != 0)
    {
        m_pDevice->SetEnum (LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC1);
        m_pDevice->GetInt64 (LvDevice_ChunkLvExternalADCValue, &ChunkValue);
        sprintf(szValue, "ADC1=%lld | ", ChunkValue);
        strcat(szChunkLine, szValue);
    }
    if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC2) != 0)
    {
        m_pDevice->SetEnum (LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC2);
        m_pDevice->GetInt64 (LvDevice_ChunkLvExternalADCValue, &ChunkValue);
        sprintf(szValue, "ADC2=%lld | ", ChunkValue);
        strcat(szChunkLine, szValue);
    }
    if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC3) != 0)
    {
        m_pDevice->SetEnum (LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC3);
        m_pDevice->GetInt64 (LvDevice_ChunkLvExternalADCValue, &ChunkValue);
        sprintf(szValue, "ADC3=%lld | ", ChunkValue);
        strcat(szChunkLine, szValue);
    }
    if (m_pDevice->IsAvailableEnumEntry(LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC4) != 0)
    {
        m_pDevice->SetEnum (LvDevice_ChunkLvExternalADCSelector, LvChunkLvExternalADCSelector_ExternalADC4);
        m_pDevice->GetInt64 (LvDevice_ChunkLvExternalADCValue, &ChunkValue);
        sprintf(szValue, "ADC4=%lld | ", ChunkValue);
        strcat(szChunkLine, szValue);
    }
    --- %%ENDIF ------------------------------
    SetWindowTitle(szChunkLine);
    --- %%ENDIF ------------------------------
    --- %%IF IncImgDisplay=1 ------------------

    m_pRenderer->DisplayImage(pBuffer);
    --- %%ELSE --------------------------------
    // no image display demonstrated - switch it ON in the code generation Wizard if you need it
    --- %%ENDIF -------------------------------
    pBuffer->Queue();
}

--- %%IF IncUniImgPreprocess=1 ------------------
//-----------------------------------------------------------------------------
// Determines optimal pixel format for the uni-process

void CCamera::SetOptimalUniPixelFormat()
{
    LvEnum PixelFormat;
    m_pDevice->GetEnum(LvDevice_PixelFormat, &PixelFormat);
    LvEnum UniPixelFormat = PixelFormat;
    switch (PixelFormat)
    {
        case LvPixelFormat_Mono8:
        case LvPixelFormat_Mono10:
        case LvPixelFormat_Mono12:
        case LvPixelFormat_Mono16:
            UniPixelFormat = LvPixelFormat_Mono8;
            break;

        case LvPixelFormat_BayerGR8:
        case LvPixelFormat_BayerRG8:
        case LvPixelFormat_BayerGB8:
        case LvPixelFormat_BayerBG8:
        case LvPixelFormat_BayerGR10:
        case LvPixelFormat_BayerRG10:
        case LvPixelFormat_BayerGB10:
        case LvPixelFormat_BayerBG10:
        case LvPixelFormat_BayerGR12:
        case LvPixelFormat_BayerRG12:
        case LvPixelFormat_BayerGB12:
        case LvPixelFormat_BayerBG12:
        case LvPixelFormat_RGB8Packed:
        case LvPixelFormat_RGBA8Packed:
            UniPixelFormat = LvPixelFormat_BGRA8Packed;
            break;
    }
    m_pDevice->SetEnum(LvDevice_LvUniPixelFormat, UniPixelFormat);
}

--- %%ENDIF -------------------------------------

//=============================================================================
//=== Xlib application skeleton ===============================================
//=============================================================================

const char* g_pszTitle = "SynView Sample Skeleton: Sample 01";
const char *g_apszMenu[] = {
    "------------------------------------",
    "Exit: X",
    "------------------------------------",
    "Open camera: O",
    "Close camera: C",
    "------------------------------------",
    "Start acquisition: Left mouse button",
    "Stop acquisition: Right mouse button",
    --- %%IF IncSimpleImgProcess=1 ------------------
    "------------------------------------",
    "Toggle image processing: P",
    --- %%ENDIF -------------------------------------
    "------------------------------------",
    0
    };

static void SetWindowTitle (const char *szTitle)
{
    if (!g_hWindow) return;

    XTextProperty win_title;
    XStringListToTextProperty ((char**)&szTitle, 1, &win_title);
    XSetWMName (g_pDisplay, g_hWindow, &win_title);
}

static int ErrorHandler (Display *display, XErrorEvent *event)
{
    if (event->error_code!=BadMatch || event->request_code!=X_PutImage)
    {
        //unknown error, display message and exit
        char error_text[512], message[1024];
        XGetErrorText (display, event->error_code, error_text, 512);
        char msg[256];
        sprintf (msg, "X error occured: %s\n\tMajor opcode: %d\n", error_text, (int) event->request_code);
        DisplayError (msg);
        exit (1);
    }

    // this is our well known bad match caused in XPutImage
    g_pCamera->StopAcquisition();
    DisplayError ("Error displaying the image, unexpected image format, freezing acquisition");
    UpdateStatus ();
    return 0;
}


int main ()
{
    if (LvOpenLibrary() != LVSTATUS_OK)
    {
        DisplayError ("Error: Opening the library failed");
        exit (1);
    }
    if (LvSystem::Open("", g_pSystem) != LVSTATUS_OK)
    {
        DisplayError ("Error: Opening the system failed");
        LvCloseLibrary();
        exit (1);
    }
    // Create one global CCamera class instance
    g_pCamera = new CCamera();
    XInitThreads ();
    g_pDisplay =  XOpenDisplay (NULL);
    int iScreenWidth  = DisplayWidth(g_pDisplay, DefaultScreen(g_pDisplay)); 
    int iScreenHeight = DisplayHeight(g_pDisplay, DefaultScreen(g_pDisplay)); 
    int iAppWidth = (iScreenWidth * 9)/10; // 90% of width 
    int iAppHeight = (iScreenHeight * 9)/10; // 90% of height
    int iAppXOffset = (iScreenWidth - iAppWidth)/2;
    int iAppYOffset = (iScreenHeight - iAppHeight)/2;
    g_hWindow = XCreateSimpleWindow (g_pDisplay, DefaultRootWindow(g_pDisplay), iAppXOffset, iAppYOffset, iAppWidth, iAppHeight, 0, WhitePixel(g_pDisplay, DefaultScreen(g_pDisplay)), WhitePixel(g_pDisplay, DefaultScreen(g_pDisplay)));
    SetWindowTitle (g_pszTitle);
    XSelectInput (g_pDisplay, g_hWindow, KeyPressMask | ButtonPressMask | ExposureMask | StructureNotifyMask);
    Atom protocols[1];
    protocols[0] = XInternAtom (g_pDisplay, "WM_DELETE_WINDOW", True);
    XSetWMProtocols (g_pDisplay, g_hWindow, protocols, 1);
    XSetErrorHandler (ErrorHandler);
    XMapWindow (g_pDisplay, g_hWindow);
    GC gc = XCreateGC (g_pDisplay, g_hWindow, 0, 0);

    XEvent expose_event;
    expose_event.type = Expose;

    XEvent event;
    //message loop
    while (true)
    {
        XNextEvent (g_pDisplay, &event);
        switch (event.type)
        {
            case KeyPress:
            {
                ClearError ();

                KeySym key = XKeycodeToKeysym (g_pDisplay, event.xkey.keycode, 0);
                switch (key)
                {
                    case XK_F1:
                    case XK_Alt_L:
                    case XK_Meta_L:
                    case XK_Alt_R:
                    case XK_Meta_R:
                    {
                        if (g_pCamera) g_pCamera->StopAcquisition();
                        XSendEvent (g_pDisplay, g_hWindow, 0, 0, &expose_event);
                    }
                    break;

                    case XK_x:
                    case XK_X:
                    {
                        if (g_pCamera) 
                        {
                            g_pCamera->StopAcquisition();
                            g_pCamera->CloseCamera();
                            delete g_pCamera;
                            g_pCamera = NULL;
                        }
                        XFreeGC (g_pDisplay, gc);
                        XDestroyWindow (g_pDisplay, g_hWindow);
                        g_hWindow = 0;
                    }
                    break;

                    case XK_o:
                    case XK_O:
                    {
                        g_pCamera->OpenCamera(g_pDisplay, g_hWindow, g_pSystem);
                    }
                    break;

                    case XK_c:
                    case XK_C:
                    {
                        g_pCamera->CloseCamera();
                        XClearWindow (g_pDisplay, g_hWindow);
                    }
                    break;
                    --- %%IF IncSimpleImgProcess=1 ------------------

                    case XK_p:
                    case XK_P:
                    {
                        g_bProcessImage = !g_bProcessImage;
                        g_pCamera->SetProcessing(g_bProcessImage);
                    }
                    break;
                    --- %%ENDIF -------------------------------------
                }

                UpdateStatus ();
            }
            break;
            
            case ButtonPress:
            {
                ClearError ();

                switch (event.xbutton.button)
                {
                    case Button1:
                    {
                        g_pCamera->StartAcquisition();
                    }
                    break;

                    case Button2:
                    case Button3:
                    {
                        g_pCamera->StopAcquisition();
                    }
                    break;
                }

                UpdateStatus ();
            }
            break;

            case Expose:
                if (g_hWindow)
                {
                    XClearWindow (g_pDisplay, g_hWindow);
                    {
                        int x = 20, y = 20, item = 0;
                        while (g_apszMenu[item])
                        {
                            XDrawString (g_pDisplay, g_hWindow, gc, x, y, g_apszMenu[item], strlen (g_apszMenu[item]));
                            y += 20;
                            item++;
                        }
                        y += 100;
                        XDrawString (g_pDisplay, g_hWindow, gc, x, y, g_ErrMsg, strlen (g_ErrMsg));
                    }
                    XFlush (g_pDisplay);
                }
            break;

            case ClientMessage:
                if (strcmp (XGetAtomName(g_pDisplay, event.xclient.data.l[0]), "WM_DELETE_WINDOW") == 0)
                {
                    XDestroyWindow (g_pDisplay, g_hWindow);
                }
            break;

            case UnmapNotify:
                if (g_pCamera)
                {
                    g_pCamera->StopAcquisition();
                    g_pCamera->CloseCamera();
                }
            break;

            case DestroyNotify:
                if (g_pCamera) delete g_pCamera;
                LvSystem::Close(g_pSystem);
                LvCloseLibrary();
                XCloseDisplay (g_pDisplay);
                exit (0);
            break;

        }
    }

    // Cleanup is performed in the Unmap/DestroyNotify handlers, see above
         
    return 0;
}

//-----------------------------------------------------------------------------
