#ifndef SV_SYNVIEW_STATUS_H
#define SV_SYNVIEW_STATUS_H

#include <sv.synview.defs.h>


/** General typedef for the error status. Status values are available as defines
 *  prefixed with LVSTATUS_xxx. Value of 0 (LVSTATUS_OK) indicates no error.
 *  Most SynView functions are returning the status value to indicate the
 *  success of the function call.
 *  See also the LvGetErrorMessage() function.
 *  @ingroup GroupSynview_LvStatus
 */
typedef uint32_t LvStatus;

//------------------------------------------------------------------------------
//--- sv.synview library status codes ------------------------------------------
//------------------------------------------------------------------------------

/** @defgroup GroupSynview_LvStatus LvStatus definitions
 *  @ingroup GroupSynview_Defines
 */

/** @ingroup GroupSynview_LvStatus */
/*@{*/

/** No error. */
#define LVSTATUS_OK                                  0

/** The base library is not loaded. Failure to load the library may be
 *  caused by missing DLLs, check the SynView installation. */
#define LVSTATUS_LIBRARY_NOT_LOADED               1000

/** The SynView library was not open by the LvOpenLibrary() function. */
#define LVSTATUS_LIBRARY_NOT_OPEN                 1001

/** Too many concurrent instances of the AviSaver class. */
#define LVSTATUS_AVISAVER_TOO_MANY_INSTANCES      1010

/** Too many concurrent instances of the Device class. */
#define LVSTATUS_DEVICE_TOO_MANY_INSTANCES        1020

/** The GenTL library cannot be loaded. Check its name specification
 *  and compatibility. */
#define LVSTATUS_CANNOT_LOAD_GENTL                1030

/** This function is disabled, because the callback function was registered. */
#define LVSTATUS_DISABLED_BY_CALLBACK             1040

/** The lv.display library is not loaded. Failure to load this library may be
 *  caused by the operating system environment. */
#define LVSTATUS_DISPLAY_LIBRARY_NOT_LOADED       1050

/** The buffer contents cannot be displayed, either because its pixel
 *  format cannot be converted to a displayable one, or the display options
 *  are not supported by the operating system. */
#define LVSTATUS_DISPLAY_CANNOT_DISPLAY           1055

/** The display is not open - an invalid window handle (and/or in Linux
 *  display pointer) was set. */
#define LVSTATUS_DISPLAY_NOT_OPEN                 1060

/** The requested enum entry name or value does not exist. */
#define LVSTATUS_ENUM_ENTRY_INVALID               1070

/** The requested functionality cannot be used for this Event type. */
#define LVSTATUS_EVENT_NOT_POSSIBLE               1080

/** Too many concurrent instances of the Event class. */
#define LVSTATUS_EVENT_TOO_MANY_INSTANCES         1090

/** Cannot create file. Might be caused by a wrong path specification,
 *  insufficient rights or protected file existence. */
#define LVSTATUS_FILE_CANNOT_CREATE               1100

/** Cannot open file. Might be caused by a wrong path specification,
 *  insufficient rights or file nonexistence. */
#define LVSTATUS_FILE_CANNOT_OPEN                 1110

/** An exception occured when using the GenICam GenApi library. More info
 *  about the exception nature is recorded to the sv.synview.log file. */
#define LVSTATUS_GENICAM_EXCEPTION                1120

/** An invalid handle was passed to a SynView API function. */
#define LVSTATUS_HANDLE_INVALID                   1140

/** The cunk data cannot be parsed, because the ChunkAdapter could not be obtained. */
#define LVSTATUS_CHUNK_ADAPTER_NOT_AVAILABLE      1150

/** The specified index is out of range. */
#define LVSTATUS_INDEX_OUT_OF_RANGE               1160

/** Too many concurrent instances of the Interface class. */
#define LVSTATUS_INTERFACE_TOO_MANY_INSTANCES     1180

/** This function cannot be used in this module. */
#define LVSTATUS_INVALID_IN_THIS_MODULE           1190

/** Invalid FtrGroup specified. */
#define LVSTATUS_ITEM_GROUP_INVALID               1200

/** Invalid Item ID specified. */
#define LVSTATUS_ITEM_INVALID                     1210

/** SynView constant does not exists for this enum entry.
  * Use the enum entry string identifier to handle it. */
#define LVSTATUS_NO_CONSTANT_FOR_THIS_ENUMENTRY   1211

/** The specified enum entry ID is not a valid SynView constant. Use the enum entry 
 *  string identifiers to handle enum entries which do not have a SynView constant defined. */
#define LVSTATUS_INVALID_ENUMENTRY_ID             1212

/** This function is not applicable to this item. */
#define LVSTATUS_ITEM_NOT_APPLICABLE              1220

/** This function requires availability of specific item, but it is not available.
 *  More info in the sv.synview.log file. */
#define LVSTATUS_ITEM_NOT_AVAILABLE               1230

/** The item is not readable. */
#define LVSTATUS_ITEM_NOT_READABLE                1240

/** The item is not writable. */
#define LVSTATUS_ITEM_NOT_WRITABLE                1250

/** Cannot obtain a feature node map for the device. More info in the
 *  sv.synview.log file. */
#define LVSTATUS_NODE_MAP_CANNOT_GET              1260

/** The functionality is not implemented for the requested parameters. */
#define LVSTATUS_NOT_IMPLEMENTED                  1270

/** Invalid parameter passed to a SynView API function (for example an
 *  invalid pointer). */
#define LVSTATUS_PARAMETER_INVALID                1280

/** Too many concurrent instances of the Renderer class. */
#define LVSTATUS_RENDERER_TOO_MANY_INSTANCES      1290

/** The stream was already started. */
#define LVSTATUS_STREAM_ALREADY_STARTED           1300

/** The stream was already stopped. */
#define LVSTATUS_STREAM_ALREADY_STOPPED           1310

/** Too many concurrent instances of the Stream class. */
#define LVSTATUS_STREAM_TOO_MANY_INSTANCES        1320

/** Too many concurrent instances of the System class. */
#define LVSTATUS_SYSTEM_TOO_MANY_INSTANCES        1330

/** The device is not accessible. It is either used by other application or,
  * if it is a GigE device, 
  * it can be on a different network or there can be an IP address confict.*/
#define LVSTATUS_DEVICE_NOT_ACCESSIBLE            1340

/** The device (camera) does not permit a read-write access. The application 
  * attempted to open the device for controllable or exclusive access, but it
  * is probably currently used by another application. */
#define LVSTATUS_DEVICE_NOT_READWRITE             1341

/** The requested function is not supported by this event type. */
#define LVSTATUS_NOT_SUPPORTED_FOR_THIS_EVENT     1350

/** The number of allocated buffers is smaller than required minimum 
 *  number of buffers. See also the @ref LvStream_StreamAnnounceBufferMinimum. */
#define LVSTATUS_NOT_ENOUGH_BUFFERS               1360

/** Some of the allocated buffers are of smaller size, than is the payload size 
 *  needed for the acquisition. See also @ref LvStream_LvCalcPayloadSize.  */
#define LVSTATUS_INSUFFICIENT_BUFFER_SIZE         1370

/** The IP or MAC Address used in the LvSetString() has wrong format. The proper  
 *  format is N.N.N.N for IP address and XX:XX:XX:XX:XX:XX for MAC address, where
 *  N is decadic number between 0 and 255, XX is 2 digit hexdecimal number.*/
#define LVSTATUS_INVALID_IP_OR_MAC_ADDRESS_FORMAT 1380

/** The XML file with feature definitions could not be loaded. This may indicate
 *  a misconfiguration of the SynView system - reinstallation recommended. */
#define LVSTATUS_CANNOT_LOAD_XML                  1390

/** The string buffer passed to the function does not have the size big enough 
 *  to hold the returned string. */
#define LVSTATUS_INSUFFICIENT_STRING_BUFFER_SIZE  1400

/** The Interface or Device was not found according to the search criteria. */
#define LVSTATUS_NOT_FOUND                        1410

/** A parameter passed to the function is not applicable at this function. 
 *  For example in the LvSystemFindInterface() the constant LvFindBy_UserID 
 *  is not applicable, because this constant can be used only for devices.
 */
#define LVSTATUS_PARAM_NOT_APPLICABLE             1420

/** The requested enumeration entry is not available. */
#define LVSTATUS_ENUM_ENTRY_NOT_AVAILABLE         1430

/** The function has returned because a timeout has expired. */
#define LVSTATUS_TIMEOUT                          1500

/** The lookup table is not available for the current pixel format. */
#define LVSTATUS_LUT_NOT_AVAILABLE                1510

/** The lookup table has unsupported size. */
#define LVSTATUS_LUT_UNSUPPORTED_SIZE             1520

/** The XML file with camera remote features could not be extracted from the ZIP file. */
#define LVSTATUS_XML_UNZIP_ERROR                  1530

/** The AcquisitionStart command cannot be executed. This is probably because the acquisition 
 *  is already running, or the conditions for starting the acquisition are not satisfied. */
#define LVSTATUS_ACQUISITION_CANNOT_BE_STARTED    1540

/** The AcquisitionStop command cannot be executed. This is probably because the acquisition 
 *  already had been stopped, or the conditions for stopping the acquisition are not satisfied. */
#define LVSTATUS_ACQUISITION_CANNOT_BE_STOPPED    1550

/** The file with settings was created by different remote device vendor/model. This 
 *  may cause its incompatibility with the current remote device. */
#define LVSTATUS_SETTINGS_INCOMPATIBLE_MODEL      1560

/** The file with settings was created by different remote device firmware version. This 
 *  may cause its incompatibility with the current remote device. */
#define LVSTATUS_SETTINGS_INCOMPATIBLE_VERSION    1570

/** The file with settings were saved with an ID. The ID specified when loading is different. */
#define LVSTATUS_SETTINGS_INCOMPATIBLE_ID         1580

/** The operation on the buffer is not possible, the buffer is queued for acquisition. */
#define LVSTATUS_BUFFER_IS_QUEUED                 1590

/** The buffer was not yet filled with image data. */
#define LVSTATUS_BUFFER_NOT_FILLED                1600

/** The SynView library cannot be reopened after it was once closed. */
#define LVSTATUS_CANNOT_REOPEN_LIBRARY            1610

// GenTL errors 
/** The GenTL library returned the GC_ERR_UNKNOWN error. More info in the
 *  sv.synview.log file. */
#define LVSTATUS_GC_UNKNOWN                       2000 

/** The GenTL library returned the GC_ERR_ERROR error. More info in the
 *  sv.synview.log file. */
#define LVSTATUS_GC_ERROR                         2001 

/** The GenTL library returned the GC_ERR_NOT_INITIALIZED error. More info in
 *  the sv.synview.log file. */
#define LVSTATUS_GC_NOT_INITIALIZED               2002 

/** The GenTL library returned the GC_ERR_NOT_IMPLEMENTED error. More info in
 *  the sv.synview.log file. */
#define LVSTATUS_GC_NOT_IMPLEMENTED               2003 

/** The GenTL library returned the GC_ERR_RESOURCE_IN_USE error. More info in
 *  the sv.synview.log file. */
#define LVSTATUS_GC_RESOURCE_IN_USE               2004 

/** The GenTL library returned the GC_ERR_ACCESS_DENIED error. More info in
 *  the sv.synview.log file. */
#define LVSTATUS_GC_ACCESS_DENIED                 2005 

/** The GenTL library returned the GC_ERR_INVALID_HANDLE error. More info in
 *  the sv.synview.log file. */
#define LVSTATUS_GC_INVALID_HANDLE                2006 

/** The GenTL library returned the GC_ERR_INVALID_ID error. More info in the
 *  sv.synview.log file. */
#define LVSTATUS_GC_INVALID_ID                    2007 

/** The GenTL library returned the GC_ERR_NO_DATA error. More info in the
 *  sv.synview.log file. */
#define LVSTATUS_GC_NO_DATA                       2008 

/** The GenTL library returned the GC_ERR_INVALID_PARAMETER error. More info
 *  in the sv.synview.log file. */
#define LVSTATUS_GC_INVALID_PARAMETER             2009 

/** The GenTL library returned the GC_ERR_IO error. More info in the
 *  sv.synview.log file. */
#define LVSTATUS_GC_IO                            2010 

/** The GenTL library returned the GC_ERR_TIMEOUT error. More info in the
 *  sv.synview.log file. */
#define LVSTATUS_GC_TIMEOUT                       2011

/** The GenTL library returned the GC_ERR_ABORT error. More info in the
 *  sv.synview.log file. */
#define LVSTATUS_GC_ABORT                         2012 

/** The GenTL library returned the GC_ERR_INVALID_BUFFER error. More info in the
 *  sv.synview.log file. */
#define LVSTATUS_GC_INVALID_BUFFER                2013 

/** The GenTL library returned the GC_ERR_NOT_AVAILABLE error. More info in the
 *  sv.synview.log file. */
#define LVSTATUS_GC_NOT_AVAILABLE                 2014 

/** The GenTL library returned the GC_ERR_INVALID_ADDRESS error. More info in the
 *  sv.synview.log file. */
#define LVSTATUS_GC_INVALID_ADDRESS               2015 

/** The GenTL library returned the GC_ERR_BUFFER_TOO_SMALL error. More info in the
 *  sv.synview.log file. */
#define LVSTATUS_GC_BUFFER_TOO_SMALL              2016 

/** The GenTL library returned the GC_ERR_INVALID_INDEX error. More info in the
 *  sv.synview.log file. */
#define LVSTATUS_GC_INVALID_INDEX                 2017 

/** The GenTL library returned the GC_ERR_PARSING_CHUNK_DATA error. More info in the
 *  sv.synview.log file. */
#define LVSTATUS_GC_PARSING_CHUNK_DATA            2018 

/** The GenTL library returned the GC_ERR_INVALID_VALUE error. More info in the
 *  sv.synview.log file. */
#define LVSTATUS_GC_INVALID_VALUE                 2019 

/** The GenTL library returned the GC_ERR_RESOURCE_EXHAUSTED error. More info in the
 *  sv.synview.log file. */
#define LVSTATUS_GC_RESOURCE_EXHAUSTED            2020 

/** The GenTL library returned the GC_ERR_OUT_OF_MEMORY error. More info in the
 *  sv.synview.log file. */
#define LVSTATUS_GC_OUT_OF_MEMORY                 2021 

/** The GenTL library returned the GC_ERR_BUSY error. More info in the
 *  sv.synview.log file. */
#define LVSTATUS_GC_BUSY                          2022 

/** The GenTL library returned the GC_ERR_CUSTOM_ID error. More info in the
 *  sv.synview.log file. */
#define LVSTATUS_GC_CUSTOM_ID                     2100 
/** The GenTL library returned the LVSTATUS_GC_GIGEVERSION_NOT_SUPPORTED error.
 * More info in the sv.synview.log file. */
#define LVSTATUS_GC_GIGEVERSION_NOT_SUPPORTED     2100 

// SrcGen errors 
/** The template for requested item is not available. */
#define LVSTATUS_SRCGEN_TEMPLATE_NOT_AVAILABLE    3000

/** The symbolic for requested item is not available. */
#define LVSTATUS_SRCGEN_SYMBOLIC_NOT_AVAILABLE    3001

/** License error. License not available */
#define LVSTATUS_LICENSE_NOT_AVAILABLE            4000

/** The last error status could not be recorded. This is most probably caused by 
 *  too many threads used by the application. */
#define LVSTATUS_LAST_ERROR_NOT_AVAILABLE         9998

/** Undefined error. More info in the sv.synview.log file. */
#define LVSTATUS_ERROR                            9999

/*@}*/



//------------------------------------------------------------------------------
//--- sv.imgproc library status codes ------------------------------------------
//------------------------------------------------------------------------------

/** @defgroup GroupImgproc_LvStatus LvStatus definitions
 *  @ingroup GroupImgproc_Defines
 */

/** @ingroup GroupImgproc_LvStatus */
/*@{*/

/** Invalid pointer. One of the pointers used by the function is NULL or invalid.
 */
#define LVSTATUS_LVIP_INVALID_POINTER                              100001
/** Pointer to the source data image info or its data is invalid.
 */
#define LVSTATUS_LVIP_INVALID_SRC_POINTER                          100002
/** Pointer to the destination data image info or its data is invalid.
 */
#define LVSTATUS_LVIP_INVALID_DST_POINTER                          100003
/** The source or destination image info has invalid or unexpected pixel format or the dwPixelIncrement value
 * in the of @ref LvipImgInfo structure.
 * Another reason when this error code could be retrieved is when using the LVIP_BmpInfoToImgInfo() function
 * with BITMAPINFOHEADER containing in its \a biBitCount member a different value than is supported.
 * Supported values in BITMAPINFOHEADER are:
 * @code
 *   8            LVIP_PIXEL_FORMAT_MONO8
 *   16           LVIP_PIXEL_FORMAT_RGB555_PACKED or LVIP_PIXEL_FORMAT_RGB565_PACKED
 *                (depends on biCompression member of the BITMAPINFOHEADER)
 *   24           LVIP_PIXEL_FORMAT_RGB8_PACKED
 *   32           LVIP_PIXEL_FORMAT_RGBA8_PACKED
 * @endcode
 */
#define LVSTATUS_LVIP_INVALID_PIXEL_FORMAT                         100004
/** The LvipImgInfo parameter is not initialized.
 * The supplied @ref LvipImgInfo parameter has invalid contents. Either initialize it
 * with the LVIP_InitImgInfo() function or setup in your code all the members to appropriate values.
 */
#define LVSTATUS_LVIP_IMAGEINFO_NOT_INITIALIZED                    100005
/** Memory allocation failed.
 * This error code happens when the operating system does not allow to allocate any new memory to this library.
 * When this error happens, there is a critical insufficience of memory; it might indicate
 * a huge memory leak, typically caused by not deallocating used images when processing in a loop.
 * Another cause could be big image dimensions resulting in an attempt to allocate a huge memory amount.
 * Typically this error code could be retrieved from LVIP_AllocateImageData() or LVIP_SaveToTiff() functions.
 */  
#define LVSTATUS_LVIP_MEMORY_ALLOC_FAILED                          100006
/** BMP header is unsupported.
 * Some members of BITMAPINFOHEADER (part of BITMAPINFO) have unexpected or unsupported values.
 * Check if the BITMAPINFO has correct data and that is correctly filled up.
 * Members \a biSize and \a biCompression have to be correctly filled up. The biCompression member has to be filled up with 
 * BI_RGB or BI_BITFIELDS values.
 */
#define LVSTATUS_LVIP_UNSUPPORTED_BMP_HEADER                       100007
/** Source pixel format is incompatible to BITMAP pixel format possibilities.
 * BITMAP could be created only using one of these pixel formats:
 * @code
 *    LVIP_PIXEL_FORMAT_MONO8
 *    LVIP_PIXEL_FORMAT_RGB555_PACKED
 *    LVIP_PIXEL_FORMAT_RGB565_PACKED
 *    LVIP_PIXEL_FORMAT_RGB8_PACKED
 *    LVIP_PIXEL_FORMAT_RGBA8_PACKED
 * @endcode
 * If you have the image in other pixel format, you should convert the image by the
 * LVIP_ConvertToPixelFormat() function.
 */
#define LVSTATUS_LVIP_BMP_INCOMPATIBLE_PIXEL_FORMAT                100008
/** Color format has incompatible line increment to BMP possibilities.
 * The BMP has to have line increment aligned to 4 bytes.
 */
#define LVSTATUS_LVIP_BMP_INCOMPATIBLE_LINE_INCREMENT              100009
/** Image info not equal.
 */
#define LVSTATUS_LVIP_IMAGEINFO_NOT_EQUAL                          100011
/** The requested function or format is not supported.
 * If you are not sure which functionality is not supported, see New Electronic Technology 
 * Log Messages Receiver application for details.
 */
#define LVSTATUS_LVIP_UNSUPPORTED                                  100012
/** Unsupported pixel format of the source.
 * See documentation of the function which returns this error code for supported source pixel formats.
 */
#define LVSTATUS_LVIP_UNSUPPORTED_SRC_PIXEL_FORMAT                 100013
/** Unsupported pixel format of the destination.
 * See documentation of function which returns this error code for supported destination pixel formats.
 */
#define LVSTATUS_LVIP_UNSUPPORTED_DST_PIXEL_FORMAT                 100014
/** The image uses color planes and the called function doesn't support it.
 */
#define LVSTATUS_LVIP_UNSUPPORTED_COLOR_PLANES                     100015
/** The image uses reversion and the called function doesn't support it.
 */
#define LVSTATUS_LVIP_UNSUPPORTED_REVERSION                        100016
/** Image line increment is too big.
 * This some functions support only limited line increment. 
 * This currently applies only to LVIP_Deinterlace() function, which has a limit of 
 * 2048 * 32bit RGB image - it means that line increment could be more than 8192 bytes.
 */
#define LVSTATUS_LVIP_LINEINCREMENT_TOO_BIG                        100017
/** Destination image info is incompatible to expected output format.
 * You are trying to call a function and the supplied destination image info has
 * other than expected contents.
 *
 * Possible reason is that you don'did not specify the @ref LvipOption_ReallocateDst flag, so
 * if the function has different output format than specified in the destination
 * image info, the image info could not be rellocated and this error happens.
 *
 * Note that some functions may require different destination parameters than expected,
 * the size of the destination image in the LVIP_CopyArea() function depends on
 * whether the desired rectangle lies fully in the image or not.
 */
#define LVSTATUS_LVIP_DST_IMG_INFO_INCOMPATIBLE                    100018
/** Destination image info size is different from the source image info size.
 * It means that the function expects that the source and destination image info size is the same. Because it is not,
 * and the destination image info cannot be changed using the @ref LvipOption_ReallocateDst flag, this error happens.
 */
#define LVSTATUS_LVIP_INCOMPATIBLE_SRC_AND_DST_SIZE                100020
/** Rotated destination image info size is different to the expected size.
 * It means that the function expects that the source width of image info has to be the same to the height of the destination
 * image info (and vice versa) - and it is not - and additionaly destination image info cannot be changed using
 * the @ref LvipOption_ReallocateDst flag.
 */
#define LVSTATUS_LVIP_INCOMPATIBLE_SRC_AND_DST_SIZE_ROTATED        100021
/** The source and destination image info have different pixel format.
 * You are trying to call some function which expects that both of source and destination image info has the same pixel format.
 * But the destination image info has different pixel format set and it is not
 * permitted to change this (by the @ref LvipOption_ReallocateDst flag).
 */
#define LVSTATUS_LVIP_INCOMPATIBLE_SRC_AND_DST_PIXEL_FORMAT        100023
/** Source and destination image info have incompatible flags.
 * Some functions need to have both of source and destination image info flags compatible.
 * This applies for example for the bitmap orientation (top-down versus bottom-up).
 */
#define LVSTATUS_LVIP_INCOMPATIBLE_SRC_AND_DST_FLAGS               100024
/** The specified rectangle is outside of source image data.
 * It means that you are trying to copy an area which not exist in the source image.
 * To create a destination image, the rectangle must at least partially overlap the source image.
 */
#define LVSTATUS_LVIP_DST_RECT_OUTSIDE_SRC                         100025
/** The source image info structure has no data.
 * The source image info has no image data - it means that the source image info does not point to any valid image data.
 *
 * Use LVIP_AllocateImageData() function to allocate the buffer for the image or point
 * the pData member(s) to a valid image.
 */
#define LVSTATUS_LVIP_SRC_IMAGEINFO_NO_DATA                        100026
/** The destination image info structure has no data.
 * The destination image info has no image data - it means that the destination image info does not point to any valid image data
 * and it is not permitted to allocate the buffer automatically (the @ref LvipOption_ReallocateDst flag was not specified).
 */
#define LVSTATUS_LVIP_DST_IMAGEINFO_NO_DATA                        100027
/** The image isn't in the displayable format.
 * If there is a need to display an image, there is a need to have this image in a displayable format
 * (in Windows one of the BMP pixel formats and the line increment alighed to 4 bytes).
 * The image does not have such format and the automatic conversion to a displayable format
 * was not enabled (using the third parameter of the LVIP_DisplayImage() function and
 * optionally the @ref LvipOption_ReallocateDst flag).
 */
#define LVSTATUS_LVIP_NOT_DISPLAYABLE_FORMAT                       100028
/** Invalid LUT handle passed as a parameter of the function.
 * The LUT is in incompatible format which couldn't be used in this function.
 *
 * General this happens when trying to use some Bayer-decoding function and a non-Bayer LUT is used
 * or the LUT has been created for the different pixel format.
 */
#define LVSTATUS_LVIP_INVALID_LUT_HANDLE                           100029
/** Invalid LUT type passed as a parameter of the function.
 * The LUT is in incompatible format which couldn't be used in this function.
 *
 * General this happens when trying to use some Bayer-decoding function and a non-Bayer LUT is used
 * or the LUT has been created for the different pixel format.
 */
#define LVSTATUS_LVIP_INVALID_LUT_TYPE                             100030
/** The reference image is in incompatible pixel format.
 * When using the reference image info (in the LVIP_ApplyShadingCorrection() function), it must be
 * in the same pixel format as the source image.
 */
#define LVSTATUS_LVIP_INCOMPATIBLE_REF_PIXEL_FORMAT                100032
/** The reference image has incompatible flags.
 * When using the reference image info (in the LVIP_ApplyShadingCorrection() function), it must have
 * compatible flags as the source image.
 * For example: @ref LvipImgAttr_BottomUp flags must be same.
 */
#define LVSTATUS_LVIP_INCOMPATIBLE_REF_FLAGS                       100033
/** A system error occured when trying to open or read the file.
 * Possible reasons could be a wrong path and/or file name, the file currently locked or without rights to be opened/read
 * or the file does not exist yet.
 */
#define LVSTATUS_LVIP_CANNOT_OPEN_READ_FILE                        100034
/** A system error occured when trying to create or write to the file.
 * Possible reasons could be an invalid path and/or file name, without rights to be created/written.
 */
#define LVSTATUS_LVIP_CANNOT_CREATE_WRITE_FILE                     100035
/** The contents of the TIFF file is in the invalid or in the unsupported form.
 * You are probably trying to read a TIFF file, which was not created by this library (see the LVIP_SaveToTiff() function)
 */
#define LVSTATUS_LVIP_TIFF_CONTENTS_INVALID                        100036
/** The contents of the BMP file is in the invalid or in the unsupported form.
 * You are trying to read a BMP file which has different (or invalid) format than which is
 * supported by this library.
 */
#define LVSTATUS_LVIP_BMP_CONTENTS_INVALID                         100037

/** The PixelFormat is not BayerArray
 */
#define LVSTATUS_LVIP_NOT_BAYER_PIXEL_FORMAT                       100038

/** Saving to JPEG failed. More info in the LOG file.
 */
#define LVSTATUS_LVIP_JPEG_SAVE_FAILED                             100039

/** Loading from JPEG failed. More info in the LOG file.
 */
#define LVSTATUS_LVIP_JPEG_LOAD_FAILED                             100040

/*@}*/



#endif
