#ifndef SV_SYNVIEW_H
#define SV_SYNVIEW_H

#include "sv.synview.defs.h"
#include "sv.synview.enums.h"
#include "sv.synview.imgproc.h"

//------------------------------------------------------------------------------

/** @defgroup GroupSynview_PlainC SynView Plain C API functions
 *  @ingroup GroupSynview
 */

/** @defgroup GroupSynview_GeneralFunctions  General purpose functions
 *  @ingroup GroupSynview_PlainC
 */

/** @defgroup GroupSynview_SystemFunctions  System module functions
 *  @ingroup GroupSynview_PlainC
 */

/** @defgroup GroupSynview_InterfaceFunctions  Interface module functions
 *  @ingroup GroupSynview_PlainC
 */

/** @defgroup GroupSynview_DeviceFunctions  Device module functions
 *  @ingroup GroupSynview_PlainC
 */

/** @defgroup GroupSynview_StreamFunctions  Stream module functions
 *  @ingroup GroupSynview_PlainC
 */

/** @defgroup GroupSynview_BufferFunctions  Buffer module functions
 *  @ingroup GroupSynview_PlainC
 */

/** @defgroup GroupSynview_EventFunctions  Event module functions
 *  @ingroup GroupSynview_PlainC
 */

/** @defgroup GroupSynview_RendererFunctions  Renderer module functions
 *  @ingroup GroupSynview_PlainC
 */

/** @defgroup GroupSynview_FeatureFunctions  Feature control functions
 *  @ingroup GroupSynview_PlainC
 */

/** @defgroup GroupSynview_FirmwareFunctions  Firmware update functions
 *  @ingroup GroupSynview_DeviceFunctions
 */

//------------------------------------------------------------------------------
/** Returns SynView version.
 *  @return The returned doubleword contains the build version
 *          in the low word and the high word is the major version in the
 *          upper byte and subversion in the lower byte.
 *  For example:
 *  @code
 *  uint32_t Version = LvGetVersion();
 *  printf("SynView %d.%02d.%03d",
 *          ((Version >> 24) & 0xFF),
 *          ((Version >> 16) & 0xFF),
 *          (Version & 0xFFFF));
 *  @endcode
 *  @ingroup GroupSynview_GeneralFunctions
 */
LV_DLLENTRY uint32_t LV_STDC LvGetVersion();

//------------------------------------------------------------------------------
/** Opens the SynView library. This must be done before you can use any other
 *  SynView function (with the exception of LvGetVersion() and LvGetErrorMessage()).
 *  If you are using SynView in Windows DLL, avoid calling this in Windows DllMain()
 *  function - for proper functionality this function must be called when the
 *  application or DLL is already fully initialized and there are no restrictions
 *  about synchronization (DllMain has such restrictions).
 *  If you call this function multiple times, you must balance it by the same
 *  number of the LvCloseLibrary() calls. Only the first call will actually do
 *  the initialization. IMPORTANT: The library must not be opened again
 *  once it was already uninitialized.
 *
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_GeneralFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvOpenLibrary();

//------------------------------------------------------------------------------
/** Closes the SynView library. This must be done before you exit your
 *  application. Be sure to close first all dependent modules (System).
 *  If you are using SynView in a Windows DLL, avoid calling this in Windows DllMain()
 *  function - for proper functionality this function must be called when the
 *  application or DLL is still fully functional, which is not the case of
 *  PROCESS_DETACH in the DllMain(). If you have called LvOpenLibrary() multiple
 *  times, you must balance it by the same number of calls of this function. Only the last call
 *  actually does the uninitialization. IMPORTANT: The library must not be opened again
 *  once it was already uninitialized.
 *
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_GeneralFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvCloseLibrary();

//------------------------------------------------------------------------------
/** Returns a short description of the error. Note that only some of the errors are
 *  suitable for direct display to the user, many error values indicate states
 *  which are understandable to the programmer, but may not be understandable to
 *  the end user.
 *  @param  Error     The error code (the return value of most SynView functions).
 *  @param  pMessage  Pointer to the text buffer.
 *  @param  Size      Size of the buffer.
 *
 *  @sa @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_GeneralFunctions
 */
LV_DLLENTRY void LV_STDC LvGetErrorMessage (LvStatus Error,
                                            char* pMessage, 
                                            size_t Size);
                               
//------------------------------------------------------------------------------
/** Returns more detailed description of the last error, which happened in the thread
 *  from which this function was called. As the info is recorded inside SynView for each 
 *  error, the desription provides more detailed info, including the name of the function, in which 
 *  the error happened, and possibly more diagnostic info. The difference to LvGetErrorMessage() 
 *  is that LvGetErrorMessage() returns a static string from a numbered table of errors
 *  while this function returns additionally info recorded at the time the error happened.
 *  If a function returns LVSTATUS_OK, it does not reset this error message (for speed reasons)
 *  so the correct approach is to get the error number as the function return value and if this
 *  return value is not LVSTATUS_OK, then you can get more info about the error using this 
 *  function. be sure to call it from the same thread.
 *  @param  pMessage  Pointer to the text buffer.
 *  @param  Size      Size of the buffer.
 *
 *  @sa @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_GeneralFunctions
 */
LV_DLLENTRY void LV_STDC LvGetLastErrorMessage (char* pMessage, 
                                                size_t Size);
                               
//------------------------------------------------------------------------------
/** Adds a line to the sv.synview.log. The SynView log is a tool for New Electronic
 *  Technology technical support, in some cases may be useful to put to the log
 *  additional info from your code.
 *  @param  pLogMessage  Pointer to the null teminated string with the message.
 *  @ingroup GroupSynview_GeneralFunctions
 */
LV_DLLENTRY void LV_STDC LvLog (const char* pLogMessage);

//------------------------------------------------------------------------------
/** Gets a general info in form of a 32-bit integer value.
 *  @param   Info      One of the @ref LvLibInfo values.
 *  @param   pInfo     The value is returned in this parameter.
 *  @param   Param     Additional parameter, required by some types of info.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_GeneralFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvGetLibInfo (LvEnum Info,
                                           int32_t* pInfo,
                                           int32_t Param = 0);
#else
LV_DLLENTRY LvStatus LV_STDC LvGetLibInfo (LvEnum Info,
                                           int32_t* pInfo,
                                           int32_t Param);
#endif

//------------------------------------------------------------------------------
/** Gets a general info in form of a string value.
 *  @param   Info      One of the @ref LvLibInfo values.
 *  @param   pInfoStr  The string value is returned in this parameter.
 *  @param   Size      Size of the buffer (to which pInfoStr points).
 *  @param   Param     Additional parameter, required by some types of info.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_GeneralFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvGetLibInfoStr (LvEnum Info,
                                              char* pInfoStr,
                                              size_t Size,
                                              int32_t Param = 0);
#else
LV_DLLENTRY LvStatus LV_STDC LvGetLibInfoStr (LvEnum Info,
                                              char* pInfoStr,
                                              size_t Size,
                                              int32_t Param);
#endif

//------------------------------------------------------------------------------
/** Gets a buffer size needed for a general info in form of a string value.
 *  @param   Info      One of the @ref LvLibInfo values.
 *  @param   pSize     Size of the buffer is returned in this parameter.
 *  @param   Param     Additional parameter, required by some types of info.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_GeneralFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvGetLibInfoStrSize (LvEnum Info,
                                                  size_t* pSize,
                                                  int32_t Param = 0);
#else
LV_DLLENTRY LvStatus LV_STDC LvGetLibInfoStrSize (LvEnum Info,
                                                  size_t* pSize,
                                                  int32_t Param);
#endif

//##############################################################################
//  LvSystem
//##############################################################################

/** Updates the list of systems available. This function must be called before iterating
 *  through the systems by the LvGetNumberOfSystems() and LvGetSystemId() functions.
 *  The systems are physically represented by GenTL libraries available in the
 *  operating systems, this call searches for them in standard locations. See also
 *  the description of the sv.synview.ini file in the SynView User's Guide.
 *  Note that this function is seldom needed, most applications will work with the
 *  default system (see LvSystemOpen() for details).
 *
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_SystemFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvUpdateSystemList ();

//------------------------------------------------------------------------------
/** Returns the number of systems found after the LvUpdateSystemList() call.
 *  Typical use of this function is in iterating systems using the LvGetSystemId()
 *  function.
 *
 *  @param   pNumberOfSystems  The number of systems found.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_SystemFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvGetNumberOfSystems (uint32_t* pNumberOfSystems);

//------------------------------------------------------------------------------
/** Returns the string ID of the system at given index. This ID is used in the
 *  LvSystemOpen() function for opening the system.
 *  @param   Index      Zero-based index of the system, a value >= 0 and < number of systems,
 *                      returned by the LvGetNumberOfSystems() function.
 *  @param   pSystemId  Pointer to a string buffer, where the system ID will be placed.
 *  @param   Size       Size of the buffer.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_SystemFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvGetSystemId (uint32_t Index,
                                            char* pSystemId,
                                            size_t Size);

//------------------------------------------------------------------------------
/** Returns the size of the string buffer needed to hold the system ID string,
 *  including the terminating zero character.
 *  @param   Index      Zero-based index of the system, a value >= 0 and < number of systems,
 *                      returned by the LvGetNumberOfSystems() function.
 *  @param   pSize      Size of the buffer is returned in this parameter.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_SystemFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvGetSystemIdSize (uint32_t Index,
                                                size_t* pSize);

//------------------------------------------------------------------------------
/** Opens the System module. Opening the system actually means loading the
 *  corresponding GenTL library. Note that before you can open the System, the
 *  LvOpenLibrary() must be called. The same system can be open multiple times
 *  (there is a reference counter inside); in such case there must be also
 *  the same number of LvSystemClose() calls used (every open increase the
 *  reference count and every close decreases it).
 *  @param   pSystemId  A string ID of the system. This can be either an empty string - then
 *                      the default system is opened, or it can be a string obtained from the
 *                      LvGetSystemId() function.
 *  @param   phSystem   Pointer to a handle to the opened System module.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_SystemFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvSystemOpen (const char* pSystemId,
                                           LvHSystem* phSystem);

//------------------------------------------------------------------------------
/** Closes the opened system. Actually it means freeing the coresponding GenTL
 *  library. Be sure you first close all dependent modules (Interface, Event etc.).
 *  If the System was opened multiple times, it only decreases the reference counter
 *  (see the note by the LvSystemOpen()).
 *  @param   phSystem  Pointer to a handle to the System module, obtained from the
 *                     LvSystemOpen() function. The handle is assigned 0 after
 *                     the operation.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_SystemFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvSystemClose (LvHSystem* phSystem);

//------------------------------------------------------------------------------
/** Updates the internal list of available interfaces. You can then iterate
 *  through them by LvSystemGetNumberOfInterfaces() and LvSystemGetInterfaceId().
 *  @param   hSystem   A handle to the System module, obtained from the
 *                     LvSystemOpen() function.
 *  @param   Timeout   Specifies a timeout in ms for searching the interfaces. This applies
 *                     only to special cases of interfaces, where some delay can happen;
 *                     common interfaces are detected without any significant delays.
 *  @return  If the timeout has expired while waiting for the completion, the function returns 
             @ref LVSTATUS_TIMEOUT, otherwise it returns the @ref LvStatus value indicating the 
             success of the requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_SystemFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvSystemUpdateInterfaceList (LvHSystem hSystem,
                                                          uint32_t Timeout = 0xFFFFFFFF);
#else
LV_DLLENTRY LvStatus LV_STDC LvSystemUpdateInterfaceList (LvHSystem hSystem,
                                                          uint32_t Timeout);
#endif

//------------------------------------------------------------------------------
/** Returns the number of found interfaces, after the LvSystemUpdateInterfaceList()
 *  call.
 *  @param   hSystem   A handle to the System module, obtained from the
 *                     LvSystemOpen() function.
 *  @param   pNumberOfInterfaces  Number of interfaces found.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_SystemFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvSystemGetNumberOfInterfaces (LvHSystem hSystem,
                                                            uint32_t* pNumberOfInterfaces);

//------------------------------------------------------------------------------
/** Returns a string ID of the interface, which is used by the LvInterfaceOpen()
 *  function.
 *  @param   hSystem      A handle to the System module, obtained from the
 *                        LvSystemOpen() function.
 *  @param   Index        Zero-based index of the interface, a value >= 0 and < number of interfaces,
 *                        returned by the LvSystemGetNumberOfInterfaces() function.
 *  @param   pInterfaceId Pointer to a string buffer, where the interface ID will be placed.
 *  @param   Size         Size of the string buffer.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_SystemFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvSystemGetInterfaceId (LvHSystem hSystem,
                                                     uint32_t Index,
                                                     char* pInterfaceId,
                                                     size_t Size);

//------------------------------------------------------------------------------
/** Returns the size of the string buffer needed to hold the Interface ID string,
 *  including the terminating zero character.
 *  @param   hSystem   A handle to the System module, obtained from the
 *                     LvSystemOpen() function.
 *  @param   Index     Zero-based index of the interface, a value >= 0 and < number of interfaces,
 *                     returned by the LvSystemGetNumberOfInterfaces() function.
 *  @param   pSize     Size of the buffer is returned in this parameter.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_SystemFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvSystemGetInterfaceIdSize (LvHSystem hSystem,
                                                         uint32_t Index,
                                                         size_t* pSize);

//------------------------------------------------------------------------------
/** Finds the interface according specified criteria and returns a string ID of the 
 *  interface, which is used by the LvInterfaceOpen() function. This function does not
 *  update the interface list - if you need to do so, call the LvSystemUpdateInterfaceList()
 *  function before calling this function.
 *  @param   hSystem      A handle to the System module, obtained from the
 *                        LvSystemOpen() function.
 *  @param   FindBy       Specifies by which criteria to find the interface. Use one of the 
 *                        @ref LvFindBy constants.
 *  @param   pFindStr     Specifies the find string, the meaning of which is determined by the 
 *                        FindBy parameter, for example when using the LvFindBy_IPAddress,
 *                        this string should contain the IP address searched for. The searched
 *                        string is not case sensitive and need not be complete (is searched as
 *                        a substring).
 *  @param   pInterfaceId Pointer to a string buffer, where the interface ID will be placed.
 *  @param   Size         Size of the string buffer.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus. If the Interface is found,
 *           the returned status is LVSTATUS_OK.
 *  @ingroup GroupSynview_SystemFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvSystemFindInterface (LvHSystem hSystem,
                                                    LvEnum FindBy, 
                                                    const char* pFindStr,
                                                    char* pInterfaceId,
                                                    size_t Size);

//##############################################################################
//  LvInterface
//##############################################################################

/** Opens the Interface module. The same Interface can be open multiple times
 *  (there is a reference counter inside); in such case there must be also
 *  the same number of LvInterfaceClose() calls used (every open increase the
 *  reference count and every close decreases it) .
 *  @param   hSystem       A handle to the System module, obtained from the
 *                         LvSystemOpen() function.
 *  @param   pInterfaceId  A string interface ID, obtained by the LvSystemGetInterfaceId().
 *  @param   phInterface   In this parameter the handle to the interface is
 *                         returned.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_InterfaceFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvInterfaceOpen (LvHSystem hSystem,
                                              const char* pInterfaceId,
                                              LvHInterface* phInterface);

//------------------------------------------------------------------------------
/** Closes the interface. If the Interface was opened multiple times, it only
 *  decreases the reference counter (see a note by the LvInterfaceOpen()).
 *  Be sure you first close all dependent modules (Device, Event etc.).
 *  @param   phInterface  Pointer to a handle to the Interface module, obtained from the
 *                        LvInterfaceOpen() function. The handle is assigned 0 after
 *                        the operation.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_InterfaceFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvInterfaceClose (LvHInterface* phInterface);

//------------------------------------------------------------------------------
/** Updates the Device list. The available devices are searched.
 *  @param   hInterface  A handle to the Interface module, obtained from the
 *                       LvInterfaceOpen() function.
 *  @param   Timeout     Specifies a timeout in ms for searching the devices.
 *  @return  If the timeout has expired while waiting for the completion, the function returns 
             @ref LVSTATUS_TIMEOUT, otherwise it returns the @ref LvStatus value indicating the 
             success of the requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_InterfaceFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvInterfaceUpdateDeviceList (LvHInterface hInterface,
                                                          uint32_t Timeout = 0xFFFFFFFF);
#else
LV_DLLENTRY LvStatus LV_STDC LvInterfaceUpdateDeviceList (LvHInterface hInterface,
                                                          uint32_t Timeout);
#endif

//------------------------------------------------------------------------------
/** Returns the number of devices found by the LvInterfaceUpdateDeviceList()
 *  function.
 *  @param   hInterface  A handle to the Interface module, obtained from the
 *                       LvInterfaceOpen() function.
 *  @param   pDevices    Number of devices found.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_InterfaceFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvInterfaceGetNumberOfDevices (LvHInterface hInterface,
                                                            uint32_t* pDevices);

//------------------------------------------------------------------------------
/** Returns a string ID of the device at specified position in the list.
 *  Note that this device ID is stable (the same physical device has always the
 *  same ID) and it is unique (no other physical device can have the same ID).
 *  To hardcode directly the device ID in your application is not recommended, as
 *  the application would not be usable, when a deffective device needs to be
 *  replaced. The SynView User's Guide discuss the ways, how to solve such
 *  maintainability demands.
 *  @param   hInterface  A handle to the Interface module, obtained from the
 *                       LvInterfaceOpen() function.
 *  @param   Index       Zero-based index of the device, a value >= 0 and < number of devices,
 *                       returned by the LvInterfaceGetNumberOfDevices() function.
 *  @param   pDeviceId   Pointer to a string buffer, where the device ID will be placed.
 *  @param   Size        Size of the buffer.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_InterfaceFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvInterfaceGetDeviceId (LvHInterface hInterface,
                                                     uint32_t Index, 
                                                     char* pDeviceId,
                                                     size_t Size);

//----------------------------------------------------------------------------
/** Returns the size of the string buffer needed to hold the Device ID string,
 *  including the terminating zero character.
 *  @param   hInterface  A handle to the Interface module, obtained from the
 *                       LvInterfaceOpen() function.
 *  @param   Index       Zero-based index of the device, a value >= 0 and < number of devices,
 *                       returned by the LvInterfaceGetNumberOfDevices() function.
 *  @param   pSize       Size of the buffer is returned in this parameter.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_InterfaceFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvInterfaceGetDeviceIdSize (LvHInterface hInterface,
                                                         uint32_t Index,
                                                         size_t* pSize);

//------------------------------------------------------------------------------
/** Finds the device according specified criteria and returns a string ID of the 
 *  device, which can be used by the LvDeviceOpen() function. This function does not
 *  update the device list - if you need to do so, call the LvInterfaceUpdateDeviceList()
 *  function before calling this function.
 *  @param   hInterface  A handle to the Interface module, obtained from the
 *                       LvInterfaceOpen() function.
 *  @param   FindBy      Specifies by which criteria to find the interface. Use one of the 
 *                       @ref LvFindBy constants.
 *  @param   pFindStr    Specifies the find string, the meaning of which is determined by the 
 *                       FindBy parameter, for example when using the LvFindBy_IPAddress,
 *                       this string should contain the IP address searched for. The searched
 *                       string is not case sensitive and need not be complete (is searched as
 *                       a substring).
 *  @param   pDeviceId   Pointer to a string buffer, where the device ID will be placed.
 *  @param   Size        Size of the buffer.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus. If the Device is found,
 *           the returned status is LVSTATUS_OK.
 *  @ingroup GroupSynview_InterfaceFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvInterfaceFindDevice (LvHInterface hInterface,
                                                    LvEnum FindBy, 
                                                    const char* pFindStr,
                                                    char* pDeviceId,
                                                    size_t Size);

//##############################################################################
//  LvDevice
//##############################################################################

/** Opens the Device module. This physically means opening a connection with the
 *  device and retrieving a list of device remote features. Always check the
 *  success of this function call; the opening may fail for example when you
 *  request an exclusive access and the device is already open by some other
 *  application.
 *  @param   hInterface  A handle to the Interface module, obtained from the
 *                       LvInterfaceOpen() function.
 *  @param   pDeviceId   A string ID of the device, obtained by LvInterfaceGetDeviceId()
 *                       function.
 *  @param   phDevice    In this parameter the handle to the Device is
 *                       returned.
 *  @param   Access      Desired device access, one of the @ref LvDeviceAccess constants.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_DeviceFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvDeviceOpen (LvHInterface hInterface,
                                           const char* pDeviceId,
                                           LvHDevice* phDevice,
                                           LvEnum Access = LvDeviceAccess_Exclusive);
#else
LV_DLLENTRY LvStatus LV_STDC LvDeviceOpen (LvHInterface hInterface,
                                           const char* pDeviceId,
                                           LvHDevice* phDevice,
                                           LvEnum Access);
#endif

//------------------------------------------------------------------------------
// try (mw)
/** Re-Opens the Device. does not create a new class, just tries to reopen the connection
 *  @param   hInterface  A handle to the Interface module, obtained from the
 *                       LvInterfaceOpen() function.
 *  @param   pDeviceId   A string ID of the device, obtained by LvInterfaceGetDeviceId()
 *                       function.
 *  @param   hDevice     The handle to the Device
 *  @param   Access      Desired device access, one of the @ref LvDeviceAccess constants.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_DeviceFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvDeviceReOpen (LvHInterface hInterface,
                                           const char* pDeviceId,
                                           LvHDevice hDevice,
                                           LvEnum Access = LvDeviceAccess_Exclusive);
#else
LV_DLLENTRY LvStatus LV_STDC LvDeviceReOpen (LvHInterface hInterface,
                                           const char* pDeviceId,
                                           LvHDevice hDevice,
                                           LvEnum Access);
#endif

//------------------------------------------------------------------------------
/** Closes the Device. Be sure you first close all dependent modules (Stream, Event etc.).
 *  @param   phDevice  Pointer to a handle to the Device module, obtained from the
 *                     LvDeviceOpen() function. This handle is assigned 0 after
 *                     the operation.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_DeviceFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvDeviceClose (LvHDevice* phDevice);


//------------------------------------------------------------------------------
/** Returns the number of available stream types for this device. You can then
 *  iterate the streams by the LvDeviceGetStreamId() function.
 *  @param   hDevice           A handle to the Device module, obtained from the
 *                             LvDeviceOpen() function.
 *  @param   pNumberOfStreams  The number of streams is returned here.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_DeviceFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvDeviceGetNumberOfStreams (LvHDevice hDevice,
                                                         uint32_t* pNumberOfStreams);

//------------------------------------------------------------------------------
/** Returns a string Stream ID, needed for opening the stream.
 *  @param   hDevice    A handle to the Device module, obtained from the
 *                      LvDeviceOpen() function.
 *  @param   Index      Zero-based index of the stream type, a value >= 0 and < number of streams,
 *                      returned by the LvDeviceGetNumberOfStreams() function.
 *  @param   pStreamId  Pointer to a string buffer, where the stream ID will be placed.
 *  @param   Size       Size of the buffer.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_DeviceFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvDeviceGetStreamId (LvHDevice hDevice,
                                                  uint32_t Index,
                                                  char* pStreamId,
                                                  size_t Size);

//------------------------------------------------------------------------------
/** Returns the size of the string buffer needed to hold the stream ID at given
 *  index, including the space for the terminating zero character.
 *  @param   hDevice  A handle to the Device module, obtained from the
 *                    LvDeviceOpen() function.
 *  @param   Index    Zero-based index of the stream type, a value >= 0 and < number of streams,
 *                    returned by the LvDeviceGetNumberOfStreams() function.
 *  @param   pSize    Size of the buffer is returned in this parameter.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_DeviceFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvDeviceGetStreamIdSize (LvHDevice hDevice,
                                                      uint32_t Index,
                                                      size_t* pSize);

//------------------------------------------------------------------------------
/** Starts the acquisition. This function includes more than just calling the
 *  AcquisitionStart remote command on the device - it checks the size of the buffers,
 *  prepares the streams for the start, locks GenTL params and then starts the
 *  acquisition on the device itself.
 *  Always check the success of this function call.
 *  @param   hDevice  A handle to the Device module, obtained from the
 *                    LvDeviceOpen() function.
 *  @param   Options  Reserved for future use, must be 0 or ommitted.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_DeviceFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvDeviceAcquisitionStart (LvHDevice hDevice, uint32_t Options = 0);
#else
LV_DLLENTRY LvStatus LV_STDC LvDeviceAcquisitionStart (LvHDevice hDevice, uint32_t Options);
#endif

//------------------------------------------------------------------------------
/** Stops the acquisition.
 *  @param   hDevice  A handle to the Device module, obtained from the
 *                    LvDeviceOpen() function.
 *  @param   Options  Reserved for future use, must be 0 or ommitted.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_DeviceFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvDeviceAcquisitionStop (LvHDevice hDevice, uint32_t Options = 0);
#else
LV_DLLENTRY LvStatus LV_STDC LvDeviceAcquisitionStop (LvHDevice hDevice, uint32_t Options);
#endif

//------------------------------------------------------------------------------
/** Aborts athe acquisition immediately, without completing the current Frame
 *  or waiting on a trigger.
 *  @param   hDevice  A handle to the Device module, obtained from the
 *                    LvDeviceOpen() function.
 *  @param   Options  Reserved for future use, must be 0 or ommitted.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_DeviceFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvDeviceAcquisitionAbort (LvHDevice hDevice, uint32_t Options = 0);
#else
LV_DLLENTRY LvStatus LV_STDC LvDeviceAcquisitionAbort (LvHDevice hDevice, uint32_t Options);
#endif

//------------------------------------------------------------------------------
/** Prepares the device for acquisition, so that the acquisition using the
 *  LvDeviceAcquisitionStart() function then can start fast. If it is not called
 *  before LvDeviceAcquisitionStart(), it is called automatically inside the
 *  LvDeviceAcquisitionStart().
 *  @param   hDevice  A handle to the Device module, obtained from the
 *                    LvDeviceOpen() function.
 *  @param   Options  Reserved for future use, must be 0 or ommitted.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_DeviceFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvDeviceAcquisitionArm (LvHDevice hDevice, uint32_t Options = 0);
#else
LV_DLLENTRY LvStatus LV_STDC LvDeviceAcquisitionArm (LvHDevice hDevice, uint32_t Options);
#endif


#ifdef SIMPLON_INC_OBSOLETE
    //------------------------------------------------------------------------------
    // This function enables to directly set the hardware Lookup Table (LUT) on the
    // device. The LUT is also available through the @ref LvDevice_LUTIndex and
    // @ref LvDevice_LUTValue fetaures. This function frees you from necessity to
    // iterate through all LUT items and makes the LUT setting faster.
    // @param   hDevice  A handle to the Device module, obtained from the
    //                   LvDeviceOpen() function.
    // @param   Selector Lut type, one of the @ref LvLUTSelector values.
    // @param   pLUT     Pointer to the LUT buffer.
    // @param   Size     Size of the buffer. The only valid values are
    //                    - 256 for 8-bit LUT
    //                    - 2048 for 10-bit LUT
    //                    - 8192 for 12-bit LUT
    // @return  Returns the @ref LvStatus value indicating the success of the
    //          requested operation. See @ref GroupSynview_LvStatus.
    // @ingroup GroupSynview_DeviceFunctions

    LV_DLLENTRY LvStatus LV_STDC LvDeviceSetHwLUT (LvHDevice hDevice,
                                                   LvEnum Selector,
                                                   void* pLUT,
                                                   size_t Size);

    //------------------------------------------------------------------------------
    //  This function enables to directly get the hardware Lookup Table (LUT) on the
    //  device. The LUT is also available through the @ref LvDevice_LUTIndex and
    //  @ref LvDevice_LUTValue fetaures. This function frees you from necessity to
    //  iterate through all LUT items and makes the LUT getting faster.
    //  @param   hDevice  A handle to the Device module, obtained from the
    //                    LvDeviceOpen() function.
    //  @param   Selector Lut type, one of the @ref LvLUTSelector values.
    //  @param   pLUT     Pointer to the LUT buffer.
    //  @param   Size     Size of the buffer. The only valid values are
    //                     - 256 for 8-bit LUT
    //                     - 2048 for 10-bit LUT
    //                     - 8192 for 12-bit LUT
    //  @return  Returns the @ref LvStatus value indicating the success of the
    //           requested operation. See @ref GroupSynview_LvStatus.
    //  @ingroup GroupSynview_DeviceFunctions

    LV_DLLENTRY LvStatus LV_STDC LvDeviceGetHwLUT (LvHDevice hDevice,
                                                   LvEnum Selector,
                                                   void* pLUT,
                                                   size_t Size);
#endif
//------------------------------------------------------------------------------
/** Saves the device settings to a file. In the Options can be specified which
 *  parts of the device configuration are to be saved - the Remote, Local and/or
 *  GenTL part. See alo notes by LvDeviceLoadSettings().
 *  @param   hDevice    A handle to the Device module, obtained from the
 *                      LvDeviceOpen() function.
 *  @param   pId        A string ID enabling to protect the file. If you 
 *                      specify a non-empty ID in LvDeviceSaveSettings(), 
 *                      you must use the same ID in LvDeviceLoadSettings(), 
 *                      otherwise the settings are not loaded.
 *  @param   pFileName  The file specification, to which the configuration is
 *                      stored. It is a text file.
 *  @param   Options    One or or-ed combination of @ref GroupSynview_SaveFlags.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_DeviceFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvDeviceSaveSettings (LvHDevice hDevice,
                                                   const char* pId,
                                                   const char* pFileName,
                                                   uint32_t Options);

//------------------------------------------------------------------------------
/** Loads the device settings from a file. In the Options can be specified which
 *  parts of the device configuration are to be loaded - the Remote, Local and/or
 *  GenTL part. Note that there are several factors, which can break the 
 *  compatibility of the settings file with the current device:
 *    - When the current device is of different vendor/model, the settings
 *      file is most probably not compatible.
 *    - When the current device is of the same vendor/model, but uses a different
 *      firmware version - this could mean that some remote device features
 *      are not present or behave differently.
 *    - When the XML version of the Local and GenTL features changes - again
 *      this could mean that some features are not present or behave differently.
 *  For this reason this function checks the versions and if not the same,
 *  it returns either the @ref LVSTATUS_SETTINGS_INCOMPATIBLE_MODEL or @ref 
 *  LVSTATUS_SETTINGS_INCOMPATIBLE_VERSION error states. 
 *  As the difference in versions might not necessarily mean a real incompability,
 *  you can use the @ref LvSaveFlag_IgnoreVersion and @ref LvSaveFlag_IgnoreModel
 *  flags in the Options parameter in order to force this function to try to
 *  load the settings even if the possible incompatibility is found.
 *  @param   hDevice    A handle to the Device module, obtained from the
 *                      LvDeviceOpen() function.
 *  @param   pId        A string ID enabling to protect the file. If you 
 *                      specify a non-empty ID in LvDeviceSaveSettings(), 
 *                      you must use the same ID in LvDeviceLoadSettings(), 
 *                      otherwise the settings are not loaded.
 *  @param   pFileName  The file specification, where the configuration is
 *                      stored. It is a text file.
 *  @param   Options    One or or-ed combination of @ref GroupSynview_SaveFlags.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_DeviceFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvDeviceLoadSettings (LvHDevice hDevice,
                                                   const char* pId,
                                                   const char* pFileName,
                                                   uint32_t Options);

//------------------------------------------------------------------------------
/** Loads device batch commands from a file.
 *  @param   hDevice    A handle to the Device module, obtained from the
 *                      LvDeviceOpen() function.
 *  @param   pFileName  The file specification, where the configuration is
 *                      stored. It is a text file.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_DeviceFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvDeviceLoadBatch (LvHDevice hDevice,
                                                   const char* pFileName);

//------------------------------------------------------------------------------
/** Sets the lookup table. If the hardware lookup table is available, it is used,
 *  otherwise a software lookup table is set.
 *  This function belongs to a set of functions, which unify the functionality
 *  of devices with real-time processing embedded in hardware (RTF) and devices
 *  without real-time processing, for which the processing is made by software.
 *  The LUT is automatically recalculated to appropriate type, if you use different
 *  LUT bit depth than is the actually used for the current pixel format.
 *  @param   hDevice   A handle to the Device module, obtained from the
 *                     LvDeviceOpen() function.
 *  @param   Selector  Lookup table selector, see @ref LvLUTSelector.
 *  @param   pLUT      Pointer to the lookup table.
 *  @param   Size      Size of the lookup table. The only valid values are
 *                     - 256 for 8-bit LUT
 *                     - 2048 for 10-bit LUT
 *                     - 8192 for 12-bit LUT
 *  @param   Options   The LvUniLutFlags_HwLut option can be used to apply to function directly on HW LUT.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_DeviceFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvDeviceUniSetLut (LvHDevice hDevice,
                                                LvEnum Selector,
                                                void* pLUT, 
                                                size_t Size,
                                                uint32_t Options = 0);
#else
LV_DLLENTRY LvStatus LV_STDC LvDeviceUniSetLut (LvHDevice hDevice,
                                                LvEnum Selector,
                                                void* pLUT, 
                                                size_t Size,
                                                uint32_t Options);
#endif

//------------------------------------------------------------------------------
/** Gets the lookup table. See LvDeviceUniSetLut() for details.
 *  The LUT is automatically recalculated to appropriate type, if you use different
 *  LUT bit depth than is the actually used for the current pixel format. So you can
 *  for example read the 12-bit LUT to 8-bit LUT array.
 *  @param   hDevice   A handle to the Device module, obtained from the
 *                     LvDeviceOpen() function.
 *  @param   Selector  Lookup table selector, see @ref LvLUTSelector.
 *  @param   pLUT      Pointer to the lookup table.
 *  @param   Size      Size of the lookup table. The only valid values are
 *                     - 256 for 8-bit LUT
 *                     - 2048 for 10-bit LUT
 *                     - 8192 for 12-bit LUT
 *  @param   Options   The LvUniLutFlags_HwLut option can be used to apply to function directly on HW LUT.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_DeviceFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvDeviceUniGetLut (LvHDevice hDevice,
                                                LvEnum Selector,
                                                void* pLUT,
                                                size_t Size,
                                                uint32_t Options = 0);
#else
LV_DLLENTRY LvStatus LV_STDC LvDeviceUniGetLut (LvHDevice hDevice,
                                                LvEnum Selector,
                                                void* pLUT,
                                                size_t Size,
                                                uint32_t Options);
#endif
                                                  
//##############################################################################
//  LvStream
//##############################################################################

/** Opens the stream module, associated with the device.
 *  @param   hDevice     A handle to the Device module, obtained from the
 *                       LvDeviceOpen() function.
 *  @param   pStreamId   A string ID of the stream, obtained from LvDeviceGetStreamId().
 *                       If an empty string is used, the first found stream is
 *                       opened. This is usually the image data stream.
 *  @param   phStream    In this parameter the handle to the Stream is
 *                       returned.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_StreamFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvStreamOpen (LvHDevice hDevice,
                                           const char* pStreamId,
                                           LvHStream* phStream);

//------------------------------------------------------------------------------
/** Closes the Stream. Be sure you first close all dependent modules (Buffers, Event,
 *  Renderer etc.).
 *  @param   phStream  Pointer to a handle to the Stream module, obtained from the
 *                     LvStreamOpen() function. This handle is assigned 0 after
 *                     the operation.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_StreamFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvStreamClose (LvHStream* phStream);

//------------------------------------------------------------------------------
/** Returns the buffer handle at given index.
 *  @param   hStream      A handle to the Stream module, obtained from the
 *                        LvStreamOpen() function.
 *  @param   BufferIndex  Zero-based index.
 *  @param   phBuffer     In this parameter the buffer handle is returned.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_StreamFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvStreamGetBufferAt (LvHStream hStream,
                                                  uint32_t BufferIndex,
                                                  LvHBuffer* phBuffer);

//------------------------------------------------------------------------------
/** Moves the buffers according to the LvQueueOperation specified.
 *  @param   hStream    A handle to the Stream module, obtained from the
 *                      LvStreamOpen() function.
 *  @param   Operation  One of the @ref LvQueueOperation.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_StreamFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvStreamFlushQueue (LvHStream hStream,
                                                 LvEnum Operation);

//------------------------------------------------------------------------------
/** Starts the stream. This function need not be used on the image stream, where
 *  it is called automatically in the LvDeviceAcquisitionStart() function.
 *  @param   hStream         A handle to the Stream module, obtained from the
 *                           LvStreamOpen() function.
 *  @param   StartFlags      One of the GroupSynview_StreamStartFlags.
 *  @param   ImagesToAcquire Number of images to acquire.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_StreamFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvStreamStart (LvHStream hStream,
                                            uint32_t StartFlags = LvStreamStartFlags_Default,
                                            uint32_t ImagesToAcquire = 0xFFFFFFFF);
#else
LV_DLLENTRY LvStatus LV_STDC LvStreamStart (LvHStream hStream,
                                            uint32_t StartFlags,
                                            uint32_t ImagesToAcquire);
#endif

//------------------------------------------------------------------------------
/** Stops the stream. This function need not be used on the image stream, where
 *  it is called automatically in the LvDeviceAcquisitionStop() function.
 *  @param   hStream    A handle to the Stream module, obtained from the
 *                      LvStreamOpen() function.
 *  @param   StopFlags  One of the GroupSynview_StreamStopFlags.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_StreamFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvStreamStop (LvHStream hStream,
                                           uint32_t StopFlags = LvStreamStopFlags_Default);
#else
LV_DLLENTRY LvStatus LV_STDC LvStreamStop (LvHStream hStream,
                                           uint32_t StopFlags);
#endif

//##############################################################################
//  LvBuffer
//##############################################################################

/** Opens a buffer. On the GenTL level it corresponds to DSAnnounceBuffer() or
 *  DSAllocAndAnnounceBuffer().
 *  @param   hStream       A handle to the Stream module, obtained from the
 *                         LvStreamOpen() function.
 *  @param   pDataPointer  Pointer to image data buffer. This can be supplied by
 *                         the application (in such case the DataSize must be set
 *                         to the actual size of the buffer), or can be left NULL -
 *                         in such case the buffer is allocated by SynView.
 *  @param   DataSize      Size of the buffer supplied, or 0 if the pDataPointer
 *                         is NULL.
 *  @param   pUserPointer  A user pointer, which is then passed back in the
 *                         LvEventCallbackNewBufferFunct(). It enables the application
 *                         to reference some own data structure associated with the
 *                         buffer.
 *  @param   Options       Currently unused, must be 0.
 *  @param   phBuffer      To this parameter the handle to the buffer is returned.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_BufferFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvBufferOpen (LvHStream hStream,
                                           void* pDataPointer, 
                                           size_t DataSize, 
                                           void* pUserPointer, 
                                           uint32_t Options, 
                                           LvHBuffer* phBuffer);

//------------------------------------------------------------------------------
/** Closes the buffer. On the GenTL level it corresponds to the DSRevokeBuffer()
 *  function.
 *  @param   phBuffer  A handle to the Buffer module, obtained from the
 *                     LvBufferOpen() function. This handle is assigned 0 after
                       the operation.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_BufferFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvBufferClose (LvHBuffer* phBuffer);

//------------------------------------------------------------------------------
/** Attaches a process buffer to a buffer. The process buffer may be needed for
 *  software processing, for example Bayer decoding, if the device hardware is
 *  not capable of it. The process buffer can be either supplied by the application
 *  by this function, or allocated automatically by SynView upon need.
 *  @param   hBuffer       A handle to the Buffer module, obtained from the
 *                         LvBufferOpen() function.
 *  @param   pDataPointer  Pointer to the supplied buffer.
 *  @param   DataSize      Size of the buffer.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_BufferFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvBufferAttachProcessBuffer (LvHBuffer hBuffer,
                                                          void* pDataPointer,
                                                          size_t DataSize);

//------------------------------------------------------------------------------
/** Puts the buffer to the input buffer pool. This is an important part of the
 *  image handling loop: after the buffer with the acquired image is passed
 *  to the application, the application must return it to the input buffer
 *  pool by this function after processing.
 *  @param   hBuffer  A handle to the Buffer module, obtained from the
 *                    LvBufferOpen() function.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_BufferFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvBufferQueue (LvHBuffer hBuffer);

//------------------------------------------------------------------------------
/** Parses the chunk data of the image. The chunk data are then accessible as
 *  device remote features (for example @ref LvDevice_ChunkTimestamp).
 *  @param   hBuffer       A handle to the Buffer module, obtained from the
 *                         LvBufferOpen() function.
 *  @param   UpdateLayout  If set to 1, the layout of chunk data is decoded. If set
 *                         to 0, the data are only read from already decoded layout -
 *                         this is faster. Usually, the layout of the chunk data is
 *                         constant, so it needs to be decoded only at first call
 *                         of this function.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_BufferFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvBufferParseChunkData (LvHBuffer hBuffer,
                                                     uint32_t UpdateLayout = 0);
#else
LV_DLLENTRY LvStatus LV_STDC LvBufferParseChunkData (LvHBuffer hBuffer,
                                                     uint32_t UpdateLayout);
#endif

//------------------------------------------------------------------------------
/** Saves the image to a file in Windows BMP format. If the image is in the
 *  pixel format not compatible with the BMP format, it is automatically converted.
 *  @param   hBuffer    A handle to the Buffer module, obtained from the
 *                      LvBufferOpen() function.
 *  @param   pFileName  The file name. Be sure to specify it with the full path.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_BufferFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvBufferSaveImageToBmpFile (LvHBuffer hBuffer,
                                                         const char* pFileName);

//------------------------------------------------------------------------------
/** Saves the image to a file in JPEG format. If the image is in the
 *  pixel format not compatible with the JPEG format, it is automatically converted.
 *  @param   hBuffer    A handle to the Buffer module, obtained from the
 *                      LvBufferOpen() function.
 *  @param   pFileName  The file name. Be sure to specify it with the full path.
 *  @param   Quality    The quality factor in range from from 1 to 100. The higher
 *                      is the factor, the higher is the quality and lower the
 *                      compression.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_BufferFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvBufferSaveImageToJpgFile (LvHBuffer hBuffer,
                                                         const char* pFileName,
                                                         uint32_t Quality);

//------------------------------------------------------------------------------
/** Saves the image to a file in the TIFF format. If the image is in the
 *  pixel format not compatible with the TIF format, it is automatically converted.
 *  @param   hBuffer    A handle to the Buffer module, obtained from the
 *                      LvBufferOpen() function.
 *  @param   pFileName  The file name. Be sure to specify it with the full path.
 *  @param   Options    Options for saved pixel format. The @ref LvipOption_TiffConvertTo16Bit flag can be used there.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_BufferFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvBufferSaveImageToTifFile (LvHBuffer hBuffer,
                                                         const char* pFileName,
                                                         uint32_t Options = 0);
#else
LV_DLLENTRY LvStatus LV_STDC LvBufferSaveImageToTifFile (LvHBuffer hBuffer,
                                                         const char* pFileName,
                                                         uint32_t Options);
#endif

//------------------------------------------------------------------------------
/** Fills the @ref LvipImgInfo structure for the image in the buffer. This simplifies
 *  a direct use of the @ref GroupImgproc. If the image is processed, the image info
 *  points to the processed image, otherwise it points to the original image.
 *  @param   hBuffer  A handle to the Buffer module, obtained from the
 *                    LvBufferOpen() function.
 *  @param   pImgInfo Pointer to the ImgInfo structure, to which are the image
 *                    parameters stored.
 *  @param   Options  Currently unused, must be 0.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_BufferFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvBufferGetImgInfo (LvHBuffer hBuffer,
                                                 LvipImgInfo* pImgInfo,
                                                 uint32_t Options = 0);
#else
LV_DLLENTRY LvStatus LV_STDC LvBufferGetImgInfo (LvHBuffer hBuffer,
                                                 LvipImgInfo* pImgInfo,
                                                 uint32_t Options);
#endif

//------------------------------------------------------------------------------
/** Returns the rectangle to which the buffer was last painted. This is useful
 *  namely in case you have a tile mode and want to identify the buffer according
 *  a mouse click location. If the buffer was not yet painted by the renderer, the 
 *  returned values are 0.
 *  @param   hBuffer  A handle to the Buffer module, obtained from the
 *                    LvBufferOpen() function.
 *  @param   pX       Pointer to X offset in pixels.
 *  @param   pY       Pointer to Y offset in pixels.
 *  @param   pWidth   Pointer to Width in pixels.
 *  @param   pHeight  Pointer to Height in pixels.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_BufferFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvBufferGetLastPaintRect (LvHBuffer hBuffer,
                                                       int32_t* pX, 
                                                       int32_t* pY, 
                                                       int32_t* pWidth, 
                                                       int32_t* pHeight);

//------------------------------------------------------------------------------
/** Calculates white balance factors from the current image.
 *  @param   hBuffer  A handle to the Buffer module, obtained from the
 *                    LvBufferOpen() function.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_BufferFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvBufferUniCalculateWhiteBalance (LvHBuffer hBuffer);

//##############################################################################
//  LvEvent
//##############################################################################

//------------------------------------------------------------------------------
/** Opens the Event module for specified owner module.
 *  @param   hModule    A handle to the System, Device or Stream module.
 *  @param   EventType  One of the @ref LvEventType.
 *  @param   phEvent    To this parameter the Event handle is stored.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_EventFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvEventOpen (LvHModule hModule,
                                          LvEnum EventType,
                                          LvHEvent* phEvent);

//------------------------------------------------------------------------------
/** Closes the Event module.
 *  @param   phEvent A handle to the Event module, obtained from the
 *                   LvEventOpen() function. This handle is assigned 0 after
                     the operation.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_EventFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvEventClose (LvHEvent* phEvent);

//------------------------------------------------------------------------------
/** Terminates a single wait in the LvEventWaitAndGetData() function.
 *  @param   hEvent  A handle to the Event module, obtained from the
 *                   LvEventOpen() function.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_EventFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvEventKill (LvHEvent hEvent);

//------------------------------------------------------------------------------
/** Discards all buffers in the output buffer queue (waiting to be delivered
 *  to the application).
 *  @param   hEvent  A handle to the Event module, obtained from the
 *                   LvEventOpen() function.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_EventFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvEventFlush (LvHEvent hEvent);

//------------------------------------------------------------------------------
/** Waits for the event and gets its data in one atomic operation. Use this function 
 *  only for events other than LvEventType_NewBuffer, for the the LvEventType_NewBuffer
 *  event type use the LvEventWaitAndGetNewBuffer() function instead. Do not use
 *  this function if you use the callback - see @ref LvEventSetCallback() or
 *  @ref LvEventSetCallbackNewBuffer().
 *  @param   hEvent  A handle to the Event module, obtained from the
 *                   LvEventOpen() function.
 *  @param   pBuffer Pointer to a user allocated buffer to receive the event data.
 *                   The buffer can be parsed by the LvEventGetDataInfo() function.
 *  @param   pSize   Size of the buffer must be specified in this parameter and
 *                   after the function returns, the actual size is returned in
 *                   this parameter.
 *  @param   Timeout The wait timeout in milliseconds. The value 0xFFFFFFFF is
 *                   considered as infinite. Note that you can also kill waiting
 *                   from another thread using the LvEventKill() function.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_EventFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvEventWaitAndGetData (LvHEvent hEvent,
                                                    void* pBuffer,
                                                    size_t* pSize,
                                                    uint32_t Timeout=0xFFFFFFFF);
#else
LV_DLLENTRY LvStatus LV_STDC LvEventWaitAndGetData (LvHEvent hEvent,
                                                    void* pBuffer,
                                                    size_t* pSize,
                                                    uint32_t Timeout);
#endif

//------------------------------------------------------------------------------
/** Waits for the event and gets its data in one atomic operation. Use this function 
 *  only for events of the LvEventType_NewBuffer type. Do not use
 *  this function if you use the callback - see @ref LvEventSetCallback() or
 *  @ref LvEventSetCallbackNewBuffer().
 *  @param   hEvent   A handle to the Event module, obtained from the
 *                    LvEventOpen() function.
 *  @param   phBuffer The handle to the received buffer is returned in this parameter.
 *  @param   Timeout  The wait timeout in milliseconds. The value 0xFFFFFFFF is
 *                    considered as infinite. Note that you can also kill waiting
 *                    from another thread using the LvEventKill() function.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_EventFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvEventWaitAndGetNewBuffer (LvHEvent hEvent,
                                                         LvHBuffer* phBuffer,
                                                         uint32_t Timeout=0xFFFFFFFF);
#else
LV_DLLENTRY LvStatus LV_STDC LvEventWaitAndGetNewBuffer (LvHEvent hEvent,
                                                         LvHBuffer* phBuffer,
                                                         uint32_t Timeout);
#endif

//------------------------------------------------------------------------------
/** Enables to parse the buffer from LvEventWaitAndGetData.
 *  @param   hEvent     A handle to the Event module, obtained from the
 *                      LvEventOpen() function.
 *  @param   pInBuffer  Pointer to a buffer containing event data. This value must
 *                      not be NULL.
 *  @param   InSize     Size of the provided pInBuffer in bytes.
 *  @param   Info       One of the @ref LvEventDataInfo.
 *  @param   pBuffer    Pointer to a user allocated buffer to receive the requested
 *                      information. If this parameter is NULL, pSize will contain
 *                      the minimal size of pBuffer in bytes. If the pType is
 *                      a string, the size includes the terminating 0.
 *  @param   pSize      Size of the buffer is returned in this parameter.
 *  @param   pType      One of the @ref LvInfoDataType.
 *  @param   Param      Additional parameter, if used, its role is explained by the
 *                      @ref LvEventDataInfo.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_EventFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvEventGetDataInfo (LvHEvent hEvent,
                                                 void* pInBuffer, 
                                                 size_t InSize,
                                                 LvEnum Info,
                                                 void* pBuffer, 
                                                 size_t* pSize,
                                                 LvEnum* pType = NULL,
                                                 int32_t Param = 0);
#else
LV_DLLENTRY LvStatus LV_STDC LvEventGetDataInfo (LvHEvent hEvent,
                                                 void* pInBuffer, 
                                                 size_t InSize,
                                                 LvEnum Info,
                                                 void* pBuffer, 
                                                 size_t* pSize,
                                                 LvEnum* pType,
                                                 int32_t Param);
#endif

//------------------------------------------------------------------------------
/** Puts a new event to Event ouptut queue. This function can be used only for
 *  user-defined events.
 *  @param   hEvent   A handle to the Event module, obtained from the
 *                    LvEventOpen() function.
 *  @param   pBuffer  Pointer to event data.
 *  @param   Size     Size of the event data.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_EventFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvEventPutData (LvHEvent hEvent,
                                             void* pBuffer,
                                             size_t Size);

//------------------------------------------------------------------------------
/** Specifies a callback function for the event thread. Note that the callback function
 *  cannot be a method of a class.
 *  @param   hEvent     A handle to the Event module, obtained from the
 *                      LvEventOpen() function.
 *  @param   pFunction  The callback function in the forms of @ref LvEventCallbackFunct.
 *  @param   pUserParam User parameter, which will be passed to each callback call.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_EventFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvEventSetCallback (LvHEvent hEvent,
                                                 LvEventCallbackFunct pFunction,
                                                 void* pUserParam);

//------------------------------------------------------------------------------
/** Specifies a callback function for the thread of the Event of the @ref LvEventType_NewBuffer.
 *  Once the application specifies this callback, it becomes responsible for returning
 *  the image buffers to the input buffer pool. Note that the callback function
 *  cannot be a method of a class.
 *  @param   hEvent  A handle to the Event module, obtained from the
 *                   LvEventOpen() function.
 *  @param   pFunction  The callback function in the forms of LvEventCallbackNewBufferFunct.
 *  @param   pUserParam User parameter, which will be passed to each callback call.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_EventFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvEventSetCallbackNewBuffer (LvHEvent hEvent,
                                                          LvEventCallbackNewBufferFunct pFunction,
                                                          void* pUserParam);

//------------------------------------------------------------------------------
/** Starts an internal thread, which waits for events and passes them to specified
 *  callback function. When the thread is started, the application must no longer
 *  call the LvEventWaitAndGetData() or LvEventWaitAndGetNewBufer() functions -
 *  this is called internally in the thread and upon return from this function a
 *  callback function is called.
 *  @param   hEvent  A handle to the Event module, obtained from the
 *                   LvEventOpen() function.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_EventFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvEventStartThread (LvHEvent hEvent);

//------------------------------------------------------------------------------
/** Stops the event internal thread.
 *  @param   hEvent  A handle to the Event module, obtained from the
 *                   LvEventOpen() function.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_EventFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvEventStopThread  (LvHEvent hEvent);

//------------------------------------------------------------------------------
/** Indicates that the event callback function for LvEventSetCallback or 
 *  LvEventSetCallbackNewBuffer needs to exit as soon as possible
 *  @param   hEvent  A handle to the Event module, obtained from the
 *                   LvEventOpen() function.
 *  @return  Returns the must exit status
 *  @ingroup GroupSynview_EventFunctions
 */
LV_DLLENTRY uint32_t LV_STDC LvEventCallbackMustExit  (LvHEvent hEvent);


//##############################################################################
//  LvRenderer
//##############################################################################

/** Opens the Renderer module for image display. The renderer attempts to load
 *  the sv.synview.display library. In case of SynView installation in an operating
 *  system without possibility to graphically display (for example Linux without
 *  XWindows), the load of this library fails and the calls to Renderer functions
 *  will return errors.
 *  @param   hStream      A handle to the Stream module, obtained from the
 *                        LvStreamOpen() function.
 *  @param   phRenderer   In this parameter the handle to the renderer is returned.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_RendererFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvRendererOpen (LvHStream hStream, LvHRenderer* phRenderer);

//------------------------------------------------------------------------------
/** Starts the Renderer module.
 *  @param   phRenderer A handle to the Renderer module, obtained from the
 *                      LvRendererOpen() function.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_RendererFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvRendererStart (LvHRenderer* phRenderer);

//------------------------------------------------------------------------------
/** Stops the Renderer module.
 *  @param   phRenderer A handle to the Renderer module, obtained from the
 *                      LvRendererOpen() function.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_RendererFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvRendererStop (LvHRenderer* phRenderer);

//------------------------------------------------------------------------------
/** Closes the Renderer module.
 *  @param   phRenderer A handle to the Renderer module, obtained from the
 *                      LvRendererOpen() function.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_RendererFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvRendererClose (LvHRenderer* phRenderer);

#ifdef _WIN32

//------------------------------------------------------------------------------
/** Sets the target window, in which the renderer has to display. Note that
 *  the application itself assure any repainting (when the window need to be
 *  repainted due to a movement of overlapping) - use LvRendererRepaint()
 *  in such case.
 *  @param   hRenderer  A handle to the Renderer module, obtained from the
 *                      LvRendererOpen() function.
 *  @param   hWnd       Handle to the window.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_RendererFunctions
 */
    LV_DLLENTRY LvStatus LV_STDC LvRendererSetWindow (LvHRenderer hRenderer,
                                                      HWND hWnd);
#else

//------------------------------------------------------------------------------
/** Sets the target window, in which the renderer has to display. Note that
 *  the application itself assure any repainting (when the window need to be
 *  repainted due to a movement of overlapping) - use LvRendererRepaint()
 *  in such case.
 *  @param   hRenderer  A handle to the Renderer module, obtained from the
 *                      LvRendererOpen() function.
 *  @param   pDisplay   Pointer to the display. It is defined as void* in order
 *                      to make SynView header files independent on XWindows.
 *  @param   hWindow    Handle to the window.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_RendererFunctions
 */
    LV_DLLENTRY LvStatus LV_STDC LvRendererSetWindow (LvHRenderer hRenderer,
                                                      void* pDisplay,
                                                      int64_t hWindow);
#endif

//------------------------------------------------------------------------------
/** Checks, if the image can be displayed. Namely the possibility to convert 
 *  the image to desired display pixel format is checked.
 *  @param   hRenderer    A handle to the Renderer module, obtained from the
 *                        LvRendererOpen() function.
 *  @param   hBuffer      The buffer to be displayed.
 *  @param   RenderFlags  Zero or a combination of @ref LvRenderFlags.
 *  @return  Returns the @ref LvStatus value; the value LVSTATUS_OK indicates
 *           the display is possible, the value LVSTATUS_DISPLAY_CANNOT_DISPLAY
 *           indicates impossibility of pixel format conversion or a misconfiguration
 *           of the renderer. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_RendererFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvRendererCanDisplayImage (LvHRenderer hRenderer,
                                                        LvHBuffer hBuffer,
                                                        uint32_t RenderFlags = 0);
#else
LV_DLLENTRY LvStatus LV_STDC LvRendererCanDisplayImage (LvHRenderer hRenderer,
                                                        LvHBuffer hBuffer,
                                                        uint32_t RenderFlags);
#endif

//------------------------------------------------------------------------------
/** Displays the image. The image display mode is set by Renderer features,
 *  see @ref LvRendererFtr.
 *  @param   hRenderer    A handle to the Renderer module, obtained from the
 *                        LvRendererOpen() function.
 *  @param   hBuffer      The buffer to be displayed.
 *  @param   RenderFlags  Zero or a combination of @ref LvRenderFlags.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_RendererFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvRendererDisplayImage (LvHRenderer hRenderer,
                                                     LvHBuffer hBuffer,
                                                     uint32_t RenderFlags = 0);
#else
LV_DLLENTRY LvStatus LV_STDC LvRendererDisplayImage (LvHRenderer hRenderer,
                                                     LvHBuffer hBuffer,
                                                     uint32_t RenderFlags);
#endif

//------------------------------------------------------------------------------
/** Repaints the contents of the display window. In order to be able to repaint,
 *  all images to be displayed must be still held by the application,
 *  i.e. must not be returned to the input buffer pool. See also
 *  @ref LvStream_LvPostponeQueueBuffers feature.
 *  A typical usage of this function is in the WM_PAINT handler in a Windows
 *  application.
 *  @param   hRenderer    A handle to the Renderer module, obtained from the
 *                        LvRendererOpen() function.
 *  @param   RenderFlags  Zero or a combination of @ref LvRenderFlags.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_RendererFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvRendererRepaint (LvHRenderer hRenderer,
                                                uint32_t RenderFlags = 0);
#else
LV_DLLENTRY LvStatus LV_STDC LvRendererRepaint (LvHRenderer hRenderer,
                                                uint32_t RenderFlags);
#endif


//##############################################################################
//  LvAviSaver - not implemented
//##############################################################################

#ifdef LVAVISAVER
    LV_DLLENTRY LvStatus LV_STDC LvAviSaverOpen (LvHStream hStream, LvHAviSaver* phAviSaver);
    LV_DLLENTRY LvStatus LV_STDC LvAviSaverClose (LvHAviSaver* phAviSaver);
    LV_DLLENTRY LvStatus LV_STDC LvAviSaverStoreImage (LvHAviSaver hAviSaver,
                                                       LvHBuffer hBuffer,
                                                       uint32_t Options);
    LV_DLLENTRY LvStatus LV_STDC LvAviSaverOpenFile (LvHAviSaver hAviSaver,
                                                     const char* pFileName,
                                                     uint32_t Options);
    LV_DLLENTRY LvStatus LV_STDC LvAviSaverCloseFile (LvHAviSaver hAviSaver);
    LV_DLLENTRY LvStatus LV_STDC LvAviSaverSaveSettings (LvHAviSaver hAviSaver,
                                                         const char* pId,
                                                         const char* pFileName,
                                                         uint32_t Options);
    LV_DLLENTRY LvStatus LV_STDC LvAviSaverLoadSettings (LvHAviSaver hAviSaver,
                                                         const char* pId,
                                                         const char* pFileName,
                                                         uint32_t Options);

#endif

//=== General Feature Manipulation ===============================================

/** Returns a number of features for specified group. This is useful for building
 *  a list of all available features (like the tree in lv.explorer).
 *  @param   hModule      A handle to the System, Interface, Device, Stream, Event,
 *                        Buffer or Renderer module.
 *  @param   FeatureGroup One of the @ref LvFtrGroup.
 *  @param   pNumItems    The number of features is returned in this parameter.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvGetNumFeatures (LvHModule hModule,
                                               LvEnum FeatureGroup,
                                               uint32_t* pNumItems);

//------------------------------------------------------------------------------
/** Returns the feature ID at specified position. Can be used to iterate all the features
 *  in a list.
 *  @param   hModule      A handle to the System, Interface, Device, Stream, Event,
 *                        Buffer or Renderer module.
 *  @param   FeatureGroup One of the @ref LvFtrGroup.
 *  @param   Index        Zero based index of the feature in the list.
 *  @param   pItem        Feature ID is returned in this parameter.
 *  @param   pLevel       Feature Level expressing its position in the tree is returned
 *                        in this parameter. The base level has value 1.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvGetFeatureAt (LvHModule hModule,
                                             LvEnum FeatureGroup,
                                             uint32_t Index,
                                             LvFeature* pItem,
                                             uint32_t* pLevel);

//------------------------------------------------------------------------------
/** Returns a feature ID based on the feature name. This function is a substantial
 *  function for the generic approach to the feature - by this function you can
 *  get the ID of any existing feature, that means also for those, for which
 *  a SynView constant is not defined. Be sure to check the success of this function -
 *  if the feature is not mandatory, it may not exist.
 *  @param   hModule      A handle to the System, Interface, Device, Stream, Event,
 *                        Buffer or Renderer module.
 *  @param   FeatureGroup One of the @ref LvFtrGroup.
 *  @param   pName        Name of the feature.
 *  @param   pItem        Feature ID is returned in this parameter.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvGetFeatureByName (LvHModule hModule,
                                                 LvEnum FeatureGroup,
                                                 const char* pName,
                                                 LvFeature* pItem);

//------------------------------------------------------------------------------
/** A helper function, allowing simply to determine, if a feature is implemented.
 *  It is a wrapper around the LvGetAccess() function.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @return  If the feature is implemented, returns 1, otherwise 0.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY uint32_t LV_STDC LvIsImplemented (LvHModule hModule,
                                              LvFeature Feature);

//------------------------------------------------------------------------------
/** A helper function, allowing simply to determine, if a feature is implemented.
 *  It is a wrapper around the LvGetAccess() and LvGetFeatureByName() functions.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   FeatureGroup One of the @ref LvFtrGroup.
 *  @param   pName        Name of the feature.
 *  @return  If the feature is implemented, returns 1, otherwise 0.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY uint32_t LV_STDC LvIsImplementedByName (LvHModule hModule,
                                                    LvEnum FeatureGroup,
                                                    const char* pName);

//------------------------------------------------------------------------------
/** A helper function, allowing simply to determine, if an enum entry of an enum 
 *  feature is implemented.
 *  @param   hModule    A handle to the System, Interface, Device, Stream, Event,
 *                      Buffer or Renderer module.
 *  @param   Feature    The feature ID - use a symbolic constant (one of the
 *                      @ref GroupSynview_Features) or an ID obtained by the
 *                      LvGetFeatureByName() function.
 *  @param   EnumEntry  The SynView constant for the enum entry.
 *  @return  If the enum entry is implemented, returns 1, otherwise 0.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY uint32_t LV_STDC LvIsImplementedEnumEntry (LvHModule hModule,
                                                       LvFeature Feature,
                                                       LvEnum EnumEntry);

//------------------------------------------------------------------------------
/** A helper function, allowing simply to determine, if a feature is available.
 *  It is a wrapper around the LvGetAccess() function.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @return  If the feature is available, returns 1, otherwise 0.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY uint32_t LV_STDC LvIsAvailable (LvHModule hModule,
                                            LvFeature Feature);

//------------------------------------------------------------------------------
/** A helper function, allowing simply to determine, if a feature is available.
 *  It is a wrapper around the LvGetAccess() and LvGetFeatureByName() functions.
 *  @param   hModule      A handle to the System, Interface, Device, Stream, Event,
 *                        Buffer or Renderer module.
 *  @param   FeatureGroup One of the @ref LvFtrGroup.
 *  @param   pName        Name of the feature.
 *  @return  If the feature is available, returns 1, otherwise 0.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY uint32_t LV_STDC LvIsAvailableByName (LvHModule hModule,
                                                  LvEnum FeatureGroup,
                                                  const char* pName);

//------------------------------------------------------------------------------
/** A helper function, allowing simply to determine, if an enum entry of an enum 
 *  feature is available.
 *  @param   hModule    A handle to the System, Interface, Device, Stream, Event,
 *                      Buffer or Renderer module.
 *  @param   Feature    The feature ID - use a symbolic constant (one of the
 *                      @ref GroupSynview_Features) or an ID obtained by the
 *                      LvGetFeatureByName() function.
 *  @param   EnumEntry  The SynView constant for the enum entry.
 *  @return  If the enum entry is available, returns 1, otherwise 0.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY uint32_t LV_STDC LvIsAvailableEnumEntry (LvHModule hModule,
                                                     LvFeature Feature,
                                                     LvEnum EnumEntry);

//------------------------------------------------------------------------------
/** A helper function, allowing simply to determine, if a feature is readable.
 *  It is a wrapper around the LvGetAccess() function.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @return  If the feature is readable, returns 1, otherwise 0.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY uint32_t LV_STDC LvIsReadable (LvHModule hModule,
                                           LvFeature Feature);

//------------------------------------------------------------------------------
/** A helper function, allowing simply to determine, if a feature is writable.
 *  It is a wrapper around the LvGetAccess() function.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @return  If the feature is writable, returns 1, otherwise 0.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY uint32_t LV_STDC LvIsWritable (LvHModule hModule,
                                           LvFeature Feature);

//------------------------------------------------------------------------------
/** Returns the feature type, GUI representation and group.
 *  @param   hModule    A handle to the System, Interface, Device, Stream, Event,
 *                      Buffer or Renderer module.
 *  @param   Feature    The feature ID - use a symbolic constant (one of the
 *                      @ref GroupSynview_Features) or an ID obtained by the
 *                      LvGetFeatureByName() function.
 *  @param   pFtrType   The feature type is returned in this parameter.
 *                      The returned value is one of the @ref LvFtrType.
 *                      Can be NULL.
 *  @param   pFtrGui    The feature GUI representation is returned in this parameter.
 *                      The returned value is one of the @ref LvFtrGui.
 *                      Can be NULL.
 *  @param   pFtrGroup  The feature group, to which the feature belongs.
 *                      The returned value is one of the @ref LvFtrGroup.
 *                      Can be NULL.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvGetType (LvHModule hModule,
                                        LvFeature Feature,
                                        LvEnum* pFtrType,
                                        LvEnum* pFtrGui = NULL,
                                        LvEnum* pFtrGroup = NULL);
#else
LV_DLLENTRY LvStatus LV_STDC LvGetType (LvHModule hModule,
                                        LvFeature Feature,
                                        LvEnum* pFtrType,
                                        LvEnum* pFtrGui,
                                        LvEnum* pFtrGroup);
#endif

//------------------------------------------------------------------------------
/** Gets a Boolean value (as 32-bit integer).
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   pValue    The bool value (as 32-bit integer) is returned in this parameter.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvGetBool (LvHModule hModule,
                                        LvFeature Feature,
                                        int32_t* pValue);

//------------------------------------------------------------------------------
/** Sets a Boolean value.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   Value     Value to be set (in form of 32_bit integer value).
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvSetBool (LvHModule hModule,
                                        LvFeature Feature,
                                        int32_t Value);

//------------------------------------------------------------------------------
/** Gets a 32-bit integer value.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   pValue    The integer value is returned in this parameter.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @note    The value is internally kept always as a 64-bit value; the
 *           functions for setting and getting a 32-bit value are provided
 *           just for convenience.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvGetInt32 (LvHModule hModule,
                                         LvFeature Feature,
                                         int32_t* pValue);

//------------------------------------------------------------------------------
/** Sets a 32-bit value.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   Value     Value to be set.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @note    The value is internally kept always as a 64-bit value; the
 *           functions for setting and getting a 32-bit value are provided
 *           just for convenience.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvSetInt32 (LvHModule hModule,
                                         LvFeature Feature,
                                         int32_t Value);

//------------------------------------------------------------------------------
/** Returns a range and increment of an 32-bit integer feature.
 *  @param   hModule     A handle to the System, Interface, Device, Stream, Event,
 *                       Buffer or Renderer module.
 *  @param   Feature     The feature ID - use a symbolic constant (one of the
 *                       @ref GroupSynview_Features) or an ID obtained by the
 *                       LvGetFeatureByName() function.
 *  @param   pMinValue   The minimum value is returned in this parameter. Can be NULL.
 *  @param   pMaxValue   The maximum value is returned in this parameter. Can be NULL.
 *  @param   pIncrement  The increment value is returned in this parameter. Can be NULL.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @note    The value is internally kept always as a 64-bit value; the
 *           functions for setting and getting a 32-bit value are provided
 *           just for convenience.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvGetInt32Range (LvHModule hModule,
                                              LvFeature Feature, 
                                              int32_t* pMinValue,
                                              int32_t* pMaxValue,
                                              int32_t* pIncrement);

//------------------------------------------------------------------------------
/** Gets a 64-bit integer value.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   pValue    The integer value is returned in this parameter.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @note    This function is equal to the LvGetInt() function.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvGetInt64 (LvHModule hModule,
                                         LvFeature Feature, 
                                         int64_t* pValue);

//------------------------------------------------------------------------------
/** Sets a 64-bit integer value.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   Value     Value to be set.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @note    This function is equal to the LvSetInt() function.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvSetInt64 (LvHModule hModule,
                                         LvFeature Feature, 
                                         int64_t Value);

//------------------------------------------------------------------------------
/** Returns a range and increment of an 64-bit integer feature.
 *  @param   hModule     A handle to the System, Interface, Device, Stream, Event,
 *                       Buffer or Renderer module.
 *  @param   Feature     The feature ID - use a symbolic constant (one of the
 *                       @ref GroupSynview_Features) or an ID obtained by the
 *                       LvGetFeatureByName() function.
 *  @param   pMinValue   The minimum value is returned in this parameter. Can be NULL.
 *  @param   pMaxValue   The maximum value is returned in this parameter. Can be NULL.
 *  @param   pIncrement  The increment value is returned in this parameter. Can be NULL.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @note    This function is equal to the LvGetIntRange() function.
 *  @ingroup GroupSynview_FeatureFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvGetInt64Range (LvHModule hModule,
                                              LvFeature Feature, 
                                              int64_t* pMinValue,
                                              int64_t* pMaxValue,
                                              int64_t* pIncrement = NULL);
#else
LV_DLLENTRY LvStatus LV_STDC LvGetInt64Range (LvHModule hModule,
                                              LvFeature Feature, 
                                              int64_t* pMinValue,
                                              int64_t* pMaxValue,
                                              int64_t* pIncrement);
#endif

//------------------------------------------------------------------------------
/** Gets a 64-bit integer value. 
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   pValue    The integer value is returned in this parameter.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @note    This function is equal to the LvGetInt64() function.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvGetInt (LvHModule hModule,
                                       LvFeature Feature, 
                                       int64_t* pValue);

//------------------------------------------------------------------------------
/** Sets a 64-bit integer value.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   Value     Value to be set.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @note    This function is equal to the LvSetInt64() function.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvSetInt (LvHModule hModule,
                                       LvFeature Feature, 
                                       int64_t Value);

//------------------------------------------------------------------------------
/** Returns a range and increment of an 64-bit integer feature.
 *  @param   hModule     A handle to the System, Interface, Device, Stream, Event,
 *                       Buffer or Renderer module.
 *  @param   Feature     The feature ID - use a symbolic constant (one of the
 *                       @ref GroupSynview_Features) or an ID obtained by the
 *                       LvGetFeatureByName() function.
 *  @param   pMinValue   The minimum value is returned in this parameter. Can be NULL.
 *  @param   pMaxValue   The maximum value is returned in this parameter. Can be NULL.
 *  @param   pIncrement  The increment value is returned in this parameter. Can be NULL.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @note    This function is equal to the LvGetInt64Range() function.
 *  @ingroup GroupSynview_FeatureFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvGetIntRange (LvHModule hModule,
                                            LvFeature Feature, 
                                            int64_t* pMinValue,
                                            int64_t* pMaxValue,
                                            int64_t* pIncrement = NULL);
#else
LV_DLLENTRY LvStatus LV_STDC LvGetIntRange (LvHModule hModule,
                                            LvFeature Feature, 
                                            int64_t* pMinValue,
                                            int64_t* pMaxValue,
                                            int64_t* pIncrement);
#endif

//------------------------------------------------------------------------------
/** Gets a float value.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   pValue    The float value is returned in this parameter.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvGetFloat (LvHModule hModule,
                                         LvFeature Feature,
                                         double* pValue);

//------------------------------------------------------------------------------
/** Sets a float value.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   Value     The value to be set.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvSetFloat (LvHModule hModule,
                                         LvFeature Feature,
                                         double Value);

//------------------------------------------------------------------------------
/** Returns a range of a float feature.
 *  @param   hModule    A handle to the System, Interface, Device, Stream, Event,
 *                      Buffer or Renderer module.
 *  @param   Feature    The feature ID - use a symbolic constant (one of the
 *                      @ref GroupSynview_Features) or an ID obtained by the
 *                      LvGetFeatureByName() function.
 *  @param   pMinValue  The minimum value is returned in this parameter. Can be NULL.
 *  @param   pMaxValue  The maximum value is returned in this parameter. Can be NULL.
 *  @param   pIncrement The increment value is returned in this parameter. If the increment is 
 *                      not defined, 0 is returned. Can be NULL.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvGetFloatRange (LvHModule hModule,
                                              LvFeature Feature,
                                              double* pMinValue,
                                              double* pMaxValue,
                                              double* pIncrement = NULL);
#else
LV_DLLENTRY LvStatus LV_STDC LvGetFloatRange (LvHModule hModule,
                                              LvFeature Feature,
                                              double* pMinValue,
                                              double* pMaxValue,
                                              double* pIncrement);
#endif

//------------------------------------------------------------------------------
/** Gets a string value. If you need first to get the string size, use the
 *  LvGetStringSize() function.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   pValue    Pointer to a null-terminated string buffer.
 *  @param   Size      Size of the buffer.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvGetString (LvHModule hModule,
                                          LvFeature Feature,
                                          char* pValue,
                                          size_t Size);

//------------------------------------------------------------------------------
/** Gets a buffer size needed for a string.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   pSize     Size of the buffer (including space for terminating zero)
 *                     is returned in this parameter.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvGetStringSize (LvHModule hModule,
                                              LvFeature Feature,
                                              size_t* pSize);

//------------------------------------------------------------------------------
/** Sets a string value.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   pValue    The string value (null-terminated).
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvSetString (LvHModule hModule,
                                          LvFeature Feature,
                                          const char* pValue);

//------------------------------------------------------------------------------
/** Gets a block of data.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   pBuffer   Pointer to a buffer, to which the data will be stored.
 *  @param   Size      Size of the buffer.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvGetBuffer (LvHModule hModule,
                                          LvFeature Feature,
                                          void* pBuffer,
                                          size_t Size);

//------------------------------------------------------------------------------
/** Gets the block data size.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   pSize     The needed size of the buffer is returned in this parameter.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvGetBufferSize (LvHModule hModule,
                                              LvFeature Feature,
                                              size_t* pSize);

//------------------------------------------------------------------------------
/** Sets a block of data.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   pBuffer   Pointer to the data.
 *  @param   Size      Size of the data.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvSetBuffer (LvHModule hModule,
                                          LvFeature Feature,
                                          void* pBuffer,
                                          size_t Size);

//------------------------------------------------------------------------------
/** Gets a pointer.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   ppValue   The pointer is returned in this parameter.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvGetPtr (LvHModule hModule,
                                       LvFeature Feature,
                                       void** ppValue);

//------------------------------------------------------------------------------
/** Sets a pointer.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   pValue    The pointer to be set.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvSetPtr (LvHModule hModule,
                                       LvFeature Feature, 
                                       void* pValue);

//------------------------------------------------------------------------------
/** Gets the SynView constant for the enumeration entry, if exists. If does not exist,
 *  you must work with the string enumeration entry value.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   pValue    SynView constant for the enum entry is returned in this parameter.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvGetEnum (LvHModule hModule,
                                        LvFeature Feature,
                                        LvEnum* pValue);

//------------------------------------------------------------------------------
/** Sets the enumeration entry by the SynView constant. If the SynView constant is not
 *  defined for the feature, then use LvSetEnumStr() to set the enum entry by a string.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   Value     SynView constant for the requested enumeration entry.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
// sets the SynView constant for the enum, if exists
LV_DLLENTRY LvStatus LV_STDC LvSetEnum (LvHModule hModule, 
                                        LvFeature Feature,
                                        LvEnum Value);

//------------------------------------------------------------------------------
/** Gets the enumeration entry as string (symbolic name). It is not possible to
 *  get the needed size for this single feature, instead, it is possible to get the
 *  maximum size of the all enum values of this feature, by the
 *  LvGetInfo(LvFtrInfo_EnumEntryNameMaxSize) function.
 *  @param   hModule        A handle to the System, Interface, Device, Stream, Event,
 *                          Buffer or Renderer module.
 *  @param   Feature        The feature ID - use a symbolic constant (one of the
 *                          @ref GroupSynview_Features) or an ID obtained by the
 *                          LvGetFeatureByName() function.
 *  @param   pSymbolicName  A pointer to a string buffer, where the symbolic name will be returned.
 *  @param   Size           Size of the buffer.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvGetEnumStr (LvHModule hModule,
                                           LvFeature Feature,
                                           char* pSymbolicName,
                                           size_t Size);

//------------------------------------------------------------------------------
/** Sets enumeration entry by its string symbolic name.
 *  @param   hModule        A handle to the System, Interface, Device, Stream, Event,
 *                          Buffer or Renderer module.
 *  @param   Feature        The feature ID - use a symbolic constant (one of the
 *                          @ref GroupSynview_Features) or an ID obtained by the
 *                          LvGetFeatureByName() function.
 *  @param   pSymbolicName  A pointer to a string with the symbolic name of the
 *                          enumeration entry.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvSetEnumStr (LvHModule hModule,
                                           LvFeature Feature,
                                           const char* pSymbolicName);

//------------------------------------------------------------------------------
/** Gets the SynView constant for the enumeration entry, if exists.
 *  @param   hModule        A handle to the System, Interface, Device, Stream, Event,
 *                          Buffer or Renderer module.
 *  @param   Feature        The feature ID - use a symbolic constant (one of the
 *                          @ref GroupSynview_Features) or an ID obtained by the
 *                          LvGetFeatureByName() function.
 *  @param   pSymbolicName  A string with symbolic name of the enum entry.
 *  @param   pValue         The SynView constant for the enum entry is returned
 *                          in this parameter. If the SynView constant does not exist for 
 *                          this enumeration entry, 0 is returned (no error in indicated).
 *  @param   pFtrAccess     The feature access is returned in this parameter - one of
 *                          GroupSynview_LvFtrAccess. Can be NULL.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvGetEnumValByStr (LvHModule hModule, 
                                                LvFeature Feature,
                                                const char* pSymbolicName,
                                                LvEnum* pValue,
                                                LvEnum* pFtrAccess = NULL);
#else
LV_DLLENTRY LvStatus LV_STDC LvGetEnumValByStr (LvHModule hModule, 
                                                LvFeature Feature,
                                                const char* pSymbolicName,
                                                LvEnum* pValue,
                                                LvEnum* pFtrAccess);
#endif

//------------------------------------------------------------------------------
/** Returns a string symbolic name of the enum entry for the SynView constant.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   Value     The SynView constant for the enum entry.
 *  @param   pSymbolicName     Pointer to string buffer, where the symbolic name is returned.
 *                             Can be NULL.
 *  @param   SymbolicNameSize  Size of pSymbolicName buffer.
 *  @param   pFtrAccess    The access mode of the enum entry is returned in this parameter - one of
 *                          @ref LvFtrAccess. Can be NULL.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvGetEnumStrByVal (LvHModule hModule,
                                                LvFeature Feature,
                                                LvEnum Value,
                                                char* pSymbolicName,
                                                size_t SymbolicNameSize,
                                                LvEnum* pFtrAccess = NULL);
#else
LV_DLLENTRY LvStatus LV_STDC LvGetEnumStrByVal (LvHModule hModule,
                                                LvFeature Feature,
                                                LvEnum Value,
                                                char* pSymbolicName,
                                                size_t SymbolicNameSize,
                                                LvEnum* pFtrAccess);
#endif

//------------------------------------------------------------------------------
/** Executes a command.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   Timeout   If greater than 0, the LvCmdIsDone() is called in a loop to wait
 *                     for the command completion, until the LvCmdIsDone() returns true or 
 *                     the Timeout (in milliseconds) expires. If set to 0, no wait is done.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvCmdExecute (LvHModule hModule,
                                           LvFeature Feature,
                                           uint32_t Timeout = 0);
#else
LV_DLLENTRY LvStatus LV_STDC LvCmdExecute (LvHModule hModule,
                                           LvFeature Feature,
                                           uint32_t Timeout);
#endif

//------------------------------------------------------------------------------
/** Checks if the command execution has completed.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   pIsDone   In this parameter is returned 1, if the command is
 *                     completed, otherwise 0.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvCmdIsDone (LvHModule hModule,
                                          LvFeature Feature,
                                          uint32_t* pIsDone);

//------------------------------------------------------------------------------
/** Gets the access mode of the feature.
 *  @param   hModule      A handle to the System, Interface, Device, Stream, Event,
 *                        Buffer or Renderer module.
 *  @param   Feature      The feature ID - use a symbolic constant (one of the
 *                        @ref GroupSynview_Features) or an ID obtained by the
 *                        LvGetFeatureByName() function.
 *  @param   pFtrAccess   The access is returned in this parameter. One of the
 *                        @ref LvFtrAccess.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvGetAccess (LvHModule hModule,
                                          LvFeature Feature,
                                          LvEnum* pFtrAccess);

//------------------------------------------------------------------------------
/** Gets the feature visibility (beginner-expert-guru).
 *  @param   hModule          A handle to the System, Interface, Device, Stream, Event,
 *                            Buffer or Renderer module.
 *  @param   Feature          The feature ID - use a symbolic constant (one of the
 *                            @ref GroupSynview_Features) or an ID obtained by the
 *                            LvGetFeatureByName() function.
 *  @param   pFtrVisibility   The visibility is returned in this parameter. One of the
 *                            @ref LvFtrVisibility.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvGetVisibility (LvHModule hModule,
                                              LvFeature Feature,
                                              LvEnum* pFtrVisibility);

//------------------------------------------------------------------------------
/** Gets an info in form of a 32-bit integer value.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   FtrInfo   One of the @ref LvFtrInfo.
 *  @param   pInfo     The value is returned in this parameter.
 *  @param   Param     Additional parameter, required by some types of info.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvGetInfo (LvHModule hModule,
                                        LvFeature Feature,
                                        LvEnum FtrInfo,
                                        int32_t* pInfo,
                                        int32_t Param = 0);
#else
LV_DLLENTRY LvStatus LV_STDC LvGetInfo (LvHModule hModule,
                                        LvFeature Feature,
                                        LvEnum FtrInfo,
                                        int32_t* pInfo,
                                        int32_t Param);
#endif

//------------------------------------------------------------------------------
/** Gets an info in form of a string value.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   FtrInfo   One of the @ref LvFtrInfo.
 *  @param   pInfoStr  The string value is returned in this parameter.
 *  @param   Size      Size of the buffer (to which pInfoStr points).
 *  @param   Param     Additional parameter, required by some types of info.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvGetInfoStr (LvHModule hModule,
                                           LvFeature Feature,
                                           LvEnum FtrInfo,
                                           char* pInfoStr,
                                           size_t Size,
                                           int32_t Param = 0);
#else
LV_DLLENTRY LvStatus LV_STDC LvGetInfoStr (LvHModule hModule,
                                           LvFeature Feature,
                                           LvEnum FtrInfo,
                                           char* pInfoStr,
                                           size_t Size,
                                           int32_t Param);
#endif

//------------------------------------------------------------------------------
/** Gets a buffer size needed for an info in form of a string value.
 *  @param   hModule   A handle to the System, Interface, Device, Stream, Event,
 *                     Buffer or Renderer module.
 *  @param   Feature   The feature ID - use a symbolic constant (one of the
 *                     @ref GroupSynview_Features) or an ID obtained by the
 *                     LvGetFeatureByName() function.
 *  @param   FtrInfo  One of the @ref LvFtrInfo.
 *  @param   pSize     Size of the buffer is returned in this parameter.
 *  @param   Param     Additional parameter, required by some types of info.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvGetInfoStrSize (LvHModule hModule,
                                               LvFeature Feature,
                                               LvEnum FtrInfo,
                                               size_t* pSize,
                                               int32_t Param = 0);
#else
LV_DLLENTRY LvStatus LV_STDC LvGetInfoStrSize (LvHModule hModule,
                                               LvFeature Feature,
                                               LvEnum FtrInfo,
                                               size_t* pSize,
                                               int32_t Param);
#endif

//------------------------------------------------------------------------------
/** Registers or unregisters a callback function for the feature.
 *  This callback is produced by GenApi when a feature changes its value or status.
 *  The application should process this callback fast. Note that the callback can be
 *  called also from another thread - see @ref LvEventType_FeatureDevEvent.
 *  Important note: The feature callback function should never set any other feature.
 *  Doing so can lead to recursions, which would be probably hard to diagnose and
 *  could cause unexpected behavior.
 *  
 *  @param   hModule       A handle to the System, Interface, Device, Stream or Renderer module.
 *  @param   Feature       The feature ID - use a symbolic constant (one of the
 *                         @ref GroupSynview_Features) or an ID obtained by the
 *                         LvGetFeatureByName() function.
 *  @param   pFunction     The callback function in the form of LvFeatureCallbackFunct.
 *                         If you want to unregister the function, use NULL at this parameter.
 *  @param   pUserParam    User parameter, which will be passed to each callback call.
 *  @param   pFeatureParam Second user parameter, which will be passed to each callback call.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvRegisterFeatureCallback(LvHModule hModule,
                                                       LvFeature Feature,
                                                       LvFeatureCallbackFunct pFunction,
                                                       void* pUserParam = NULL, 
                                                       void* pFeatureParam = NULL);
#else
LV_DLLENTRY LvStatus LV_STDC LvRegisterFeatureCallback(LvHModule hModule,
                                                       LvFeature Feature,
                                                       LvFeatureCallbackFunct pFunction,
                                                       void* pUserParam, 
                                                       void* pFeatureParam);
#endif

//------------------------------------------------------------------------------
/** Starts a thread, which in a loop polls the non-cached features. If the feature
 *  polling interval expires, the value is read and the feature callback is called.
 *  
 *  @param   hModule          A handle to the System, Interface, Device, Stream or Renderer module.
 *  @param   PollingTime      A time in milliseconds between 2 calls to poll the features.
 *  @param   PollChildren     If set to true, also the features in all children modules
 *                            are polled. For example, if your application uses only one System 
 *                            module, then it is a parent of all other modules, so the polling
 *                            will be propagated to all modules from a single thread.
 *                            If a module has started own polling thread, then it is excluded
 *                            from the propagating.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
#ifdef __cplusplus
LV_DLLENTRY LvStatus LV_STDC LvStartPollingThread(LvHModule hModule,
                                                  uint32_t PollingTime = 1000,
                                                  int32_t PollChildren = 0);
#else
LV_DLLENTRY LvStatus LV_STDC LvStartPollingThread(LvHModule hModule,
                                                  uint32_t PollingTime,
                                                  int32_t PollChildren);
#endif

//------------------------------------------------------------------------------
/** Stops the polling thread. See LvStartPollingThread() for details.
 *  
 *  @param   hModule          A handle to the System, Interface, Device, Stream or Renderer module.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvStopPollingThread(LvHModule hModule);


//------------------------------------------------------------------------------
/** Polls all the non-cached features of the module. If the feature
 *  polling interval expires, the value is read and the feature callback is called.
 *  
 *  @param   hModule          A handle to the System, Interface, Device, Stream or Renderer module.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FeatureFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvPoll(LvHModule hModule);


//=== Firmware update functions ===============================================

/** Returns the file name mask (with wildcard characters), for searching the
 *  file with the appropriate firmware update. The files with the FW update
 *  have in their names coded the hardware IDs, so using this mask (for example
 *  in a filter in a file open dialog box) assures the file appropriate for this
 *  device is used.
 *  @param   hModule       A handle to the Device module.
 *  @param   Which         An ID specific for a hardware. Discussed in the SynView
 *                         User's Guide.
 *  @param   pFilePattern  In this parameter the file pattern is returned.
 *  @param   Size          Size of the buffer (to which the pFilePattern points).
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FirmwareFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvFwGetFilePattern (LvHModule hModule,
                                                 uint32_t Which,
                                                 char* pFilePattern,
                                                 size_t Size);

//------------------------------------------------------------------------------
/** Loads the firmware from a file to the hardware. It can be very long process
 *  (taking minutes) and this functions blocks the thread druting this process.
 *  It is recommended to check the load status from another thread using the
 *  LvFwGetLoadStatus() function.
 *  @param   hModule     A handle to the Device module.
 *  @param   Which       An ID specific for a hardware. Discussed in the SynView
 *                       User's Guide.
 *  @param   pFilePath   File specification, with full path.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FirmwareFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvFwLoad (LvHModule hModule,
                                       uint32_t Which,
                                       const char* pFilePath);

//------------------------------------------------------------------------------
/** Returns the byte count and whether the loading is still in progress.
 *  @param   hModule            A handle to the Device module.
 *  @param   Which              An ID specific for a hardware. Discussed in the SynView
 *                              User's Guide.
 *  @param   pCurrentByteCount  Returns number of bytes transferred so far.
 *  @param   pIsLoading         Returns 1 if the loading is still in progress.
 *  @return  Returns the @ref LvStatus value indicating the success of the
 *           requested operation. See @ref GroupSynview_LvStatus.
 *  @ingroup GroupSynview_FirmwareFunctions
 */
LV_DLLENTRY LvStatus LV_STDC LvFwGetLoadStatus (LvHModule hModule,
                                                uint32_t Which,
                                                uint32_t* pCurrentByteCount,
                                                uint32_t* pIsLoading);



//=== Typedefs for dynamic load ===============================================

typedef uint32_t (LV_STDC *TfnLvGetVersion                               ) ();
typedef LvStatus (LV_STDC *TfnLvOpenLibrary                              ) ();
typedef LvStatus (LV_STDC *TfnLvCloseLibrary                             ) ();
typedef void     (LV_STDC *TfnLvGetErrorMessage                          ) (LvStatus Error, char* pMessage, size_t Size);
typedef void     (LV_STDC *TfnLvGetLastErrorMessage                      ) (char* pMessage, size_t Size);
typedef void     (LV_STDC *TfnLvLog                                      ) (const char* pLogMessage);
typedef LvStatus (LV_STDC *TfnLvGetLibInfo                               ) (LvEnum Info, int32_t* pInfo, int32_t Param);
typedef LvStatus (LV_STDC *TfnLvGetLibInfoStr                            ) (LvEnum Info, char* pInfoStr, size_t Size, int32_t Param);
typedef LvStatus (LV_STDC *TfnLvGetLibInfoStrSize                        ) (LvEnum Info, size_t* pSize, int32_t Param);
typedef LvStatus (LV_STDC *TfnLvUpdateSystemList                         ) ();
typedef LvStatus (LV_STDC *TfnLvGetNumberOfSystems                       ) (uint32_t* pNumberOfSystems);
typedef LvStatus (LV_STDC *TfnLvGetSystemId                              ) (uint32_t Index, char* pSystemId, size_t Size);
typedef LvStatus (LV_STDC *TfnLvGetSystemIdSize                          ) (uint32_t Index, size_t* pSize);
typedef LvStatus (LV_STDC *TfnLvSystemOpen                               ) (const char* pSystemId, LvHSystem* phSystem);
typedef LvStatus (LV_STDC *TfnLvSystemClose                              ) (LvHSystem* phSystem);
typedef LvStatus (LV_STDC *TfnLvSystemUpdateInterfaceList                ) (LvHSystem hSystem, uint32_t Timeout);
typedef LvStatus (LV_STDC *TfnLvSystemGetNumberOfInterfaces              ) (LvHSystem hSystem, uint32_t* pNumberOfInterfaces);
typedef LvStatus (LV_STDC *TfnLvSystemGetInterfaceId                     ) (LvHSystem hSystem, uint32_t Index, char* pInterfaceId, size_t Size);
typedef LvStatus (LV_STDC *TfnLvSystemGetInterfaceIdSize                 ) (LvHSystem hSystem, uint32_t Index, size_t* pSize);
typedef LvStatus (LV_STDC *TfnLvSystemFindInterface                      ) (LvHSystem hSystem, LvEnum FindBy,  const char* pFindStr, char* pInterfaceId, size_t Size);
typedef LvStatus (LV_STDC *TfnLvInterfaceOpen                            ) (LvHSystem hSystem, const char* pInterfaceId, LvHInterface* phInterface);
typedef LvStatus (LV_STDC *TfnLvInterfaceClose                           ) (LvHInterface* phInterface);
typedef LvStatus (LV_STDC *TfnLvInterfaceUpdateDeviceList                ) (LvHInterface hInterface, uint32_t Timeout);
typedef LvStatus (LV_STDC *TfnLvInterfaceGetNumberOfDevices              ) (LvHInterface hInterface, uint32_t* pDevices);
typedef LvStatus (LV_STDC *TfnLvInterfaceGetDeviceId                     ) (LvHInterface hInterface, uint32_t Index, char* pDeviceId, size_t Size);
typedef LvStatus (LV_STDC *TfnLvInterfaceGetDeviceIdSize                 ) (LvHInterface hInterface, uint32_t Index, size_t* pSize);
typedef LvStatus (LV_STDC *TfnLvInterfaceFindDevice                      ) (LvHInterface hInterface, LvEnum FindBy, const char* pFindStr, char* pDeviceId, size_t Size);
typedef LvStatus (LV_STDC *TfnLvDeviceOpen                               ) (LvHInterface hInterface, const char* pDeviceId, LvHDevice* phDevice, LvEnum Access);
typedef LvStatus (LV_STDC *TfnLvDeviceReOpen                             ) (LvHInterface hInterface, const char* pDeviceId, LvHDevice hDevice, LvEnum Access);
typedef LvStatus (LV_STDC *TfnLvDeviceClose                              ) (LvHDevice* phDevice);
typedef LvStatus (LV_STDC *TfnLvDeviceGetNumberOfStreams                 ) (LvHDevice hDevice, uint32_t* pNumberOfStreams);
typedef LvStatus (LV_STDC *TfnLvDeviceGetStreamId                        ) (LvHDevice hDevice, uint32_t Index, char* pStreamId, size_t Size);
typedef LvStatus (LV_STDC *TfnLvDeviceGetStreamIdSize                    ) (LvHDevice hDevice, uint32_t Index, size_t* pSize);
typedef LvStatus (LV_STDC *TfnLvDeviceAcquisitionStart                   ) (LvHDevice hDevice, uint32_t Options);
typedef LvStatus (LV_STDC *TfnLvDeviceAcquisitionStop                    ) (LvHDevice hDevice, uint32_t Options);
typedef LvStatus (LV_STDC *TfnLvDeviceAcquisitionAbort                   ) (LvHDevice hDevice, uint32_t Options);
typedef LvStatus (LV_STDC *TfnLvDeviceAcquisitionArm                     ) (LvHDevice hDevice, uint32_t Options);
typedef LvStatus (LV_STDC *TfnLvDeviceSetHwLUT                           ) (LvHDevice hDevice, LvEnum Selector, void* pLUT, size_t Size);
typedef LvStatus (LV_STDC *TfnLvDeviceGetHwLUT                           ) (LvHDevice hDevice, LvEnum Selector, void* pLUT, size_t Size);
typedef LvStatus (LV_STDC *TfnLvDeviceSaveSettings                       ) (LvHDevice hDevice, const char* pId, const char* pFileName, uint32_t Options);
typedef LvStatus (LV_STDC *TfnLvDeviceLoadSettings                       ) (LvHDevice hDevice, const char* pId, const char* pFileName, uint32_t Options);
typedef LvStatus (LV_STDC *TfnLvDeviceLoadBatch                          ) (LvHDevice hDevice, const char* pFileName);
typedef LvStatus (LV_STDC *TfnLvDeviceUniSetLut                          ) (LvHDevice hDevice, LvEnum Selector, void* pLUT, size_t Size, uint32_t Options);
typedef LvStatus (LV_STDC *TfnLvDeviceUniGetLut                          ) (LvHDevice hDevice, LvEnum Selector, void* pLUT, size_t Size, uint32_t Options);
typedef LvStatus (LV_STDC *TfnLvStreamOpen                               ) (LvHDevice hDevice, const char* pStreamId, LvHStream* phStream);
typedef LvStatus (LV_STDC *TfnLvStreamClose                              ) (LvHStream* phStream);
typedef LvStatus (LV_STDC *TfnLvStreamGetBufferAt                        ) (LvHStream hStream, uint32_t BufferIndex, LvHBuffer* phBuffer);
typedef LvStatus (LV_STDC *TfnLvStreamFlushQueue                         ) (LvHStream hStream, LvEnum Operation);
typedef LvStatus (LV_STDC *TfnLvStreamStart                              ) (LvHStream hStream, uint32_t StartFlags, uint32_t ImagesToAcquire);
typedef LvStatus (LV_STDC *TfnLvStreamStop                               ) (LvHStream hStream, uint32_t StopFlags);
typedef LvStatus (LV_STDC *TfnLvBufferOpen                               ) (LvHStream hStream, void* pDataPointer, size_t DataSize, void* pUserPointer, uint32_t Options, LvHBuffer* phBuffer);
typedef LvStatus (LV_STDC *TfnLvBufferClose                              ) (LvHBuffer* phBuffer);
typedef LvStatus (LV_STDC *TfnLvBufferAttachProcessBuffer                ) (LvHBuffer hBuffer, void* pDataPointer, size_t DataSize);
typedef LvStatus (LV_STDC *TfnLvBufferQueue                              ) (LvHBuffer hBuffer);
typedef LvStatus (LV_STDC *TfnLvBufferParseChunkData                     ) (LvHBuffer hBuffer, uint32_t UpdateLayout);
typedef LvStatus (LV_STDC *TfnLvBufferSaveImageToBmpFile                 ) (LvHBuffer hBuffer, const char* pFileName);
typedef LvStatus (LV_STDC *TfnLvBufferSaveImageToJpgFile                 ) (LvHBuffer hBuffer, const char* pFileName, uint32_t Quality);
typedef LvStatus (LV_STDC *TfnLvBufferSaveImageToTifFile                 ) (LvHBuffer hBuffer, const char* pFileName, uint32_t Options);
typedef LvStatus (LV_STDC *TfnLvBufferGetImgInfo                         ) (LvHBuffer hBuffer, LvipImgInfo* pImgInfo, uint32_t Options);
typedef LvStatus (LV_STDC *TfnLvBufferGetLastPaintRect                   ) (LvHBuffer hBuffer, int32_t* pX, int32_t* pY, int32_t* pWidth, int32_t* pHeight);
typedef LvStatus (LV_STDC *TfnLvBufferUniCalculateWhiteBalance           ) (LvHBuffer hBuffer);
typedef LvStatus (LV_STDC *TfnLvEventOpen                                ) (LvHSystem hSystem, LvEnum EventType, LvHEvent* phEvent);
typedef LvStatus (LV_STDC *TfnLvEventClose                               ) (LvHEvent* phEvent); 
typedef LvStatus (LV_STDC *TfnLvEventKill                                ) (LvHEvent hEvent);
typedef LvStatus (LV_STDC *TfnLvEventFlush                               ) (LvHEvent hEvent);
typedef LvStatus (LV_STDC *TfnLvEventWaitAndGetData                      ) (LvHEvent hEvent, void* pBuffer, size_t* pSize, uint32_t Timeout);
typedef LvStatus (LV_STDC *TfnLvEventWaitAndGetNewBuffer                 ) (LvHEvent hEvent, LvHBuffer* phBuffer, uint32_t Timeout);
typedef LvStatus (LV_STDC *TfnLvEventGetDataInfo                         ) (LvHEvent hEvent, void* pInBuffer, size_t InSize, LvEnum Info, void* pBuffer, size_t* pSize, LvEnum* pType, int32_t Param);
typedef LvStatus (LV_STDC *TfnLvEventPutData                             ) (LvHEvent hEvent, void* pBuffer, size_t Size);
typedef LvStatus (LV_STDC *TfnLvEventSetCallback                         ) (LvHEvent hEvent, LvEventCallbackFunct pFunction, void* pUserParam);
typedef LvStatus (LV_STDC *TfnLvEventSetCallbackNewBuffer                ) (LvHEvent hEvent, LvEventCallbackNewBufferFunct pFunction, void* pUserParam);
typedef LvStatus (LV_STDC *TfnLvEventRegisterFeatureForCallback          ) (LvHEvent hEvent, LvFeature Feature, void* pFeatureParam, LvEnum Register);
typedef LvStatus (LV_STDC *TfnLvEventStartThread                         ) (LvHEvent hEvent);
typedef LvStatus (LV_STDC *TfnLvEventStopThread                          ) (LvHEvent hEvent);
typedef uint32_t (LV_STDC *TfnLvEventCallbackMustExit                    ) (LvHEvent hEvent);
typedef LvStatus (LV_STDC *TfnLvGetNumFeatures                           ) (LvHModule hModule, LvEnum FeatureGroup, uint32_t* pNumItems);
typedef LvStatus (LV_STDC *TfnLvGetFeatureAt                             ) (LvHModule hModule, LvEnum FeatureGroup, uint32_t Index, LvFeature* pItem, uint32_t* pLevel);
typedef LvStatus (LV_STDC *TfnLvGetFeatureByName                         ) (LvHModule hModule, LvEnum FeatureGroup, const char* pName, LvFeature* pItem);
typedef uint32_t (LV_STDC *TfnLvIsImplemented                            ) (LvHModule hModule, LvFeature Feature);
typedef uint32_t (LV_STDC *TfnLvIsImplementedByName                      ) (LvHModule hModule, LvEnum FeatureGroup, const char* pName);
typedef uint32_t (LV_STDC *TfnLvIsImplementedEnumEntry                   ) (LvHModule hModule, LvFeature Feature, LvEnum EnumEntry);
typedef uint32_t (LV_STDC *TfnLvIsAvailable                              ) (LvHModule hModule, LvFeature Feature);
typedef uint32_t (LV_STDC *TfnLvIsAvailableByName                        ) (LvHModule hModule, LvEnum FeatureGroup, const char* pName);
typedef uint32_t (LV_STDC *TfnLvIsAvailableEnumEntry                     ) (LvHModule hModule, LvFeature Feature, LvEnum EnumEntry);
typedef uint32_t (LV_STDC *TfnLvIsReadable                               ) (LvHModule hModule, LvFeature Feature);
typedef uint32_t (LV_STDC *TfnLvIsWritable                               ) (LvHModule hModule, LvFeature Feature);
typedef LvStatus (LV_STDC *TfnLvGetType                                  ) (LvHModule hModule, LvFeature Feature, LvEnum* pFtrType, LvEnum* pFtrGui, LvEnum* pFtrGroup);
typedef LvStatus (LV_STDC *TfnLvGetInt32                                 ) (LvHModule hModule, LvFeature Feature, int32_t* pValue);
typedef LvStatus (LV_STDC *TfnLvSetInt32                                 ) (LvHModule hModule, LvFeature Feature, int32_t Value);
typedef LvStatus (LV_STDC *TfnLvGetInt32Range                            ) (LvHModule hModule, LvFeature Feature, int32_t* pMinValue, int32_t* pMaxValue, int32_t* pIncrement);
typedef LvStatus (LV_STDC *TfnLvGetInt64                                 ) (LvHModule hModule, LvFeature Feature, int64_t* pValue);
typedef LvStatus (LV_STDC *TfnLvSetInt64                                 ) (LvHModule hModule, LvFeature Feature, int64_t Value);
typedef LvStatus (LV_STDC *TfnLvGetInt64Range                            ) (LvHModule hModule, LvFeature Feature, int64_t* pMinValue, int64_t* pMaxValue, int64_t* pIncrement);
typedef LvStatus (LV_STDC *TfnLvGetInt                                   ) (LvHModule hModule, LvFeature Feature, int64_t* pValue);
typedef LvStatus (LV_STDC *TfnLvSetInt                                   ) (LvHModule hModule, LvFeature Feature, int64_t Value);
typedef LvStatus (LV_STDC *TfnLvGetIntRange                              ) (LvHModule hModule, LvFeature Feature, int64_t* pMinValue, int64_t* pMaxValue, int64_t* pIncrement);
typedef LvStatus (LV_STDC *TfnLvGetFloat                                 ) (LvHModule hModule, LvFeature Feature, double* pValue);
typedef LvStatus (LV_STDC *TfnLvSetFloat                                 ) (LvHModule hModule, LvFeature Feature, double Value);
typedef LvStatus (LV_STDC *TfnLvGetFloatRange                            ) (LvHModule hModule, LvFeature Feature, double* pMinValue, double* pMaxValue, double* pIncrement);
typedef LvStatus (LV_STDC *TfnLvGetBool                                  ) (LvHModule hModule, LvFeature Feature, int32_t* pValue);
typedef LvStatus (LV_STDC *TfnLvSetBool                                  ) (LvHModule hModule, LvFeature Feature, int32_t Value);
typedef LvStatus (LV_STDC *TfnLvGetString                                ) (LvHModule hModule, LvFeature Feature, char* pValue, size_t Size);
typedef LvStatus (LV_STDC *TfnLvGetStringSize                            ) (LvHModule hModule, LvFeature Feature, size_t* pSize);
typedef LvStatus (LV_STDC *TfnLvSetString                                ) (LvHModule hModule, LvFeature Feature, const char* pValue);
typedef LvStatus (LV_STDC *TfnLvGetBuffer                                ) (LvHModule hModule, LvFeature Feature, void* pBuffer, size_t Size);
typedef LvStatus (LV_STDC *TfnLvGetBufferSize                            ) (LvHModule hModule, LvFeature Feature, size_t* pSize);
typedef LvStatus (LV_STDC *TfnLvSetBuffer                                ) (LvHModule hModule, LvFeature Feature, void* pBuffer, size_t Size);
typedef LvStatus (LV_STDC *TfnLvGetPtr                                   ) (LvHModule hModule, LvFeature Feature, void** ppValue);
typedef LvStatus (LV_STDC *TfnLvSetPtr                                   ) (LvHModule hModule, LvFeature Feature, void* pValue);
typedef LvStatus (LV_STDC *TfnLvGetEnum                                  ) (LvHModule hModule, LvFeature Feature, LvEnum* pValue);
typedef LvStatus (LV_STDC *TfnLvSetEnum                                  ) (LvHModule hModule, LvFeature Feature, LvEnum Value);
typedef LvStatus (LV_STDC *TfnLvGetEnumStr                               ) (LvHModule hModule, LvFeature Feature, char* pSymbolicName, size_t SymbolicNameSize);
typedef LvStatus (LV_STDC *TfnLvSetEnumStr                               ) (LvHModule hModule, LvFeature Feature, const char* pSymbolicName);
typedef LvStatus (LV_STDC *TfnLvGetEnumValByStr                          ) (LvHModule hModule, LvFeature Feature, const char* pSymbolicName, LvEnum* pValue, LvEnum* pFtrAccess);
typedef LvStatus (LV_STDC *TfnLvGetEnumStrByVal                          ) (LvHModule hModule, LvFeature Feature, LvEnum Value, char* pSymbolicName, size_t SymbolicNameSize, LvEnum* pFtrAccess);
typedef LvStatus (LV_STDC *TfnLvCmdExecute                               ) (LvHModule hModule, LvFeature Feature, uint32_t Timeout);
typedef LvStatus (LV_STDC *TfnLvCmdIsDone                                ) (LvHModule hModule, LvFeature Feature, uint32_t* pIsDone);
typedef LvStatus (LV_STDC *TfnLvGetAccess                                ) (LvHModule hModule, LvFeature Feature, LvEnum* pFtrAccess);
typedef LvStatus (LV_STDC *TfnLvGetVisibility                            ) (LvHModule hModule, LvFeature Feature, LvEnum* pFtrVisibility);
typedef LvStatus (LV_STDC *TfnLvGetInfo                                  ) (LvHModule hModule, LvFeature Feature, LvEnum FtrInfo, int32_t* pInfo, int32_t Param);
typedef LvStatus (LV_STDC *TfnLvGetInfoStr                               ) (LvHModule hModule, LvFeature Feature, LvEnum FtrInfo, char* pInfoStr, size_t Size, int32_t Param);
typedef LvStatus (LV_STDC *TfnLvGetInfoStrSize                           ) (LvHModule hModule, LvFeature Feature, LvEnum FtrInfo, size_t* pSize, int32_t Param);
typedef LvStatus (LV_STDC *TfnLvRegisterFeatureCallback                  ) (LvHModule hModule, LvFeature Feature, LvFeatureCallbackFunct pFunction, void* pUserParam, void* pFeatureParam);
typedef LvStatus (LV_STDC *TfnLvStartPollingThread                       ) (LvHModule hModule, uint32_t PollingTime, int32_t PollChildren);
typedef LvStatus (LV_STDC *TfnLvStopPollingThread                        ) (LvHModule hModule);
typedef LvStatus (LV_STDC *TfnLvPoll                                     ) (LvHModule hModule);
typedef LvStatus (LV_STDC *TfnLvFwGetFilePattern                         ) (LvHModule hModule, uint32_t Which, char* pFilePattern, size_t Size);
typedef LvStatus (LV_STDC *TfnLvFwLoad                                   ) (LvHModule hModule, uint32_t Which, const char* pFilePath);
typedef LvStatus (LV_STDC *TfnLvFwGetLoadStatus                          ) (LvHModule hModule, uint32_t Which, uint32_t* pCurrentByteCount, uint32_t* pIsLoading);
typedef LvStatus (LV_STDC *TfnLvRendererOpen                             ) (LvHStream hStream, LvHRenderer* phRenderer);
typedef LvStatus (LV_STDC *TfnLvRendererStart                            ) (LvHRenderer* phRenderer);
typedef LvStatus (LV_STDC *TfnLvRendererStop                             ) (LvHRenderer* phRenderer);
typedef LvStatus (LV_STDC *TfnLvRendererClose                            ) (LvHRenderer* phRenderer);
#ifdef _WIN32
typedef LvStatus (LV_STDC *TfnLvRendererSetWindow                        ) (LvHRenderer hRenderer, HWND hWnd);
typedef LvStatus (LV_STDC *TfnLvRendererGetWindow                        ) (LvHRenderer hRenderer, HWND* phWnd);
#else
typedef LvStatus (LV_STDC *TfnLvRendererSetWindow                        ) (LvHRenderer hRenderer, void* pDisplay, int64_t hWindow);
#endif                                                                 
typedef LvStatus (LV_STDC *TfnLvRendererDisplayImage                     ) (LvHRenderer hRenderer, LvHBuffer hBuffer, uint32_t RenderFlags);
typedef LvStatus (LV_STDC *TfnLvRendererCanDisplayImage                  ) (LvHRenderer hRenderer, LvHBuffer hBuffer, uint32_t RenderFlags);
typedef LvStatus (LV_STDC *TfnLvRendererRepaint                          ) (LvHRenderer hRenderer, uint32_t RenderFlags);
#ifdef LVAVISAVER
typedef LvStatus (LV_STDC *TfnLvAviSaverOpen                             ) (LvHStream hStream, LvHAviSaver* phAviSaver);
typedef LvStatus (LV_STDC *TfnLvAviSaverClose                            ) (LvHAviSaver* phAviSaver);
typedef LvStatus (LV_STDC *TfnLvAviSaverStoreImage                       ) (LvHAviSaver hAviSaver, LvHBuffer hBuffer, uint32_t Options);
typedef LvStatus (LV_STDC *TfnLvAviSaverOpenFile                         ) (LvHAviSaver hAviSaver, const char* pFileName, uint32_t Options);
typedef LvStatus (LV_STDC *TfnLvAviSaverCloseFile                        ) (LvHAviSaver hAviSaver);
typedef LvStatus (LV_STDC *TfnLvAviSaverSaveSettings                     ) (LvHAviSaver hAviSaver, const char* pId, const char* pFileName, uint32_t Options);
typedef LvStatus (LV_STDC *TfnLvAviSaverLoadSettings                     ) (LvHAviSaver hAviSaver, const char* pId, const char* pFileName, uint32_t Options);
#endif



#endif
