#ifndef SV_SYNVIEW_ENUMS_H
#define SV_SYNVIEW_ENUMS_H
                                       
#include "sv.synview.defs.h"

/** @defgroup GroupSynview_Features Features
 *  @ingroup GroupSynview_Enums
 */

/** @defgroup GroupSynview_EnumEntries Enumeration entries
 *  @ingroup GroupSynview_Enums
 */

//------------------------------------------------------------------------------

/** @defgroup GroupSynview_StreamStartFlags  LvStreamStart() flags definitions
 *  @ingroup GroupSynview_Defines
 */
/*@{*/
/** Default stream start flag */
#define LvStreamStartFlags_Default 0x00000000
/*@}*/

//------------------------------------------------------------------------------

/** @defgroup GroupSynview_StreamStopFlags  LvStreamStop() flags definitions
 *  @ingroup GroupSynview_Defines
 */
/*@{*/
/** Stop the acquisition engine when the currently running tasks like filling a
 *  buffer are completed. This is the default. */
#define LvStreamStopFlags_Default  0x00000000

/** Stop the acquisition engine immediately and leave buffers currently being
 *  filled in the Input Buffer Pool. */
#define LvStreamStopFlags_Kill     0x00000001
/*@}*/

//------------------------------------------------------------------------------

/** @defgroup GroupSynview_UniLutFlags  LvDeviceUniSetLut() and LvDeviceUniGetLut() flags definitions
 *  @ingroup GroupSynview_Defines
 */
/*@{*/
/** If present, the operation is done directly on HW LUT, passing the UniProcess 
  * mechanism. */
#define LvUniLutFlags_HwLut 0x00000001

/*@}*/

//------------------------------------------------------------------------------

/** @defgroup GroupSynview_SaveFlags  LvSaveFlag definitions
 *  @ingroup GroupSynview_Defines
 */
/*@{*/
/** Save/load device remote XML features. */
#define LvSaveFlag_RemoteFtr        0x00000001 
/** Save/load device local XML features. */
#define LvSaveFlag_LocalFtr         0x00000002 
/** Save/load device GenTL XML features. */
#define LvSaveFlag_GenTlFtr         0x00000004 
/** Save/load device all features (combines all flags above). */
#define LvSaveFlag_All              0x0000000F 
/** If specified, the remote device FW version check is not done when reading the file - the file is read
 *  even if it was created by device with a different FW version (this may lead to errors by some features).
 */
#define LvSaveFlag_IgnoreVersion    0x00000100 
/** If specified, the remote device model check is not done when reading the file - the file is read
 *  even if it was created by different device model (this may lead to errors by some features).
 */
#define LvSaveFlag_IgnoreModel      0x00000200 
/*@}*/

//------------------------------------------------------------------------------

/** Enum values for the Info parameter of the LvGetLibInfo(), LvGetLibInfoStr() and
 *  LvGetLibInfoStrSize() functions.
 *  @ingroup GroupSynview_Enums
 *  @hideinitializer
 */
enum LvLibInfo
{
    /** Returns the full path to the SynView binaries (applications and libraries -
     *  in Windows the Bin folder of SynView). LvFtrType_String. */
    LvInfo_BinPath       = 0,

    /** Returns the full path to the SynView application data. This folder may be
     *  different from the BinPath, for example in Windows Vista the BinPath is
     *  write protected, while AppDataPath is at the read-write location and
     *  contains files like sv.synview.log etc. LvFtrType_String. */
    LvInfo_AppDataPath   = 1,

    /** Returns the full path to the SynView user data. In Windows this is equal to 
     *  AppDataPath. LvFtrType_String. */
    LvInfo_UserDataPath  = 2,

    /** Returns the full path to the SynView config data. In Windows this is equal to 
     *  AppDataPath. LvFtrType_String. */
    LvInfo_CfgPath       = 3,

    /** Returns the full path to the SynView installation root folder. LvFtrType_String. */
    LvInfo_InstPath      = 4,

    /** Returns the full path to the lv.SynView.ini file. LvFtrType_String. */
    LvInfo_IniFile       = 5,

    /** Returns the build date of the library. */
    LvInfo_BuildDate     = 10
};

//------------------------------------------------------------------------------


/** LvFtrVersion constants.
 *  Define the version of SFNC (Standard Features Naming Convention).
 */
enum LvFtrVersion
{
    LvFtrVersion_Unknown = 0xFFFFFFFF,
	LvFtrVersion_151     = 0x00010501,
	LvFtrVersion_200     = 0x00020000
};


/** LvFtrGroup constants.
 *  Define the group of features. The group is composed of the module and the
 *  feature origin. The richest set is belonging to the Device module:
 *    - Device remote features are those, which are provided by the device itself through GenICam GenApi.
 *    - Device GenTL  features are those, which are provided by the GenTL library through GenICam GenApi.
 *    - Device local  features are those, which are implemented directly in the SynView library.
 *  Used in LvGetNumFeatures(), LvGetFeatureAt(), LvGetFeatureByName().
 *  @ingroup GroupSynview_Enums
 *  @hideinitializer
 */
enum LvFtrGroup
{
    LvFtrGroup_DeviceRemote    =   3, /**< Device remote features obtained from the device GenApi node tree. */

    LvFtrGroup_SystemGtl       =  11, /**< System    features obtained from the GenTL GenApi node tree. */
    LvFtrGroup_InterfaceGtl    =  12, /**< Interface features obtained from the GenTL GenApi node tree. */
    LvFtrGroup_DeviceGtl       =  13, /**< Device    features obtained from the GenTL GenApi node tree. */
    LvFtrGroup_StreamGtl       =  14, /**< Stream    features obtained from the GenTL GenApi node tree. */
    LvFtrGroup_BufferGtl       =  15, /**< Buffer    features obtained from the GenTL GenApi node tree. */

    LvFtrGroup_SystemLocal     =  21, /**< System    local features, implemented in SynView. */
    LvFtrGroup_InterfaceLocal  =  22, /**< Interface local features, implemented in SynView. */
    LvFtrGroup_DeviceLocal     =  23, /**< Device    local features, implemented in SynView. */
    LvFtrGroup_StreamLocal     =  24, /**< Stream    local features, implemented in SynView. */
    LvFtrGroup_BufferLocal     =  25, /**< Buffer    local features, implemented in SynView. */
    LvFtrGroup_RendererLocal   =  26, /**< Renderer  local features, implemented in SynView. */
    #ifdef LVAVISAVER
        LvFtrGroup_AviSaverLocal   =  27, 
    #endif
    LvFtrGroup_EventLocal      =  28, /**< Event     local features, implemented in SynView. */

    #ifdef SIMPLON_INC_OBSOLETE
    LvFtrGroup_Buffer          =  35, /**< Obsolete - will be removed. Buffer local features, implemented in SynView. */
    LvFtrGroup_Event           =  38, /**< Obsolete - will be removed. Event local features, implemented in SynView. */
    #endif

    LvFtrGroup_BufferItemsGtl  =  86, /**< Obsolete - will be removed. Buffer local GenTL features obtained from the GenTL plain C API. */
    LvFtrGroup_EventItemsGtl   =  85, /**< Obsolete - will be removed. Event  local GenTL features obtained from the GenTL plain C API. */

    LvFtrGroup_SystemHidden    =  91, /**< System    hidden features. Do not use, reserved for special purposes. */
    LvFtrGroup_InterfaceHidden =  92, /**< Interface hidden features. Do not use, reserved for special purposes. */
    LvFtrGroup_DeviceHidden    =  93, /**< Device    hidden features. Do not use, reserved for special purposes. */
    LvFtrGroup_StreamHidden    =  94, /**< Stream    hidden features. Do not use, reserved for special purposes. */
    LvFtrGroup_BufferHidden    =  95, /**< Buffer    hidden features. Do not use, reserved for special purposes. */
    LvFtrGroup_RendererHidden  =  96, /**< Renderer  hidden features. Do not use, reserved for special purposes. */
    #ifdef LVAVISAVER
        LvFtrGroup_AviSaverHidden  =  97,
    #endif
    LvFtrGroup_EventHidden     =  98  /**< Event hidden features. Do not use, reserved for special purposes. */
};

//------------------------------------------------------------------------------

/** LvFtrType constants. Define the type of the feature. Used in the LvGetType() function.
 *  @ingroup GroupSynview_Enums
 *  @hideinitializer
 */
enum LvFtrType
{
    /** Integer type, use LvGetInt32(), LvSetInt32(), LvGetInt64(), LvSetInt64() to get/set a value. */
    LvFtrType_Integer        = 1,
    
    /** Float type, use LvGetFloat() and LvSetFloat() to get/set a value. */
    LvFtrType_Float          = 2,
    
    /** String type, use LvGetString() and LvSetString() to get/set a value. */
    LvFtrType_String         = 3,
    
    /** Enumeration type, use LvGetEnum(), LvSetEnum(), LvGetEnumStr() and LvSetEnumStr() to get/set a value. */
    LvFtrType_Enumeration    = 4,
    
    /** Boolean type, use LvGetInt32() and LvSetInt32() to get/set a value. */
    LvFtrType_Boolean        = 5,
    
    /** Command type, use LvCmdExecute() and LvCmdIsDone() to execute and check. */
    LvFtrType_Command        = 6,
    
    /** Category type, used in the tree of feaures build. */
    LvFtrType_Category       = 7,
    
    /** String list type (multiple strings in one, separated by terminating 0), use LvGetString() and LvSetString() to get/set a value. */
    LvFtrType_StringList     = 8,
    
    /** Pointer type, use LvGetPtr() and LvSetPtr() to get/set a value. */
    LvFtrType_Pointer        = 9,
    
    /** Buffer type (in GenICam it corresponds with the Register type), use LvGetBuffer() and LvSetBuffer() to get/set a value. Do not confuse it with the LvBuffer module. */
    LvFtrType_Buffer         = 10,
    
    /** Unknown type, cannot be accessed. */
    LvFtrType_Other          = 11
};

//------------------------------------------------------------------------------

/** LvFtrGui constants. Define the recommended GUI representation of the feature. Used in the LvGetType() function.
 *  @ingroup GroupSynview_Enums
 *  @hideinitializer
 */
enum LvFtrGui
{
    LvFtrGui_IntEdit         = 0,  /**< The recommended representation is an edit box with a decimal value. Used by @ref LvFtrType_Integer. */
    LvFtrGui_IntEditHex      = 1,  /**< The recommended representation is an edit box with a hexadecimal value. Used by @ref LvFtrType_Integer. */
    LvFtrGui_IntSlider       = 2,  /**< The recommended representation is a linear slider. Used by @ref LvFtrType_Integer. */
    LvFtrGui_IntSliderLog    = 3,  /**< The recommended representation is a logarithmic slider. Used by @ref LvFtrType_Integer. */
    LvFtrGui_FloatEdit       = 4,  /**< The recommended representation is an edit box. Used by @ref LvFtrType_Float. */
    LvFtrGui_FloatSlider     = 5,  /**< The recommended representation is a linear slider. Used by @ref LvFtrType_Float. */
    LvFtrGui_FloatSliderLog  = 6,  /**< The recommended representation is a logarithmic slider. Used by @ref LvFtrType_Float. */
    LvFtrGui_Label           = 7,  /**< The recommended representation is read-only label. Used by @ref LvFtrType_Category. */
    LvFtrGui_StringEdit      = 8,  /**< The recommended representation is an edit box for a string. Used by @ref LvFtrType_String. */
    LvFtrGui_CheckBox        = 9,  /**< The recommended representation is a check box. Used by @ref LvFtrType_Boolean. */
    LvFtrGui_ComboBox        = 10, /**< The recommended representation is a combo box. Used by @ref LvFtrType_Boolean. */
    LvFtrGui_Button          = 11, /**< The recommended representation is a button. Used by @ref LvFtrType_Command. */
    LvFtrGui_IpV4Address     = 12, /**< The recommended representation is an edit box for a string with an IP address in the form N.N.N.N. Used by @ref LvFtrType_Integer. */
    LvFtrGui_IpMacAddress    = 13, /**< The recommended representation is an edit box for a string with a MAC address in the form XX:XX:XX:XX:XX:XX. Used by @ref LvFtrType_Integer. */
    LvFtrGui_Undefined       = 0xFF  /**< The recommended representation is not defined. */
};

//------------------------------------------------------------------------------

/** LvFtrVisibility constants. Define the visibility level of the feature. Used in LvGetVisibility(). 
 *  Should be used for displaying the feature tree (or list).
 *  @ingroup GroupSynview_Enums
 *  @hideinitializer
 */
enum LvFtrVisibility
{
    LvFtrVisibility_Beginner  = 0,  /**< Beginner level - the feature should be displayed always. */
    LvFtrVisibility_Expert    = 1,  /**< Expert level - the feature should be displayed if at least the Expert level is selected.*/
    LvFtrVisibility_Guru      = 2,  /**< Guru level - the feature should be displayed if at least the Guru level is selected. */
    LvFtrVisibility_Invisible = 3   /**< Invisible - the feature should not be displayed. */
};

//------------------------------------------------------------------------------

/** LvFtrAccess constants.
 *  Define the current feature access mode. Used in the LvGetAccess(). Also used for
 *  enumeration features in functions LvGetEnumValByStr() and LvGetEnumStrByVal().
 *  @ingroup GroupSynview_Enums
 *  @hideinitializer
 */
enum LvFtrAccess
{
    LvFtrAccess_NotImplemented = 0, /**< The feature is not implemented at all. */
    LvFtrAccess_NotAvailable   = 1, /**< The feature is implemented, but under the current conditions is not available. */
    LvFtrAccess_WriteOnly      = 2, /**< The feature is available and is write only. */
    LvFtrAccess_ReadOnly       = 3, /**< The feature is available and is read only. */
    LvFtrAccess_ReadWrite      = 4  /**< The feature is available and is fully accessible. */
};

//------------------------------------------------------------------------------

/** LvFtrInfo constants. 
 *  Define the info type when querying for feature info by the LvGetInfo() and LvGetInfoStr()
 *  functions.
 *  @ingroup GroupSynview_Enums
 *  @hideinitializer
 */
enum LvFtrInfo
{
    /** Returns 1 if the feature has the Streamable attribute set. To be used in the LvGetInfo() function. */
    LvFtrInfo_IsStreamable         = 100,

    /** Returns 1 if the feature should not be used directly, because SynView provides
     *  for this functionality a native API. For example the
     *  AcquisitionStart and AcquisitionStop device remote features are wrapped by
     *  additional functionality in SynView (for example locking TL params before the
     *  AcquisitionStart command is issued). To be used in the LvGetInfo() function. */
    LvFtrInfo_IsWrapped            = 101,
    
    /** Returns 1 if the feature is a selector, that means subsequent features are
     *  indexed by it . To be used in the LvGetInfo() function. */
    LvFtrInfo_IsSelector           = 102,
    
    /** Returns 1 if the feature is cached. To be used in the LvGetInfo() function. */
    LvFtrInfo_IsCached             = 103,

    /** Returns the polling time for a non-cached feature. If the feature is dependent on
     *  other non-cached features, the returned polling time is the minimum found.
     *  The polling time defines recommended time to update the non-cached feature. 
     *  For example the LvDevice_DeviceTemperature is a typical non-cached feature - it changes 
     *  independentely and a as it changes slowly, the recommended polling time might be
     *  10000 = 10 seconds, i.e. the application, which displays the temperature, should
     *  update it on screen every 10 seconds.
     *  The returned value -1 means the polling time is not defined.
     *  To be used in the LvGetInfo() function. */
    LvFtrInfo_PollingTime          = 104,

    /** Returns the feature Name. Do not confuse it with the DisplayName - the Name is the string identifier,
     *  by which the feature can be identified and a numeric ID can be obtained for further actions (generic
     *  feature access). To be used in the LvGetInfoStr() function. */
    LvFtrInfo_Name                 = 200,
    
    /** Returns the feature Display name for representation in GUI. To be used in the LvGetInfoStr() function. */
    LvFtrInfo_DisplayName          = 201,
    
    /** Returns the feature Description text. To be used in the LvGetInfoStr() function. */
    LvFtrInfo_Description          = 202,

    /** Returns the feature Physical units, if defined. To be used in the LvGetInfoStr() function. */
    LvFtrInfo_PhysicalUnits        = 203,

    /** Returns the feature Tooltip (a short description to be used in the GUI).
     *  To be used in the LvGetInfoStr() function. */
    LvFtrInfo_ToolTip              = 204,
    
    /** Returns the SynView symbolic constant of the feature, as a string (utilized in
     *  the Source code generator). To be used in the LvGetInfoStr() function. */
    LvFtrInfo_SymbolicConst        = 205,

    /** Returns the SynView symbolic constant of the enumeration feature, as a string (utilized in
     *  the Source code generator). To be used in the LvGetInfoStr() function. */
    LvFtrInfo_SymbolicEnumConst    = 206,
    
    /** Returns the string ID of selected features belonging under this selector. Param = index (utilized in
     *  the Source code generator). To be used in the LvGetInfoStr() function. */
    LvFtrInfo_SelectedFeatures     = 208,

    /** Returns the string ID of selecting features under which this feature belongs. Param = index (utilized in
     *  the Source code generator). To be used in the LvGetInfoStr() function. */
    LvFtrInfo_SelectingFeatures    = 209,

    /** Returns the SynView symbolic constant for the feature group, to which the feature belongs, 
     *  as a string (utilized in the Source code generator). To be used in the LvGetInfoStr() function. */
    LvFtrInfo_SymbolicGroupConst   = 210,

    /** Returns the string indicating the type of module, to which the fature belongs, 
     *  for example "System", "Interface", "Device", ...
     *  (utilized in the Source code generator). To be used in the LvGetInfoStr() function. */
    LvFtrInfo_ModuleName           = 211,

    /** Returns 1 if for this feature can be safely used 32-bit integer instead of 64-bit (if the 
      * feaure is of the LvFtrType_Integer type).
     *  This info is utilized in the source code generator. To be used in the LvGetInfo() function. */
    LvFtrInfo_FitsTo32Bit          = 212,

    /** Returns 1 if this feature is either permanently read-only (cannot become read-write depending on 
     *  other features), or the feature is writable, but it is not usual to set its value from code.
     *  This info is utilized in the source code generator. To be used in the LvGetInfo() function. */
    LvFtrInfo_TakeAsReadOnly       = 213,

    /** Returns the symbolic name of the enum entry. To be used in the LvGetInfoStr() function.
      * The Param specifies a zero based index of the entry or the 
      * SynView enum entry constant. You can obtain the number of 
      * entries by the LvGetInfo() function with the @ref LvFtrInfo_EnumEntryCount 
      * parameter. If the Param is set to @ref LV_ENUMENTRY_CURRENT, the returned info is for the currently 
      * selected enum entry. */
    LvFtrInfo_EnumEntryName        = 300,

    /** Returns the display name of the enum entry. To be used in the LvGetInfoStr() function.
      * The Param specifies a zero based index of the entry or the 
      * SynView enum entry constant. You can obtain the number of 
      * entries by the LvGetInfo() function with the @ref LvFtrInfo_EnumEntryCount 
      * parameter. If the Param is set to @ref LV_ENUMENTRY_CURRENT, the returned info is for the currently 
      * selected enum entry. */
    LvFtrInfo_EnumEntryDisplayName = 301,

    /** Returns the description of the enum entry. To be used in the LvGetInfoStr() function.
      * The Param specifies a zero based index of the entry or the 
      * SynView enum entry constant. You can obtain the number of 
      * entries by the LvGetInfo() function with the @ref LvFtrInfo_EnumEntryCount 
      * parameter. If the Param is set to @ref LV_ENUMENTRY_CURRENT, the returned info is for the currently 
      * selected enum entry. */
    LvFtrInfo_EnumEntryDescription = 302,

    /** Returns the tooltip of the enum entry. To be used in the LvGetInfoStr() function.
      * The Param specifies a zero based index of the entry or the 
      * SynView enum entry constant. You can obtain the number of 
      * entries by the LvGetInfo() function with the @ref LvFtrInfo_EnumEntryCount 
      * parameter. If the Param is set to @ref LV_ENUMENTRY_CURRENT, the returned info is for the currently 
      * selected enum entry. */
    LvFtrInfo_EnumEntryToolTip     = 303,

    /** Returns the access of the enum entry (one of the @ref LvFtrAccess constants). 
      * To be used in the LvGetInfo() function.
      * The Param specifies a zero based index of the entry or the 
      * SynView enum entry constant. You can obtain the number of 
      * entries by the LvGetInfo() function with the @ref LvFtrInfo_EnumEntryCount 
      * parameter. If the Param is set to @ref LV_ENUMENTRY_CURRENT, the returned info is for the currently 
      * selected enum entry. */
    LvFtrInfo_EnumEntryAccess      = 304,

    /** Returns the SynView constant for the enum entry (if exists). To be used in the LvGetInfo() function.
      * The Param specifies a zero based index of the entry. You can obtain the number of 
      * entries by the LvGetInfo() function with the @ref LvFtrInfo_EnumEntryCount 
      * parameter. If the Param is set to @ref LV_ENUMENTRY_CURRENT, the returned info is for the currently 
      * selected enum entry. */
    LvFtrInfo_EnumEntryValue       = 305,

    /** Returns the number of enum entries for the enum. To be used in the LvGetInfo() function. */
    LvFtrInfo_EnumEntryCount       = 306,

    /** Returns the maximum string size needed (including terminating zero) 
     *  for any entry name of the enum To be used in the LvGetInfo() function. */
    LvFtrInfo_EnumEntryNameMaxSize = 307,

    /** Returns the string ID of the interface. Param = interface index. This constant
     *  can be used only in the LvSystem module for enumerating unopened interfaces
     *  (LvGetInfoStr() function, as the Feature use @ref LvSystem_Info). */
    LvFtrInfo_InterfaceID          = 0 | LV_ITF_GTL,

    /** Returns the Display name of the interface. Param = interface index. This constant
     *  can be used only in the LvSystem module for enumerating unopened interfaces
     *  (LvGetInfoStr() function, as the Feature use @ref LvSystem_Info). */
    LvFtrInfo_InterfaceDisplayName = 1 | LV_ITF_GTL,

    /** Returns the interface Transport layer type. Param = interface index. This constant
     *  can be used only in the LvSystem module for enumerating unopened interfaces
     *  (LvGetInfoStr() function, as the Feature use @ref LvSystem_Info).
     *  For example a standard interface TL type is "GEV" for
     *  GigE-Vision devices. */
    LvFtrInfo_InterfaceTlType      = 2 | LV_ITF_GTL,

  
    /** Returns the string ID of the device. Param = device index. This constant
     *  can be used only in the LvInterface module for enumerating unopened devices
     *  (LvGetInfoStr() function, as the Feature use @ref LvInterface_Info). */
    LvFtrInfo_DeviceID             = 0 | LV_DEV_GTL,

    /** Returns the Vendor name of the device. Param = device index. This constant
     *  can be used only in the LvInterface module for enumerating unopened devices
     *  (LvGetInfoStr() function, as the Feature use @ref LvInterface_Info). */
    LvFtrInfo_DeviceVendor         = 1 | LV_DEV_GTL,

    /** Returns the Model name of the device. Param = device index. This constant
     *  can be used only in the LvInterface module for enumerating unopened devices
     *  (LvGetInfoStr() function, as the Feature use @ref LvInterface_Info). */
    LvFtrInfo_DeviceModel          = 2 | LV_DEV_GTL,

    /** Returns the Transport layer type of the device. Param = device index. This constant
     *  can be used only in the LvInterface module for enumerating unopened devices
     *  (LvGetInfoStr() function, as the Feature use @ref LvInterface_Info). */
    LvFtrInfo_DeviceTlType         = 3 | LV_DEV_GTL,

    /** Returns the Display name of the device. Param = device index. This constant
     *  can be used only in the LvInterface module for enumerating unopened devices
     *  (LvGetInfoStr() function, as the Feature use @ref LvInterface_Info). */
    LvFtrInfo_DeviceDisplayName    = 4 | LV_DEV_GTL,

    /** Returns the device access status. Param = device index. The returned value is
     *  one of the @ref LvDeviceAccessStatus constants. Can be used only in the
     *  LvInterface module for enumerating unopened devices (LvGetInfo() function,
     *  as the Feature use @ref LvInterface_Info). */
    LvFtrInfo_DeviceAccessStatus   = 5 | LV_DEV_GTL,

    /** Returns the User name of the device. Param = device index. This constant
     *  can be used only in the LvInterface module for enumerating unopened devices
     *  (LvGetInfoStr() function, as the Feature use @ref LvInterface_Info). */
    LvFtrInfo_DeviceUserName       = 6 | LV_DEV_GTL,

    /** Returns the serial number of the device. Param = device index. This constant
     *  can be used only in the LvInterface module for enumerating unopened devices
     *  (LvGetInfoStr() function, as the Feature use @ref LvInterface_Info). */
    LvFtrInfo_DeviceSerialNumber   = 7 | LV_DEV_GTL,

    /** Returns the version number of the device. Param = device index. This constant
     *  can be used only in the LvInterface module for enumerating unopened devices
     *  (LvGetInfoStr() function, as the Feature use @ref LvInterface_Info). */
    LvFtrInfo_DeviceVersion        = 8 | LV_DEV_GTL

};

//------------------------------------------------------------------------------

/** LvSystemFtr constants. 
 *  @ingroup GroupSynview_Features
 *  @hideinitializer
 */
enum LvSystemFtr
{
    // --- System GenTL features -----------------------------------------------

    /** GenTL producer vendor name. @ref LvFtrType_String.
     *  System GenTL feature (@ref LvFtrGroup_SystemGtl). */
    LvSystem_TLVendorName                                =     0 | LV_SYS_GFTR,

    /** GenTL producer model name. @ref LvFtrType_String.
     *  System GenTL feature (@ref LvFtrGroup_SystemGtl). */
    LvSystem_TLModelName                                 =     1 | LV_SYS_GFTR,

    /** GenTL producer ID. @ref LvFtrType_String.
     *  System GenTL feature (@ref LvFtrGroup_SystemGtl). */
    LvSystem_TLID                                        =     2 | LV_SYS_GFTR,

    /** GenTL producer version. @ref LvFtrType_String.
     *  System GenTL feature (@ref LvFtrGroup_SystemGtl). */
    LvSystem_TLVersion                                   =     3 | LV_SYS_GFTR,

    /** GenTL producer path. @ref LvFtrType_String.
     *  System GenTL feature (@ref LvFtrGroup_SystemGtl). */
    LvSystem_TLPath                                      =     4 | LV_SYS_GFTR,

    /** GenTL producer type. @ref LvFtrType_Enumeration.
     *  Values: @ref LvTLType.
     *  System GenTL feature (@ref LvFtrGroup_SystemGtl). */
    LvSystem_TLType                                      =     5 | LV_SYS_GFTR,

    /** GenTL version major. @ref LvFtrType_Integer.
     *  System GenTL feature (@ref LvFtrGroup_SystemGtl). */
    LvSystem_GenTLVersionMajor                           =     6 | LV_SYS_GFTR,

    /** GenTL version minor. @ref LvFtrType_Integer.
     *  System GenTL feature (@ref LvFtrGroup_SystemGtl). */
    LvSystem_GenTLVersionMinor                           =     7 | LV_SYS_GFTR,

    /** GigE Vision version major. @ref LvFtrType_Integer.
     *  System GenTL feature (@ref LvFtrGroup_SystemGtl). */
    LvSystem_GevVersionMajor                             =     8 | LV_SYS_GFTR,

    /** GigE Vision version minor. @ref LvFtrType_Integer.
     *  System GenTL feature (@ref LvFtrGroup_SystemGtl). */
    LvSystem_GevVersionMinor                             =     9 | LV_SYS_GFTR,

    /** Update internal list of interfaces. @ref LvFtrType_Command.
     *  System GenTL feature (@ref LvFtrGroup_SystemGtl). */
    LvSystem_InterfaceUpdateList                         =    10 | LV_SYS_GFTR,

    /** Interface selector. @ref LvFtrType_Integer.
     *  System GenTL feature (@ref LvFtrGroup_SystemGtl). */
    LvSystem_InterfaceSelector                           =    11 | LV_SYS_GFTR,

    /** Interface ID. @ref LvFtrType_String.
     *  Depends on @ref LvSystem_InterfaceSelector.
     *  System GenTL feature (@ref LvFtrGroup_SystemGtl). */
    LvSystem_InterfaceID                                 =    12 | LV_SYS_GFTR,

    /** Interface MAC address. @ref LvFtrType_Integer.
     *  Depends on @ref LvSystem_InterfaceSelector.
     *  System GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvSystem_GevInterfaceMACAddress                      =    13 | LV_SYS_GFTR,

    /** Interface default IP address. @ref LvFtrType_Integer.
     *  Depends on @ref LvSystem_InterfaceSelector.
     *  System GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvSystem_GevInterfaceDefaultIPAddress                =    14 | LV_SYS_GFTR,

    /** Interface default subnet mask. @ref LvFtrType_Integer. 
     *  Depends on @ref LvSystem_InterfaceSelector.
     *  System GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvSystem_GevInterfaceDefaultSubnetMask               =    15 | LV_SYS_GFTR,

    /** Interface default gateway. @ref LvFtrType_Integer.
     *  Depends on @ref LvSystem_InterfaceSelector.
     *  System GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvSystem_GevInterfaceDefaultGateway                  =    16 | LV_SYS_GFTR,

    /** GenTL SFNC version major. @ref LvFtrType_Integer.
     *  System GenTL feature (@ref LvFtrGroup_SystemGtl). */
    LvSystem_GenTLSFNCVersionMajor                       =    17 | LV_SYS_GFTR,

    /** GenTL SFNC version minor. @ref LvFtrType_Integer.
     *  System GenTL feature (@ref LvFtrGroup_SystemGtl). */
    LvSystem_GenTLSFNCVersionMinor                       =    18 | LV_SYS_GFTR,

    /** Interface Type. @ref LvFtrType_Enumeration.
     *  Depends on @ref LvSystem_InterfaceSelector.
     *  System GenTL feature (@ref LvFtrGroup_SystemGtl). */
    LvSystem_InterfaceType                               =    19 | LV_SYS_GFTR,

    // --- System local features -----------------------------------------------

    /** User readable name of the system. @ref LvFtrType_String.
     *  System local feature (@ref LvFtrGroup_SystemLocal). */
    LvSystem_LvSystemDisplayName                         =     0 | LV_SYS_LFTR,

    // --- System special constants --------------------------------------------

    /** Constant to be used in LvGetInfo() and LvGetInfoStr() to obtain
     *  various info about the system. */
    LvSystem_Info                                        =     0 | LV_SYS_NFTR
};

//------------------------------------------------------------------------------

/** LvInterfaceFtr constants. 
 *  @ingroup GroupSynview_Features
 *  @hideinitializer
 */
enum LvInterfaceFtr
{
    // --- Interface GenTL features --------------------------------------------

    /** Interface ID. @ref LvFtrType_String.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_InterfaceID                              =     0 | LV_ITF_GFTR,

    /** Interface type. @ref LvFtrType_Enumeration.
     *  Values: @ref LvInterfaceType.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_InterfaceType                            =     1 | LV_ITF_GFTR,

    /** Interface gateway selector. @ref LvFtrType_Integer.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_GevInterfaceGatewaySelector              =     2 | LV_ITF_GFTR,

    /** Interface gateway. @ref LvFtrType_Integer.
     *  Depends on @ref LvInterface_GevInterfaceGatewaySelector.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_GevInterfaceGateway                      =     3 | LV_ITF_GFTR,

    /** Interface MAC address. @ref LvFtrType_Integer.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_GevInterfaceMACAddress                   =     4 | LV_ITF_GFTR,
    LvInterface_GevMACAddress                            =     4 | LV_ITF_GFTR,

    /** Interface subnet selector. @ref LvFtrType_Integer.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_GevInterfaceSubnetSelector               =     5 | LV_ITF_GFTR,

    /** Interface subnet IP address. @ref LvFtrType_Integer.
     *  Depends on @ref LvInterface_GevInterfaceSubnetSelector.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_GevInterfaceSubnetIPAddress              =     6 | LV_ITF_GFTR,

    /** Interface subnet mask. @ref LvFtrType_Integer.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_GevInterfaceSubnetMask                   =     7 | LV_ITF_GFTR,

    /** Update internal list ofdevices. @ref LvFtrType_Command.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_DeviceUpdateList                         =     8 | LV_ITF_GFTR,

    /** Device selector. @ref LvFtrType_Integer.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_DeviceSelector                           =     9 | LV_ITF_GFTR,

    /** Device ID. @ref LvFtrType_String.
     *  Depends on @ref LvInterface_DeviceSelector.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_DeviceID                                 =    10 | LV_ITF_GFTR,

    /** Device vendor name. @ref LvFtrType_String.
     *  Depends on @ref LvInterface_DeviceSelector.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_DeviceVendorName                         =    11 | LV_ITF_GFTR,

    /** Device Model name. @ref LvFtrType_String.
     *  Depends on @ref LvInterface_DeviceSelector.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_DeviceModelName                          =    12 | LV_ITF_GFTR,

    /** Device access status. @ref LvFtrType_Enumeration.
     *  Values: @ref LvDeviceAccessStatus.
     *  Depends on @ref LvInterface_DeviceSelector.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_DeviceAccessStatus                       =    13 | LV_ITF_GFTR,

    /** Device IP address. @ref LvFtrType_Integer.
     *  Depends on @ref LvInterface_DeviceSelector.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_GevDeviceIPAddress                       =    14 | LV_ITF_GFTR,

    /** Device subnet mask. @ref LvFtrType_Integer.
     *  Depends on @ref LvInterface_DeviceSelector.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_GevDeviceSubnetMask                      =    15 | LV_ITF_GFTR,

    /** Device MAC address. @ref LvFtrType_Integer.
     *  Depends on @ref LvInterface_DeviceSelector.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_GevDeviceMACAddress                      =    16 | LV_ITF_GFTR,

    /** Device User ID. @ref LvFtrType_String.
     *  Depends on @ref LvInterface_DeviceSelector.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_DeviceUserID                             =    17 | LV_ITF_GFTR,
    LvInterface_LvDeviceUserID                           =    17 | LV_ITF_GFTR,

    /** Device identifier (serial number). @ref LvFtrType_String.
     *  Depends on @ref LvInterface_DeviceSelector.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_DeviceSerialNumber                       =    18 | LV_ITF_GFTR,
    LvInterface_LvDeviceSerialNumber                     =    18 | LV_ITF_GFTR,

    /** Interface TL version major. @ref LvFtrType_Integer.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_InterfaceTLVersionMajor                  =    19 | LV_ITF_GFTR,

    /** Interface TL version minor. @ref LvFtrType_Integer.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_InterfaceTLVersionMinor                  =    20 | LV_ITF_GFTR,

    /** Device TL version major. @ref LvFtrType_Integer.
     *  Depends on @ref LvInterface_DeviceSelector.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_DeviceTLVersionMajor                     =    21 | LV_ITF_GFTR,

    /** Device TL version minor. @ref LvFtrType_Integer.
     *  Depends on @ref LvInterface_DeviceSelector.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_DeviceTLVersionMinor                     =    22 | LV_ITF_GFTR,

    /** Device type. @ref LvFtrType_Enumeration.
     *  Values: @ref LvDeviceType.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_DeviceType                               =    23 | LV_ITF_GFTR,

    /** Device version. @ref LvFtrType_String.
     *  Depends on @ref LvInterface_DeviceSelector.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_DeviceVersion                            =    24 | LV_ITF_GFTR,

    /** Device display name. @ref LvFtrType_String.
     *  Depends on @ref LvInterface_DeviceSelector.
     *  Interface GenTL feature (@ref LvFtrGroup_InterfaceGtl). */
    LvInterface_DeviceDisplayName                        =    25 | LV_ITF_GFTR,


    // --- Interface local features --------------------------------------------

    /** User readable name of the interface. @ref LvFtrType_String.
     *  Interface local feature (@ref LvFtrGroup_InterfaceLocal).*/
    LvInterface_InterfaceDisplayName                     =     0 | LV_ITF_LFTR,
    LvInterface_LvInterfaceDisplayName                   =     0 | LV_ITF_LFTR,


    // --- Interface special constants -----------------------------------------

    /** Constant to be used in LvGetInfo() and LvGetInfoStr() to obtain
     *  various info about the interface. */
    LvInterface_Info                                     =     0 | LV_ITF_NFTR
};

//------------------------------------------------------------------------------

/** LvDeviceFtr constants. 
 *  @ingroup GroupSynview_Features
 *  @hideinitializer
 */
enum LvDeviceFtr
{
    // --- Device remote features ----------------------------------------------

    /** Name of the manufacturer of the device. @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_DeviceVendorName                            =     0 | LV_DEV_RFTR,

    /** Model name of the device. @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_DeviceModelName                             =     1 | LV_DEV_RFTR,

    /** Manufacturer information about the device. @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_DeviceManufacturerInfo                      =     2 | LV_DEV_RFTR,

    /** Version of the device. @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_DeviceVersion                               =     3 | LV_DEV_RFTR,

    /** Version of the firmware loaded in the device. @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_DeviceFirmwareVersion                       =     4 | LV_DEV_RFTR,

    /** String that indicates the version of the firmware and
     *  software to which the device would recover. @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvRecoveryFirmwareVersion                   =     5 | LV_DEV_RFTR,

    /** Device identifier (serial number). @ref LvFtrType_String.
     *  Note: This feature is called DeviceID in the SFNC, but we use 
     *  rather the DeviceSerialNumber in order not to confuse it with the
     *  GenTL DeviceID, which is used for the device opening.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_DeviceSerialNumber                          =     6 | LV_DEV_RFTR,

    /** User-programmable device identifier. @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_DeviceUserID                                =     7 | LV_DEV_RFTR,

    /** Serial number of the sensor board. @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSensorID                                  =     8 | LV_DEV_RFTR,

    /** Serial number of the grabber board. @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvGrabberID                                 =     9 | LV_DEV_RFTR,

    /** Scan type of the sensor. @ref LvFtrType_Enumeration.
     *  Values: @ref LvDeviceScanType.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_DeviceScanType                              =    10 | LV_DEV_RFTR,

    /** Prepare the device for registers streaming without
     *  checking for consistency. @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_DeviceRegistersStreamingStart               =    11 | LV_DEV_RFTR,

    /** Announce the end of registers streaming. This will
     *  do a register set validation for consistency and
     *  activate it. @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_DeviceRegistersStreamingEnd                 =    12 | LV_DEV_RFTR,

    /** Perform the validation of the current register set for
     *  consistency. @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_DeviceRegistersCheck                        =    13 | LV_DEV_RFTR,

    /** Reports if the current register set is valid and
     *  consistent. @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_DeviceRegistersValid                        =    14 | LV_DEV_RFTR,

    /** Resets the device and to put it in its power up
     *  state. @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_DeviceReset                                 =    15 | LV_DEV_RFTR,

    /** Selects a device clock frequency to be configured. @ref LvFtrType_Enumeration.
     *  Values: @ref LvDeviceClockSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_DeviceClockSelector                         =    16 | LV_DEV_RFTR,

    /** Frequency of the selected clock in Hz. @ref LvFtrType_Float.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_DeviceClockFrequency                        =    17 | LV_DEV_RFTR,

    /** Selects the location within the device, where the
     *  temperature will be measured. @ref LvFtrType_Enumeration.
     *  Values: @ref LvDeviceTemperatureSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_DeviceTemperatureSelector                   =    18 | LV_DEV_RFTR,

    /** Current temperature at the selected location in
     *  degrees of Celcius @ref LvFtrType_Float.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_DeviceTemperature                           =    19 | LV_DEV_RFTR,

    /** Current up-time of the device in milliseconds. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvDeviceUpTime                              =    20 | LV_DEV_RFTR,

    #ifdef SIMPLON_INC_OBSOLETE
    /** Number of watchdog resets in camera's lifetime. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvDeviceNrWatchdogResets                    =    21 | LV_DEV_RFTR,
    #endif

    /** String that indicates the basic type of the device. @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvDeviceType                                =    22 | LV_DEV_RFTR,

    /** Effective width of the sensor in pixels. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_SensorWidth                                 =    23 | LV_DEV_RFTR,

    /** Effective height of the sensor in pixels. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_SensorHeight                                =    24 | LV_DEV_RFTR,

    /** Maximum width of the image in pixels. The dimension is
     *  calculated after applying horizontal binning, decimation
     *  or readout width. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_WidthMax                                    =    25 | LV_DEV_RFTR,

    /** Maximum height of the image in pixels. The dimension is
     *  calculated after applying vertical binning, decimation
     *  or readout height. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_HeightMax                                   =    26 | LV_DEV_RFTR,

    /** Image width provided by the device in pixels. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_Width                                       =    27 | LV_DEV_RFTR,

    /** Image height provided by the device in pixels. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_Height                                      =    28 | LV_DEV_RFTR,

    /** Horizontal offset from the origin of the AOI (area of interest)
     *  in pixels. The AOI is applied to the result of binning
     *  and or decimation. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_OffsetX                                     =    29 | LV_DEV_RFTR,

    /** Vertical offset from the origin of the AOI (area of interest)
     *  in pixels. The AOI is applied to the result of binning and or
     *  decimation. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_OffsetY                                     =    30 | LV_DEV_RFTR,

    /** Pixel format provided by the device. The feature combines pixel
     *  coding, size and color filter attributes. @ref LvFtrType_Enumeration.
     *  Values: see @ref LvPixelFormat.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_PixelFormat                                 =    31 | LV_DEV_RFTR,

    /** Horizontal binning, number of horizontal pixels to combine together.
     *  This increases the intensity (and S/N ratio) of the pixels and reduces
     *  the horizontal resolution (width) of the image. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_BinningHorizontal                           =    32 | LV_DEV_RFTR,

    /** Vertical binning, number of vertical pixels to combine together.
     *  This increases the intensity (and S/N ratio) of the pixels and reduces
     *  the vertical resolution (height) of the image. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_BinningVertical                             =    33 | LV_DEV_RFTR,

    /** Horizontal decimation (sub-sampling) of the image. This reduces the
     *  horizontal resolution (width) of the image by the specified factor. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_DecimationHorizontal                        =    34 | LV_DEV_RFTR,

    /** Vertical decimation (sub-sampling) of the image. This reduces the
     *  vertical resolution (height) of the image by the specified factor. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_DecimationVertical                          =    35 | LV_DEV_RFTR,

    /** Selects the mode of controlling the area of interest @ref LvFtrType_Enumeration.
     *  Values: @ref LvAOIMode.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvAOIMode                                   =    36 | LV_DEV_RFTR,

    /** Width of the sensor-side area of interest in pixels. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvReadoutWidth                              =    37 | LV_DEV_RFTR,

    /** Height of the sensor-side area of interest in pixels. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvReadoutHeight                             =    38 | LV_DEV_RFTR,

    /** X offset (left offset) for the sensor-side area of interest in
     *  pixels. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvReadoutOffsetX                            =    39 | LV_DEV_RFTR,

    /** Y offset (top offset) for the sensor-side area of interest
     *  in pixels. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvReadoutOffsetY                            =    40 | LV_DEV_RFTR,

    /** This flag controls, whether the payload size can change during acquisition. 
     *  When set, the image dimensions and other paramenters can vary during 
     *  acquisition.
     *  @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvVariablePayloadSize                       =    41 | LV_DEV_RFTR,

    /** Sets the acquisition mode of the device. It defines mainly the number
     *  of frames to capture during an acquisition and the way the acquisition
     *  stops. @ref LvFtrType_Enumeration.
     *  Values: @ref LvAcquisitionMode.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_AcquisitionMode                             =    42 | LV_DEV_RFTR,

    /** Selects the type of trigger to configure. @ref LvFtrType_Enumeration.
     *  Values: @ref LvTriggerSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_TriggerSelector                             =    43 | LV_DEV_RFTR,

    /** Controls if the selected trigger is active. @ref LvFtrType_Enumeration.
     *  Values: @ref LvTriggerMode.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_TriggerMode                                 =    44 | LV_DEV_RFTR,

    /** Generates a software trigger when trigger source is set to 'software'
     *  or any physical line. @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_TriggerSoftware                             =    45 | LV_DEV_RFTR,

    /** Specifies the internal signal or physical input line to use as the
     *  trigger source. @ref LvFtrType_Enumeration.
     *  Values: @ref LvTriggerSource.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_TriggerSource                               =    46 | LV_DEV_RFTR,

    /** Activation mode of the trigger - specifies which edge of the signal
     *  is active. @ref LvFtrType_Enumeration.
     *  Values: @ref LvTriggerActivation.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_TriggerActivation                           =    47 | LV_DEV_RFTR,

    /** Trigger delay in microseconds, specifies a delay introduced between
     *  the trigger reception and its actual activation. @ref LvFtrType_Float.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_TriggerDelay                                =    48 | LV_DEV_RFTR,

    /** Used to divide the number of incoming trigger pulses by an
     *  integer factor. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_TriggerDivider                              =    49 | LV_DEV_RFTR,

    /** Sets the caching mode for the selected trigger. The feature controls
     *  how early triggers are treated by the device. @ref LvFtrType_Enumeration.
     *  Values: @ref LvTriggerCaching.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvTriggerCaching                            =    50 | LV_DEV_RFTR,

    /** Controls the exposure (shutter) mode applied for each
     *  acquisition. @ref LvFtrType_Enumeration.
     *  Values: @ref LvExposureMode.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ExposureMode                                =    51 | LV_DEV_RFTR,

    /** Switches to mode with wider range of exposure times, but
     *  slightly higher jitter. @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvLongRangeExposureMode                     =    52 | LV_DEV_RFTR,

    /** Switches to mode with wider range of exposure times, but
     *  slightly higher jitter. @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvGlobalResetMode                           =    53 | LV_DEV_RFTR,

    /** Exposure time in microseconds. The feature controls how long
     *  are the pixels exposed to illumination. @ref LvFtrType_Float.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ExposureTime                                =    54 | LV_DEV_RFTR,

    /** Selects the automatic exposure mode. @ref LvFtrType_Enumeration.
     *  Values: @ref LvExposureAuto.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ExposureAuto                                =    55 | LV_DEV_RFTR,

    /** Switches the acquisition frame rate control on or off. The camera
     *  might internally switch to different working mode, which can decrease
     *  the maximum frame rate. @ref LvFtrType_Enumeration.
     *  Values: @ref LvAcquisitionFrameRateControlMode.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvAcquisitionFrameRateControlMode           =    56 | LV_DEV_RFTR,

    /** Acquisition frame rate in frames per second (Hz) - the frequency
     *  with which the image frames are captured. @ref LvFtrType_Float.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_AcquisitionFrameRate                        =    57 | LV_DEV_RFTR,

    /** Selects the I/O line for querying and configuration. Note that
     *  to use given line to drive a device feature (trigger, counter,
     *  etc.), source of the given feature has to refer to the line. @ref LvFtrType_Enumeration.
     *  Values: @ref LvLineSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LineSelector                                =    58 | LV_DEV_RFTR,

    /** Line mode - controls, whether given line is used as signal input or
     *  output. @ref LvFtrType_Enumeration.
     *  Values: @ref LvLineMode.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LineMode                                    =    59 | LV_DEV_RFTR,

    /** This feature controls the current electrical format of the
     *  selected physical input or output Line. @ref LvFtrType_Enumeration. 
     *  Values: @ref LvLineFormat.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LineFormat                                  =    60 | LV_DEV_RFTR,

    /** Selects a device internal signal that should drive the output
     *  signal of the selected line. LineMode must be Output. Not applicable
     *  for input lines. @ref LvFtrType_Enumeration.
     *  Values: @ref LvLineSource.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LineSource                                  =    61 | LV_DEV_RFTR,

    /** Inverts the signal output on the selected line. @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LineInverter                                =    62 | LV_DEV_RFTR,

    /** Reports the current status of the selected line. 
     *  @ref LvFtrType_Boolean.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LineStatus                                  =    63 | LV_DEV_RFTR,

    /** Bit field indicating status of all i/o lines. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LineStatusAll                               =    64 | LV_DEV_RFTR,

    /** Selects the user output for querying and configuration. 
     *  @ref LvFtrType_Enumeration.  
     *  Values: @ref LvUserOutputSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_UserOutputSelector                          =    65 | LV_DEV_RFTR,

    /** Reports the current status of the selected user output. 
     *  @ref LvFtrType_Boolean.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_UserOutputValue                             =    66 | LV_DEV_RFTR,

    /** Bit field indicating status of all user outputs. Only the bits defined 
     *  in the User Output Value All Mask are used, the others are ignored. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_UserOutputValueAll                          =    67 | LV_DEV_RFTR,

    /** Mask for the User Output Value All bitfield - defines which bits are 
     *  used to change a user output value and which are ignored. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_UserOutputValueAllMask                      =    68 | LV_DEV_RFTR,




    /** Selects which counter to configure. @ref LvFtrType_Enumeration.
     *  Values: @ref LvCounterSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_CounterSelector                             =    69 | LV_DEV_RFTR,

    /** Selects working mode of the selected counter. @ref LvFtrType_Enumeration.
     *  Values: @ref LvCounterMode.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCounterMode                               =    70 | LV_DEV_RFTR,

    /** Internal device signal incrementing the selected counter. @ref LvFtrType_Enumeration.
     *  Values: @ref LvCounterEventSource.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_CounterEventSource                          =    71 | LV_DEV_RFTR,

    /** This command resets the selected counter @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_CounterReset                                =    72 | LV_DEV_RFTR,

    /** Reads or sets the current value of the selected counter. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_CounterValue                                =    73 | LV_DEV_RFTR,

    /** Duration (or number of events) before the counter end event is
     *  generated and the counter expires. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_CounterDuration                             =    74 | LV_DEV_RFTR,

    /** Selects which timer to configure. @ref LvFtrType_Enumeration.
     *  Values: @ref LvTimerSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_TimerSelector                               =    75 | LV_DEV_RFTR,

    /** Sets the duration (in microseconds) of the timer active pulse. @ref LvFtrType_Float.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_TimerDuration                               =    76 | LV_DEV_RFTR,

    /** Sets the delay (in microseconds) applied between activating the
     *  timer and issuing the timer active signal. @ref LvFtrType_Float.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_TimerDelay                                  =    77 | LV_DEV_RFTR,

    /** Internal device signal activating the selected timer. @ref LvFtrType_Enumeration.
     *  Values: @ref LvTimerTriggerSource.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_TimerTriggerSource                          =    78 | LV_DEV_RFTR,

    /** Selects the special purpose trigger type to configure. @ref LvFtrType_Enumeration.
     *  Values: @ref LvSpecialPurposeTriggerSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSpecialPurposeTriggerSelector             =    79 | LV_DEV_RFTR,

    /** Specifies the internal signal or physical input line to use as the
     *  trigger source. @ref LvFtrType_Enumeration.
     *  Values: @ref LvSpecialPurposeTriggerSource.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSpecialPurposeTriggerSource               =    80 | LV_DEV_RFTR,

    /** Activation mode of the trigger - specifies which edge of the signal
     *  is active. @ref LvFtrType_Enumeration.
     *  Values: @ref LvSpecialPurposeTriggerActivation.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSpecialPurposeTriggerActivation           =    81 | LV_DEV_RFTR,

    /** Generates a software trigger for the selected trigger action when
     *  trigger source is set to 'software' or any physical line. @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSpecialPurposeTriggerSoftware             =    82 | LV_DEV_RFTR,

    /** A single bitfield that selects which features will be reset by the
     *  timestamp reset trigger in one access. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvImageStampsResetMask                      =    83 | LV_DEV_RFTR,

    /** Selects an image stamp type for configuration.
     *  @ref LvFtrType_Enumeration.
     *  Values: @ref LvImageStampSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvImageStampSelector                        =    84 | LV_DEV_RFTR,

    /** Enables/disables the reset trigger functionality for the selected image stamp type.
     *  @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvImageStampResetEnable                     =    85 | LV_DEV_RFTR,

    /** Selects the firmware type to load on next boot. @ref LvFtrType_Enumeration.
     *  Values: @ref LvBootSwitch.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvBootSwitch                                =    86 | LV_DEV_RFTR,

    /** Selects the algorithm used by the Bayer decoder. @ref LvFtrType_Enumeration.
     *  Values: @ref LvBayerDecoderAlgorithm.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvBayerDecoderAlgorithm                     =    87 | LV_DEV_RFTR,

    /** Sets the threshold controlling the performance of the variable
     *  gradient Bayer decoder algorithm. @ref LvFtrType_Float.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvBayerDecoderThreshold                     =    88 | LV_DEV_RFTR,

    /** Enables the watchdog reset function. 
     *  @ref LvFtrType_Boolean.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvWatchdogEnable                            =    89 | LV_DEV_RFTR,

    /** When watchdog is enabled, the device reboots when the timeout specified expires.
     *  @ref LvFtrType_Float.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvWatchdogTimerDuration                     =    90 | LV_DEV_RFTR,

    /** Resets the watchdog timer, the watchdog starts counting the specified timeout again. 
     *  @ref LvFtrType_Command.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvWatchdogTimerReset                        =    91 | LV_DEV_RFTR,

    /** Signals that the last device reboot was initiated by the watchdog function. 
     *  After reading, reset this flag explicitly, it wouldn't be affected 
     *  by a 'warm' system reboot. 
     *  @ref LvFtrType_Boolean.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvWatchdogFailed                            =    92 | LV_DEV_RFTR,

    /** Selects which gain type to configure. @ref LvFtrType_Enumeration.
     *  Values: @ref LvGainSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GainSelector                                =    93 | LV_DEV_RFTR,

    /** Gain value for the selected gain type in dB. This is an amplification
     *  factor applied to the video signal. @ref LvFtrType_Float.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_Gain                                        =    94 | LV_DEV_RFTR,

    /** Controls the automatic gain control (AGC) mode. @ref LvFtrType_Enumeration.
     *  Values: @ref LvGainAuto.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GainAuto                                    =    95 | LV_DEV_RFTR,

    /** Selects which black level type to configure. @ref LvFtrType_Enumeration.
     *  Values: @ref LvBlackLevelSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_BlackLevelSelector                          =    96 | LV_DEV_RFTR,

    /** Controls the analog black level. This represents a DC offset applied
     *  to the video signal. @ref LvFtrType_Float.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_BlackLevel                                  =    97 | LV_DEV_RFTR,

    /** Controls the automatic black level mode. @ref LvFtrType_Enumeration.
     *  Values: @ref LvBlackLevelAuto.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_BlackLevelAuto                              =    98 | LV_DEV_RFTR,

    /** Selects which color transformation module is controlled by the
     *  color transformation features. It also gives particular meaning to
     *  individual color transformation gains. @ref LvFtrType_Enumeration.
     *  Values: @ref LvColorTransformationSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ColorTransformationSelector                 =    99 | LV_DEV_RFTR,

    /** Activates the selected Color Transformation module. @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ColorTransformationEnable                   =   100 | LV_DEV_RFTR,

    /** Selects the gain factor or offset of the transformation matrix to
     *  configure @ref LvFtrType_Enumeration.
     *  Values: @ref LvColorTransformationValueSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ColorTransformationValueSelector            =   101 | LV_DEV_RFTR,

    /** Value of the selected color transformation matrix entry. @ref LvFtrType_Float.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ColorTransformationValue                    =   102 | LV_DEV_RFTR,

    /** Selects the operation mode of external device control. @ref LvFtrType_Enumeration.
     *  Values: @ref LvExternalDeviceControlMode.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvExternalDeviceControlMode                 =   103 | LV_DEV_RFTR,

    /** Selects the external ADC to configure. @ref LvFtrType_Enumeration.
     *  Values: @ref LvExternalADCSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvExternalADCSelector                       =   104 | LV_DEV_RFTR,

    /** Reads the value of the selected external ADC. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvExternalADCValue                          =   105 | LV_DEV_RFTR,

    #ifdef SIMPLON_INC_OBSOLETE
    /** Selects the operating mode of the power switches @ref LvFtrType_Enumeration.
     *  Values: @ref LvPowerSwitchMode.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvPowerSwitchMode                           =   106 | LV_DEV_RFTR,
    #endif

    /** Reports the automated action currently performed by a
     *  power switch. @ref LvFtrType_Enumeration.
     *  Values: @ref LvPowerSwitchCurrentAction.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvPowerSwitchCurrentAction                  =   107 | LV_DEV_RFTR,

    /** Selects the power switch to configure. @ref LvFtrType_Enumeration.
     *  Values: @ref LvPowerSwitchSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvPowerSwitchSelector                       =   108 | LV_DEV_RFTR,

    #ifdef SIMPLON_INC_OBSOLETE
    /** Activates the selected power switch for execution of the
     *  available control commands. @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvPowerSwitchActive                         =   109 | LV_DEV_RFTR,
    #endif

    /** Sets an external ADC to the selected power switch. The bound pair
     *  will work together during the automatic operation. @ref LvFtrType_Enumeration.
     *  Values: @ref LvPowerSwitchBoundADC.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvPowerSwitchBoundADC                       =   110 | LV_DEV_RFTR,

    /** Drives the selected power switch with desired polarity. @ref LvFtrType_Enumeration.
     *  Values: @ref LvPowerSwitchDrive.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvPowerSwitchDrive                          =   111 | LV_DEV_RFTR,

    /** Pulses the selected power switch with plus polarity. Available in
     *  the automatic operation mode. @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvPowerSwitchPulsePlus                      =   112 | LV_DEV_RFTR,

    /** Pulses the selected power switch with minus polarity. Available in
     *  the automatic operation mode. @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvPowerSwitchPulseMinus                     =   113 | LV_DEV_RFTR,

    /** Starts an automatic calibration on the selected power switch and
     *  bounded ADCs. @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvLensControlCalibrate                      =   114 | LV_DEV_RFTR,

    /** Represents the calibrated minimal ADC achievable by driving the power 
     *  switch's with minus polarity (plus if the polarity is inverted).
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvLensControlMinusEnd                       =   115 | LV_DEV_RFTR,

    /** Represents the calibrated maximal ADC achievable by driving the power 
     *  switch's with plus polarity (minus if the polarity is inverted).
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvLensControlPlusEnd                        =   116 | LV_DEV_RFTR,

    /** Represents the calibrated slow motion pulse period for the
     *  selected power switch, in microsesonds. @ref LvFtrType_Float.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvLensControlPulsePeriod                    =   117 | LV_DEV_RFTR,

    /** Represents the calibrated slow motion duty cycle for the selected power 
     *  switch (in %). Defines how much of the pulse period is the power switch 
     *  actually active. 
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvLensControlDutyCycle                      =   118 | LV_DEV_RFTR,

    /** Selects how the target lens position should be
     *  approached. @ref LvFtrType_Enumeration.
     *  Values: @ref LvLensControlTargetApproach.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvLensControlTargetApproach                 =   119 | LV_DEV_RFTR,

    /** Sets the number of slow steps required before reaching the
     *  target position @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvLensControlNrSlowSteps                    =   120 | LV_DEV_RFTR,

    /** Sets the target position (value) of the ADC bound to the
     *  selected power switch @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvLensControlTargetPosition                 =   121 | LV_DEV_RFTR,

    /** Adjusts the required target position (value) of the ADC bound
     *  to the selected power switch. @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvLensControlAdjustPosition                 =   122 | LV_DEV_RFTR,

    #ifdef SIMPLON_INC_OBSOLETE
    /** Drives the active power switches with desired
     *  polarity. @ref LvFtrType_Enumeration.
     *  Values: @ref LvPowerSwitchDriveAll.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvPowerSwitchDriveAll                       =   123 | LV_DEV_RFTR,
    #endif

    /** Duration (in microseconds) of the pulses issued at
     *  the power switch. @ref LvFtrType_Float.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvPowerSwitchPulseDuration                  =   124 | LV_DEV_RFTR,

    #ifdef SIMPLON_INC_OBSOLETE
    /** Pulses the active power switches with plus polarity. Available
     *  in the automatic operation mode. @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvPowerSwitchPulseAllPlus                   =   125 | LV_DEV_RFTR,

    /** Pulses the active power switches with minus polarity.
     *  Available in the automatic operation mode. @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvPowerSwitchPulseAllMinus                  =   126 | LV_DEV_RFTR,
    #endif

    /** Minimum value range that has to be reached on the external
     *  ADC to count the calibration as valid. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvLensControlMinCalibrationRange            =   127 | LV_DEV_RFTR,

    /** Starts an automatic calibration on the active power switches and
     *  bounded ADCs. @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvLensControlCalibrateAll                   =   128 | LV_DEV_RFTR,

    /** Selects which LUT to configure. @ref LvFtrType_Enumeration.
     *  Values: @ref LvLUTSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LUTSelector                                 =   129 | LV_DEV_RFTR,

    /** Activates the selected LUT.. @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LUTEnable                                   =   130 | LV_DEV_RFTR,

    /** Index of the element to access in the selected LUT @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LUTIndex                                    =   131 | LV_DEV_RFTR,

    /** Value of the element for the current index
     *  in the selected LUT. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LUTValue                                    =   132 | LV_DEV_RFTR,

    /** This register accesses the entire content of the 
      * selected LUT in one chunk access. 
      * @ref LvFtrType_Buffer.  
      * Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LUTValueAll                                 =   133 | LV_DEV_RFTR,

    #ifdef SIMPLON_INC_OBSOLETE
    /** Applies the LUT settings. @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvLUTUpdate                                 =   134 | LV_DEV_RFTR,
    #endif

    /** Provides the number of bytes transferred for each image by the
     *  device, including image and chunk data. The value defines the
     *  required size of the target buffer used for acquisition. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_PayloadSize                                 =   135 | LV_DEV_RFTR,

    /** Major version of the GigE Vision specification implemented
     *  by the device. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevVersionMajor                             =   136 | LV_DEV_RFTR,

    /** Minor version of the GigE Vision specification implemented
     *  by the device. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevVersionMinor                             =   137 | LV_DEV_RFTR,

    /** Endianess of the device registers. @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevDeviceModeIsBigEndian                    =   138 | LV_DEV_RFTR,

    /** Character set used by all the strings of the device registers. @ref LvFtrType_Enumeration.
     *  Values: @ref LvGevDeviceModeCharacterSet.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevDeviceModeCharacterSet                   =   139 | LV_DEV_RFTR,

    /** Selects which physical network interface to control. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevInterfaceSelector                        =   140 | LV_DEV_RFTR,

    /** MAC address of the network interface. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevInterfaceMACAddress                      =   141 | LV_DEV_RFTR,
    LvDevice_GevMACAddress                               =   141 | LV_DEV_RFTR,

    /** Selects the GEV option to interrogate for existing support. @ref LvFtrType_Enumeration.
     *  Values: @ref LvGevSupportedOptionSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevSupportedOptionSelector                  =   142 | LV_DEV_RFTR,

    /** Returns if the selected GEV option is supported. @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevSupportedOption                          =   143 | LV_DEV_RFTR,

    /** Indicates if Link Local Address IP configuration scheme is
     *  activated on the given network interface. @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevCurrentIPConfigurationLLA                =   144 | LV_DEV_RFTR,

    /** Indicates if DHCP IP configuration scheme is activated on the
     *  given network interface. @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevCurrentIPConfigurationDHCP               =   145 | LV_DEV_RFTR,

    /** Indicates if persistent IP configuration scheme is activated
     *  on the given network interface. @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevCurrentIPConfigurationPersistentIP       =   146 | LV_DEV_RFTR,

    /** Reports the IP address for the given network interface. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevCurrentIPAddress                         =   147 | LV_DEV_RFTR,

    /** Provides the subnet mask of the given interface. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevCurrentSubnetMask                        =   148 | LV_DEV_RFTR,

    /** Indicates the default gateway IP address to be used on
     *  the given network interface. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevCurrentDefaultGateway                    =   149 | LV_DEV_RFTR,

    /** Indicates the persistent IP address for this network interface.
     *  It is only used when the device boots with the persistent IP
     *  configuration scheme. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevPersistentIPAddress                      =   150 | LV_DEV_RFTR,

    /** Indicates the persistent subnet mask associated with the persistent
     *  IP address on this network interface. It is only used when the
     *  device boots with the Persistent IP configuration scheme. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevPersistentSubnetMask                     =   151 | LV_DEV_RFTR,

    /** Indicates the persistent default gateway for this network interface.
     *  It is only used when the device boots with the persistent IP
     *  configuration scheme. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevPersistentDefaultGateway                 =   152 | LV_DEV_RFTR,

    #ifdef SIMPLON_INC_OBSOLETE
    /** Indicates the first URL to the XML device description file. @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevFirstURL                                 =   153 | LV_DEV_RFTR,

    /** Indicates the second URL to the XML device description file. @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevSecondURL                                =   154 | LV_DEV_RFTR,
    #endif

    /** Indicates the number of physical network interfaces supported by this
     *  device. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevNumberOfInterfaces                       =   155 | LV_DEV_RFTR,

    /** Indicates the number of message channels supported by
     *  this device. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevMessageChannelCount                      =   156 | LV_DEV_RFTR,

    /** Indicates the number of stream channels supported
     *  by this device. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevStreamChannelCount                       =   157 | LV_DEV_RFTR,

    /** Indicates the current heartbeat timeout in milliseconds. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevHeartbeatTimeout                         =   158 | LV_DEV_RFTR,

    /** Indicates the number of timestamp ticks during 1
     *  second (frequency in Hz). @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevTimestampTickFrequency                   =   159 | LV_DEV_RFTR,

    /** Latches current timestamp counter into GevTimestampValue. @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevTimestampControlLatch                    =   160 | LV_DEV_RFTR,

    /** Resets the Timestamp counter to 0. @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevTimestampControlReset                    =   161 | LV_DEV_RFTR,

    /** Reset and latch in a single command. @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevTimestampControlLatchReset               =   162 | LV_DEV_RFTR,

    /** Returns the latched 64-bit value of the timestamp
     *  counter. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevTimestampValue                           =   163 | LV_DEV_RFTR,

    /** Controls the device access privilege of an application. @ref LvFtrType_Enumeration.
     *  Values: @ref LvGevCCP.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevCCP                                      =   164 | LV_DEV_RFTR,

    /** Selects the stream channel to control. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevStreamChannelSelector                    =   165 | LV_DEV_RFTR,

    /** Index of network interface to use. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevSCPInterfaceIndex                        =   166 | LV_DEV_RFTR,

    /** Indicates the port to which the device must send data stream.
     *  Setting this value to 0 closes the stream channel. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevSCPHostPort                              =   167 | LV_DEV_RFTR,

    /** Sends a test packet. When this feature is set, the device will fire
     *  one test packet. @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevSCPSFireTestPacket                       =   168 | LV_DEV_RFTR,

    /** The state of this feature is copied into the "do not fragment" bit
     *  of IP header of each stream packet. It can be used by the application
     *  to prevent IP fragmentation of packets on the stream
     *  channel. @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevSCPSDoNotFragment                        =   169 | LV_DEV_RFTR,

    /** Specifies the stream packet size in bytes to send on
     *  this channel. @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevSCPSBigEndian                            =   170 | LV_DEV_RFTR,

    /** Specifies the stream packet size in bytes to send on
     *  this channel. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevSCPSPacketSize                           =   171 | LV_DEV_RFTR,

    /** Indicates the delay (in timestamp counter unit, which is currently a microsecond) 
     *  to insert between each packet for this stream channel. This can be used as a crude 
     *  flow-control mechanism if the application or the network infrastructure cannot 
     *   keep up with the packets coming from the device. 
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevSCPD                                     =   172 | LV_DEV_RFTR,

    /** Indicates the destination IP address for this stream
     *  channel. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevSCDA                                     =   173 | LV_DEV_RFTR,

    /** Indicates the speed of transmission negotiated by the
     *  given network interface in Mb/s. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevLinkSpeed                                =   174 | LV_DEV_RFTR,

    /** Selects the feature configuration user set to load, save or
     *  configure. @ref LvFtrType_Enumeration.
     *  Values: @ref LvUserSetSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_UserSetSelector                             =   175 | LV_DEV_RFTR,

    /** Loads the selected user configuration set and makes
     *  it active @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_UserSetLoad                                 =   176 | LV_DEV_RFTR,

    /** Saves the current device configuration into the selected
     *  user configuration set. @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_UserSetSave                                 =   177 | LV_DEV_RFTR,

    /** Selects the default feature configuration set to be loaded
     *  and activated upon camera boot or reset. @ref LvFtrType_Enumeration.
     *  Values: @ref LvUserSetDefaultSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_UserSetDefaultSelector                      =   178 | LV_DEV_RFTR,

    /** Activates the chunk mode, ie. inclusion of chunk data in
     *  the payload data. @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkModeActive                             =   179 | LV_DEV_RFTR,

    /** Selects the chunk to configure. @ref LvFtrType_Enumeration.
     *  Values: @ref LvChunkSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkSelector                               =   180 | LV_DEV_RFTR,

    /** Enables the inclusion of the selected chunk in the
     *  payload data. @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkEnable                                 =   181 | LV_DEV_RFTR,

    /** X offset applied the image included in the payload. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkOffsetX                                =   182 | LV_DEV_RFTR,

    /** Y offset applied the image included in the payload. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkOffsetY                                =   183 | LV_DEV_RFTR,

    /** Width of the image included in the payload. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkWidth                                  =   184 | LV_DEV_RFTR,

    /** Height of the image included in the payload. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkHeight                                 =   185 | LV_DEV_RFTR,

    /** Pixel format of the image included in the payload. @ref LvFtrType_Enumeration.
     *  Values: see @ref LvPixelFormat.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkPixelFormat                            =   186 | LV_DEV_RFTR,

    /** Line pitch of the image included in the payload. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkLinePitch                              =   187 | LV_DEV_RFTR,

    /** Frame id of the image included in the payload. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkFrameID                                =   188 | LV_DEV_RFTR,

    /** Timestamp associated with the image included in the payload. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkTimestamp                              =   189 | LV_DEV_RFTR,

    /** Exposure time used to acquire the image included in the payload. @ref LvFtrType_Float.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkExposureTime                           =   190 | LV_DEV_RFTR,

    /** Selects the gain type to be reported in chunk data. 
     *  @ref LvFtrType_Enumeration.
     *  Values: @ref LvChunkGainSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkGainSelector                           =   191 | LV_DEV_RFTR,

    /** Gain used to acquire the image included in the payload. @ref LvFtrType_Float.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkGain                                   =   192 | LV_DEV_RFTR,

    /** Black level used to acquire the image included in the payload. @ref LvFtrType_Float.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkBlackLevel                             =   193 | LV_DEV_RFTR,

    /** Bit field indicating staut of all i/o lines at the time the image
     *  included in the payload was acquired. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkLineStatusAll                          =   194 | LV_DEV_RFTR,

    /** Selects the external ADC to be reported in chunk data. 
     *  @ref LvFtrType_Enumeration. 
     *  Values: @ref LvChunkLvExternalADCSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkLvExternalADCSelector                  =   195 | LV_DEV_RFTR,

    /** Reads the value of the selected external ADC at time of acquisition 
     *  of the image included in the payload. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkLvExternalADCValue                     =   196 | LV_DEV_RFTR,

    /** Selects which event to signal to the host application. @ref LvFtrType_Enumeration.
     *  Values: @ref LvEventSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventSelector                               =   197 | LV_DEV_RFTR,

    /** Activate or deactivate the notification to the host application of the 
     *  selected event occurrence. 
     *  @ref LvFtrType_Enumeration.
     *  Values: @ref LvEventNotification.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventNotification                           =   198 | LV_DEV_RFTR,

    /** ID string the smart application @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppID                                =   199 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt1                              =   200 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt2                              =   201 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt3                              =   202 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt4                              =   203 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt5                              =   204 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt6                              =   205 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt7                              =   206 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt8                              =   207 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt9                              =   208 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt10                             =   209 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt11                             =   210 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt12                             =   211 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt13                             =   212 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt14                             =   213 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt15                             =   214 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt16                             =   215 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt17                             =   216 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt18                             =   217 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt19                             =   218 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt20                             =   219 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt21                             =   220 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt22                             =   221 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt23                             =   222 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt24                             =   223 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt25                             =   224 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt26                             =   225 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt27                             =   226 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt28                             =   227 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt29                             =   228 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt30                             =   229 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt31                             =   230 | LV_DEV_RFTR,

    /** Generic signed integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInt32                             =   231 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint1                             =   232 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint2                             =   233 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint3                             =   234 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint4                             =   235 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint5                             =   236 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint6                             =   237 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint7                             =   238 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint8                             =   239 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint9                             =   240 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint10                            =   241 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint11                            =   242 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint12                            =   243 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint13                            =   244 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint14                            =   245 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint15                            =   246 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint16                            =   247 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint17                            =   248 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint18                            =   249 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint19                            =   250 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint20                            =   251 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint21                            =   252 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint22                            =   253 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint23                            =   254 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint24                            =   255 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint25                            =   256 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint26                            =   257 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint27                            =   258 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint28                            =   259 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint29                            =   260 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint30                            =   261 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint31                            =   262 | LV_DEV_RFTR,

    /** Generic unsigned integer register controlling a smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUint32                            =   263 | LV_DEV_RFTR,

    /** Characters of the ASCII command for the smart application. @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppAsciiCmdString                    =   264 | LV_DEV_RFTR,

    /** Executes the ASCII command for the smart application. @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppAsciiCmdExecute                   =   265 | LV_DEV_RFTR,

    /** Response to the ASCII command for the smart application. @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppAsciiCmdFeedback                  =   266 | LV_DEV_RFTR,

    /** Numeric return code of the ASCII command for the smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppAsciiCmdRetCode                   =   267 | LV_DEV_RFTR,

    /** Path of the smart application to be started @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppPath                              =   268 | LV_DEV_RFTR,

    /** Starts the smart application defined by the path. @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppStart                             =   269 | LV_DEV_RFTR,

    /** Returns the unique identifier of the log type of event. This feature
     *  can be used to register a callback function to be notified of the
     *  event occurrence. Its value uniquely identifies the type of event
     *  that will be received. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventLvLog                                  =   270 | LV_DEV_RFTR,

    /** Returns the timestamp of the log event. It can be used to determine
     *  precisely when the event occured. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventLvLogTimestamp                         =   271 | LV_DEV_RFTR,

    /** The log message coming with the event @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventLvLogMessage                           =   272 | LV_DEV_RFTR,

    /** Returns the unique identifier of the smart application log
     *  type of event. This feature can be used to register a callback
     *  function to be notified of the event occurrence. Its value uniquely
     *  identifies the type of event that will be received. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventLvSmartAppLog                          =   273 | LV_DEV_RFTR,

    /** Returns the timestamp of the smart application Smart Application
     *  Log Event. It can be used to determine precisely when the event
     *  occured. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventLvSmartAppLogTimestamp                 =   274 | LV_DEV_RFTR,

    /** The smart application log message coming with the event. 
     *  @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventLvSmartAppLogMessage                   =   275 | LV_DEV_RFTR,

    /** Baud rate used for the serial port communication.   
     *  @ref LvFtrType_Enumeration.
     *  Values: @ref LvSerialPortBaudRate.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSerialPortBaudRate                        =   276 | LV_DEV_RFTR,

    /** Parity used for the serial port communication.  
     *  @ref LvFtrType_Enumeration.
     *  Values: @ref LvSerialPortParity.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSerialPortParity                          =   277 | LV_DEV_RFTR,

    /** Data bits per character for the serial port communication. 
     *  @ref LvFtrType_Enumeration.
     *  Values: @ref LvSerialPortDataBits.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSerialPortDataBits                        =   278 | LV_DEV_RFTR,

    /** Stop bits per character for the serial port communication. 
     *  @ref LvFtrType_Enumeration.
     *  Values: @ref LvSerialPortStopBits.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSerialPortStopBits                        =   279 | LV_DEV_RFTR,

    /** Timeout value used to finish waiting for command response 
     *  @ref LvFtrType_Float.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSerialPortTimeout                         =   280 | LV_DEV_RFTR,

    /** Short string (or single character) marking end of transmission. 
     *  @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSerialPortEOTMarker                       =   281 | LV_DEV_RFTR,

    /** Maximal expected length of the command response.
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSerialPortMaxResponseLength               =   282 | LV_DEV_RFTR,

    /** String of the ASCII command to be sent over the serial port.
     *  @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSerialPortCommandString                   =   283 | LV_DEV_RFTR,

    /** Sends the ASCII command over the serial port Command.  
     *  @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSerialPortCommandSend                     =   284 | LV_DEV_RFTR,

    /** Response to the ASCII command sent over the serial port StringReg.  
     *  @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSerialPortCommandResponse                 =   285 | LV_DEV_RFTR,

    /** Status code indicating success of the last command. 
     *  Values: @ref LvSerialPortCommandStatus.
     *  @ref LvFtrType_Enumeration.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSerialPortCommandStatus                   =   286 | LV_DEV_RFTR,

    /** Sends an exit event to the running smart application. 
     *  @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppExitEvent                         =   287 | LV_DEV_RFTR,

    #ifdef SIMPLON_INC_OBSOLETE
    // Obsolete - do not use
    LvDevice_LvSensorExposureMode                        =   288 | LV_DEV_RFTR,
    #endif

    /** Current watchdog timer value - reports the current value, 
     *  after which the timer expires and the device reboots.
     *  @ref LvFtrType_Float.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvWatchdogTimerValue                        =   290 | LV_DEV_RFTR,

    // added in 1.00.006

    /** Indicates if the lens is wired with inverted polarity, 
     *  meaning that driving the power switch to the plus side 
     *  decreases the external ADC feedback. 
     *  @ref LvFtrType_Boolean. 
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvLensControlInvertedPolarity               =   291 | LV_DEV_RFTR,

    /** Controls the port to which the device must send messages. 
     *  Setting this value to 0 closes the message channel. 
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevMCPHostPort                              =   292 | LV_DEV_RFTR,

    /** Controls the destination IP address for the message 
     *  channel. 
     *  @ref LvFtrType_Integer. 
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevMCDA                                     =   293 | LV_DEV_RFTR,

    /** Provides the transmission timeout value in milliseconds. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevMCTT                                     =   294 | LV_DEV_RFTR,

    /** Controls the number of retransmissions allowed when 
     *  a message channel message times out. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevMCRC                                     =   295 | LV_DEV_RFTR,

    /** The smart application string related to the delivered payload. 
     *  @ref LvFtrType_String
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkLvSmartAppString                       =   296 | LV_DEV_RFTR,

    /** Selects one of the signed integer values related to the 
     *  delivered payload. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkLvSmartAppIntSelector                  =   297 | LV_DEV_RFTR,

    /** The selected smart application signed integer related to 
     *  the delivered payload. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkLvSmartAppInt                          =   298 | LV_DEV_RFTR,

    /** Selects one of the unsigned integer values related to the 
     *  delivered payload.
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkLvSmartAppUintSelector                 =   299 | LV_DEV_RFTR,

    /** The selected smart application unsigned integer related 
     *  to the delivered payload. 
     *  @ref LvFtrType_Integer. 
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkLvSmartAppUint                         =   300 | LV_DEV_RFTR,

    /** The smart application raw register related to the delivered payload. 
     *  @ref LvFtrType_Buffer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkLvSmartAppRegister                     =   301 | LV_DEV_RFTR,

    /** Returns the unique identifier of the smart application string 
     *  type of event. This feature can be used to register a callback 
     *  function to be notified of the event occurrence. Its value 
     *  uniquely identifies the type of event that will be received. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventLvSmartAppString                       =   302 | LV_DEV_RFTR,

    /** Returns the timestamp of the smart application string event. 
     *  It can be used to determine precisely when the event occured. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventLvSmartAppStringTimestamp              =   303 | LV_DEV_RFTR,

    /** The smart application string value coming with the event.
     *  @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventLvSmartAppStringValue                  =   304 | LV_DEV_RFTR,

    /** Returns the unique identifier of the smart application signed 
     *  integer type of event. This feature can be used to register a 
     *  callback function to be notified of the event occurrence. 
     *  Its value uniquely identifies the type of event that will be received. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventLvSmartAppInt                          =   305 | LV_DEV_RFTR,

    /** Returns the timestamp of the smart application signed integer 
     *  event. It can be used to determine precisely when the event occured. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventLvSmartAppIntTimestamp                 =   306 | LV_DEV_RFTR,

    /** Selects one of the signed integer values coming with the event. 
     *  @ref LvFtrType_Integer. 
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventLvSmartAppIntSelector                  =   307 | LV_DEV_RFTR,

    /** Value of the selected signed integer coming with the event. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventLvSmartAppIntValue                     =   308 | LV_DEV_RFTR,

    /** Returns the unique identifier of the smart application unsigned 
     *  integer type of event. This feature can be used to register a callback 
     *  function to be notified of the event occurrence. Its value uniquely 
     *  identifies the type of event that will be received. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventLvSmartAppUint                         =   309 | LV_DEV_RFTR,

    /** Returns the timestamp of the smart application unsigned integer 
     *  event. It can be used to determine precisely when the event occured. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventLvSmartAppUintTimestamp                =   310 | LV_DEV_RFTR,

    /** Selects one of the unsigned integer values coming with the event. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventLvSmartAppUintSelector                 =   311 | LV_DEV_RFTR,

    /** Value of the selected unsigned integer coming with the event.  
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventLvSmartAppUintValue                    =   312 | LV_DEV_RFTR,

    /** Returns the unique identifier of the smart application raw register 
     *  type of event. This feature can be used to register a callback 
     *  function to be notified of the event occurrence. Its value uniquely 
     *  identifies the type of event that will be received. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventLvSmartAppRegister                     =   313 | LV_DEV_RFTR,

    /** Returns the timestamp of the smart application raw register event. 
     *  It can be used to determine precisely when the event occured. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventLvSmartAppRegisterTimestamp            =   314 | LV_DEV_RFTR,

    /** The smart application raw register value coming with the event Register.
     *  @ref LvFtrType_String.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventLvSmartAppRegisterValue                =   315 | LV_DEV_RFTR,

    // added in 1.00.036

    /** Major version of the Standard Feature Naming Convention that 
     *  was used to create the device's XML. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_DeviceSFNCVersionMajor                      =   316 | LV_DEV_RFTR,

    /** Minor version of the Standard Feature Naming Convention that 
     *  was used to create the device's XML. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_DeviceSFNCVersionMinor                      =   317 | LV_DEV_RFTR,

    /** Sub-minor version of Standard Feature Naming Convention that 
     *  was used to create the device's XML. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_DeviceSFNCVersionSubMinor                   =   318 | LV_DEV_RFTR,

    /** Sets the duration (in microseconds) of the line debounce period. 
     *  Value of 0.0 switches the debouncer off. 
     *  @ref LvFtrType_Float.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvLineDebounceDuration                      =   319 | LV_DEV_RFTR,

    /** Provides the device key that allows the device to check the validity of 
     *  action commands. The device internal assertion of an action signal is only 
     *  authorized if the ActionDeviceKey and the action device key value in the 
     *  protocol message are equal. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ActionDeviceKey                             =   320 | LV_DEV_RFTR,

    /** Selects to which action signal further action settings apply. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ActionSelector                              =   321 | LV_DEV_RFTR,

    /** Provides the key that the device will use to validate the action on 
     *  reception of the action protocol message. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ActionGroupKey                              =   322 | LV_DEV_RFTR,

    /** Provides the mask that the device will use to validate the action on 
     *  reception of the action protocol message. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ActionGroupMask                             =   323 | LV_DEV_RFTR,

    /** Reports current calibration status of the selected power switch and its bound ADC. 
     *  The status is computed from the current ADC range, no matter if it is a result 
     *  of calibration operation or configured manually. 
     *  @ref LvFtrType_Enumeration.  
     *  Values: @ref LvLensControlCalibrationStatus.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvLensControlCalibrationStatus              =   324 | LV_DEV_RFTR,

    /** Selects the LUT control mode. 
     *  @ref LvFtrType_Enumeration.  
     *  Values: @ref LvLUTMode.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvLUTMode                                   =   325 | LV_DEV_RFTR,

    /** Selects which color channel to configure for white-balancing. 
     *  @ref LvFtrType_Enumeration.  
     *  Values: @ref LvBalanceRatioSelector.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_BalanceRatioSelector                        =   326 | LV_DEV_RFTR,

    /** Controls white balance ratio coefficient to be applied on the selected color channel. 
     *  Note that the white balance functionality is implemented using the LUT. 
     *  @ref LvFtrType_Float.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_BalanceRatio                                =   327 | LV_DEV_RFTR,

    /** Controls the mode for automatic white balancing between the color channels. 
     *  The white balancing ratios are automatically adjusted. Note that the white balance 
     *  functionality is implemented using the LUT. 
     *  @ref LvFtrType_Enumeration.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_BalanceWhiteAuto                            =   328 | LV_DEV_RFTR,

    /** Returns the class of the device. 
     *  @ref LvFtrType_Enumeration.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevDeviceClass                              =   329 | LV_DEV_RFTR,

    /** Reports the current IP configuration status, ie. the method through which the 
     *  network interface was configured. 
     *  @ref LvFtrType_Enumeration.  
     *  Values: @ref LvGevIPConfigurationStatus.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevIPConfigurationStatus                    =   330 | LV_DEV_RFTR,

    /** Indicates the maximum randomized delay the device will wait to acknowledge a discovery command. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevDiscoveryAckDelay                        =   331 | LV_DEV_RFTR,

    /** Enables the generation of extended status codes. 
     *  @ref LvFtrType_Boolean.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevGVCPExtendedStatusCodes                  =   332 | LV_DEV_RFTR,

    /** Enables the generation of PENDING_ACK. 
     *  @ref LvFtrType_Boolean.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevGVCPPendingAck                           =   333 | LV_DEV_RFTR,

    /** Disables the GVCP heartbeat. 
     *  @ref LvFtrType_Boolean.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevGVCPHeartbeatDisable                     =   334 | LV_DEV_RFTR,

    /** Indicates the longest GVCP command execution time before a device returns a PENDING_ACK. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevGVCPPendingTimeout                       =   335 | LV_DEV_RFTR,

    /** Controls the key to use to authenticate primary application switchover requests. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevPrimaryApplicationSwitchoverKey          =   336 | LV_DEV_RFTR,

    /** Returns the UDP source port of the primary application. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevPrimaryApplicationSocket                 =   337 | LV_DEV_RFTR,

    /** Returns the address of the primary application. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevPrimaryApplicationIPAddress              =   338 | LV_DEV_RFTR,

    /** This feature indicates the source port for the message channel. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevMCSP                                     =   339 | LV_DEV_RFTR,

    /** Enables the camera to continue to stream, for this stream channel, if its control channel 
     *  is closed or regardless of the reception of any ICMP messages (such as destination unreachable messages). 
     *  @ref LvFtrType_Boolean.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevSCCFGUnconditionalStreaming              =   340 | LV_DEV_RFTR,

    /** Enables cameras to use the extended chunk data payload type for this stream channel. 
     *  @ref LvFtrType_Boolean.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevSCCFGExtendedChunkData                   =   341 | LV_DEV_RFTR,

    /** Reports the direction of the stream channel. 
     *  @ref LvFtrType_Enumeration.  
     *  Values: @ref LvGevSCPDirection
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevSCPDirection                             =   342 | LV_DEV_RFTR,

    /** Indicates the source port of the stream channel. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_GevSCSP                                     =   343 | LV_DEV_RFTR,

    /** Flag indicating if the trigger was delayed when acquiring the image included in the payload.   
     *  @ref LvFtrType_Boolean.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkLvTriggerDelayed                       =   344 | LV_DEV_RFTR,

    /** Returns the unique identifier of the dropped trigger type of event. 
     *  This feature can be used to register a callback function to be notified 
     *  of the event occurrence. Its value uniquely identifies the type of event that will be received. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventLvTriggerDropped                       =   345 | LV_DEV_RFTR,

    /** Returns the timestamp of the dropped trigger event. It can be 
     *  used to determine precisely when the event occured. 
     *  @ref LvFtrType_Integer.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_EventLvTriggerDroppedTimestamp              =   346 | LV_DEV_RFTR,

    // added in 1.00.087

    /** Selects the LED clusters of the strobe light that should be enabled. 
     *  @ref LvFtrType_Enumeration.  
     *  Values: @ref LvStrobeEnable.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvStrobeEnable                              =   347 | LV_DEV_RFTR,

    /** Controls the way how the maximum time of strobe duration is calculated. 
     *  @ref LvFtrType_Enumeration.  
     *  Values: @ref LvStrobeDurationMode.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvStrobeDurationMode                        =   348 | LV_DEV_RFTR,

    /** Duration of the strobe pulse in usec. The maximum time depends on the setting of 
     *  Strobe Duration Mode feature. 
     *  @ref LvFtrType_Float. 
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvStrobeDuration                            =   349 | LV_DEV_RFTR,

    /** A delay before the strobe pulse starts after frame trigger is applied in usec. 
     *  @ref LvFtrType_Float. 
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvStrobeDelay                               =   350 | LV_DEV_RFTR,

    /** Brighness (in %) of the strobe light. Allows to lower the full brighness of the strobe. 
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvStrobeBrightness                          =   351 | LV_DEV_RFTR,

    /** Sets mode of handling of strobes not matching the device hardware constraints. 
     *  If a strobe is required (activated by a frame trigger) before the strobe device 
     *  is ready, the strobe must be dropped or delayed. 
     *  @ref LvFtrType_Enumeration.  
     *  Values: @ref LvStrobeDropMode.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvStrobeDropMode                            =   352 | LV_DEV_RFTR,

    /** Resets the LUT settings. 
     *  @ref LvFtrType_Command.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvLUTReset                                  =   353 | LV_DEV_RFTR,

    /** Flag indicating if the configured strobe was dropped when 
     *  acquiring the image included in the payload. 
     *  @ref LvFtrType_Boolean. 
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkLvStrobeDropped                        =   354 | LV_DEV_RFTR,

    // added (mw) 

    /** Flip horizontally the image sent by the device. The AOI is applied after the flipping.
     *  @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ReverseX                                    =   355 | LV_DEV_RFTR,

    /** Flip vertically the image sent by the device. The AOI is applied after the flipping.
     *  @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ReverseY                                    =   356 | LV_DEV_RFTR,

    /** Selects the region of interest to control. The RegionSelector feature allows devices 
     *  that are able to extract multiple regions out of an image, to configure the features of 
     *  those individual regions independently. 
     *  @ref LvFtrType_Enumeration.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_RegionSelector                              =   357 | LV_DEV_RFTR,

    /** Controls if the selected Region of interest is active and streaming. 
     *  @ref LvFtrType_Enumeration.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_RegionMode                                  =   358 | LV_DEV_RFTR,

    /** Control the destination of the selected region.
     *  @ref LvFtrType_Enumeration.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_RegionDestination                           =   359 | LV_DEV_RFTR,

    // added (mw) 02.04.2014 

    /** Number of frames to acquire in MultiFrame Acquisition mode. 
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_AcquisitionFrameCount                      =    360 | LV_DEV_RFTR,

    /** Number of frames to acquire for each FrameBurstStart trigger.
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_AcquisitionBurstFrameCount                 =    361 | LV_DEV_RFTR,

    /** Revision number of the custom module.
	 *  @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomID                                 =    362 | LV_DEV_RFTR,

    /** Info register in the custom module. 
	 *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomInfo                               =    363 | LV_DEV_RFTR,

    /** Controls the way of addressing a register in the custom module. 
	 *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomRegMode                            =    364 | LV_DEV_RFTR,

    /** Defines the address of a register in the custom module. 
	 *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomRegAddr                            =    365 | LV_DEV_RFTR,

    /** Transfers data to and from a register in the custom module. 
	 *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomRegData                            =    366 | LV_DEV_RFTR,

	/** Defines the address and transfers data to and from a register in the custom module. 
	 *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomRegMux                             =    367 | LV_DEV_RFTR,

    /** Total number of bytes between 2 successive lines.
     *  This feature is used to facilitate alignment of image data.
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LinePitch                                  =    368 | LV_DEV_RFTR,

	/** Flag indicating if a frame was dropped when 
     *  acquiring the image included in the payload. 
     *  @ref LvFtrType_Boolean. 
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkLvFrameAbort                          =    369 | LV_DEV_RFTR,

    /** Flag indicating if a trigger was dropped when
     *  acquiring the image included in the payload.
     *  @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkLvTriggerDropped                      =    370 | LV_DEV_RFTR,

    /** Flag indicating if a missmatch between trigger and sensor data
     *  was detected when acquiring the image included in the payload.
     *  @ref LvFtrType_Boolean.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkLvTriggerError                        =    371 | LV_DEV_RFTR,

    /** Encoder position generating the trigger
     *  for the image included in the payload.
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkLvEncoderPosition                     =    372 | LV_DEV_RFTR,

    /** Encoder rotation generating the trigger
     *  for the image included in the payload.
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_ChunkLvEncoderRotation                     =    373 | LV_DEV_RFTR,

    /** Defines the number of regions (Sequence of exposures).
     *  @ref LvFtrType_Enumeration.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_RegionNumber                               =    374 | LV_DEV_RFTR,

    /** Selects the default feature configuration set to be loaded
     *  and activated upon camera boot or reset. @ref LvFtrType_Enumeration.
     *  Values: @ref LvUserSetDefault.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_UserSetDefault                              =   375 | LV_DEV_RFTR,

    /** Sets the mode of the line debounce filter. 
     *  @ref LvFtrType_Enumeration.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvLineDebounceMode                         =    376 | LV_DEV_RFTR,

    /** Sets the gamma value. 
     *  @ref LvFtrType_Float.  
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_Gamma                                      =    377 | LV_DEV_RFTR,

    /** Transfers data to and from a register in the custom module. 
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomReg1                               =    378 | LV_DEV_RFTR,

    /** Transfers data to and from a register in the custom module. 
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomReg2                               =    379 | LV_DEV_RFTR,

    /** Transfers data to and from a register in the custom module. 
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomReg3                               =    380 | LV_DEV_RFTR,

    /** Transfers data to and from a register in the custom module. 
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomReg4                               =    381 | LV_DEV_RFTR,

    /** Transfers data to and from a register in the custom module. 
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomReg5                               =    382 | LV_DEV_RFTR,

    /** Transfers data to and from a register in the custom module. 
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomReg6                               =    383 | LV_DEV_RFTR,

    /** Transfers data to and from a register in the custom module. 
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomReg7                               =    384 | LV_DEV_RFTR,

    /** Transfers data to and from a register in the custom module. 
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomReg8                               =    385 | LV_DEV_RFTR,

    /** Transfers data to and from a register in the custom module. 
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomReg9                               =    386 | LV_DEV_RFTR,

    /** Transfers data to and from a register in the custom module. 
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomReg10                              =    387 | LV_DEV_RFTR,

    /** Transfers data to and from a register in the custom module. 
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomReg11                              =    388 | LV_DEV_RFTR,

    /** Transfers data to and from a register in the custom module. 
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomReg12                              =    389 | LV_DEV_RFTR,

    /** Transfers data to and from a register in the custom module. 
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomReg13                              =    390 | LV_DEV_RFTR,

    /** Transfers data to and from a register in the custom module. 
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomReg14                              =    391 | LV_DEV_RFTR,

    /** Transfers data to and from a register in the custom module. 
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomReg15                              =    392 | LV_DEV_RFTR,

    /** Transfers data to and from a register in the custom module. 
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomReg16                              =    393 | LV_DEV_RFTR,

    /** Bypass the custom module. 
     *  @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvCustomBypass                             =    394 | LV_DEV_RFTR,

    /** max temperature at the selected location in
     *  degrees of Celcius @ref LvFtrType_Float.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvDeviceTemperatureMax                     =    395 | LV_DEV_RFTR,

    /** min temperature at the selected location in
     *  degrees of Celcius @ref LvFtrType_Float.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvDeviceTemperatureMin                     =    396 | LV_DEV_RFTR,

    /** Presence inquiry bits for the smart application integer register. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppIntInq                           =    397 | LV_DEV_RFTR,

    /** Presence inquiry bits for the smart application unsigned integer register. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppUIntInq                          =    398 | LV_DEV_RFTR,

    /** Presence inquiry bits for the Numeric return code of the ASCII command for the smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppAsciiCmdInq                      =    399 | LV_DEV_RFTR,

    /** Presence inquiry bits for the smart application info. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppInfoInq                          =    400 | LV_DEV_RFTR,

    /** General Presence inquiry bits for smart application. @ref LvFtrType_Integer.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartGeneralInq                          =    401 | LV_DEV_RFTR,

    /** Path of the smart application XML file @ref LvFtrType_String.
     *  Device remote feature (@ref LvFtrGroup_DeviceRemote). */
    LvDevice_LvSmartAppXMLPath                          =    402 | LV_DEV_RFTR,



    // --- Device GenTL features -----------------------------------------------

    /** Device ID. @ref LvFtrType_String.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_DeviceID                                    =     0 | LV_DEV_GFTR,

    /** Device type. @ref LvFtrType_Enumeration.
     *  Values: @ref LvDeviceType.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_DeviceType                                  =     1 | LV_DEV_GFTR,

    /** Device IP address. @ref LvFtrType_Integer.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_GevDeviceIPAddress                          =     2 | LV_DEV_GFTR,

    /** Device subnet mask. @ref LvFtrType_Integer.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_GevDeviceSubnetMask                         =     3 | LV_DEV_GFTR,

    /** Device MAC address. @ref LvFtrType_Integer.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_GevDeviceMACAddress                         =     4 | LV_DEV_GFTR,

    /** Device gateway. @ref LvFtrType_Integer.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_GevDeviceGateway                            =     5 | LV_DEV_GFTR,

    /** Device stream capture mode. Controls, how the GVSP (image) stream
     *  is processed by the host machinge. @ref LvFtrType_Enumeration.
     *  Values: @ref LvGevDeviceStreamCaptureMode.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_LvGevDeviceStreamCaptureMode                =     6 | LV_DEV_GFTR,

    /** Stream selector. @ref LvFtrType_Integer.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_StreamSelector                              =     7 | LV_DEV_GFTR,

    /** Stream ID. @ref LvFtrType_String.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_StreamID                                    =     8 | LV_DEV_GFTR,

    // added in 1.00.061
    /** Identifies the endianess mode. This feature should be set to 'standard' 
     *  for all GigE Vision remote devices based on GenICam schema version 1.1 (and newer). 
     *  The value 'legacy' is intended for devices using GenICam schema version 1.0. 
     *  Values: @ref LvDeviceEndianessMechanism.
     *  @ref LvFtrType_Enumeration.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_DeviceEndianessMechanism                    =     9 | LV_DEV_GFTR,

    /** Determine the maximal usable packet size for streaming. The procedure is 
     *  already applied automatically when opening the device. Do not use during 
     *  active streaming. 
     *  @ref LvFtrType_Command.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_LvGevFindMaxPacketSize                      =    10 | LV_DEV_GFTR,

    /** Streaming packet size to be verified using the Test Packet Size command. Do not use 
     *  during active streaming. 
     *  @ref LvFtrType_Integer.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_LvGevPacketSizeValue                        =    11 | LV_DEV_GFTR,

    /** Test if the selected packet size is suitable for streaming in current network 
     *  configuration. 
     *  @ref LvFtrType_Command.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_LvGevTestPacketSize                         =    12 | LV_DEV_GFTR,

    /** Reports success of the last packet size test command. 
     *  @ref LvFtrType_Boolean.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_LvGevPacketSizeTestSuccess                  =    13 | LV_DEV_GFTR,

    /** Controls the GigE Vision control channel transmission timeout value in milliseconds. 
     *  When it expires, the acknowledge from the device will be considered as missing and the 
     *  command eventually sent again. 
     *  @ref LvFtrType_Command.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_LvGevCCTT                                   =    14 | LV_DEV_GFTR,

    /** Controls the number of GigE Vision control channel command retransmissions allowed when 
     *  a control channel command times out. 
     *  @ref LvFtrType_Command.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_LvGevCCRC                                   =    15 | LV_DEV_GFTR,

    /** Control channel status. @ref LvFtrType_Integer.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_LvCCStatus                                  =    16 | LV_DEV_GFTR,         // (mw)

    /** Device access status. @ref LvFtrType_Integer.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_DeviceAccessStatus                          =    17 | LV_DEV_GFTR,         // (mj)

    /** Control channel link status. @ref LvFtrType_Enumeration.
     *  Values: @ref LvCCLinkStatus.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_LvCCLinkStatus                              =    18 | LV_DEV_GFTR,         // (mj)

    /** GigEVision Server Capability. @ref LvFtrType_Integer.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_LvGigEServerCapability                      =    19 | LV_DEV_GFTR,         // (mw)

    /** GigEVision Server Enable. @ref LvFtrType_Integer.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_LvGigEServerEnable                          =    20 | LV_DEV_GFTR,         // (mw)

    /** GigEVision Server Event. @ref LvFtrType_Buffer.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_LvGigEServerEvent                           =    21 | LV_DEV_GFTR,         // (mw)

    /** GigEVision Server Image. @ref LvFtrType_Buffer.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_LvGigEServerImage                           =    22 | LV_DEV_GFTR,         // (mw)

    /** Device Timestamp Frequency. @ref LvFtrType_Integer.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_DeviceTimestampFrequency                    =    23 | LV_DEV_GFTR,         // (mj)

    /** Device display name. @ref LvFtrType_String.
     *  Device GenTL feature (@ref LvFtrGroup_DeviceGtl). */
    LvDevice_DeviceDisplayName                           =    24 | LV_DEV_GFTR,         // (mj)

    // --- Device local features -----------------------------------------------

    /** User readable name of the device. 
     *  @ref LvFtrType_String.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvDeviceDisplayName                         =     0 | LV_DEV_LFTR,

    /** Returns true if the acquisition was started. Note that this feature does not tell
     *  whether the images are actually delivered to the output buffer queue; it simply
     *  informs that your application is between the AcquisitionStart and AcquisitionStop
     *  actions.
     *  @ref LvFtrType_Boolean.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvDeviceIsAcquiring                         =     1 | LV_DEV_LFTR,

    #ifdef SIMPLON_INC_OBSOLETE
    /** Performs a test of the connection by asking the camera
     *  to send sending a series of packets with increasing size.
     *  The test finishes when a packet is not received, indicating
     *  it is too large for current connection. The GevMaximumPacketSize
     *  is then determined as maximum successful packet size.
     *  @ref LvFtrType_Command.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvGevFindMaximumPacketSize                  =     2 | LV_DEV_LFTR,

    /** The result of the FindMaximumPacketSize command. @ref LvFtrType_Integer.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvGevMaximumPacketSize                      =     3 | LV_DEV_LFTR,
    #endif

    /** The UniProcessing provides unified API for image preprocessing, which is
     *  done either on the device itself, if it is supported by the hardware, or
     *  by software, if not. The preprocessing includes Bayer decoding or pixel format
     *  conversion, application of LUT and Color Correction.
     *  @ref LvFtrType_Enumeration.
     *  Values: @ref LvUniProcessMode.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvUniProcessMode                            =     4 | LV_DEV_LFTR,

    /** If possible, the software image preprocessing will be preferably done
     *  in the same image (not to another buffer). This is possible only in case the
     *  preprocessing does not change the pixel format, that means the LvUniPixelFormat
     *  must be equal to PixelFormat (for example the Bayer decoding is not done by
     *  software).
     *  @ref LvFtrType_Boolean.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvUniProcessEnableInPlace                   =     5 | LV_DEV_LFTR,

    /** If the image preprocessing is enabled, this is the desired
     *  pixel format, to which the image is to be converted. Only monochrome and RGB/BGR
     *  color pixel formats are supported. The processing chain is set so that:
     *  - if the PixelFormat is undecoded Bayer, the Bayer decoding to desired LvUniPixelFormat
     *    is included
     *  - otherwise if the PixelFormat is not equal to LvUniPixelFormat, a pixel format conversion
     *    is included.
     *  @ref LvFtrType_Enumeration.
     *  Values: see @ref LvPixelFormat.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvUniPixelFormat                            =     6 | LV_DEV_LFTR,

    #ifdef SIMPLON_INC_OBSOLETE
    /** This command checks the capabilities of the device and sets on
     *  it such pixel format, which minimizes the need of processing
     *  or format conversion to Unified Pixel Format. @ref LvFtrType_Command.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvUniResolvePixelFormat                     =     7 | LV_DEV_LFTR,
    #endif

    /** Returns the size needed for the processing output buffer, which is to be used when the
     *  in-place processing is not enabled or not possible. Normally is this buffer
     *  allocated automatically for each acquisition buffer, but your application can also provide
     *  your own buffers and this feature gives the size of the buffers needed.
     *  @ref LvFtrType_Integer.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvUniProcessPayloadSize                     =     8 | LV_DEV_LFTR,

    /** The line increment of the process buffer, if the processing
     *  is active, or of the source buffer, if processing is not active.
     *  To access the image regardless whether it was processed to the process buffer
     *  or not, you need 5 independent values:
     *  - Pointer to the data - use LvUniBase feature of the Buffer, which points
     *    either to the acquired image (if no processing was done), or to the processed
     *    image (if it was processed).
     *  - Width and height - these are the same for the acquired and processed image,
     *    so use the Width and Height from the remote device, or ChunkWidth and ChunkHeight
     *    if these can change during the acquisition.
     *  - Pixel format - use LvUniPixelFormat - if this is different from the PixelFormat
     *    then processing is done, so LvUniPixelFormat is always correct.
     *  - Line pitch - use LvUniLinePitch, which returns proper line pitch of the
     *    buffer, to which the LvUniBase pointer points.
     *  Note that the above is valid only in case the processing can be successfully done
     *  (for example the source image is not in unsupported PixelFormat) and is not disabled
     *  (for example by LvUniProcessExecution=OnExplicitRequest).
     *  @ref LvFtrType_Integer.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvUniLinePitch                              =     9 | LV_DEV_LFTR,

    /** Selects the Bayer array decoding method for the software processing. This does not
     *  apply to the hardware Bayer decoding on the device, which is usually fixed to one method.
     *  @ref LvFtrType_Enumeration.
     *  Values: @ref LvBayerDecoderAlgorithm.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvUniBayerDecoderAlgorithm                  =    10 | LV_DEV_LFTR,

    /** Brightness of the image. It is realized by the LUT. Values
     *  under 1.0 means darker than original, above 1.0 lighter
     *  than the original.
     *  The LvUniLUTMode must be Generated, in order to enable this feature.
     *  @ref LvFtrType_Float.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvUniBrightness                             =    11 | LV_DEV_LFTR,

    /** Contrast of the image. It is realized by the LUT. Values under
     *  1.0 means lower contrast than original, above 1.0 higher
     *  contrast than the original.
     *  The LvUniLUTMode must be Generated, in order to enable this feature.
     *  @ref LvFtrType_Float.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvUniContrast                               =    12 | LV_DEV_LFTR,

    /** Gamma correction of the image. It is realized by the LUT.
     *  Values under 1.0 make the middle tones darker, above
     *  1.0 lighter.
     *  The LvUniLUTMode must be Generated, in order to enable this feature.
     *  @ref LvFtrType_Float.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvUniGamma                                  =    13 | LV_DEV_LFTR,

    /** Selects which color channel will be accessed by the LvUniBalanceRatio feature.
     *  The LvUniLUTMode must be Generated, in order to enable this feature.
     *  @ref LvFtrType_Enumeration.
     *  Values: @ref LvUniBalanceRatioSelector.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvUniBalanceRatioSelector                   =    14 | LV_DEV_LFTR,

    /** The white balance factor to be applied on the selected color channel.
     *  The selected color channel of all pixels will be multiplied by this value
     *  (not directly, but through the precalucated LUT). If the value is < 1.0, the saturated
     *  pixels will become gray (white is no more white). Thus it is better if all 3
     *  factors are greater than or equal to 1.0.
     *  @ref LvFtrType_Float.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvUniBalanceRatio                           =    15 | LV_DEV_LFTR,

    /** Selects the action for automatic white balance calculation. Currently only the
     *  option Once is available. Setting this option causes the following:
     *  - if there is already acquired image available, the white balance factors
     *    are calculated from this image and LUT is updated to reflect the changes
     *  - if there is no image acquired yet, an internal flag is set and the calculation
     *    is done when the image is acquired.
     *  Note that the enumeration is self-clearing, that means its value is automatically
     *  changed to Off, when the white balance calculation is finished.
     *  The newly calculated white balance is applied to to newly acquired images,
     *  not to the existing ones, unless you explicitly call the ExecProcess command
     *  for the already acquired buffers.
     *  At the time of calculation the camera should look at a neutral grey (not white) object, which
     *  should fill the whole image area. Making white balance from normal image can bring
     *  less satisfactory results.
     *  @ref LvFtrType_Enumeration.
     *  Values: @ref LvUniBalanceWhiteAuto.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvUniBalanceWhiteAuto                       =    16 | LV_DEV_LFTR,

    /** Sets all the white balance factors (LvUniBalanceRatio) to 1. The advantage of this feature in comparison
     *  with setting the 3 factors to 1 is that the LUT is updated only once, so
     *  it is faster.
     *  @ref LvFtrType_Command.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvUniBalanceWhiteReset                      =    17 | LV_DEV_LFTR,

    /** Selects which color transformation module is controlled
     *  by the color transformation features. It also gives
     *  particular meaning to individual color transformation
     *  gains.
     *  @ref LvFtrType_Enumeration.
     *  Values: @ref LvUniColorTransformationSelector.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvUniColorTransformationSelector            =    18 | LV_DEV_LFTR,

    /** Enables the Color Transformation in the processing. When disabled, the Color Transformation
     *  matrix does not lose its values; when enabling it, the original values are retained.
     *  @ref LvFtrType_Boolean.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvUniColorTransformationEnable              =    19 | LV_DEV_LFTR,

    /** Selects the cell of the Color Transformation matrix to be accessed by
     *  LvUniColorTransformationValue.
     *  @ref LvFtrType_Enumeration.
     *  Values: @ref LvColorTransformationValueSelector.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvUniColorTransformationValueSelector       =    20 | LV_DEV_LFTR,

    /** The value of the selected cell of the Color Transformation matrix.
     *  @ref LvFtrType_Float.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvUniColorTransformationValue               =    21 | LV_DEV_LFTR,

    /** Sets the Color Correction matrix according to specified saturation.
     *  The saturation set to 0 causes a conversion to greyscale,
     *  1.0 leaves the image identical, 2.0 emphasizes the colors.
     *  @ref LvFtrType_Float.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvUniSaturation                             =    22 | LV_DEV_LFTR,

    /** Defines the point, when the software image processing of the buffer is done. You may
     *  need to define this point in case you do not need all the images to be processed.
     *  Note that this applies only to the software processing; the hardware processing is
     *  done on the remote device always.
     *  @ref LvFtrType_Enumeration.
     *  Values: @ref LvUniProcessExecution.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvUniProcessExecution                       =    23 | LV_DEV_LFTR,

    // added in 1.00.061
    /** Selects the LUT control mode. The mode determines, if the LUT can be directly modified
     *  by the application, or if the LUT is to be reserved for implementation of white balance,
     *  gamma, brightnesss and contrast - in such case the LUT is filled with precalculated
     *  values by SynView library and cannot be directly modified.
     *  @ref LvFtrType_Enumeration.
     *  Values: @ref LvUniLUTMode.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
     LvDevice_LvUniLUTMode                               =    24 | LV_DEV_LFTR,

    /** This selector selects for which LUT is applied LvUniLUTIndex/LvUniLUTValue.
     *  In case of monochrome image the LUT has only one array = Luminance.
     *  In case of color images, the LUT consists of 3 arrays, for Red, Green and Blue.
     *  @ref LvFtrType_Enumeration.
     *  Values: @ref LvUniLUTSelector.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
     LvDevice_LvUniLUTSelector                           =    25 | LV_DEV_LFTR,

    /** Enables the LUT in the processing. When disabled, the LUT does not lose its values, the
     *  disabled LUT is substitued by a linear LUT, and when enabling the LUT, the original
     *  values are retained.
     *  @ref LvFtrType_Boolean.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
     LvDevice_LvUniLUTEnable                             =    26 | LV_DEV_LFTR,

    /** Index of the element to be accessed in the selected LUT via the LvUniLUTValue feature.
     *  Note that accessing the whole LUT by this approach can be very time consuming, namely
     *  on GigE cameras. If possible, it is better to use the LvUniLUTValueAll or SynView
     *  dedicated LUT functions.
     *  @ref LvFtrType_Integer.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
     LvDevice_LvUniLUTIndex                              =    27 | LV_DEV_LFTR,

    /** Value of the element for the current LvUniLUTIndex in the selected LUT.
     *  Note that accessing the whole LUT by this approach can be very time consuming, namely
     *  on GigE cameras. If possible, it is better to use the LvUniLUTValueAll or SynView
     *  dedicated LUT functions.
     *  @ref LvFtrType_Integer.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
     LvDevice_LvUniLUTValue                              =    28 | LV_DEV_LFTR,

    /** This feature enables to get/set the entire content of the selected LUT in one block access.
     *  Beware that the LUT buffer structure is vendor and model dependent, so take care
     *  if your application is exptected to work with various types od devices or
     *  devices from various vendors.
     *  @ref LvFtrType_Buffer.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
     LvDevice_LvUniLUTValueAll                           =    29 | LV_DEV_LFTR,
     
    // added in 1.00.085
    /** Selects the Color Transformation matrix control mode. The mode determines, if the matrix can be directly modified
     *  by the application, or if the matrix is to be reserved for implementation of the Saturation
     *  or other higher level features - in such case the matrix is filled with precalculated
     *  values by SynView library and cannot be directly modified.
     *  @ref LvFtrType_Enumeration.
     *  Values: @ref LvUniColorTransformationMode.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
     LvDevice_LvUniColorTransformationMode               =    30 | LV_DEV_LFTR,

    //(mw) added in 1.01.111
    /** expiring date of the device. 
     *  @ref LvFtrType_String.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvDeviceExpiringDate                        =    31 | LV_DEV_LFTR,

    //(mj) added in 1.03.005
    /** Prepare the device for registers streaming without
     *  checking for consistency. @ref LvFtrType_Command.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvDeviceRegistersStreamingStart             =    32 | LV_DEV_LFTR,

    //(mj) added in 1.03.005
    /** Announce the end of registers streaming. This will
     *  do a register set validation for consistency and
     *  activate it. @ref LvFtrType_Command.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvDeviceRegistersStreamingEnd               =    33 | LV_DEV_LFTR,

    //(mj) added in 1.03.005
    /** Resets to linear LUT.
     *  @ref LvFtrType_Command.
     *  Device local feature (@ref LvFtrGroup_DeviceLocal). */
    LvDevice_LvUniLUTReset                               =    34 | LV_DEV_LFTR,


    // --- Device special constants --------------------------------------------

    /** Constant to be used in LvGetInfo() and LvGetInfoStr() to obtain
     *  various info about the device. */
    LvDevice_Info                                        =     0 | LV_DEV_NFTR
};

// alias definitions for compatibility
#define LvDevice_LvGevCCStatus LvDevice_LvCCStatus


//------------------------------------------------------------------------------

/** LvStreamFtr constants. 
 *  @ingroup GroupSynview_Features
 *  @hideinitializer
 */
enum LvStreamFtr
{
    // --- Stream GenTL features -----------------------------------------------

    /** Stream ID. @ref LvFtrType_String.
     *  Stream GenTL feature (@ref LvFtrGroup_StreamGtl). */
    LvStream_StreamID                                    =     0 | LV_STM_GFTR,

    /** Number of buffers announced for the data stream. @ref LvFtrType_Integer.
     *  Stream GenTL feature (@ref LvFtrGroup_StreamGtl). */
    LvStream_StreamAnnouncedBufferCount                  =     1 | LV_STM_GFTR,

    /** Selects desired acquisition mode. @ref LvFtrType_Enumeration.
     *  Values: @ref LvStreamAcquisitionModeSelector.
     *  Stream GenTL feature (@ref LvFtrGroup_StreamGtl). */
    LvStream_StreamAcquisitionModeSelector               =     2 | LV_STM_GFTR,

    /** Minimum number of buffers to be announced for selected
     *  acquisition mode. @ref LvFtrType_Integer.
     *  Stream GenTL feature (@ref LvFtrGroup_StreamGtl). */
    LvStream_StreamAnnounceBufferMinimum                 =     3 | LV_STM_GFTR,

    /** Stream type. @ref LvFtrType_Enumeration.
     *  Values: @ref LvStreamType.
     *  Stream GenTL feature (@ref LvFtrGroup_StreamGtl). */
    LvStream_StreamType                                  =     4 | LV_STM_GFTR,


    // --- Stream local features -----------------------------------------------

    /** Returns the display name of the stream.
     *  @ref LvFtrType_String.
     *  Stream local feature (@ref LvFtrGroup_StreamLocal). */
    LvStream_LvStreamDisplayName                         =     0 | LV_STM_LFTR,

    /** Returns the payload size (size of buffer to hold the image data). If the payload size is
     *  not provided by the stream or device, it is calculated, so this feature returns
     *  always a valid value.
     *  @ref LvFtrType_Integer.
     *  Stream local feature (@ref LvFtrGroup_StreamLocal). */
    LvStream_LvCalcPayloadSize                           =     1 | LV_STM_LFTR,

    /** Number of buffers to be kept postponed before returning
     *  to the input buffer pool. 
     *  @ref LvFtrType_Integer.
     *  Stream local feature (@ref LvFtrGroup_StreamLocal). */
    LvStream_LvPostponeQueueBuffers                      =     2 | LV_STM_LFTR,

    /** Limit for images in the output buffer. Applicable only if the event thread is
     *  running - then if there is more than this number of buffers in the output
     *  queue, the oldest buffers are discarded and returned to input buffer pool.
     *  This is useful in case the application is not able to process all the images in
     *  time.
     *  @ref LvFtrType_Integer.
     *  If the value is 0, this limit is inactive.
     *  Stream local feature (@ref LvFtrGroup_StreamLocal). */
    LvStream_LvAwaitDeliveryLimit                        =     3 | LV_STM_LFTR,

    /** Enable the auto allocation of process buffers. The process buffers are allocated
     *  only if they are needed for the image processing or conversion. You can disable
     *  the automatic buffer allocation and provide own buffers, using the LvBufferAttachProcessBuffer()
     *  function.
     *  @ref LvFtrType_Boolean.
     *  Stream local feature (@ref LvFtrGroup_StreamLocal). */
    LvStream_LvAutoAllocateProcessBuffers                =     4 | LV_STM_LFTR,

    /** Preallocates all the process buffers, even if it is not yet sure if they
     *  will be needed. With this command you can avoid time delays when allocating the buffers
     *  during the acquisition.
     *  @ref LvFtrType_Command.
     *  Stream local feature (@ref LvFtrGroup_StreamLocal). */
    LvStream_LvPreallocateProcessBuffers                 =     5 | LV_STM_LFTR,

    /** Number of acquired frames since last acquisition start.
     *  It is equivalent to the GenTL STREAM_INFO_NUM_DELIVERED info.
     *  @ref LvFtrType_Integer. Stream local feature (@ref LvFtrGroup_StreamLocal). */
    LvStream_LvNumDelivered                              =     6 | LV_STM_LFTR,

    /** Number of lost frames due to input buffer pool underrun since stream open.
     *  It is equivalent to the GenTL STREAM_INFO_NUM_UNDERRUN info.
     *  @ref LvFtrType_Integer. Stream local feature (@ref LvFtrGroup_StreamLocal). */
    LvStream_LvNumUnderrun                               =     7 | LV_STM_LFTR,

    /** Number of announced buffers.
     *  It is equivalent to the GenTL STREAM_INFO_NUM_ANNOUNCED info.
     *  @ref LvFtrType_Integer. Stream local feature (@ref LvFtrGroup_StreamLocal). */
    LvStream_LvNumAnnounced                              =     8 | LV_STM_LFTR,

    /** Number of buffers currently in the input pool.
     *  It is equivalent to the GenTL STREAM_INFO_NUM_QUEUED info.
     *  @ref LvFtrType_Integer. Stream local feature (@ref LvFtrGroup_StreamLocal). */
    LvStream_LvNumQueued                                 =     9 | LV_STM_LFTR,

    /** Number of buffers currently in the output queue.
     *  It is equivalent to the GenTL STREAM_INFO_NUM_AWAIT_DELIVERY info.
     *  @ref LvFtrType_Integer. Stream local feature (@ref LvFtrGroup_StreamLocal). */
    LvStream_LvNumAwaitDelivery                          =    10 | LV_STM_LFTR,

    /** Flag indicating whether the acquisition engine is started or not. This is
     *  independent from the acquisition status of the remote device.
     *  It is equivalent to the GenTL STREAM_INFO_IS_GRABBING info.
     *  @ref LvFtrType_Boolean. Stream local feature (@ref LvFtrGroup_StreamLocal). */
    LvStream_LvIsGrabbing                                =    11 | LV_STM_LFTR,

    /** Number of aborted frames since last acquisition start.
     *  @ref LvFtrType_Integer. Stream local feature (@ref LvFtrGroup_StreamLocal). */
    LvStream_LvNumAborted                                =    12 | LV_STM_LFTR,

	/** Number of started frames since stream open.
     *  It is equivalent to the GenTL STREAM_INFO_NUM_STARTED info.
	 *  @ref LvFtrType_Integer. Stream local feature (@ref LvFtrGroup_StreamLocal). */
    LvStream_LvNumStarted                                =    13 | LV_STM_LFTR,


    // --- Stream special constants --------------------------------------------

    /** Constant to be used in LvGetInfo() and LvGetInfoStr() to obtain
     *  various info about the stream. */
    LvStream_Info                                        =     0 | LV_STM_NFTR
};

//------------------------------------------------------------------------------

/** LvRendererFtr constants. 
 *  @ingroup GroupSynview_Features
 *  @hideinitializer
 */
enum LvRendererFtr
{
    // --- Renderer local features ---------------------------------------------


    /** If set, the image is automatically displayed before it is passed to the supplied
     *  callback. This is functional only in case the Event thread is started.
     *  @ref LvFtrType_Boolean. Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvAutoDisplay                             =     0 | LV_RND_LFTR,

    /** Controls way how the acquired images are rendered
     *  on the screen.
     *  Note that all the Scale- options require scaling capability of the display
     *  and might not be supported in all operating systems.
     *  @ref LvFtrType_Enumeration.
     *  Values: @ref LvRenderType.
     *  Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvRenderType                              =     1 | LV_RND_LFTR,

    /** Sets the horizontal offset of the image to be rendered, i.e. the distance from the
     *  left edge of the display window.
     *  @ref LvFtrType_Integer.
     *  Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvOffsetX                                 =     2 | LV_RND_LFTR,

    /** Sets the vertical offset of the image to be rendered, i.e. the distance from the
     *  top edge of the display window.
     *  @ref LvFtrType_Integer.
     *  Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvOffsetY                                 =     3 | LV_RND_LFTR,

    /** Sets the width of the rectangle to which the image is to be rendered. Note that if
     *  the LvIgnoreAspectRatio feature is False, the real image width can be smaller, in order
     *  to keep the aspect ratio.
     *  @ref LvFtrType_Integer.
     *  Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvWidth                                   =     4 | LV_RND_LFTR,

    /** Sets the height of the rectangle to which the image is to be rendered. Note that
     *  if the LvIgnoreAspectRatio feature is False, the real image height can be smaller, in
     *  order to keep the aspect ratio.
     *  @ref LvFtrType_Integer.
     *  Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvHeight                                  =     5 | LV_RND_LFTR,

    /** Allows to ignore the original aspect ratio while rendering the image, so the image
     *  can be scaled up/down in one dimension with different factor than in the other
     *  dimension.
     *  @ref LvFtrType_Boolean.
     *  Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvIgnoreAspectRatio                       =     6 | LV_RND_LFTR,

    /** Disables scaling the image up.
     *  @ref LvFtrType_Boolean.
     *  Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvDisableScaleUp                          =     7 | LV_RND_LFTR,

    /** Disables scaling the image down.
     *  @ref LvFtrType_Boolean.
     *  Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvDisableScaleDown                        =     8 | LV_RND_LFTR,

    /** If the image is smaller than required window client size or the 
     *  specified rectangle, the image is placed to the center.
     *  @ref LvFtrType_Boolean.
     *  Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvCenterImage                             =     9 | LV_RND_LFTR,

    /** Sets the number of tiles used for image rendering. Note that for the tile
     *  repaint is needed that the corresponding buffers are still in the application
     *  ownership; once the buffer is placed to the input buffer pool, it should not be
     *  accessed for paint anymore (see also LvPostponeQueueBuffers).
     *  @ref LvFtrType_Integer.
     *  Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvNumberOfTiles                           =    10 | LV_RND_LFTR,

    /** Sets the number of columns used for image rendering. When the value is 0, the
     *  number of columns is calculated automatically.
     *  @ref LvFtrType_Integer.
     *  Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvColumns                                 =    11 | LV_RND_LFTR,

    /** Sets the number of rows used for image rendering. When the value is 0, the
     *  number of rows is calculated automatically.
     *  @ref LvFtrType_Integer.
     *  Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvRows                                    =    12 | LV_RND_LFTR,

    /** Sets the gap between the tiles in pixels.
     *  @ref LvFtrType_Integer.
     *  Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvTileGap                                 =    13 | LV_RND_LFTR,

    /** When set to True, the tile sizes and positions are calculated automatically.
     *  When the LvColumns and/or LvRows are 0, also the number of columns
     *  and/or rows is calculated automatically.
     *  @ref LvFtrType_Boolean.
     *  Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvAutoTileCalculation                     =    14 | LV_RND_LFTR,

    /** Activate the Pixel Coordinate section for getting pixel
     *  values at given coordinates. 
     *  @ref LvFtrType_Boolean.
     *  Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvShowPixel                               =    15 | LV_RND_LFTR,

    /** Sets the horizontal coordinate of the pixel for which the value
     *  should be retrieved.
     *  @ref LvFtrType_Integer.
     *  Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvPixelCoordinateX                        =    16 | LV_RND_LFTR,

    /** Sets the vertical coordinate of the pixel for which the value
     *  should be retrieved.
     *  @ref LvFtrType_Integer.
     *  Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvPixelCoordinateY                        =    17 | LV_RND_LFTR,

    /** First Value of the retrieved Pixel.
     *  @ref LvFtrType_Integer.
     *  Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvPixelVal1                               =    18 | LV_RND_LFTR,

    /** Second Value of the retrieved Pixel.
     *  @ref LvFtrType_Integer.
     *  Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvPixelVal2                               =    19 | LV_RND_LFTR,

    /** Third Value of the retrieved Pixel.
     *  @ref LvFtrType_Integer.
     *  Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvPixelVal3                               =    20 | LV_RND_LFTR,

    /** Activate a cross hair function
     *  Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvShowCrossHair                           =    21 | LV_RND_LFTR,

    /** Activate a histogram window. 
     *  @ref LvFtrType_Boolean.
     *  Renderer local feature (@ref LvFtrGroup_RendererLocal). */
    LvRenderer_LvShowHistogram                           =    22 | LV_RND_LFTR,


    // --- Renderer special constants ------------------------------------------

    /** Constant to be used in LvGetInfo() and LvGetInfoStr() to obtain
     *  various info about the device. */
    LvRenderer_Info                                      =     0 | LV_RND_NFTR
};

//------------------------------------------------------------------------------

/** LvEventFtr constants. 
 *  @ingroup GroupSynview_Features
 *  @hideinitializer
 */
enum LvEventFtr
{
    /** Represents the GenTL EVENT_EVENT_TYPE info - The event type.
     *  LvFtrType_Integer. */
    LvEvent_EventType  = 0 | LV_EVT_GTL,
    
    /** Represents the GenTL EVENT_NUM_IN_QUEUE info - Number of events in the
     *  event data queue.  LvFtrType_Integer. */
    LvEvent_NumInQueue = 1 | LV_EVT_GTL,

    /** Represents the GenTL EVENT_NUM_FIRED info - Number of events, that were
     *  fired since the creation of the event. LvFtrType_Integer */
    LvEvent_NumFired   = 2 | LV_EVT_GTL
};

//------------------------------------------------------------------------------

#ifdef LVAVISAVER
    enum LvAviSaverFtr
    {
        LvAviSaver_Dummy = 0 | LV_AVI_GTL
    };
#endif

//------------------------------------------------------------------------------

/** LvBufferFtr constants. 
 *  @ingroup GroupSynview_Features
 *  @hideinitializer
 */
enum LvBufferFtr
{
    /** Represents the GenTL BUFFER_INFO_BASE info - Base address of the buffer
     *  memory. LvFtrType_Pointer. */
    LvBuffer_Base          = 0 | LV_BUF_GTL,

    /** Represents the GenTL BUFFER_INFO_SIZE info - Size of the buffer in
     *  bytes. LvFtrType_Integer. */
    LvBuffer_Size          = 1 | LV_BUF_GTL,

    /** Represents the GenTL BUFFER_INFO_USER_PTR info - The user pointer
     *  (supplied by the application when the buffer was allocated).
     *  LvFtrType_Pointer. 
     *  Note: This pointer should not be used in the C++ API and .Net Class Library, 
     *  where this pointer is utilized internally for the LvBuffer class instance. 
     *  The actual User pointer is available by the LvBuffer::GetUserPtr() function. */
    LvBuffer_UserPtr       = 2 | LV_BUF_GTL,

    /** Represents the GenTL BUFFER_INFO_TIMESTAMP info - Timestamp the buffer
     *  was acquired. The unit is device/implementation dependent.
     *  LvFtrType_Integer. */
    LvBuffer_TimeStamp     = 3 | LV_BUF_GTL,

    /** Represents the GenTL BUFFER_INFO_NEW_DATA info - Flag to indicate that
     *  the buffer contains new data since the last delivery. LvFtrType_Boolean. */
    LvBuffer_NewData       = 4 | LV_BUF_GTL,

    /** Represents the GenTL BUFFER_INFO_IS_QUEUED info - Flag to indicate if
     *  the buffer is in the input pool or output queue. LvFtrType_Boolean. */
    LvBuffer_IsQueued      = 5 | LV_BUF_GTL,

    /** Represents the GenTL BUFFER_INFO_ISACQUIRING info - Flag to indicate
     *  that the buffer is currently being filled with data. LvFtrType_Boolean. */
    LvBuffer_IsAcquiring   = 6 | LV_BUF_GTL,

    /** Represents the GenTL BUFFER_INFO_ISINCOMPLETE info - Flag to indicate
     *  that a buffer was filled, but an error occurred during that process.
     *  LvFtrType_Boolean. */
    LvBuffer_IsIncomplete  = 7 | LV_BUF_GTL,

    /** Represents the GenTL BUFFER_INFO_TLTYPE info - Transport layer
     *  technologies that are supported. LvFtrType_String. */
    LvBuffer_TlType        = 8 | LV_BUF_GTL,

    /** Represents the GenTL BUFFER_INFO_SIZE_FILLED info - Number of bytes
     *  written into the buffer last time it has been filled. This value is
     *  reset to 0 when the buffer is placed into the Input Buffer Pool.
     *  LvFtrType_Integer. */
    LvBuffer_SizeFilled    = 9 | LV_BUF_GTL,

    /** Represents the GenTL 1.2 BUFFER_INFO_WIDTH info -
     *  Width of the data in the buffer in number of pixels. This information refers
     *  for example to the width entry in the GigE Vision image stream data leader.
     *  LvFtrType_Integer. */
    LvBuffer_Width        = 10 | LV_BUF_GTL,

    /** Represents the GenTL 1.2 BUFFER_INFO_HEIGHT info -
     *  Height of the data in the buffer in number of pixels as configured. For
     *  variable size images this is the max Height of the buffer. For example
     *  this information refers to the height entry in the GigE Vision image
     *  stream data leader.
     *  LvFtrType_Integer. */
    LvBuffer_Height       = 11 | LV_BUF_GTL,

    /** Represents the GenTL 1.2 BUFFER_INFO_XOFFSET info -
     *  XOffset of the data in the buffer in number of pixels from the image origin to
     *  handle areas of interest. This information refers for example to the information
     *  provided in the GigE Vision image stream data leader.
     *  LvFtrType_Integer. */
    LvBuffer_XOffset      = 12 | LV_BUF_GTL,

    /** Represents the GenTL 1.2 BUFFER_INFO_YOFFSET info - 
     *  YOffset of the data in the buffer in number of lines from the image origin to handle
     *  areas of interest. This information refers for example to the information
     *  provided in the GigE Vision image stream data leader.
     *  LvFtrType_Integer. */
    LvBuffer_YOffset      = 13 | LV_BUF_GTL,

    /** Represents the GenTL 1.2 BUFFER_INFO_XPADDING info -
     *  XPadding of the data in the buffer in number of bytes. This information refers
     *  for example to the information provided in the GigE Vision image stream data leader.
     *  LvFtrType_Integer. */
    LvBuffer_XPadding     = 14 | LV_BUF_GTL,

    /** Represents the GenTL 1.2 BUFFER_INFO_YPADDING info - 
     *  YPadding of the data in the buffer in number of bytes. This information refers
     *  for example to the information provided in the GigE Vision image stream data leader.
     *  LvFtrType_Integer. */
    LvBuffer_YPadding     = 15 | LV_BUF_GTL,

    /** Represents the GenTL 1.2 BUFFER_INFO_FRAMEID info -
     *  A sequentially incremented number of the frame. This information refers for example
     *  to the information provided in the GigE Vision image stream block id.
     *  The wrap around of this number is transportation technology dependent.
     *  For GigE Vision it is (so far) 16bit wrapping to 1.
     *  LvFtrType_Integer. */
    LvBuffer_FrameId      = 16 | LV_BUF_GTL,

    /** Represents the GenTL 1.2 BUFFER_INFO_IMAGEPRESENT info -
     *  Flag to indicate if the current data in the buffer contains image data. This information
     *  refers for example to the information provided in the GigE Vision image stream data leader.
     *  LvFtrType_Boolean. */
    LvBuffer_ImagePresent = 17 | LV_BUF_GTL,

    /** Represents the GenTL 1.2 BUFFER_INFO_IMAGEOFFSET info -
     *  Offset of the image data from the beginning of the delivered buffer in bytes.
     *  Applies for example when delivering the image as part of chunk data or on
     *  technologies requiring specific buffer alignment.
     *  LvFtrType_Integer. */
    LvBuffer_ImageOffset  = 18 | LV_BUF_GTL,

    /** Represents the GenTL 1.2 BUFFER_INFO_PAYLOADTYPE info -
     *  Payload type of the data. This information refers to the constants defined in GenTL PAYLOADTYPE_IDs
     *  (UNKNOWN=0, IMAGE=1, RAW_DATA=2, FILE=3, CHUNK_DATA=4, CUSTOM=1000)
     *  LvFtrType_Integer. */
    LvBuffer_PayloadType  = 19 | LV_BUF_GTL,

    /** Represents the GenTL 1.2 BUFFER_INFO_PIXELFORMAT info -
     *  This information refers for example to the information provided in the GigE Vision
     *  image stream data leader. The interpretation of the pixel format depends on the namespace
     *  the pixel format belongs to. This can be inquired using the LvBuffer_PixelFormatNameSpace feature.
     *  LvFtrType_Integer. */
    LvBuffer_PixelFormat  = 20 | LV_BUF_GTL,

    /** Represents the GenTL 1.2 BUFFER_INFO_PIXELFORMAT_NAMESPACE info -
     *  This information refers to the constants defined in GenTL 1.2 PIXELFORMAT_NAMESPACE_IDs to allow
     *  interpretation of LvBuffer_PixelFormat (UNKNOWN=0, GEV=1, IIDC=2, CUSTOM=1000).
     *  LvFtrType_Integer. */
    LvBuffer_PixelFormatNameSpace = 21 | LV_BUF_GTL,

    /** Represents the GenTL 1.2 BUFFER_INFO_DELIVERED_IMAGEHEIGHT info -
     *  The number of lines in the current buffer as delivered by the transport mechanism.
     *  For area scan type images this is usually the number of lines configured in the device.
     *  For variable size linescan images this number may be lower than the configured image height.
     *  This information refers for example to the information provided in the GigE Vision image stream data trailer.
     *  LvFtrType_Integer. */
    LvBuffer_DeliveredImageHeight = 22 | LV_BUF_GTL,

    /** Represents the GenTL 1.2 BUFFER_INFO_DELIVERED_CHUNKPAYLOADSIZE info -
     *  This information refers for example to the information provided in the GigE Vision image stream data leader.
     *  LvFtrType_Integer. */
    LvBuffer_DeliveredChunkPayloadSize = 23 | LV_BUF_GTL,

    /** Represents the GenTL 1.2 BUFFER_INFO_CHUNKLAYOUTID info - 
     *  This information refers for example to the information provided in the GigE
     *  Vision image stream data leader. The chunk layout id serves as an indicator that
     *  the chunk layout has changed and the application should re-parse the chunk layout
     *  in the buffer. When a chunk layout (availability or position of individual chunks)
     *  changes since the last buffer delivered by the device through the same stream,
     *  the device MUST change the chunk layout id. As long as the chunk layout remains
     *  stable, the camera MUST keep the chunk layout id intact. When switching back to
     *  a layout, which was already used before, the camera can use the same id again
     *  or use a new id. A chunk layout id value of 0 is invalid. It is reserved for use
     *  by cameras not supporting the layout id functionality.
     *  LvFtrType_Integer. */
    LvBuffer_ChunkLayoutId = 24 | LV_BUF_GTL,

    /** Represents the GenTL 1.2 BUFFER_INFO_FILENAME info - 
     *  This information refers for example to the information provided in the GigE Vision image stream data leader.
     *  For other technologies this is to be implemented accordingly. Since this is GigE Vision related
     *  information and the filename in GigE Vision is UTF8 coded, this filename is also UTF8 coded.
     *  LvFtrType_Integer. */
    LvBuffer_FileName     = 25 | LV_BUF_GTL,

    /** Unified base address of the buffer. If the image was processed to
     *  the output buffer, the pointer to the output buffer is returned,
     *  otherwise the pointer to the acquisition buffer is returned. This enables
     *  to write simple universal code for image handling.
     *  LvFtrType_Pointer. SynView feature. */
    LvBuffer_UniBase       = 0 | LV_BUF_LOC,

    /** Size of the buffer returned on @ref LvBuffer_UniBase call.
     *  LvFtrType_Integer. SynView feature. */
    LvBuffer_UniSize       = 1 | LV_BUF_LOC,

    /** Pointer to the process buffer, attached to this acquisition buffer.
     *  LvFtrType_Pointer. SynView feature. */
    LvBuffer_ProcessBase   = 2 | LV_BUF_LOC,

    /** Size of the process buffer, attached to this acquisition buffer.
     *  LvFtrType_Integer. SynView feature. */
    LvBuffer_ProcessSize   = 3 | LV_BUF_LOC,

    /** Executes the SW image processing of the buffer. To be used when the
     *  @ref LvDevice_LvUniProcessExecution is set to @ref LvUniProcessExecution_OnExplicitRequest.
     *  @ref LvFtrType_Command.
     *  SynView feature. */
    LvBuffer_ExecProcess   = 4 | LV_BUF_LOC,

    /** Unified image offset. If the image was processed to
     *  the output buffer, the image offset to the output buffer is returned,
     *  otherwise the image offset to the acquisition buffer is returned. This enables
     *  to write simple universal code for image handling.
     *  LvFtrType_Integer. SynView feature. */
    LvBuffer_UniImageOffset= 5 | LV_BUF_LOC,

    /** Executes the SW image processing of the buffer with copy to attached process buffer. To be used 
     *  when the @ref LvDevice_LvUniProcessExecution is set to @ref LvUniProcessExecution_OnExplicitRequest.
     *  @ref LvFtrType_Command.
     *  SynView feature. */
    LvBuffer_ExecProcessCopy = 6 | LV_BUF_LOC

    // Buffer local feature not currently supported
    // LvBuffer_BufferData      = 0 | LV_BUF_GFTR,           //  LvFtrType_Register. Entire buffer data.
    // LvBuffer_BufferUserData  = 1 | LV_BUF_GFTR,           //  LvFtrType_Integer. User data provided by user data pointer at buffer announcement.
};

//------------------------------------------------------------------------------

/** LvInfoDataType constants. 
 *  The enum is used only by the LvEventGetDataInfo() function - this function
 *  follows the GenTL EventGetDataInfo() function, which uses different
 *  data types, than the GenApi.
 *  @ingroup GroupSynview_Enums
 *  @hideinitializer
 */
enum LvInfoDataType
{
    /** Represents the GenTL INFO_DATATYPE_UNKNOWN info - Unknown data type. */
    LvInfoDataType_Unknown     = 0,

    /** Represents the GenTL INFO_DATATYPE_STRING info - 0-terminated C string
     *  (ASCII encoded). */
    LvInfoDataType_String      = 1,

    /** Represents the GenTL INFO_DATATYPE_STRINGLIST info - Concatenated
     *  INFO_DATATYPE_STRING list. End of list is signaled with an additional 0. */
    LvInfoDataType_StringList  = 2,

    /** Represents the GenTL INFO_DATATYPE_INT16 info - Signed 16 bit integer. */
    LvInfoDataType_Int16       = 3,

    /** Represents the GenTL INFO_DATATYPE_UINT16 info - unsigned 16 bit integer. */
    LvInfoDataType_UInt16      = 4,

    /** Represents the GenTL INFO_DATATYPE_INT32 info - signed 32 bit integer. */
    LvInfoDataType_Int32       = 5,

    /** Represents the GenTL INFO_DATATYPE_UINT32 info - unsigned 32 bit integer. */
    LvInfoDataType_UInt32      = 6,

    /** Represents the GenTL INFO_DATATYPE_INT64 info - signed 64 bit integer. */
    LvInfoDataType_Int64       = 7,

    /** Represents the GenTL INFO_DATATYPE_UINT64 info - unsingned64 bit integer. */
    LvInfoDataType_UInt64      = 8,

    /** Represents the GenTL INFO_DATATYPE_FLOAT64 info - Signed 64 bit floating
     *  point number. */
    LvInfoDataType_Float64     = 9,

    /** Represents the GenTL INFO_DATATYPE_PTR info - Pointer type (void*). Size
     *  is platform dependent (32 bit on 32 bit platforms). */
    LvInfoDataType_Ptr         = 10,

    /** Represents the GenTL INFO_DATATYPE_BOOL8 info - Boolean value occupying
     *  8 bit. 0 for false and anything for true. */
    LvInfoDataType_Bool        = 11,

    /** Represents the GenTL INFO_DATATYPE_SIZET info - Platform dependent unsigned
     *  integer (32 bit on 32 bit platforms). */
    LvInfoDataType_SizeT       = 12,

    /** Represents the GenTL INFO_DATATYPE_BUFFER info - Like the INFO_DATATYPE_STRING
     *  but with arbitrary data and no 0 termination. */
    LvInfoDataType_Buffer      = 13,
};

//------------------------------------------------------------------------------

/** LvQueueOperation constants.
 *  Define enum values for the LvStreamFlushQueue() function.
 *  @ingroup GroupSynview_Enums
 *  @hideinitializer
 */
enum LvQueueOperation
{
    /** Represents the GenTL ACQ_QUEUE_INPUT_TO_OUTPUT. Flushes the input pool
     *  to the output queue and if necessary adds entries in the
     *  LvEventType_NewBuffer event data queue. */
    LvQueueOperation_InputToOutput   = 0,

    /** Represents the GenTL ACQ_QUEUE_OUTPUT_DISCARD. Discards all buffers in
     *  the output queue and if necessary removes the entries from the event data
     *  queue. */
    LvQueueOperation_OutputDiscard   = 1,

    /** Represents the GenTL ACQ_QUEUE_ALL_TO_INPUT. Puts all buffers in the
     *  input pool. Even those in the output queue and discard entries in the
     *  event data queue. */
    LvQueueOperation_AllToInput      = 2,

    /** Represents the GenTL ACQ_QUEUE_UNQUEUED_TO_INPUT. Puts all buffers that
     *  are not in the input pool or the output queue in the input pool. */
    LvQueueOperation_UnqueuedToInput = 3,

    /** Represents the GenTL ACQ_QUEUE_ALL_DISCARD. Discards all buffers in the
     *  input pool and output queue. */
    LvQueueOperation_AllDiscard      = 4
};

//------------------------------------------------------------------------------

/** LvEventType constants.
 *  @if PlainC
 *  Type of the event, for which the Event module is created - see @ref LvEventOpen().
 *  @endif
 *  @if Cpp
 *  Type of the event, for which the Event module is created - see @ref LvEvent::Open().
 *  @endif
 *  Currently only the @ref LvEventType_NewBuffer is supported by SynView.
 *  @ingroup GroupSynview_Enums
 *  @hideinitializer
*/
enum LvEventType
{
    /** Represents the GenTL EVENT_ERROR - Notification on module errors.
     *  For this type of event the @ref LvEventDataInfo_Id is @ref LvInfoDataType_Int32
     *  and @ref LvEventDataInfo_Value is @ref LvInfoDataType_String.*/
    LvEventType_Error              = 0,

    /** Represents the GenTL EVENT_NEW_BUFFER - Notification on newly filled
     *  buffers placed to the output queue. For this type of event the
     *  @ref LvEventDataInfo_Id is @ref LvInfoDataType_Ptr (GenTL Buffer handle) and
     *  @ref LvEventDataInfo_Value is @ref LvInfoDataType_Ptr (Private pointer). */
    LvEventType_NewBuffer          = 1,

    /** Represents the GenTL EVENT_FEATURE_INVALIDATE - Notification if a feature was 
     *  changed by the GenTL Producer library and thus needs to be invalidated in the 
     *  GenICam GenApi instance using the module.
     */
    LvEventType_FeatureInvalidate  = 2,

    /** Represents the GenTL EVENT_FEATURE_CHANGE - Notification if the GenTL Producer 
     *  library wants to manually set a feature in the GenICam GenApi instance using the module.
     */
    LvEventType_FeatureChange      = 3,

    /** Represents the GenTL EVENT_FEATURE_DEVEVENT - Notification if the GenTL Producer 
     *  wants to inform the GenICam GenApi instance of the remote device that a GenApi 
     *  compatible event was fired. This event is processed internally in SynView API -
     *  it is converted into the feature change callback - see the @ref LvModule::RegisterFeatureCallback() function. 
     *  However, the thread which checks the GenTL event and converts it into the callbacks must be 
     *  started explicitly by the application - see the @ref LvEventStartThread() function. 
     *  This event type can be opened only on the Device module.
     */
    LvEventType_FeatureDevEvent    = 4,
    LvEventType_RemoteDevice       = 4,

    /** Represents the GenTL EVENT_MODULE - Notification if the GenTL Producer 
     *  wants to inform the GenICam GenApi instance of the module that a GenApi 
     *  compatible event was fired.
     */
    LvEventType_Module             = 5
};

//------------------------------------------------------------------------------

/** LvEventDataInfo constants.
 *  Define values for the info specification in the LvEventGetDataInfo() function.
 *  @ingroup GroupSynview_Enums
 *  @hideinitializer
 */
enum LvEventDataInfo
{
    /** Represents the GenTL EVENT_DATA_ID - Event ID. See @ref LvEventType for the
     *  explanation, what this ID means according to the event type.*/
    LvEventDataInfo_Id    = 0,
    
    /** Represents the GenTL EVENT_DATA_VALUE - Event Data. See @ref LvEventType for the
     *  explanation, what this data means according to the event type. */
    LvEventDataInfo_Value = 1
};

//------------------------------------------------------------------------------

/** @defgroup GroupSynview_PixelFormatDefines LvPixelFormat definitions
 *  @ingroup GroupSynview_Defines
 *  @showinitializer
 */
/*@{*/
#define LV_PIX_MONO       0x01000000 /**< PixelFormat component: The pixel format is monochrome. */
#define LV_PIX_COLOR      0x02000000 /**< PixelFormat component: The pixel format is color. */
#define LV_PIX_CUSTOM     0x80000000 /**< PixelFormat component: The pixel format is custom. */ 
#define LV_PIX_COLOR_MASK 0xFF000000 /**< Mask for the color flag */

// Indicate effective number of bits occupied by the pixel (including padding).
#define LV_PIX_OCCUPY8BIT                 0x00080000 /**< PixelFormat component: One pixel occupies 8 bits. */
#define LV_PIX_OCCUPY12BIT                0x000C0000 /**< PixelFormat component: One pixel occupies 12 bits. */
#define LV_PIX_OCCUPY16BIT                0x00100000 /**< PixelFormat component: One pixel occupies 16 bits. */
#define LV_PIX_OCCUPY24BIT                0x00180000 /**< PixelFormat component: One pixel occupies 24 bits. */
#define LV_PIX_OCCUPY32BIT                0x00200000 /**< PixelFormat component: One pixel occupies 32 bits. */
#define LV_PIX_OCCUPY36BIT                0x00240000 /**< PixelFormat component: One pixel occupies 36 bits. */
#define LV_PIX_OCCUPY48BIT                0x00300000 /**< PixelFormat component: One pixel occupies 48 bits. */
#define LV_PIX_EFFECTIVE_PIXEL_SIZE_MASK  0x00FF0000 /**< Mask for the pixel size part. */
#define LV_PIX_EFFECTIVE_PIXEL_SIZE_SHIFT         16 /**< Shift for the pixel size part. */
/*@}*/

/** LvPixelFormat constants
 *  Enum for the @ref LvDevice_PixelFormat, @ref LvDevice_ChunkPixelFormat and @ref LvDevice_LvUniPixelFormat features.
 *  The Pixel format constants are defined by the GenICam standard. The value consists of 3 parts:
 *   - byte 1 - color/mono/custom
 *   - byte 2 - bits per pixel
 *   - byte 3 and 4 - ID of the pixel format
 *
 *  Exceptions are:
 *   - LvPixelFormat_BGR555Packed - used only in the Image Processing Library (in conversions for display).
 *  @showinitializer
 *  @ingroup GroupSynview_EnumEntries
 */
enum LvPixelFormat
{
    // Mono buffer format defines
    LvPixelFormat_Mono8            = 0x01080001,    /**<  Monochrome 8-bit.           Defined as (LV_PIX_MONO | LV_PIX_OCCUPY8BIT  | 0x0001).   @showinitializer */
    LvPixelFormat_Mono8S           = 0x01080002,    /**<  Monochrome 8-bit signed.    Defined as (LV_PIX_MONO | LV_PIX_OCCUPY8BIT  | 0x0002).   @showinitializer */
    LvPixelFormat_Mono10           = 0x01100003,    /**<  Monochrome 10-bit unpacked. Defined as (LV_PIX_MONO | LV_PIX_OCCUPY16BIT | 0x0003).   @showinitializer */
    LvPixelFormat_Mono10Packed     = 0x010C0004,    /**<  Monochrome 10-bit packed.   Defined as (LV_PIX_MONO | LV_PIX_OCCUPY12BIT | 0x0004).   @showinitializer */
    LvPixelFormat_Mono12           = 0x01100005,    /**<  Monochrome 12-bit unpacked. Defined as (LV_PIX_MONO | LV_PIX_OCCUPY16BIT | 0x0005).   @showinitializer */
    LvPixelFormat_Mono12Packed     = 0x010C0006,    /**<  Monochrome 12-bit packed.   Defined as (LV_PIX_MONO | LV_PIX_OCCUPY12BIT | 0x0006).   @showinitializer */
    LvPixelFormat_Mono14           = 0x01100025,    /**<  Monochrome 14-bit unpacked. Defined as (LV_PIX_MONO | LV_PIX_OCCUPY16BIT | 0x0025).   @showinitializer */
    LvPixelFormat_Mono16           = 0x01100007,    /**<  Monochrome 16-bit.          Defined as (LV_PIX_MONO | LV_PIX_OCCUPY16BIT | 0x0007).   @showinitializer */
    // Bayer buffer format defines
    LvPixelFormat_BayerGR8         = 0x01080008,    /**<  Undecoded 8-bit  Bayer array with the GR array position. Defined as           (LV_PIX_MONO | LV_PIX_OCCUPY8BIT  | 0x0008).   @showinitializer */
    LvPixelFormat_BayerRG8         = 0x01080009,    /**<  Undecoded 8-bit  Bayer array with the RG array position. Defined as           (LV_PIX_MONO | LV_PIX_OCCUPY8BIT  | 0x0009).   @showinitializer */
    LvPixelFormat_BayerGB8         = 0x0108000A,    /**<  Undecoded 8-bit  Bayer array with the GB array position. Defined as           (LV_PIX_MONO | LV_PIX_OCCUPY8BIT  | 0x000A).   @showinitializer */
    LvPixelFormat_BayerBG8         = 0x0108000B,    /**<  Undecoded 8-bit  Bayer array with the BG array position. Defined as           (LV_PIX_MONO | LV_PIX_OCCUPY8BIT  | 0x000B).   @showinitializer */
    LvPixelFormat_BayerGR10        = 0x0110000C,    /**<  Undecoded 10-bit Bayer array with the GR array position. Defined as           (LV_PIX_MONO | LV_PIX_OCCUPY16BIT | 0x000C).   @showinitializer */
    LvPixelFormat_BayerRG10        = 0x0110000D,    /**<  Undecoded 10-bit Bayer array with the RG array position. Defined as           (LV_PIX_MONO | LV_PIX_OCCUPY16BIT | 0x000D).   @showinitializer */
    LvPixelFormat_BayerGB10        = 0x0110000E,    /**<  Undecoded 10-bit Bayer array with the GB array position. Defined as           (LV_PIX_MONO | LV_PIX_OCCUPY16BIT | 0x000E).   @showinitializer */
    LvPixelFormat_BayerBG10        = 0x0110000F,    /**<  Undecoded 10-bit Bayer array with the BG array position. Defined as           (LV_PIX_MONO | LV_PIX_OCCUPY16BIT | 0x000F).   @showinitializer */
    LvPixelFormat_BayerGR12        = 0x01100010,    /**<  Undecoded 12-bit Bayer array with the GR array position. Defined as           (LV_PIX_MONO | LV_PIX_OCCUPY16BIT | 0x0010).   @showinitializer */
    LvPixelFormat_BayerRG12        = 0x01100011,    /**<  Undecoded 12-bit Bayer array with the RG array position. Defined as           (LV_PIX_MONO | LV_PIX_OCCUPY16BIT | 0x0011).   @showinitializer */
    LvPixelFormat_BayerGB12        = 0x01100012,    /**<  Undecoded 12-bit Bayer array with the GB array position. Defined as           (LV_PIX_MONO | LV_PIX_OCCUPY16BIT | 0x0012).   @showinitializer */
    LvPixelFormat_BayerBG12        = 0x01100013,    /**<  Undecoded 12-bit Bayer array with the BG array position. Defined as           (LV_PIX_MONO | LV_PIX_OCCUPY16BIT | 0x0013).   @showinitializer */
    LvPixelFormat_BayerGR10Packed  = 0x010C0026,    /**<  Undecoded 10-bit packed Bayer array with the GR array position. Defined as    (LV_PIX_MONO | LV_PIX_OCCUPY12BIT | 0x0026).   @showinitializer */
    LvPixelFormat_BayerRG10Packed  = 0x010C0027,    /**<  Undecoded 10-bit packed Bayer array with the RG array position. Defined as    (LV_PIX_MONO | LV_PIX_OCCUPY12BIT | 0x0027).   @showinitializer */
    LvPixelFormat_BayerGB10Packed  = 0x010C0028,    /**<  Undecoded 10-bit packed Bayer array with the GB array position. Defined as    (LV_PIX_MONO | LV_PIX_OCCUPY12BIT | 0x0028).   @showinitializer */
    LvPixelFormat_BayerBG10Packed  = 0x010C0029,    /**<  Undecoded 10-bit packed Bayer array with the BG array position. Defined as    (LV_PIX_MONO | LV_PIX_OCCUPY12BIT | 0x0029).   @showinitializer */
    LvPixelFormat_BayerGR12Packed  = 0x010C002A,    /**<  Undecoded 12-bit packed Bayer array with the GR array position. Defined as    (LV_PIX_MONO | LV_PIX_OCCUPY12BIT | 0x002A).   @showinitializer */
    LvPixelFormat_BayerRG12Packed  = 0x010C002B,    /**<  Undecoded 12-bit packed Bayer array with the RG array position. Defined as    (LV_PIX_MONO | LV_PIX_OCCUPY12BIT | 0x002B).   @showinitializer */
    LvPixelFormat_BayerGB12Packed  = 0x010C002C,    /**<  Undecoded 12-bit packed Bayer array with the GB array position. Defined as    (LV_PIX_MONO | LV_PIX_OCCUPY12BIT | 0x002C).   @showinitializer */
    LvPixelFormat_BayerBG12Packed  = 0x010C002D,    /**<  Undecoded 10-bit packed Bayer array with the BG array position. Defined as    (LV_PIX_MONO | LV_PIX_OCCUPY12BIT | 0x002D).   @showinitializer */
    LvPixelFormat_BayerGR16        = 0x0110002E,    /**<  Undecoded 16-bit Bayer array with the GR array position. Defined as           (LV_PIX_MONO | LV_PIX_OCCUPY16BIT | 0x002E).   @showinitializer */
    LvPixelFormat_BayerRG16        = 0x0110002F,    /**<  Undecoded 16-bit Bayer array with the RG array position. Defined as           (LV_PIX_MONO | LV_PIX_OCCUPY16BIT | 0x002F).   @showinitializer */
    LvPixelFormat_BayerGB16        = 0x01100030,    /**<  Undecoded 16-bit Bayer array with the GB array position. Defined as           (LV_PIX_MONO | LV_PIX_OCCUPY16BIT | 0x0030).   @showinitializer */
    LvPixelFormat_BayerBG16        = 0x01100031,    /**<  Undecoded 16-bit Bayer array with the BG array position. Defined as           (LV_PIX_MONO | LV_PIX_OCCUPY16BIT | 0x0031).   @showinitializer */
    // RGB buffer format defines
    LvPixelFormat_RGB8             = 0x02180014,    /**<  RGB 24-bit (3x8 bits).                  Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY24BIT | 0x0014).  @showinitializer */
    LvPixelFormat_BGR8             = 0x02180015,    /**<  BGR 24-bit (3x8 bits).                  Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY24BIT | 0x0015).  @showinitializer */
    LvPixelFormat_RGBA8            = 0x02200016,    /**<  RGB 32-bit (3x8 bits + 1x8 bits alpha). Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY32BIT | 0x0016).  @showinitializer */
    LvPixelFormat_BGRA8            = 0x02200017,    /**<  BGR 32-bit (3x8 bits + 1x8 bits alpha). Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY32BIT | 0x0017).  @showinitializer */
    LvPixelFormat_RGB10            = 0x02300018,    /**<  RGB 48-bit unpacked (3x10 bits).        Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY48BIT | 0x0018).  @showinitializer */
    LvPixelFormat_BGR10            = 0x02300019,    /**<  BGR 48-bit unpacked (3x10 bits).        Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY48BIT | 0x0019).  @showinitializer */
    LvPixelFormat_RGB12            = 0x0230001A,    /**<  RGB 48-bit unpacked (3x12 bits).        Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY48BIT | 0x001A).  @showinitializer */
    LvPixelFormat_BGR12            = 0x0230001B,    /**<  BGR 48-bit unpacked (3x12 bits).        Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY48BIT | 0x001B).  @showinitializer */
    LvPixelFormat_RGB16            = 0x02300033,    /**<  RGB 48-bit (3x16 bits).                 Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY48BIT | 0x0033).  @showinitializer */
    LvPixelFormat_BGR16            = 0x0230004B,    /**<  BGR 48-bit (3x16 bits).                 Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY48BIT | 0x004B).  @showinitializer */
    LvPixelFormat_RGB10V1Packed    = 0x0220001C,    /**<  RGB 32-bit packed (3x10 bits).          Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY32BIT | 0x001C).  @showinitializer */
    LvPixelFormat_RGB10P32         = 0x0220001D,    /**<  RGB 32-bit packed (3x10 bits).          Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY32BIT | 0x001D).  @showinitializer */
    LvPixelFormat_RGB12V1Packed    = 0x02240034,    /**<  RGB 36-bit packed (3x12 bits).          Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY36BIT | 0x0034).  @showinitializer */
    LvPixelFormat_RGB565P          = 0x02100035,    /**<  RGB 16-bit packed (5,6,5 bits).         Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY16BIT | 0x0035).  @showinitializer */
    LvPixelFormat_BGR565P          = 0x02100036,    /**<  BGR 16-bit packed (5,6,5 bits).         Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY16BIT | 0x0036).  @showinitializer */

    // YUV buffer format defines
    LvPixelFormat_YUV8_UYV                = 0x02180020, /**<  YUV 4:4:4 8-bit  UYV.    Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY24BIT | 0x0020).  @showinitializer */
    LvPixelFormat_YUV422_8_UYVY           = 0x0210001F, /**<  YUV 4:2:2 8-bit  UYVY.   Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY16BIT | 0x001F).  @showinitializer */
    LvPixelFormat_YUV422_8                = 0x02100032, /**<  YUV 4:2:2 8-bit  YUYV.   Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY16BIT | 0x0032).  @showinitializer */
    LvPixelFormat_YUV411_8_UYYVYY         = 0x020C001E, /**<  YUV 4:1:1 8-bit  UYYVYY. Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY12BIT | 0x001E).  @showinitializer */

    // YCbCr buffer format defines
    LvPixelFormat_YCbCr8                  = 0x0218005B, /**<  YCbCr 4:4:4 8-bit YCbCr.    Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY24BIT | 0x005B).  @showinitializer */
    LvPixelFormat_YCbCr8_CbYCr            = 0x0218003A, /**<  YCbCr 4:4:4 8-bit CbYCr.    Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY24BIT | 0x003A).  @showinitializer */
    LvPixelFormat_YCbCr422_8              = 0x0210003B, /**<  YCbCr 4:2:2 8-bit YCbYCr.   Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY16BIT | 0x003B).  @showinitializer */
    LvPixelFormat_YCbCr422_8_CbYCrY       = 0x02100043, /**<  YCbCr 4:2:2 8-bit CbYCrY.   Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY16BIT | 0x0043).  @showinitializer */
    LvPixelFormat_YCbCr411_8              = 0x020C005A, /**<  YCbCr 4:1:1 8-bit YYCbYYCr. Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY12BIT | 0x005A).  @showinitializer */
    LvPixelFormat_YCbCr411_8_CbYYCrYY     = 0x020C003C, /**<  YCbCr 4:1:1 8-bit CbYYCrYY. Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY12BIT | 0x003C).  @showinitializer */

    // YCbCr 601 buffer format defines
    LvPixelFormat_YCbCr601_8_CbYCr        = 0x0218003D, /**<  YCbCr601 4:4:4 8-bit CbYCrt.   Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY24BIT | 0x003D).  @showinitializer */
    LvPixelFormat_YCbCr601_422_8_CbYCrY   = 0x02100044, /**<  YCbCr601 4:2:2 8-bit CbYCrY.   Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY16BIT | 0x0044).  @showinitializer */
    LvPixelFormat_YCbCr601_422_8          = 0x0210003E, /**<  YCbCr601 4:2:2 8-bit YCbYCr.   Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY16BIT | 0x003E).  @showinitializer */
    LvPixelFormat_YCbCr601_411_8_CbYYCrYY = 0x020C003F, /**<  YCbCr601 4:1:1 8-bit CbYYCrYY. Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY12BIT | 0x003F).  @showinitializer */

    // YCbCr 709 buffer format defines
    LvPixelFormat_YCbCr709_8_CbYCr        = 0x02180040, /**<  YCbCr709 4:4:4 8-bit CbYCr.    Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY24BIT | 0x0040).  @showinitializer */
    LvPixelFormat_YCbCr709_422_8_CbYCrY   = 0x02100045, /**<  YCbCr709 4:2:2 8-bit CbYCrY.   Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY16BIT | 0x0045).  @showinitializer */
    LvPixelFormat_YCbCr709_422_8          = 0x02100041, /**<  YCbCr709 4:2:2 8-bit YCbYCr.   Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY16BIT | 0x0041).  @showinitializer */
    LvPixelFormat_YCbCr709_411_8_CbYYCrYY = 0x020C0042, /**<  YCbCr709 4:1:1 8-bit CbYYCrYY. Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY12BIT | 0x0042).  @showinitializer */

    // RGB Planar buffer format defines
    LvPixelFormat_RGB8_Planar      = 0x02180021,    /**<  RGB 8-bit planar.  Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY24BIT | 0x0021).  @showinitializer */
    LvPixelFormat_RGB10_Planar     = 0x02300022,    /**<  RGB 10-bit planar. Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY48BIT | 0x0022).  @showinitializer */
    LvPixelFormat_RGB12_Planar     = 0x02300023,    /**<  RGB 12-bit planar. Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY48BIT | 0x0023).  @showinitializer */
    LvPixelFormat_RGB16_Planar     = 0x02300024,    /**<  RGB 16-bit planar. Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY48BIT | 0x0024).  @showinitializer */

    // Non standard
    LvPixelFormat_BGR555P          = 0x021000E1     /**<  RGB 15-bit packed (3x5 bits). Defined as (LV_PIX_COLOR | LV_PIX_OCCUPY16BIT | 0x00E1). Not a standard GenICam format, used only in the image processing library. @showinitializer */


// alias for pixel formats according to SFNC-1.5 in order to be compatible to older apps which use old predefined constants
#define	LvPixelFormat_Mono8Signed       LvPixelFormat_Mono8S		/**< Alias for @ref LvPixelFormat_Mono8S. */
#define	LvPixelFormat_RGB8Packed	    LvPixelFormat_RGB8			/**< Alias for @ref LvPixelFormat_RGB8.   */
#define	LvPixelFormat_BGR8Packed	    LvPixelFormat_BGR8			/**< Alias for @ref LvPixelFormat_BGR8.   */
#define	LvPixelFormat_RGBA8Packed	    LvPixelFormat_RGBA8			/**< Alias for @ref LvPixelFormat_RGBA8.  */
#define	LvPixelFormat_BGRA8Packed	    LvPixelFormat_BGRA8			/**< Alias for @ref LvPixelFormat_BGRA8.  */
#define	LvPixelFormat_RGB10Packed	    LvPixelFormat_RGB10			/**< Alias for @ref LvPixelFormat_RGB10.  */
#define	LvPixelFormat_BGR10Packed	    LvPixelFormat_BGR10			/**< Alias for @ref LvPixelFormat_BGR10.  */
#define	LvPixelFormat_RGB12Packed	    LvPixelFormat_RGB12			/**< Alias for @ref LvPixelFormat_RGB12.  */
#define	LvPixelFormat_BGR12Packed	    LvPixelFormat_BGR12			/**< Alias for @ref LvPixelFormat_BGR12.  */
#define	LvPixelFormat_RGB16Packed	    LvPixelFormat_RGB16			/**< Alias for @ref LvPixelFormat_RGB16.  */
#define	LvPixelFormat_BGR16Packed	    LvPixelFormat_BGR16			/**< Alias for @ref LvPixelFormat_BGR16.  */
#define	LvPixelFormat_RGB10V2Packed	    LvPixelFormat_RGB10P32		/**< Alias for @ref LvPixelFormat_RGB10P32.  */
#define	LvPixelFormat_RGB565Packed	    LvPixelFormat_RGB565P		/**< Alias for @ref LvPixelFormat_RGB565P.  */
#define	LvPixelFormat_BGR565Packed	    LvPixelFormat_BGR565P		/**< Alias for @ref LvPixelFormat_BGR565P.  */
#define	LvPixelFormat_YUV411Packed	    LvPixelFormat_YUV411_8_UYYVYY /**< Alias for @ref LvPixelFormat_YUV411_8_UYYVYY.  */
#define	LvPixelFormat_YUV422Packed	    LvPixelFormat_YUV422_8_UYVY   /**< Alias for @ref LvPixelFormat_YUV422_8_UYVY.  */
#define	LvPixelFormat_YUV422YUYVPacked  LvPixelFormat_YUV422_8        /**< Alias for @ref LvPixelFormat_YUV422_8.  */
#define	LvPixelFormat_YUV444Packed	    LvPixelFormat_YUV8_UYV        /**< Alias for @ref LvPixelFormat_YUV8_UYV.  */
#define	LvPixelFormat_RGB8Planar	    LvPixelFormat_RGB8_Planar	/**< Alias for @ref LvPixelFormat_RGB8_Planar.  */
#define	LvPixelFormat_RGB10Planar	    LvPixelFormat_RGB10_Planar	/**< Alias for @ref LvPixelFormat_RGB10_Planar.  */
#define	LvPixelFormat_RGB12Planar	    LvPixelFormat_RGB12_Planar	/**< Alias for @ref LvPixelFormat_RGB12_Planar.  */
#define	LvPixelFormat_RGB16Planar	    LvPixelFormat_RGB16_Planar	/**< Alias for @ref LvPixelFormat_RGB16_Planar.  */

// capital letters vs lower case letters
#define	LvPixelFormat_Mono8s            LvPixelFormat_Mono8S	    /**< Alias for @ref LvPixelFormat_Mono8S.  */
#define	LvPixelFormat_RGBa8	            LvPixelFormat_RGBA8		    /**< Alias for @ref LvPixelFormat_RGBA8.  */
#define	LvPixelFormat_BGRa8	            LvPixelFormat_BGRA8		    /**< Alias for @ref LvPixelFormat_BGRA8.  */
#define	LvPixelFormat_RGB565p	        LvPixelFormat_RGB565P	    /**< Alias for @ref LvPixelFormat_RGB565P.  */
#define	LvPixelFormat_BGR565p	        LvPixelFormat_BGR565P	    /**< Alias for @ref LvPixelFormat_BGR565P.  */
#define	LvPixelFormat_RGB10p32	        LvPixelFormat_RGB10P32	    /**< Alias for @ref LvPixelFormat_RGB10P32.  */
#define	LvPixelFormat_BGR555p	        LvPixelFormat_BGR555P	    /**< Alias for @ref LvPixelFormat_BGR555P.  */

// other possible alias names
#define	LvPixelFormat_YUV411_8          LvPixelFormat_YUV411_8_UYYVYY /**< Alias for @ref LvPixelFormat_YUV411_8_UYYVYY.  */
#define	LvPixelFormat_YUV8              LvPixelFormat_YUV8_UYV        /**< Alias for @ref LvPixelFormat_YUV8_UYV.  */

};

//------------------------------------------------------------------------------

/** This enum is used for opening the Device - see LvDeviceOpen().
 *  @ingroup GroupSynview_EnumEntries
 *  @showinitializer
 */
enum LvDeviceAccess
{
    /** Represents the GenTL DEVICE_ACCESS_UNKNOWN. @showinitializer */
    LvDeviceAccess_Unknown   = 0,

    /** Represents the GenTL DEVICE_ACCESS_NONE.  This either means that the Device
     *  is not open because it was not opened before or the access to it was denied. @showinitializer */
    LvDeviceAccess_None      = 1,

    /** Represents the GenTL DEVICE_ACCESS_READONLY. Open the Device read only.
     *  All Port functions can only read from the Device. @showinitializer */
    LvDeviceAccess_ReadOnly  = 2,

    /** Represents the GenTL DEVICE_ACCESS_CONTROL. Open the Device in a way that
     *  other hosts/processes can have read only access to the Device. Device access
     *  level is read/write for this process. @showinitializer */
    LvDeviceAccess_Control   = 3,

    /** Represents the GenTL DEVICE_ACCESS_EXCLUSIVE. Open the Device in a way
     *  that only this host/process can have access to the Device. Device access
     *  level is read/write for this process. @showinitializer */
    LvDeviceAccess_Exclusive = 4
};

//------------------------------------------------------------------------------

/** Values for the @ref LvFtrInfo_DeviceAccessStatus, @ref LvInterface_DeviceAccessStatus and 
 *  @ref LvDevice_DeviceAccessStatus features.
 *  @ingroup GroupSynview_EnumEntries
 *  @showinitializer
 */
enum LvDeviceAccessStatus
{
    /** Represents the GenTL DEVICE_ACCESS_STATUS_UNKNOWN. The current availability
     *  of the Device is unknown. @showinitializer */
    LvDeviceAccessStatus_Unknown   = 0,

    /** Represents the GenTL DEVICE_ACCESS_STATUS_READWRITE - The Device is available
     *  for Read/Write. @showinitializer */
    LvDeviceAccessStatus_ReadWrite = 1,

    /** Represents the GenTL DEVICE_ACCESS_STATUS_READONLY - The Device is available
     *  only for Read access (cannot be controlled). @showinitializer */
    LvDeviceAccessStatus_ReadOnly  = 2,

    /** Represents the GenTL DEVICE_ACCESS_STATUS_NOACCESS - The Device is not
     *  available either because it is already open or because it is not reachable. @showinitializer */
    LvDeviceAccessStatus_NoAccess  = 3,

    /** Represents the GenTL DEVICE_ACCESS_STATUS_BUSY - The Device is already opened
     *  by another entity. @showinitializer */
    LvDeviceAccessStatus_Busy      = 4,

    /** Represents the GenTL DEVICE_ACCESS_STATUS_OPEN_READWRITE - The Device is open and
     *  available for Read/Write.. @showinitializer */
    LvDeviceAccessStatus_OpenReadWrite = 5,

    /** Represents the GenTL DEVICE_ACCESS_STATUS_OPEN_READ - The Device is open and
     *  available only for Read access. @showinitializer */
    LvDeviceAccessStatus_OpenReadOnly = 6
};


//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_DeviceScanType feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvDeviceScanType
{
    /** Indicates area scan sensor. */
    LvDeviceScanType_Areascan                     = LV_FTR_ENUM | 0x00010000,
    /** Indicates linr scan sensor. */
    LvDeviceScanType_Linescan                     = LV_FTR_ENUM | 0x00010001
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_DeviceClockSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvDeviceClockSelector
{
    /** Sensor digitization clock. */
    LvDeviceClockSelector_SensorDigitization      = LV_FTR_ENUM | 0x00020000
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_DeviceTemperatureSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
*/
enum LvDeviceTemperatureSelector
{
    /** Temperature on sensor */
    LvDeviceTemperatureSelector_Sensor            = LV_FTR_ENUM | 0x00030000,

    /** Temperature on main board */
    LvDeviceTemperatureSelector_Mainboard         = LV_FTR_ENUM | 0x00030001,

    /** Temperature on fpga chip */
    LvDeviceTemperatureSelector_FPGA              = LV_FTR_ENUM | 0x00030002
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvAOIMode feature.
 *  @ingroup GroupSynview_EnumEntries
 */
enum LvAOIMode
{
    /** Camera automatically applies as much of the desired AOI setting on the sensor and the rest is cut on transfer */
    LvAOIMode_Automatic                           = LV_FTR_ENUM | 0x00040000,

    /** The AOI is applied before the transfer, in camera memory */
    LvAOIMode_ClipOnTransfer                      = LV_FTR_ENUM | 0x00040001,

    /** Fine control of separate AOI setting on the sensor and before the transfer */
    LvAOIMode_Manual                              = LV_FTR_ENUM | 0x00040002
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_AcquisitionMode feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvAcquisitionMode
{
    /** Single frame acquisition - after acquisition starts, single frame is acquired and acquisition stops. */
    LvAcquisitionMode_SingleFrame                 = LV_FTR_ENUM | 0x00050000,

    /** Multiple frame acquisition - after acquisition starts, specified numbr of frames is acquired and acquisition stops. */
    LvAcquisitionMode_MultiFrame                  = LV_FTR_ENUM | 0x00050001,

    /** Continuous acquisition - after starting, the acquisition is active until explicitly stopped. */
    LvAcquisitionMode_Continuous                  = LV_FTR_ENUM | 0x00050002 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_ExposureAuto feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvExposureAuto
{
    /** Automatic exposure mode off - the automatic exposure is not applied. */
    LvExposureAuto_Off                            = LV_FTR_ENUM | 0x00060000,

    /** Automatic exposure mode once - the exposure time is once adjusted, then switches back to off. */
    LvExposureAuto_Once                           = LV_FTR_ENUM | 0x00060001,

    /** Automatic exposure mode continuous - the exposure time is continuously auto-adjusted. */
    LvExposureAuto_Continuous                     = LV_FTR_ENUM | 0x00060002 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_TriggerSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvTriggerSelector
{
    /** Frame start trigger - controls a new frame acquisition. */
    LvTriggerSelector_FrameStart                  = LV_FTR_ENUM | 0x00070000,

    /** Frame burst start trigger - Selects a trigger starting the capture of the bursts of frames. */
    LvTriggerSelector_FrameBurstStart             = LV_FTR_ENUM | 0x00070001,

    /** Line start trigger - Selects a trigger starting the capture of one Line of a Frame. */
    LvTriggerSelector_LineStart                   = LV_FTR_ENUM | 0x00070002,

    /** Virtual frame active trigger - Selects a trigger activating the capture of virtual Frames. */
    LvTriggerSelector_VirtualFrameActive          = LV_FTR_ENUM | 0x00070003 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_TriggerMode feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvTriggerMode
{
    /** Trigger mode off - disables selected trigger */
    LvTriggerMode_Off                             = LV_FTR_ENUM | 0x00080000,

    /** Trigger mode on - enables selected trigger. */
    LvTriggerMode_On                              = LV_FTR_ENUM | 0x00080001 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_TriggerSource feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvTriggerSource
{
    /** Sets the signal source for the selected trigger to line 1 (optocoupler input). */
    LvTriggerSource_Line1                         = LV_FTR_ENUM | 0x00090000,

    /** Sets the signal source for the selected trigger to line 2 (optocoupler input). */
    LvTriggerSource_Line2                         = LV_FTR_ENUM | 0x00090001,

    /** Sets the signal source for the selected trigger to line 3 (optocoupler input). */
    LvTriggerSource_Line3                         = LV_FTR_ENUM | 0x00090002,

    /** Sets the signal source for the selected trigger to line 4 (optocoupler input). */
    LvTriggerSource_Line4                         = LV_FTR_ENUM | 0x00090003,

    /** Sets the signal source for the selected trigger to line 5. */
    LvTriggerSource_Line5                         = LV_FTR_ENUM | 0x00090004,

    /** Sets the signal source for the selected trigger to line 6. */
    LvTriggerSource_Line6                         = LV_FTR_ENUM | 0x00090005,

    /** Sets the signal source for the selected trigger to line 7. */
    LvTriggerSource_Line7                         = LV_FTR_ENUM | 0x00090006,

    /** Sets the signal source for the selected trigger to line 8. */
    LvTriggerSource_Line8                         = LV_FTR_ENUM | 0x00090007,

    /** Sets the signal source for the selected trigger to line 17 (TTL input). */
    LvTriggerSource_Line17                        = LV_FTR_ENUM | 0x00090008,

    /** Sets the signal source for the selected trigger to line 18 (TTL input). */
    LvTriggerSource_Line18                        = LV_FTR_ENUM | 0x00090009,

    /** Sets the signal source for the selected trigger to line 19. */
    LvTriggerSource_Line19                        = LV_FTR_ENUM | 0x0009000A,

    /** Sets the signal source for the selected trigger to line 20. */
    LvTriggerSource_Line20                        = LV_FTR_ENUM | 0x0009000B,

    /** Sets the signal source for the selected trigger to line 21. */
    LvTriggerSource_Line21                        = LV_FTR_ENUM | 0x0009000C,

    /** Sets the signal source for the selected trigger to line 22. */
    LvTriggerSource_Line22                        = LV_FTR_ENUM | 0x0009000D,

    /** Sets the signal source for the selected trigger to line 23. */
    LvTriggerSource_Line23                        = LV_FTR_ENUM | 0x0009000E,

    /** Sets the signal source for the selected trigger to line 24. */
    LvTriggerSource_Line24                        = LV_FTR_ENUM | 0x0009000F,

    /** Sets the signal source for the selected trigger to software. */
    LvTriggerSource_Software                      = LV_FTR_ENUM | 0x00090010,

    /** Sets the signal source for the selected trigger to action signal 1. */
    LvTriggerSource_Action1                       = LV_FTR_ENUM | 0x00090011,

    /** Sets the signal source for the selected trigger to action signal 2. */
    LvTriggerSource_Action2                       = LV_FTR_ENUM | 0x00090012,

    /** Sets the signal source for the selected trigger to action signal 3. */
    LvTriggerSource_Action3                       = LV_FTR_ENUM | 0x00090013,

    /** Sets the signal source for the selected trigger to action signal 4. */
    LvTriggerSource_Action4                       = LV_FTR_ENUM | 0x00090014,

    /** Sets the signal source for the selected trigger to action signal 5. */
    LvTriggerSource_Action5                       = LV_FTR_ENUM | 0x00090015,

    /** Sets the signal source for the selected trigger to action signal 6. */
    LvTriggerSource_Action6                       = LV_FTR_ENUM | 0x00090016,

    /** Sets the signal source for the selected trigger to action signal 7. */
    LvTriggerSource_Action7                       = LV_FTR_ENUM | 0x00090017,

    /** Sets the signal source for the selected trigger to action signal 8. */
    LvTriggerSource_Action8                       = LV_FTR_ENUM | 0x00090018,

    /** Sets the signal source for the selected trigger to quadrature decoder. */
    LvTriggerSource_Quad                          = LV_FTR_ENUM | 0x00090019,

    /** Sets the signal source for the selected trigger to counter 1. */
    LvTriggerSource_Counter1                      = LV_FTR_ENUM | 0x0009001a,

    /** Sets the signal source for the selected trigger to counter 2. */
    LvTriggerSource_Counter2                      = LV_FTR_ENUM | 0x0009001b,

    /** Sets the signal source for the selected trigger to counter 3. */
    LvTriggerSource_Counter3                      = LV_FTR_ENUM | 0x0009001c,

    /** Sets the signal source for the selected trigger to counter 4. */
    LvTriggerSource_Counter4                      = LV_FTR_ENUM | 0x0009001d,

    /** Sets the signal source for the selected trigger to timer 1. */
    LvTriggerSource_Timer1                        = LV_FTR_ENUM | 0x0009001e,

    /** Sets the signal source for the selected trigger to timer 2. */
    LvTriggerSource_Timer2                        = LV_FTR_ENUM | 0x0009001f,

    /** Sets the signal source for the selected trigger to timer 3. */
    LvTriggerSource_Timer3                        = LV_FTR_ENUM | 0x00090020,

    /** Sets the signal source for the selected trigger to timer 4. */
    LvTriggerSource_Timer4                        = LV_FTR_ENUM | 0x00090021,

    /** Sets the signal source for the selected trigger to user output 1. */
    LvTriggerSource_UserOutput1                   = LV_FTR_ENUM | 0x00090022,

    /** Sets the signal source for the selected trigger to user output 2. */
    LvTriggerSource_UserOutput2                   = LV_FTR_ENUM | 0x00090023,

    /** Sets the signal source for the selected trigger to user output 3. */
    LvTriggerSource_UserOutput3                   = LV_FTR_ENUM | 0x00090024,

    /** Sets the signal source for the selected trigger to user output 4. */
    LvTriggerSource_UserOutput4                   = LV_FTR_ENUM | 0x00090025,

    /** Sets the signal source for the selected trigger to user output 5. */
    LvTriggerSource_UserOutput5                   = LV_FTR_ENUM | 0x00090026,

    /** Sets the signal source for the selected trigger to user output 6. */
    LvTriggerSource_UserOutput6                   = LV_FTR_ENUM | 0x00090027,

    /** Sets the signal source for the selected trigger to user output 7. */
    LvTriggerSource_UserOutput7                   = LV_FTR_ENUM | 0x00090028,

    /** Sets the signal source for the selected trigger to user output 8. */
    LvTriggerSource_UserOutput8                   = LV_FTR_ENUM | 0x00090029
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_TriggerActivation feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvTriggerActivation
{
    /** Selects the trigger signal rising edge as active. */
    LvTriggerActivation_RisingEdge                = LV_FTR_ENUM | 0x000A0000,

    /** Selects the trigger signal falling edge as active */
    LvTriggerActivation_FallingEdge               = LV_FTR_ENUM | 0x000A0001,

    /** Selects the trigger signal any edge as active */
    LvTriggerActivation_AnyEdge                   = LV_FTR_ENUM | 0x000A0002,

    /** Selects the trigger signal high level as active */
    LvTriggerActivation_LevelHigh                 = LV_FTR_ENUM | 0x000A0003,

    /** Selects the trigger signal low level as active */
    LvTriggerActivation_LevelLow                  = LV_FTR_ENUM | 0x000A0004 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvTriggerCaching feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvTriggerCaching
{
    /** Trigger caching mode 'cache' - early triggers are cached and applied as soon as possible. */
    LvTriggerCaching_Cache                        = LV_FTR_ENUM | 0x000B0000,

    /** Trigger caching mode 'cache' - early triggers are dropped */
    LvTriggerCaching_Drop                         = LV_FTR_ENUM | 0x000B0001 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_ExposureMode feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvExposureMode
{
    /** Timed exposure mode - the exposure time is controlled by corresponding feature. */
    LvExposureMode_Timed                          = LV_FTR_ENUM | 0x000C0000,

    #ifdef SIMPLON_INC_OBSOLETE
    /** Trigger width exposure mode - the exposure time is controlled by trigger signal width. */
    LvExposureMode_TriggerWidth                   = LV_FTR_ENUM | 0x000C0001
    #endif
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvAcquisitionFrameRateControlMode feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvAcquisitionFrameRateControlMode
{
    /** Disables frame rate control - the camera operates at maximum frame rate */
    LvAcquisitionFrameRateControlMode_Off         = LV_FTR_ENUM | 0x000D0000,

    /** Enables frame rate control - the rate can be explicitly adjusted */
    LvAcquisitionFrameRateControlMode_On          = LV_FTR_ENUM | 0x000D0001 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LineSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvLineSelector
{
    /** Selects device's logical line 1 (optocoupler input). */
    LvLineSelector_Line1                          = LV_FTR_ENUM | 0x000E0000,

    /** Selects device's logical line 2 (optocoupler input). */
    LvLineSelector_Line2                          = LV_FTR_ENUM | 0x000E0001,

    /** Selects device's logical line 3 (optocoupler input). */
    LvLineSelector_Line3                          = LV_FTR_ENUM | 0x000E0002,

    /** Selects device's logical line 4 (optocoupler input). */
    LvLineSelector_Line4                          = LV_FTR_ENUM | 0x000E0003,

    /** Selects device's logical line 5. */
    LvLineSelector_Line5                          = LV_FTR_ENUM | 0x000E0004,

    /** Selects device's logical line 6. */
    LvLineSelector_Line6                          = LV_FTR_ENUM | 0x000E0005,

    /** Selects device's logical line 7. */
    LvLineSelector_Line7                          = LV_FTR_ENUM | 0x000E0006,

    /** Selects device's logical line 8. */
    LvLineSelector_Line8                          = LV_FTR_ENUM | 0x000E0007, 

    /** Selects device's logical line 9 (optocoupler output). */
    LvLineSelector_Line9                          = LV_FTR_ENUM | 0x000E0008,

    /** Selects device's logical line 10 (optocoupler output). */
    LvLineSelector_Line10                         = LV_FTR_ENUM | 0x000E0009,

    /** Selects device's logical line 11 (optocoupler output). */
    LvLineSelector_Line11                         = LV_FTR_ENUM | 0x000E000A,

    /** Selects device's logical line 12 (optocoupler output). */
    LvLineSelector_Line12                         = LV_FTR_ENUM | 0x000E000B,

    /** Selects device's logical line 13. */
    LvLineSelector_Line13                         = LV_FTR_ENUM | 0x000E000C,

    /** Selects device's logical line 14. */
    LvLineSelector_Line14                         = LV_FTR_ENUM | 0x000E000D,

    /** Selects device's logical line 15. */
    LvLineSelector_Line15                         = LV_FTR_ENUM | 0x000E000E,

    /** Selects device's logical line 16. */
    LvLineSelector_Line16                         = LV_FTR_ENUM | 0x000E000F,

    /** Selects device's logical line 17 (TTL input). */
    LvLineSelector_Line17                         = LV_FTR_ENUM | 0x000E0010,

    /** Selects device's logical line 18 (TTL input). */
    LvLineSelector_Line18                         = LV_FTR_ENUM | 0x000E0011,

    /** Selects device's logical line 19. */
    LvLineSelector_Line19                         = LV_FTR_ENUM | 0x000E0012,

    /** Selects device's logical line 20. */
    LvLineSelector_Line20                         = LV_FTR_ENUM | 0x000E0013,

    /** Selects device's logical line 21. */
    LvLineSelector_Line21                         = LV_FTR_ENUM | 0x000E0014,

    /** Selects device's logical line 22. */
    LvLineSelector_Line22                         = LV_FTR_ENUM | 0x000E0015,

    /** Selects device's logical line 23. */
    LvLineSelector_Line23                         = LV_FTR_ENUM | 0x000E0016,

    /** Selects device's logical line 24. */
    LvLineSelector_Line24                         = LV_FTR_ENUM | 0x000E0017,

    /** Selects device's logical line 25 (TTL output). */
    LvLineSelector_Line25                         = LV_FTR_ENUM | 0x000E0018,

    /** Selects device's logical line 26 (TTL output). */
    LvLineSelector_Line26                         = LV_FTR_ENUM | 0x000E0019,

    /** Selects device's logical line 27. */
    LvLineSelector_Line27                         = LV_FTR_ENUM | 0x000E001A,

    /** Selects device's logical line 28. */
    LvLineSelector_Line28                         = LV_FTR_ENUM | 0x000E001B,

    /** Selects device's logical line 29. */
    LvLineSelector_Line29                         = LV_FTR_ENUM | 0x000E001C,

    /** Selects device's logical line 30. */
    LvLineSelector_Line30                         = LV_FTR_ENUM | 0x000E001D,

    /** Selects device's logical line 31. */
    LvLineSelector_Line31                         = LV_FTR_ENUM | 0x000E001E,

    /** Selects device's logical line 32. */
    LvLineSelector_Line32                         = LV_FTR_ENUM | 0x000E001F

};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LineMode feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvLineMode
{
    /** The line is used as signal input. */     
    LvLineMode_Input                              = LV_FTR_ENUM | 0x000F0000,

    /** The line is used as signal output. */
    LvLineMode_Output                             = LV_FTR_ENUM | 0x000F0001 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LineFormat feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvLineFormat
{
    /** Not connected line. */
    LvLineFormat_NoConnect                        = LV_FTR_ENUM | 0x00100000,

    /** The Line is currently in Tri-state mode (Not driven). */
    LvLineFormat_TriState                         = LV_FTR_ENUM | 0x00100001,

    /** The Line is currently accepting or sending TTL level signals. */
    LvLineFormat_TTL                              = LV_FTR_ENUM | 0x00100002,

    /** The Line is currently accepting or sending LVDS level signals. */
    LvLineFormat_LVDS                             = LV_FTR_ENUM | 0x00100003,

    /** The Line is currently accepting or sending RS-422 level signals. */
    LvLineFormat_RS422                            = LV_FTR_ENUM | 0x00100004,

    /** Optically isolated line (optocoupler). */
    LvLineFormat_OptoCoupled                      = LV_FTR_ENUM | 0x00100005 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LineSource feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvLineSource
{
    /** Switches the line source off. This disables the line output (disconnects the line). */
    LvLineSource_Off                              = LV_FTR_ENUM | 0x00110000,

    /** Selects exposure active signal as line source. */
    LvLineSource_ExposureActive                   = LV_FTR_ENUM | 0x00110001,

    /** Selects timer 1 active signal as line source. */
    LvLineSource_Timer1Active                     = LV_FTR_ENUM | 0x00110002,

    /** Selects timer 2 active signal as line source. */
    LvLineSource_Timer2Active                     = LV_FTR_ENUM | 0x00110003,

    /** Selects timer 3 active signal as line source. */
    LvLineSource_Timer3Active                     = LV_FTR_ENUM | 0x00110004,

    /** Selects timer 4 active signal as line source. */
    LvLineSource_Timer4Active                     = LV_FTR_ENUM | 0x00110005, 

    /** Selects user output 1 value signal as line source. */
    LvLineSource_UserOutput1                      = LV_FTR_ENUM | 0x00110006, 

    /** Selects user output 2 value signal as line source. */
    LvLineSource_UserOutput2                      = LV_FTR_ENUM | 0x00110007, 

    /** Selects user output 3 value signal as line source. */
    LvLineSource_UserOutput3                      = LV_FTR_ENUM | 0x00110008, 

    /** Selects user output 4 value signal as line source. */
    LvLineSource_UserOutput4                      = LV_FTR_ENUM | 0x00110009, 

    /** Selects user output 5 value signal as line source. */
    LvLineSource_UserOutput5                      = LV_FTR_ENUM | 0x0011000A, 

    /** Selects user output 6 value signal as line source. */
    LvLineSource_UserOutput6                      = LV_FTR_ENUM | 0x0011000B, 

    /** Selects user output 7 value signal as line source. */
    LvLineSource_UserOutput7                      = LV_FTR_ENUM | 0x0011000C, 

    /** Selects user output 8 value signal as line source. */
    LvLineSource_UserOutput8                      = LV_FTR_ENUM | 0x0011000D, 

    /** Selects counter 1 active signal as line source. */
    LvLineSource_Counter1Active                   = LV_FTR_ENUM | 0x0011000E,

    /** Selects counter 2 active signal as line source. */
    LvLineSource_Counter2Active                   = LV_FTR_ENUM | 0x0011000F,

    /** Selects counter 3 active signal as line source. */
    LvLineSource_Counter3Active                   = LV_FTR_ENUM | 0x00110010,

    /** Selects counter 4 active signal as line source. */
    LvLineSource_Counter4Active                   = LV_FTR_ENUM | 0x00110011
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LineMode feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvLineDebounceMode
{
    /** The line debounce filter is used in debounce mode. */     
    LvLineDebounceMode_Debounce                   = LV_FTR_ENUM | 0x00110100,

    /** The line debounce filter is used in deglitch mode. */
    LvLineDebounceMode_Deglitch                   = LV_FTR_ENUM | 0x00110101 
};


//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_CounterSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvCounterSelector
{
    /** Selects counter 1 for configuration. */
    LvCounterSelector_Counter1                    = LV_FTR_ENUM | 0x00120000,

    /** Selects counter 2 for configuration. */
    LvCounterSelector_Counter2                    = LV_FTR_ENUM | 0x00120001,

    /** Selects counter 3 for configuration. */
    LvCounterSelector_Counter3                    = LV_FTR_ENUM | 0x00120002,

    /** Selects counter 4 for configuration. */
    LvCounterSelector_Counter4                    = LV_FTR_ENUM | 0x00120003 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvCounterMode feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvCounterMode
{
    /** Automatic reset mode. Once completed, the counter automatically resets itself and starts counting again. */
    LvCounterMode_Autoreset                       = LV_FTR_ENUM | 0x00130000 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_CounterEventSource feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvCounterEventSource
{
    /** Switches counter event signal off - no signal will be incrementing the counter */
    LvCounterEventSource_Off                      = LV_FTR_ENUM | 0x00140000,

    /** Switches counter event signal to frame trigger - activation of the frame trigger internal signal (before counting down eventual trigger delay) increments the counter. */
    LvCounterEventSource_FrameTrigger             = LV_FTR_ENUM | 0x00140001, 

    /** Switches counter event signal to timer tick - 1MHz clock increments the counter. */
    LvCounterEventSource_TimerTick                = LV_FTR_ENUM | 0x00140002, 

    /** Switches counter event signal to line 1 (optocoupler input) - active edge of line 1 increments the counter. */
    LvCounterEventSource_Line1                    = LV_FTR_ENUM | 0x00140003, 

    /** Switches counter event signal to line 2 (optocoupler input) - active edge of line 2 increments the counter. */
    LvCounterEventSource_Line2                    = LV_FTR_ENUM | 0x00140004, 

    /** Switches counter event signal to line 3 (optocoupler input) - active edge of line 3 increments the counter. */
    LvCounterEventSource_Line3                    = LV_FTR_ENUM | 0x00140005, 

    /** Switches counter event signal to line 4 (optocoupler input) - active edge of line 4 increments the counter. */
    LvCounterEventSource_Line4                    = LV_FTR_ENUM | 0x00140006, 

    /** Switches counter event signal to line 17 (TTL input) - active edge of line 17 increments the counter. */
    LvCounterEventSource_Line17                   = LV_FTR_ENUM | 0x00140007, 

    /** Switches counter event signal to line 18 (TTL input) - active edge of line 18 increments the counter. */
    LvCounterEventSource_Line18                   = LV_FTR_ENUM | 0x00140008 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_TimerSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvTimerSelector
{
    /** Selects timer 1 for configuration. */
    LvTimerSelector_Timer1                        = LV_FTR_ENUM | 0x00150000,

    /** Selects timer 2 for configuration. */
    LvTimerSelector_Timer2                        = LV_FTR_ENUM | 0x00150001,

    /** Selects timer 3 for configuration. */
    LvTimerSelector_Timer3                        = LV_FTR_ENUM | 0x00150002,

    /** Selects timer 4 for configuration. */
    LvTimerSelector_Timer4                        = LV_FTR_ENUM | 0x00150003 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_TimerTriggerSource feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvTimerTriggerSource
{
    /** Switches timer trigger signal off - no signal will be firing the timer */
    LvTimerTriggerSource_Off                      = LV_FTR_ENUM | 0x00160000,

    /** Switches timer trigger signal to frame trigger - activation of the frame trigger internal signal (before counting down eventual trigger delay) activates the timer. */
    LvTimerTriggerSource_FrameTrigger             = LV_FTR_ENUM | 0x00160001,

    /** Switches timer trigger signal to counter 1 end - expiration of counter 1 activates the timer. */
    LvTimerTriggerSource_Counter1End              = LV_FTR_ENUM | 0x00160002,

    /** Switches timer trigger signal to counter 2 end - expiration of counter 2 activates the timer. */
    LvTimerTriggerSource_Counter2End              = LV_FTR_ENUM | 0x00160003,

    /** Switches timer trigger signal to counter 3 end - expiration of counter 3 activates the timer. */
    LvTimerTriggerSource_Counter3End              = LV_FTR_ENUM | 0x00160004,

    /** Switches timer trigger signal to counter 4 end - expiration of counter 4 activates the timer. */
    LvTimerTriggerSource_Counter4End              = LV_FTR_ENUM | 0x00160005, 

    /** Switches timer trigger signal to user output 1 - activation of user output 1 activates the timer. */
    LvTimerTriggerSource_UserOutput1              = LV_FTR_ENUM | 0x00160006, 

    /** Switches timer trigger signal to user output 2 - activation of user output 2 activates the timer. */
    LvTimerTriggerSource_UserOutput2              = LV_FTR_ENUM | 0x00160007, 

    /** Switches timer trigger signal to user output 3 - activation of user output 3 activates the timer. */
    LvTimerTriggerSource_UserOutput3              = LV_FTR_ENUM | 0x00160008,

    /** Switches timer trigger signal to user output 4 - activation of user output 4 activates the timer. */
    LvTimerTriggerSource_UserOutput4              = LV_FTR_ENUM | 0x00160009, 

    /** Switches timer trigger signal to user output 5 - activation of user output 5 activates the timer. */
    LvTimerTriggerSource_UserOutput5              = LV_FTR_ENUM | 0x0016000a, 

    /** Switches timer trigger signal to user output 6 - activation of user output 6 activates the timer. */
    LvTimerTriggerSource_UserOutput6              = LV_FTR_ENUM | 0x0016000b, 

    /** Switches timer trigger signal to user output 7 - activation of user output 7 activates the timer. */
    LvTimerTriggerSource_UserOutput7              = LV_FTR_ENUM | 0x0016000c, 

    /** Switches timer trigger signal to user output 8 - activation of user output 8 activates the timer. */
    LvTimerTriggerSource_UserOutput8              = LV_FTR_ENUM | 0x0016000d 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvSpecialPurposeTriggerSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvSpecialPurposeTriggerSelector
{
    /** Timestamps reset trigger - controls reset of timestamp, frame ID and other image stamps. */
    LvSpecialPurposeTriggerSelector_ImageStampsReset = LV_FTR_ENUM | 0x00170000 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvSpecialPurposeTriggerSource feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvSpecialPurposeTriggerSource
{
    /** Sets trigger source off - it can be still be issued by an explicit software trigger */
    LvSpecialPurposeTriggerSource_Off             = LV_FTR_ENUM | 0x00180000,

    /** Sets the signal source for the selected trigger to line 1 (optocoupler input). */
    LvSpecialPurposeTriggerSource_Line1           = LV_FTR_ENUM | 0x00180001,

    /** Sets the signal source for the selected trigger to line 2 (optocoupler input). */
    LvSpecialPurposeTriggerSource_Line2           = LV_FTR_ENUM | 0x00180002,

    /** Sets the signal source for the selected trigger to line 3 (optocoupler input). */
    LvSpecialPurposeTriggerSource_Line3           = LV_FTR_ENUM | 0x00180003,

    /** Sets the signal source for the selected trigger to line 4 (optocoupler input). */
    LvSpecialPurposeTriggerSource_Line4           = LV_FTR_ENUM | 0x00180004,

    /** Sets the signal source for the selected trigger to line 5. */
    LvSpecialPurposeTriggerSource_Line5           = LV_FTR_ENUM | 0x00180005,

    /** Sets the signal source for the selected trigger to line 6. */
    LvSpecialPurposeTriggerSource_Line6           = LV_FTR_ENUM | 0x00180006,

    /** Sets the signal source for the selected trigger to line 7. */
    LvSpecialPurposeTriggerSource_Line7           = LV_FTR_ENUM | 0x00180007,

    /** Sets the signal source for the selected trigger to line 8. */
    LvSpecialPurposeTriggerSource_Line8           = LV_FTR_ENUM | 0x00180008,

    /** Sets the signal source for the selected trigger to line 17 (TTL input). */
    LvSpecialPurposeTriggerSource_Line17          = LV_FTR_ENUM | 0x00180009,

    /** Sets the signal source for the selected trigger to line 18 (TTL input). */
    LvSpecialPurposeTriggerSource_Line18          = LV_FTR_ENUM | 0x0018000A,

    /** Sets the signal source for the selected trigger to line 19. */
    LvSpecialPurposeTriggerSource_Line19          = LV_FTR_ENUM | 0x0018000B,

    /** Sets the signal source for the selected trigger to line 20. */
    LvSpecialPurposeTriggerSource_Line20          = LV_FTR_ENUM | 0x0018000C,

    /** Sets the signal source for the selected trigger to line 21. */
    LvSpecialPurposeTriggerSource_Line21          = LV_FTR_ENUM | 0x0018000D,

    /** Sets the signal source for the selected trigger to line 22. */
    LvSpecialPurposeTriggerSource_Line22          = LV_FTR_ENUM | 0x0018000E,

    /** Sets the signal source for the selected trigger to line 23. */
    LvSpecialPurposeTriggerSource_Line23          = LV_FTR_ENUM | 0x0018000F,

    /** Sets the signal source for the selected trigger to line 24. */
    LvSpecialPurposeTriggerSource_Line24          = LV_FTR_ENUM | 0x00180010,

    /** Sets the signal source for the selected trigger to action signal 1. */
    LvSpecialPurposeTriggerSource_Action1         = LV_FTR_ENUM | 0x00180011,

    /** Sets the signal source for the selected trigger to action signal 2. */
    LvSpecialPurposeTriggerSource_Action2         = LV_FTR_ENUM | 0x00180012,

    /** Sets the signal source for the selected trigger to action signal 3. */
    LvSpecialPurposeTriggerSource_Action3         = LV_FTR_ENUM | 0x00180013,

    /** Sets the signal source for the selected trigger to action signal 4. */
    LvSpecialPurposeTriggerSource_Action4         = LV_FTR_ENUM | 0x00180014,

    /** Sets the signal source for the selected trigger to action signal 5. */
    LvSpecialPurposeTriggerSource_Action5         = LV_FTR_ENUM | 0x00180015,

    /** Sets the signal source for the selected trigger to action signal 6. */
    LvSpecialPurposeTriggerSource_Action6         = LV_FTR_ENUM | 0x00180016,

    /** Sets the signal source for the selected trigger to action signal 7. */
    LvSpecialPurposeTriggerSource_Action7         = LV_FTR_ENUM | 0x00180017,

    /** Sets the signal source for the selected trigger to action signal 8. */
    LvSpecialPurposeTriggerSource_Action8         = LV_FTR_ENUM | 0x00180018
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvSpecialPurposeTriggerActivation feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvSpecialPurposeTriggerActivation
{
    /** Selects the trigger signal rising edge as active. */
    LvSpecialPurposeTriggerActivation_RisingEdge  = LV_FTR_ENUM | 0x00190000,

    /** Selects the trigger signal falling edge as active */
    LvSpecialPurposeTriggerActivation_FallingEdge = LV_FTR_ENUM | 0x00190001 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvImageStampSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvImageStampSelector
{
    /** Selects the flag controling reset of the image timestamp */
    LvImageStampSelector_Timestamp                = LV_FTR_ENUM | 0x001A0000,

    /** Selects the flag controling reset of the image frame ID */
    LvImageStampSelector_FrameID                  = LV_FTR_ENUM | 0x001A0001 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvBootSwitch feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvBootSwitch
{
    /** Selects the pure GigE Vision mode strictly following the GigE Vision specification */
    LvBootSwitch_PureGEV                          = LV_FTR_ENUM | 0x001B0000,

    /** Selects the legacy mode allowing dual operation through GigE Vision or custom protocol. */
    LvBootSwitch_Legacy                           = LV_FTR_ENUM | 0x001B0001 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_GainSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvGainSelector
{
    /** Apply gain on all channels and taps. */
    LvGainSelector_All                            = LV_FTR_ENUM | 0x001C0000,

    /** Apply analog gain. */
    LvGainSelector_AnalogAll                      = LV_FTR_ENUM | 0x001C0001,

    /** Apply digital gain. */
    LvGainSelector_DigitalAll                     = LV_FTR_ENUM | 0x001C0002,

    /** Apply gain on all red channels. */
    LvGainSelector_Red                            = LV_FTR_ENUM | 0x001C0003,

    /** Apply gain on all green channels. */
    LvGainSelector_Green                          = LV_FTR_ENUM | 0x001C0004,

    /** Apply gain on all blue channels. */
    LvGainSelector_Blue                           = LV_FTR_ENUM | 0x001C0005,

    /** Apply analog gain on red channel. */
    LvGainSelector_AnalogRed                      = LV_FTR_ENUM | 0x001C0006,

    /** Apply analog gain on green channel. */
    LvGainSelector_AnalogGreen                    = LV_FTR_ENUM | 0x001C0007,

    /** Apply analog gain on blue channel. */
    LvGainSelector_AnalogBlue                     = LV_FTR_ENUM | 0x001C0008,

    /** Apply digital gain on red channel. */
    LvGainSelector_DigitalRed                     = LV_FTR_ENUM | 0x001C0009,

    /** Apply digital gain on green channel. */
    LvGainSelector_DigitalGreen                   = LV_FTR_ENUM | 0x001C000A,

    /** Apply digital gain on blue channel. */
    LvGainSelector_DigitalBlue                    = LV_FTR_ENUM | 0x001C000B,

    /** Apply digital gain on Y channel. */
    LvGainSelector_DigitalY                       = LV_FTR_ENUM | 0x001C000C,

    /** Apply digital gain on U channel. */
    LvGainSelector_DigitalU                       = LV_FTR_ENUM | 0x001C000D,

    /** Apply digital gain on V channel. */
    LvGainSelector_DigitalV                       = LV_FTR_ENUM | 0x001C000E

};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_GainAuto feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvGainAuto
{
    /** Automatic gain mode off - the gain value is controlled 'manually'. */
    LvGainAuto_Off                                = LV_FTR_ENUM | 0x001D0000,

    /** Automatic gain mode 'once' - the gain value is calculated and applied once and the feature switches back to 'off' (manual mode). */
    LvGainAuto_Once                               = LV_FTR_ENUM | 0x001D0001,

    /** Continuous automatic gain mode - the AGC is applied continuously. */
    LvGainAuto_Continuous                         = LV_FTR_ENUM | 0x001D0002 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_BlackLevelSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvBlackLevelSelector
{
    /** Apply black level on all channels and taps. */
    LvBlackLevelSelector_All                      = LV_FTR_ENUM | 0x001E0000,

    /** Apply black level on tap1. */
    LvBlackLevelSelector_Tap1                     = LV_FTR_ENUM | 0x001E0001,

    /** Apply black level on tap2. */
    LvBlackLevelSelector_Tap2                     = LV_FTR_ENUM | 0x001E0002
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_BlackLevelAuto feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvBlackLevelAuto
{
    /** Automatic black level mode off - the black level value is controlled 'manually'. */
    LvBlackLevelAuto_Off                          = LV_FTR_ENUM | 0x001F0000,

    /** Automatic black level mode 'once' - the black level value is calculated and applied once and the feature switches back to 'off' (manual mode). */
    LvBlackLevelAuto_Once                         = LV_FTR_ENUM | 0x001F0001,

    /** Continuous automatic black level mode - the automatic black level is applied continuously. */
    LvBlackLevelAuto_Continuous                   = LV_FTR_ENUM | 0x001F0002 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_ColorTransformationSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvColorTransformationSelector
{
    /** RGB to RGB matrix transformation */
    LvColorTransformationSelector_RGBtoRGB        = LV_FTR_ENUM | 0x00210000,

    /** RGB to YUV matrix transformation */
    LvColorTransformationSelector_RGBtoYUV        = LV_FTR_ENUM | 0x00210001 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_ColorTransformationValueSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvColorTransformationValueSelector
{
    /** Selects the gain 00 (RR, red-red) entry of the color transformation matrix. */
    LvColorTransformationValueSelector_Gain00     = LV_FTR_ENUM | 0x00230000,

    /** Selects the gain 01 (RG, red-green) entry of the colortransformation matrix. */
    LvColorTransformationValueSelector_Gain01     = LV_FTR_ENUM | 0x00230001,

    /** Selects the gain 02 (RB, red-blue) entry of the color transformation matrix. */
    LvColorTransformationValueSelector_Gain02     = LV_FTR_ENUM | 0x00230002,

    /** Selects the gain 10 (GR, green-red) entry of the color transformation matrix. */
    LvColorTransformationValueSelector_Gain10     = LV_FTR_ENUM | 0x00230003,

    /** Selects the gain 11 (GG, green-green) entry of the color transformation matrix. */
    LvColorTransformationValueSelector_Gain11     = LV_FTR_ENUM | 0x00230004,

    /** Selects the gain 12 (GB, green-blue) entry of the color transformation matrix. */
    LvColorTransformationValueSelector_Gain12     = LV_FTR_ENUM | 0x00230005,

    /** Selects the gain 20 (BR, blue-red) entry of the color transformation matrix. */
    LvColorTransformationValueSelector_Gain20     = LV_FTR_ENUM | 0x00230006,

    /** Selects the gain 21 (BG, blue-green) entry of the color transformation matrix. */
    LvColorTransformationValueSelector_Gain21     = LV_FTR_ENUM | 0x00230007,

    /** Selects the gain 22 (BB, blue-blue) entry of the color transformation matrix. */
    LvColorTransformationValueSelector_Gain22     = LV_FTR_ENUM | 0x00230008,

    /** Selects the offset 0 (red) entry of the color transformation matrix. */
    LvColorTransformationValueSelector_Offset0    = LV_FTR_ENUM | 0x00230009,

    /** Selects the offset 1 (green) entry of the color transformation matrix. */
    LvColorTransformationValueSelector_Offset1    = LV_FTR_ENUM | 0x0023000a,

    /** Selects the offset 2 (blue) entry of the color transformation matrix. */
    LvColorTransformationValueSelector_Offset2    = LV_FTR_ENUM | 0x0023000b
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvExternalDeviceControlMode feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvExternalDeviceControlMode
{
    /** Selects the custom mode. */
    LvExternalDeviceControlMode_Custom            = LV_FTR_ENUM | 0x00240000 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvExternalADCSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvExternalADCSelector
{
    /** Selects external ADC 1 for configuration. */
    LvExternalADCSelector_ExternalADC1            = LV_FTR_ENUM | 0x00250000,

    /** Selects external ADC 2 for configuration. */
    LvExternalADCSelector_ExternalADC2            = LV_FTR_ENUM | 0x00250001,

    /** Selects external ADC 3 for configuration. */
    LvExternalADCSelector_ExternalADC3            = LV_FTR_ENUM | 0x00250002,

    /** Selects external ADC 4 for configuration. */
    LvExternalADCSelector_ExternalADC4            = LV_FTR_ENUM | 0x00250003 
};

//------------------------------------------------------------------------------

#ifdef SIMPLON_INC_OBSOLETE
/** Enum values for the @ref LvDevice_LvPowerSwitchMode feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvPowerSwitchMode
{
    /** Selects manual mode in which the power switches can be controlled directly */
    LvPowerSwitchMode_Manual                      = LV_FTR_ENUM | 0x00260000,

    /** Select automatic mode in which the power switches are under software control */
    LvPowerSwitchMode_Automatic                   = LV_FTR_ENUM | 0x00260001 
};
#endif

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvPowerSwitchCurrentAction feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvPowerSwitchCurrentAction
{
    /** Reports that all power switches are idle */
    LvPowerSwitchCurrentAction_Idle               = LV_FTR_ENUM | 0x00270000,

    /** Reports that a pulse command is pending */
    LvPowerSwitchCurrentAction_Pulse              = LV_FTR_ENUM | 0x00270001,

    /** Reports that a calibration is pending */
    LvPowerSwitchCurrentAction_Calibrate          = LV_FTR_ENUM | 0x00270002,

    /** Reports that a position adjustment is pending */
    LvPowerSwitchCurrentAction_AdjustPosition     = LV_FTR_ENUM | 0x00270003,

    /** Reports that a power switch drive operation is pending */
    LvPowerSwitchCurrentAction_Drive              = LV_FTR_ENUM | 0x00270004
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvPowerSwitchSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvPowerSwitchSelector
{
    /** Selects power switch 1 for configuration. */
    LvPowerSwitchSelector_PowerSwitch1            = LV_FTR_ENUM | 0x00280000,

    /** Selects power switch 2 for configuration. */
    LvPowerSwitchSelector_PowerSwitch2            = LV_FTR_ENUM | 0x00280001,

    /** Selects power switch 3 for configuration. */
    LvPowerSwitchSelector_PowerSwitch3            = LV_FTR_ENUM | 0x00280002,

    /** Selects power switch 4 for configuration. */
    LvPowerSwitchSelector_PowerSwitch4            = LV_FTR_ENUM | 0x00280003 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvPowerSwitchDrive feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvPowerSwitchDrive
{
    /** Switches the selected power switch off. */
    LvPowerSwitchDrive_Off                        = LV_FTR_ENUM | 0x00290000,

    /** Sswitches the selected power switch to plus polarity. */
    LvPowerSwitchDrive_Plus                       = LV_FTR_ENUM | 0x00290001,

    /** Switches the selected power switch to minus polarity. */
    LvPowerSwitchDrive_Minus                      = LV_FTR_ENUM | 0x00290002 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvPowerSwitchDriveAll feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvPowerSwitchDriveAll
{
    /** Switches the active power switches off */
    LvPowerSwitchDriveAll_Off                     = LV_FTR_ENUM | 0x002A0000,

    /** Switches the active power switches to plus polarity */
    LvPowerSwitchDriveAll_Plus                    = LV_FTR_ENUM | 0x002A0001,

    /** Switches the active power switches to minus polarity */
    LvPowerSwitchDriveAll_Minus                   = LV_FTR_ENUM | 0x002A0002 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvPowerSwitchBoundADC feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvPowerSwitchBoundADC
{
    /** Binds no external ADC to the power switch */
    LvPowerSwitchBoundADC_None                    = LV_FTR_ENUM | 0x002B0000,

    /** Binds external ADC 1 to the power switch */
    LvPowerSwitchBoundADC_ExternalADC1            = LV_FTR_ENUM | 0x002B0001,

    /** Binds external ADC 2 to the power switch */
    LvPowerSwitchBoundADC_ExternalADC2            = LV_FTR_ENUM | 0x002B0002,

    /** Binds external ADC 3 to the power switch */
    LvPowerSwitchBoundADC_ExternalADC3            = LV_FTR_ENUM | 0x002B0003,

    /** Binds external ADC 4 to the power switch */
    LvPowerSwitchBoundADC_ExternalADC4            = LV_FTR_ENUM | 0x002B0004 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvLensControlTargetApproach feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvLensControlTargetApproach
{
    /** Approaches the target position directly, no matter from which side. */
    LvLensControlTargetApproach_Direct            = LV_FTR_ENUM | 0x002C0000,

    /** Approaches the target position always from the plus side to improve accuracy. */
    LvLensControlTargetApproach_FromPlus          = LV_FTR_ENUM | 0x002C0001,

    /** Approaches the target position always from the minus side to improve accuracy. */
    LvLensControlTargetApproach_FromMinus         = LV_FTR_ENUM | 0x002C0002 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LUTSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvLUTSelector
{
    /** Selects the luminance LUT for configuration. */
    LvLUTSelector_Luminance                       = LV_FTR_ENUM | 0x002D0000,

    /** Selects the red LUT for configuration. */
    LvLUTSelector_Red                             = LV_FTR_ENUM | 0x002D0001,

    /** Selects the green LUT for configuration. */
    LvLUTSelector_Green                           = LV_FTR_ENUM | 0x002D0002,

    /** Selects the blue LUT for configuration. */
    LvLUTSelector_Blue                            = LV_FTR_ENUM | 0x002D0003 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_GevDeviceModeCharacterSet feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvGevDeviceModeCharacterSet
{
    /** Indicates that the camera uses the UTF8 character set. */
    LvGevDeviceModeCharacterSet_UTF8              = LV_FTR_ENUM | 0x002E0000 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_GevSupportedOptionSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvGevSupportedOptionSelector
{
    /** Indicates whether the (first) network interface supports auto IP addressing (also known as LLA). */
    LvGevSupportedOptionSelector_IPConfigurationLLA          = LV_FTR_ENUM | 0x002F0000,

    /** Indicates whether the (first) network interface supports DHCP IP addressing. */
    LvGevSupportedOptionSelector_IPConfigurationDHCP         = LV_FTR_ENUM | 0x002F0001,

    /** Indicates whether the (first) network interface supports fixed IP addressing (also known as persistent IP addressing). */
    LvGevSupportedOptionSelector_IPConfigurationPersistentIP = LV_FTR_ENUM | 0x002F0002,

    /** Indicates whether command concatenation is supported by the device. */
    LvGevSupportedOptionSelector_CommandsConcatenation       = LV_FTR_ENUM | 0x002F0003,

    /** Indicates whether write memory scheme is supported by the device. */
    LvGevSupportedOptionSelector_WriteMem                    = LV_FTR_ENUM | 0x002F0004,

    /** Indicates whether packet resendig is supported by the device. */
    LvGevSupportedOptionSelector_PacketResend                = LV_FTR_ENUM | 0x002F0005,

    /** Indicates whether event (message channel) is supported by the device. */
    LvGevSupportedOptionSelector_Event                       = LV_FTR_ENUM | 0x002F0006,

    /** Indicates whether eventdata (message channel) is supported by the device. */
    LvGevSupportedOptionSelector_EventData                   = LV_FTR_ENUM | 0x002F0007,

    /** Indicates whether pending acknowledge is supported by the device. */
    LvGevSupportedOptionSelector_PendingAck                          = LV_FTR_ENUM | 0x002F0008,

    /** Indicates whether action commands are supported by the device. */
    LvGevSupportedOptionSelector_Action                              = LV_FTR_ENUM | 0x002F0009,

    /** Indicates whether primary application switchover is supported by the device. */
    LvGevSupportedOptionSelector_PrimaryApplicationSwitchover        = LV_FTR_ENUM | 0x002F000A,

    /** Indicates whether extended GigE Vision status codes are supported by the device. */
    LvGevSupportedOptionSelector_ExtendedStatusCodes                 = LV_FTR_ENUM | 0x002F000B,

    /** Indicates whether writable discovery acknowledge delay is supported by the device. */
    LvGevSupportedOptionSelector_DiscoveryAckDelayWritable           = LV_FTR_ENUM | 0x002F000C,

    /** Indicates whether discovery acknowledge delay is supported by the device. */
    LvGevSupportedOptionSelector_DiscoveryAckDelay                   = LV_FTR_ENUM | 0x002F000D,

    /** Indicates whether test data is supported by the device. */
    LvGevSupportedOptionSelector_TestData                            = LV_FTR_ENUM | 0x002F000E,

    /** Indicates whether manifest table is supported by the device. */
    LvGevSupportedOptionSelector_ManifestTable                       = LV_FTR_ENUM | 0x002F000F,

    /** Indicates whether the primary application port and IP address features are supported by the device. */
    LvGevSupportedOptionSelector_CCPApplicationSocket                = LV_FTR_ENUM | 0x002F0010,

    /** Indicates whether link speed feature is supported by the device. */
    LvGevSupportedOptionSelector_LinkSpeed                           = LV_FTR_ENUM | 0x002F0011,

    /** Indicates whether heartbeat disabling is supported by the device. */
    LvGevSupportedOptionSelector_HeartbeatDisable                    = LV_FTR_ENUM | 0x002F0012,

    /** Indicates whether serial number feature is supported by the device. */
    LvGevSupportedOptionSelector_SerialNumber                        = LV_FTR_ENUM | 0x002F0013,

    /** Indicates whether user defined name is supported by the device. */
    LvGevSupportedOptionSelector_UserDefinedName                     = LV_FTR_ENUM | 0x002F0014,

    /** Indicates whether the stream channel source port feature is supported by the device. */
    LvGevSupportedOptionSelector_StreamChannelSourceSocket           = LV_FTR_ENUM | 0x002F0015,

    /** Indicates whether the extended chunk data is supported by the device. */
    LvGevSupportedOptionSelector_StreamChannel0ExtendedChunkData     = LV_FTR_ENUM | 0x002F0016,

    /** Indicates whether the unconditional streaming is supported by the device. */
    LvGevSupportedOptionSelector_StreamChannel0UnconditionalStreaming= LV_FTR_ENUM | 0x002F0017,

    /** Indicates whether the reassembly of fragmented IP packets is supported by the device. */
    LvGevSupportedOptionSelector_StreamChannel0IPReassembly          = LV_FTR_ENUM | 0x002F0018,

    /** Indicates whether the big and little endian stream channel is supported by the device. */
    LvGevSupportedOptionSelector_StreamChannel0BigAndLittleEndian    = LV_FTR_ENUM | 0x002F0019,

    /** Indicates whether the message channel source port feature is supported by the device. */
    LvGevSupportedOptionSelector_MessageChannelSourceSocket          = LV_FTR_ENUM | 0x002F001A
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_GevCCP feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvGevCCP
{
    /** Sets the control channel privilege feature to open. */
    LvGevCCP_OpenAccess                           = LV_FTR_ENUM | 0x00300000,

    /** Sets the control channel privilege feature to exclusive. */
    LvGevCCP_ExclusiveAccess                      = LV_FTR_ENUM | 0x00300001,

    /** Sets the control channel privilege feature to control.  */
    LvGevCCP_ControlAccess                        = LV_FTR_ENUM | 0x00300002, 

    /** Sets the control channel privilege feature to control with switchover active. */
    LvGevCCP_ControlAccessSwitchoverActive        = LV_FTR_ENUM | 0x00300003
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_UserSetSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvUserSetSelector
{
    /** Selects the default configuration set. */
    LvUserSetSelector_Default                     = LV_FTR_ENUM | 0x00310000,

    /** Selects user set 1.  */
    LvUserSetSelector_UserSet1                    = LV_FTR_ENUM | 0x00310001,

    /** Selects user set 2.  */
    LvUserSetSelector_UserSet2                    = LV_FTR_ENUM | 0x00310002,

    /** Selects user set 3.  */
    LvUserSetSelector_UserSet3                    = LV_FTR_ENUM | 0x00310003,

    /** Selects user set 4.  */
    LvUserSetSelector_UserSet4                    = LV_FTR_ENUM | 0x00310004 
};

/** Enum values for the @ref LvDevice_UserSetDefaultSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvUserSetDefaultSelector
{
    /** Selects the default user set as the default startup set. */
    LvUserSetDefaultSelector_Default              = LV_FTR_ENUM | 0x00310000,

    /** Selects user set 1 as the default startup set. */
    LvUserSetDefaultSelector_UserSet1             = LV_FTR_ENUM | 0x00310001,

    /** Selects user set 2 as the default startup set. */
    LvUserSetDefaultSelector_UserSet2             = LV_FTR_ENUM | 0x00310002,

    /** Selects user set 3 as the default startup set. */
    LvUserSetDefaultSelector_UserSet3             = LV_FTR_ENUM | 0x00310003,

    /** Selects user set 4 as the default startup set. */
    LvUserSetDefaultSelector_UserSet4             = LV_FTR_ENUM | 0x00310004, 

    /** When resetting/connecting the camera, no user set is applied, the last camera 
     *  configuration remains. During camera boot, the default user set is applied. */
    LvUserSetDefaultSelector_None                 = LV_FTR_ENUM | 0x00310005
};

/** Enum values for the @ref LvDevice_UserSetDefault feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvUserSetDefault
{
    /** Selects the default user set as the default startup set. */
    LvUserSetDefault_Default              = LV_FTR_ENUM | 0x00310000,

    /** Selects user set 1 as the default startup set. */
    LvUserSetDefault_UserSet1             = LV_FTR_ENUM | 0x00310001,

    /** Selects user set 2 as the default startup set. */
    LvUserSetDefault_UserSet2             = LV_FTR_ENUM | 0x00310002,

    /** Selects user set 3 as the default startup set. */
    LvUserSetDefault_UserSet3             = LV_FTR_ENUM | 0x00310003,

    /** Selects user set 4 as the default startup set. */
    LvUserSetDefault_UserSet4             = LV_FTR_ENUM | 0x00310004, 

    /** When resetting/connecting the camera, no user set is applied, the last camera 
     *  configuration remains. During camera boot, the default user set is applied. */
    LvUserSetDefault_None                 = LV_FTR_ENUM | 0x00310005
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_ChunkSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvChunkSelector
{
    /** Selects the X offset chunk for configuration. */
    LvChunkSelector_OffsetX                       = LV_FTR_ENUM | 0x00330000,

    /** Selects the Y offset chunk for configuration. */
    LvChunkSelector_OffsetY                       = LV_FTR_ENUM | 0x00330001,

    /** Selects the width chunk for configuration. */
    LvChunkSelector_Width                         = LV_FTR_ENUM | 0x00330002,

    /** Selects the height chunk for configuration. */
    LvChunkSelector_Height                        = LV_FTR_ENUM | 0x00330003,

    /** Selects the pixel format chunk for configuration. */
    LvChunkSelector_PixelFormat                   = LV_FTR_ENUM | 0x00330004,

    /** Selects the line pitch chunk for configuration. */
    LvChunkSelector_LinePitch                     = LV_FTR_ENUM | 0x00330005,

    /** Selects the frame id chunk for configuration. */
    LvChunkSelector_FrameID                       = LV_FTR_ENUM | 0x00330006,

    /** Selects the time stamp chunk for configuration. */
    LvChunkSelector_Timestamp                     = LV_FTR_ENUM | 0x00330007,

    /** Selects the exposure time chunk for configuration. */
    LvChunkSelector_ExposureTime                  = LV_FTR_ENUM | 0x00330008,

    /** Selects the gain chunk for configuration. */
    LvChunkSelector_Gain                          = LV_FTR_ENUM | 0x00330009,

    /** Selects the line status all chunk for configuration. */
    LvChunkSelector_LineStatusAll                 = LV_FTR_ENUM | 0x0033000A,

    /** Selects the black level chunk for configuration. */
    LvChunkSelector_BlackLevel                    = LV_FTR_ENUM | 0x0033000B, 

    /** Selects the external ADC chunk for configuration. */
    LvChunkSelector_LvExternalADCValue            = LV_FTR_ENUM | 0x0033000C,

    /** Selects the smart application string chunk for configuration. */
    LvChunkSelector_LvSmartAppString              = LV_FTR_ENUM | 0x0033000D,

    /** Selects the smart application signed integer chunk for configuration. */
    LvChunkSelector_LvSmartAppInt                 = LV_FTR_ENUM | 0x0033000E,

    /** Selects the smart application unsigned integer chunk for configuration. */
    LvChunkSelector_LvSmartAppUint                = LV_FTR_ENUM | 0x0033000F,

    /** Selects the smart application raw register chunk for configuration. */
    LvChunkSelector_LvSmartAppRegister            = LV_FTR_ENUM | 0x00330010,

    /** Selects the trigger delayed chunk for configuration. */
    LvChunkSelector_LvTriggerDelayed              = LV_FTR_ENUM | 0x00330011,

    /** Selects the strobe dropped chunk for configuration. */
    LvChunkSelector_LvStrobeDropped               = LV_FTR_ENUM | 0x00330012,

    /** Selects the frame abort chunk for configuration. */
    LvChunkSelector_LvFrameAbort                  = LV_FTR_ENUM | 0x00330013,

    /** Selects the trigger dropped chunk for configuration. */
    LvChunkSelector_LvTriggerDropped              = LV_FTR_ENUM | 0x00330014,

    /** Selects the trigger error chunk for configuration. */
    LvChunkSelector_LvTriggerError                = LV_FTR_ENUM | 0x00330015,

    /** Selects the encoder position chunk for configuration. */
    LvChunkSelector_LvEncoderPosition             = LV_FTR_ENUM | 0x00330016,

    /** Selects the encoder rotation chunk for configuration. */
    LvChunkSelector_LvEncoderRotation             = LV_FTR_ENUM | 0x00330017,

    /** Selects the frame trigger abort chunk for configuration. */
    LvChunkSelector_LvVirtFrmTriggerAbort         = LV_FTR_ENUM | 0x00330018,

    /** Selects the frame trigger drop chunk for configuration. */
    LvChunkSelector_LvVirtFrmTriggerDrop          = LV_FTR_ENUM | 0x00330019,

    /** Selects the frame trigger delay chunk for configuration. */
    LvChunkSelector_LvVirtFrmTriggerDelay         = LV_FTR_ENUM | 0x0033001A,

    /** Selects the first virtual frame chunk for configuration. */
    LvChunkSelector_LvVirtFrmFirst                = LV_FTR_ENUM | 0x0033001B,

   /** Selects the virtual frame length chunk for configuration. */
    LvChunkSelector_LvVirtFrmLength               = LV_FTR_ENUM | 0x0033001C
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_ChunkGainSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvChunkGainSelector
{
    /** Analog gain. */
    LvChunkGainSelector_AnalogAll                 = LV_FTR_ENUM | 0x00340000,

    /** Digital gain. */
    LvChunkGainSelector_DigitalAll                = LV_FTR_ENUM | 0x00340001 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_EventSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvEventSelector
{
    /** This enumeration value selects the log event for configuration. */
    LvEventSelector_LvLog                         = LV_FTR_ENUM | 0x00350000,

    /** This enumeration value selects the smart application log event for configuration. */
    LvEventSelector_LvSmartAppLog                 = LV_FTR_ENUM | 0x00350001, 

    // added in 1.00.006
    /** This enumeration value selects the smart application string event for configuration. */
    LvEventSelector_LvSmartAppString              = LV_FTR_ENUM | 0x00350002,

    /** This enumeration value selects the smart application signed integer event for configuration. */
    LvEventSelector_LvSmartAppInt                 = LV_FTR_ENUM | 0x00350003,

    /** This enumeration value selects the smart application unsigned integer event for configuration. */
    LvEventSelector_LvSmartAppUint                = LV_FTR_ENUM | 0x00350004,

    /** This enumeration value selects the smart application raw register event for configuration. */
    LvEventSelector_LvSmartAppRegister            = LV_FTR_ENUM | 0x00350005,

    /** This enumeration value selects the dropped trigger event for configuration. */
    LvEventSelector_LvTriggerDropped              = LV_FTR_ENUM | 0x00350006
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_EventNotification feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvEventNotification
{
    /** The notifications for the selected event are deactivated. */
    LvEventNotification_Off                       = LV_FTR_ENUM | 0x00360000,

    /** The notifications for the selected event are activated. */
    LvEventNotification_On                        = LV_FTR_ENUM | 0x00360001 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvSystem_TLType feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvTLType
{
    /** GenTL producer supporting mixed technologies. */
    LvTLType_Mixed                                = LV_FTR_ENUM | 0x00370000,

    /** GenTL producer supporting custom technology devices. */
    LvTLType_Custom                               = LV_FTR_ENUM | 0x00370001,

    /** GenTL producer supporting GigE Vision devices. */
    LvTLType_GEV                                  = LV_FTR_ENUM | 0x00370002,

    /** GenTL producer supporting USB3 Vision devices. */
    LvTLType_U3V                                  = LV_FTR_ENUM | 0x00370003,

    /** GenTL producer supporting ICube devices. */
    LvTLType_ICUBE                                = LV_FTR_ENUM | 0x00370004,

    /** GenTL producer supporting Sim devices. */
    LvTLType_SIM                                  = LV_FTR_ENUM | 0x00370005,

    /** GenTL producer supporting iAM devices. */
    LvTLType_iAM                                  = LV_FTR_ENUM | 0x00370006
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvInterface_InterfaceType feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvInterfaceType   
{
    /** Interface supporting a custom technology devices. */
    LvInterfaceType_Custom                        = LV_FTR_ENUM | 0x00380000,

    /** Interface supporting GigE Vision devices. */
    LvInterfaceType_GEV                           = LV_FTR_ENUM | 0x00380001,

    /** Interface supporting USB3 Vision devices. */
    LvInterfaceType_U3V                           = LV_FTR_ENUM | 0x00380002,

    /** Interface supporting ICube devices. */
    LvInterfaceType_ICUBE                         = LV_FTR_ENUM | 0x00380003,

    /** Interface supporting Sim devices. */
    LvInterfaceType_SIM                           = LV_FTR_ENUM | 0x00380004,

    /** Interface supporting iAM devices. */
    LvInterfaceType_iAM                           = LV_FTR_ENUM | 0x00380005
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_DeviceType feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvDeviceType
{
    /** Device based on a custom technology. */
    LvDeviceType_Custom                           = LV_FTR_ENUM | 0x003A0000,

    /** GigE Vision compatible device. */
    LvDeviceType_GEV                              = LV_FTR_ENUM | 0x003A0001, 

    /** USB3 Vision compatible device. */
    LvDeviceType_U3V                              = LV_FTR_ENUM | 0x003A0002,

    /** ICube device. */
    LvDeviceType_ICUBE                            = LV_FTR_ENUM | 0x003A0003, 

    /** Sim device. */
    LvDeviceType_SIM                              = LV_FTR_ENUM | 0x003A0004,

    /** iAM device. */
    LvDeviceType_iAM                              = LV_FTR_ENUM | 0x003A0005
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvCCLinkStatus feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvCCLinkStatus
{
    /** Unknown. */
    LvCCLinkStatus_Unknown                        = LV_FTR_ENUM | 0x003A0100,

    /** Open. */
    LvCCLinkStatus_Open                           = LV_FTR_ENUM | 0x003A0101, 

    /** Close. */
    LvCCLinkStatus_Closed                         = LV_FTR_ENUM | 0x003A0102,

    /** Busy. */
    LvCCLinkStatus_Busy                           = LV_FTR_ENUM | 0x003A0103, 

    /** Connected. */
    LvCCLinkStatus_Connected                      = LV_FTR_ENUM | 0x003A0104, 

    /** ReConnected. */
    LvCCLinkStatus_ReConnected                    = LV_FTR_ENUM | 0x003A0105, 

    /** DisConnected. */
    LvCCLinkStatus_DisConnected                   = LV_FTR_ENUM | 0x003A0106, 

    /** Interrupted. */
    LvCCLinkStatus_Interrupted                    = LV_FTR_ENUM | 0x003A0107, 

    /** Lost. */
    LvCCLinkStatus_Lost                           = LV_FTR_ENUM | 0x003A0108, 

    /** LostControl. */
    LvCCLinkStatus_LostControl                    = LV_FTR_ENUM | 0x003A0109

};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvGevDeviceStreamCaptureMode feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvGevDeviceStreamCaptureMode
{
    /** System default mode, configurable in the ini file. */
    LvGevDeviceStreamCaptureMode_SystemDefault    = LV_FTR_ENUM | 0x003B0000,

    /** Socket mode, the GVSP stream is processed through the socket interface (regular operating system networks stack). */
    LvGevDeviceStreamCaptureMode_Socket           = LV_FTR_ENUM | 0x003B0001,

    /** Filter driver mode, the GVSP stream is processed through the filter driver interface (bypassing operating system network stack). */
    LvGevDeviceStreamCaptureMode_FilterDriver     = LV_FTR_ENUM | 0x003B0002,
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvStream_StreamAcquisitionModeSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvStreamAcquisitionModeSelector
{
    /** Default acquisition mode. */
    LvStreamAcquisitionModeSelector_Default       = LV_FTR_ENUM | 0x003C0000 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvStream_StreamType feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvStreamType
{
    /** Stream belonging to a custom technology device. */
    LvStreamType_Custom                           = LV_FTR_ENUM | 0x003D0000,

    /** Stream belonging to a GigE Vision compatible device. */
    LvStreamType_GEV                              = LV_FTR_ENUM | 0x003D0001, 

    /** Stream belonging to a USB3 Vision compatible device. */
    LvStreamType_U3V                              = LV_FTR_ENUM | 0x003D0002,

    /** Stream belonging to a ICube device. */
    LvStreamType_ICUBE                            = LV_FTR_ENUM | 0x003D0003, 

    /** Stream belonging to a Sim device. */
    LvStreamType_SIM                              = LV_FTR_ENUM | 0x003D0004,

    /** Stream belonging to a iAM device. */
    LvStreamType_iAM                              = LV_FTR_ENUM | 0x003D0005
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvUniProcessMode feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvUniProcessMode
{
    /** HwOnly - The processing is done only in case it is available directly on the
     *  hardware (device). The images will be delivered to the output buffer queue already
     *  processed.
     */
    LvUniProcessMode_HwOnly                       = LV_FTR_ENUM | 0x003E0000,

    /** SwOnly - The processing will be done by software even if the hardware could support
     *  the operation. The software processing is done when the buffer is passed to the output
     *  buffer queue (or later - see LvUniProcessExecution).
     */
    LvUniProcessMode_SwOnly                       = LV_FTR_ENUM | 0x003E0001,

    /** Auto - The processing will be done by hardware and by software will be processed
     *  only the part, which is not possible to do on hardware. Note that if the Bayer decoding
     *  is done by software (this happens when you select an undecoded Bayer pixel format
     *  as the device PixelFormat), the LUT must be then also done by software, even if it
     *  is available in hardware; that's because it must be applied after the Bayer decoding.
     */
    LvUniProcessMode_Auto                         = LV_FTR_ENUM | 0x003E0002,

    /** Off - The automatic processing is not available. You can use the HW features
     *  (LUT etc.) directly. */
    LvUniProcessMode_Off                          = LV_FTR_ENUM | 0x003E0003 
};
 
//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvBayerDecoderAlgorithm and 
 *  @ref LvDevice_LvUniBayerDecoderAlgorithm feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvBayerDecoderAlgorithm
{
    /** Nearest neighbour algorithm - Fastest decoding, giving the worst results, enables also decoding to a monochrome pixel format. */
    LvBayerDecoderAlgorithm_NearestNeighbour        = LV_FTR_ENUM | 0x003F0000,

    /** Bilinear interpolation algorithm - Fast common decoding, enables also decoding to a monochrome pixel format. */
    LvBayerDecoderAlgorithm_BilinearInterpolation   = LV_FTR_ENUM | 0x003F0001,

    /** Bilinear color correction algorithm - Decoding with quick enhacements on edges. */
    LvBayerDecoderAlgorithm_BilinearColorCorrection = LV_FTR_ENUM | 0x003F0002,

    /** Pixel grouping algorithm - Slower decoding, giving very good results. */
    LvBayerDecoderAlgorithm_PixelGrouping           = LV_FTR_ENUM | 0x003F0003,

    /** Variable gradient algorithm - Slowest decoding, giving the best results. */
    LvBayerDecoderAlgorithm_VariableGradient        = LV_FTR_ENUM | 0x003F0004 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvUniBalanceRatioSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvUniBalanceRatioSelector
{
    /** Selects the red channel for configuration. */
    LvUniBalanceRatioSelector_Red                 = LV_FTR_ENUM | 0x00400000,

    /** Selects the green channel for configuration. */
    LvUniBalanceRatioSelector_Green               = LV_FTR_ENUM | 0x00400001,

    /** Selects the blue channel for configuration. */
    LvUniBalanceRatioSelector_Blue                = LV_FTR_ENUM | 0x00400002 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvUniBalanceWhiteAuto feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvUniBalanceWhiteAuto
{
    /** Automatic white balance mode off - the automatic white balance is not applied. */
    LvUniBalanceWhiteAuto_Off                     = LV_FTR_ENUM | 0x00410000,

    /** Automatic white balance mode once - the white balance factors are once adjusted, then switches back to off. */
    LvUniBalanceWhiteAuto_Once                    = LV_FTR_ENUM | 0x00410001 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvUniColorTransformationSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvUniColorTransformationSelector
{
    /** RGB to RGB matrix transformation  - currently the only Color Transformation matrix type. */
    LvUniColorTransformationSelector_RGBtoRGB     = LV_FTR_ENUM | 0x00420000 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvUniColorTransformationValueSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvUniColorTransformationValueSelector        
{
    /** Selects the gain 00 (RR, red-red) entry of the color transformation matrix. */
    LvUniColorTransformationValueSelector_Gain00  = LV_FTR_ENUM | 0x00430000,

    /** Selects the gain 01 (RG, red-green) entry of the colortransformation matrix. */
    LvUniColorTransformationValueSelector_Gain01  = LV_FTR_ENUM | 0x00430001,

    /** Selects the gain 02 (RB, red-blue) entry of the color transformation matrix. */
    LvUniColorTransformationValueSelector_Gain02  = LV_FTR_ENUM | 0x00430002,

    /** Selects the gain 10 (GR, green-red) entry of the color transformation matrix. */
    LvUniColorTransformationValueSelector_Gain10  = LV_FTR_ENUM | 0x00430003,

    /** Selects the gain 11 (GG, green-green) entry of the color transformation matrix. */
    LvUniColorTransformationValueSelector_Gain11  = LV_FTR_ENUM | 0x00430004,

    /** Selects the gain 12 (GB, green-blue) entry of the color transformation matrix. */
    LvUniColorTransformationValueSelector_Gain12  = LV_FTR_ENUM | 0x00430005,

    /** Selects the gain 20 (BR, blue-red) entry of the color transformation matrix. */
    LvUniColorTransformationValueSelector_Gain20  = LV_FTR_ENUM | 0x00430006,

    /** Selects the gain 21 (BG, blue-green) entry of the color transformation matrix. */
    LvUniColorTransformationValueSelector_Gain21  = LV_FTR_ENUM | 0x00430007,

    /** Selects the gain 22 (BB, blue-blue) entry of the color transformation matrix. */
    LvUniColorTransformationValueSelector_Gain22  = LV_FTR_ENUM | 0x00430008 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvRenderer_LvRenderType feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvRenderType
{
    /** Renders the acquired image in full size. */
    LvRenderType_FullSize                         = LV_FTR_ENUM | 0x00440000,

    /** Renders the acquired image to fit into the window. */
    LvRenderType_ScaleToFit                       = LV_FTR_ENUM | 0x00440001,

    /** Renders the acquired image scaled to required size. */
    LvRenderType_ScaleToSize                      = LV_FTR_ENUM | 0x00440002,

    /** Renders the acquired images in tiles. */
    LvRenderType_ScaleToTiles                     = LV_FTR_ENUM | 0x00440003 
};

//------------------------------------------------------------------------------

/** The flags passed as parameter to the functions LvRendererDisplayImage() and LvRendererRepaint().
 *  @ingroup GroupSynview_Enums
 *  @hideinitializer
 */
enum LvRenderFlags
{
    /** Before painting the image, the window background is repainted. This
     *  is done automaticaly whenever the change of the window size is detected,
     *  or display mode is switched. You can also call LvRendererDisplayImage()
     *  with 0 as the buffer handle and this flag just to erase image painting area. */
    LvRenderFlags_RepaintBackground = 0x00000001,
    /** If the buffer LvBuffer_IsIncomplete feature is true, it is not painted. 
     *  The IsIncomplete feature indicates the contents of the buffer is a mixture of new and old image data,
     *  typically it happens when some packets with image data from a GigE camera are lost.
     *  If this flag is set simply the paint or repaint of such buffer is skipped, leaving whatever
     *  was before on the screen.*/
    LvRenderFlags_DontPaintIncomplete    = 0x00000002,
    /** This flag has a meaning ony for the LvRendererCanDisplayImage() function.
     *  If used, this function will not return an error if the window handle was not yet
     *  assigned by the LvRendererSetWindow() function. This can be utilized for checking
     *  if the image is displayable before the display window is actually used. */
    LvRenderFlags_IgnoreInvalidWinHandle = 0x00000004,
};

//------------------------------------------------------------------------------

/** Enum values for the LvSystemFindInterface() and LvInterfaceFindDevice() functions.
 *  @ingroup GroupSynview_Enums
 *  @hideinitializer
 */
enum LvFindBy
{
    /** Can be used in the LvInterfaceFindDevice() for finding the device by its User ID 
     *  (nickname). */
    LvFindBy_UserID          = 0,

    /** Can be used in the LvInterfaceFindDevice() for finding the device by its vendor name. */
    LvFindBy_VendorName      = 1,

    /** Can be used in the LvInterfaceFindDevice() for finding the device by its model name. */
    LvFindBy_ModelName       = 2, 

    /** Can be used in the LvSystemFindInterface() or LvInterfaceFindDevice() for finding the 
     *  interface or device by its Transport Layer type. The search string can be then one of the following:
     *  - "GEV" for GigE Vision,
     *  - "CL" for Camera Link,
     *  - "IIDC" for IIDC 1394,
     *  - "UVC" for USB video class devices,
     *  - "Custom" for not defined ones (for example the New Electronic Technology CorSight streaming device).
     */
    LvFindBy_TLType          = 3,

    /** Can be used in the LvSystemFindInterface() or LvInterfaceFindDevice() for finding the 
     *  interface or device by its display name. */
    LvFindBy_DisplayName     = 4,

    /** Can be used in the LvSystemFindInterface() or LvInterfaceFindDevice() for finding the 
     *  interface or device by its IP address (in case of the interface, it is the default IP address
     *  of the NIC). */
    LvFindBy_GevIPAddress    = 5,

    /** Can be used in the LvInterfaceFindDevice() for finding the device by its model name. */
    LvFindBy_GevMACAddress   = 6,

    /** Can be used in the LvInterfaceFindDevice() for finding the device by its serial number. */
    LvFindBy_SerialNumber    = 7,

    /** Tries to find the string in all available IDs (UserID, VendorName, ModelName...). */
    LvFindBy_Any             = 99

};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvSerialPortBaudRate feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvSerialPortBaudRate
{
    /** Baud rate of 2400 bauds. */
    LvSerialPortBaudRate_Baud2400                 = LV_FTR_ENUM | 0x00450000,

    /** Baud rate of 4800 bauds. */
    LvSerialPortBaudRate_Baud4800                 = LV_FTR_ENUM | 0x00450001,

    /** Baud rate of 9600 bauds. */
    LvSerialPortBaudRate_Baud9600                 = LV_FTR_ENUM | 0x00450002,

    /** Baud rate of 14400 bauds. */
    LvSerialPortBaudRate_Baud14400                = LV_FTR_ENUM | 0x00450003,

    /** Baud rate of 19200 bauds. */
    LvSerialPortBaudRate_Baud19200                = LV_FTR_ENUM | 0x00450004,

    /** Baud rate of 38400 bauds. */
    LvSerialPortBaudRate_Baud38400                = LV_FTR_ENUM | 0x00450005,

    /** Baud rate of 57600 bauds. */
    LvSerialPortBaudRate_Baud57600                = LV_FTR_ENUM | 0x00450006,

    /** Baud rate of 115200 bauds. */
    LvSerialPortBaudRate_Baud115200               = LV_FTR_ENUM | 0x00450007
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvSerialPortParity feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvSerialPortParity
{
    /** Parity method 'none', parity bit not used. */
    LvSerialPortParity_None                       = LV_FTR_ENUM | 0x00460000,

    /** Parity method 'odd', odd number of set bits in each character. */
    LvSerialPortParity_Odd                        = LV_FTR_ENUM | 0x00460001,

    /** Parity method 'even', even number of set bits in each character. */
    LvSerialPortParity_Even                       = LV_FTR_ENUM | 0x00460002
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvSerialPortDataBits feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvSerialPortDataBits
{
    /** 7 data bits per character. */
    LvSerialPortDataBits_DataBits7                = LV_FTR_ENUM | 0x00470000,

    /** 8 data bits per character. */
    LvSerialPortDataBits_DataBits8                = LV_FTR_ENUM | 0x00470001
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvSerialPortStopBits feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvSerialPortStopBits
{
    /** 1 stop bit per character. */
    LvSerialPortStopBits_StopBits1                = LV_FTR_ENUM | 0x00480000,

    /** 1.5 stop bit per character. */
    LvSerialPortStopBits_StopBits1dot5            = LV_FTR_ENUM | 0x00480001,

    /** 2 stop bits per character. */
    LvSerialPortStopBits_StopBits2                = LV_FTR_ENUM | 0x00480002
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvSerialPortCommandStatus feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvSerialPortCommandStatus
{
    /** Last command was successfully transferred. */
    LvSerialPortCommandStatus_Success             = LV_FTR_ENUM | 0x00490000,

    /** Last command ended with timeout (depending on configuration this might be problem or not). */
    LvSerialPortCommandStatus_Timeout             = LV_FTR_ENUM | 0x00490001,

    /** Last command failed: port busy. */
    LvSerialPortCommandStatus_PortBusy            = LV_FTR_ENUM | 0x00490002,

    /** Last command failed: generic communication error. */
    LvSerialPortCommandStatus_CommunicationError  = LV_FTR_ENUM | 0x00490003,

    /** Last command failed: frame error. */
    LvSerialPortCommandStatus_FrameError          = LV_FTR_ENUM | 0x00490004,

    /** Last command failed: parity error. */
    LvSerialPortCommandStatus_ParityError         = LV_FTR_ENUM | 0x00490005,

    /** Last command failed: overflow. */
    LvSerialPortCommandStatus_Overflow            = LV_FTR_ENUM | 0x00490006
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_ChunkLvExternalADCSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvChunkLvExternalADCSelector
{
    /** External ADC 1. */
    LvChunkLvExternalADCSelector_ExternalADC1     = LV_FTR_ENUM | 0x004A0000,

    /** External ADC 2. */
    LvChunkLvExternalADCSelector_ExternalADC2     = LV_FTR_ENUM | 0x004A0001,

    /** External ADC 3. */
    LvChunkLvExternalADCSelector_ExternalADC3     = LV_FTR_ENUM | 0x004A0002,

    /** External ADC 4. */
    LvChunkLvExternalADCSelector_ExternalADC4     = LV_FTR_ENUM | 0x004A0003
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_UserOutputSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvUserOutputSelector
{
    /** Selects user output 1. */
    LvUserOutputSelector_UserOutput1              = LV_FTR_ENUM | 0x004B0000, 

    /** Selects user output 2. */
    LvUserOutputSelector_UserOutput2              = LV_FTR_ENUM | 0x004B0001, 

    /** Selects user output 3. */
    LvUserOutputSelector_UserOutput3              = LV_FTR_ENUM | 0x004B0002, 

    /** Selects user output 4. */
    LvUserOutputSelector_UserOutput4              = LV_FTR_ENUM | 0x004B0003, 

    /** Selects user output 5. */
    LvUserOutputSelector_UserOutput5              = LV_FTR_ENUM | 0x004B0004, 

    /** Selects user output 6. */
    LvUserOutputSelector_UserOutput6              = LV_FTR_ENUM | 0x004B0005, 

    /** Selects user output 7. */
    LvUserOutputSelector_UserOutput7              = LV_FTR_ENUM | 0x004B0006, 

    /** Selects user output 8. */
    LvUserOutputSelector_UserOutput8              = LV_FTR_ENUM | 0x004B0007
};


//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvUniProcessExecution feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvUniProcessExecution
{
    /** The SW image processing is delayed to the time the application asks for the
     *  LvBuffer_UniBase or LvBuffer_ProcessBase pointer or for the LvipImgInfo data.
     *  This enables to the application to skip the processing  in case it is not needed. If this
     *  is queried several times for the same image, the processing is done only once.
     */
    LvUniProcessExecution_OnBufferPtrQuery        = LV_FTR_ENUM | 0x004C0000,

    /** The SW image processing is done always - at the moment the buffer is popped from the
     *  output buffer queue, before delivering it to the application.
     */
    LvUniProcessExecution_OnPopFromQueue          = LV_FTR_ENUM | 0x004C0001,

    /** The SW processing is not done automatically, but must be explicitly done
     *  by the ExecProcess command of the Buffer.
     */
    LvUniProcessExecution_OnExplicitRequest       = LV_FTR_ENUM | 0x004C0002

};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvLensControlCalibrationStatus feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvLensControlCalibrationStatus
{
    /** Current calibration parameters are invalid */
    LvLensControlCalibrationStatus_Invalid        = LV_FTR_ENUM | 0x004D0000, 

    /** Current calibration parameters are valid */
    LvLensControlCalibrationStatus_Valid          = LV_FTR_ENUM | 0x004D0001 
};


//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvLUTMode feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvLUTMode
{
    /** In this mode the LUT is controlled directly. */
    LvLUTMode_Direct                              = LV_FTR_ENUM | 0x004E0000, 

    /** In this mode the LUT is controlled through the higher level
     *  features, such as brightness, contrast, gamma or white balance.
     */
    LvLUTMode_BalanceWhite                        = LV_FTR_ENUM | 0x004E0001
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_BalanceRatioSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvBalanceRatioSelector
{
    /** Balance ratio will be applied to the red channel. */
    LvBalanceRatioSelector_Red                    = LV_FTR_ENUM | 0x004F0000, 

    /** Balance ratio will be applied to the green channel. */
    LvBalanceRatioSelector_Green                  = LV_FTR_ENUM | 0x004F0001, 

    /** Balance ratio will be applied to the blue channel. */
    LvBalanceRatioSelector_Blue                   = LV_FTR_ENUM | 0x004F0002 
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_BalanceWhiteAuto feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvBalanceWhiteAuto
{
    /** Automatic white balance mode off - the automatic white balance is not applied. */
    LvBalanceWhiteAuto_Off                        = LV_FTR_ENUM | 0x00500000,

    /** Automatic white balance mode once - the white balance factors are once adjusted,
     *  then switches the enumeration back to the Off value.
     */
    LvBalanceWhiteAuto_Once                       = LV_FTR_ENUM | 0x00500001,

    /** Automatic white balance mode continuous - the white balance is continuously auto-adjusted. */
    LvBalanceWhiteAuto_Continuous                 = LV_FTR_ENUM | 0x00500002
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_GevDeviceClass feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvGevDeviceClass
{
    /** Indicates that the device is a GigE Vision transmitter. */
    LvGevDeviceClass_Transmitter                  = LV_FTR_ENUM | 0x00510000
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_GevIPConfigurationStatus feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvGevIPConfigurationStatus
{
    /** No IP configuration method was executed or it is not known. */
    LvGevIPConfigurationStatus_None               = LV_FTR_ENUM | 0x00520000,

    /** The current device IP configuration was obtained through persistent IP. */
    LvGevIPConfigurationStatus_PersistentIP       = LV_FTR_ENUM | 0x00520001,

    /** The current device IP configuration was obtained through DHCP. */
    LvGevIPConfigurationStatus_DHCP               = LV_FTR_ENUM | 0x00520002,

    /** The current device IP configuration was obtained through LLA. */
    LvGevIPConfigurationStatus_LLA                = LV_FTR_ENUM | 0x00520003,

    /** The current device IP configuration was obtained through ForceIP. */
    LvGevIPConfigurationStatus_ForceIP            = LV_FTR_ENUM | 0x00520004
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_GevSCPDirection feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvGevSCPDirection
{
    /** Indicates that the stream channel is a transmitter. */
    LvGevSCPDirection_Transmitter                 = LV_FTR_ENUM | 0x00530000
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_DeviceEndianessMechanism feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvDeviceEndianessMechanism
{
    /** Legacy endianess handling mode, intended for GigE Vision remote devices using GenICam schema version 1.0. */
    LvDeviceEndianessMechanism_Legacy             = LV_FTR_ENUM | 0x00540000,

    /** Standard endianess handling mode, intended for GigE Vision remote devices using GenICam schema version 1.1 and newer. */
    LvDeviceEndianessMechanism_Standard           = LV_FTR_ENUM | 0x00540001
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvUniLUTMode feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvUniLUTMode
{
    /** In this mode the LUT is controlled directly. */
    LvUniLUTMode_Direct                           = LV_FTR_ENUM | 0x00550000,
    /** In this mode the LUT is controlled through the higher level features, such as brightness, contrast, gamma or white balance. */
    LvUniLUTMode_Generated                        = LV_FTR_ENUM | 0x00550001
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvUniLUTSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvUniLUTSelector
{
    /** Selects the luminance LUT for configuration. */
    LvUniLUTSelector_Luminance                    = LV_FTR_ENUM | 0x00560000,

    /** Selects the red LUT for configuration. */
    LvUniLUTSelector_Red                          = LV_FTR_ENUM | 0x00560001,

    /** Selects the green LUT for configuration. */
    LvUniLUTSelector_Green                        = LV_FTR_ENUM | 0x00560002,

    /** Selects the blue LUT for configuration. */
    LvUniLUTSelector_Blue                         = LV_FTR_ENUM | 0x00560003
};

//------------------------------------------------------------------------------
// added in 1.00.085
/** Enum values for the @ref LvDevice_LvUniColorTransformationMode feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvUniColorTransformationMode
{
    /** In this mode the Color Transformation matrix can be controlled directly. */
    LvUniColorTransformationMode_Direct           = LV_FTR_ENUM | 0x00570000,

    /** In this mode the Color Transformation matrix is set through the higher level
     *  features, such as the Saturation. */
    LvUniColorTransformationMode_Generated        = LV_FTR_ENUM | 0x00570001
};

//------------------------------------------------------------------------------
// added in 1.00.087
//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvStrobeEnable feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvStrobeEnable
{
    /** Switches the strobe off. */
    LvStrobeEnable_Off                            = LV_FTR_ENUM | 0x00580000,

    /** Switches on all LED clusters of the strobe light. On strobe lights possessing just 
     *  a single LED cluster this cluster is switche on. */
    LvStrobeEnable_AllClusters                    = LV_FTR_ENUM | 0x00580001,

    /** Switches on LED cluster 1 only. The strobe will use just the LEDs in this cluster. */
    LvStrobeEnable_LEDCluster1                    = LV_FTR_ENUM | 0x00580002,

    /** Switches on LED cluster 2 only. The strobe will use just the LEDs in this cluster. */
    LvStrobeEnable_LEDCluster2                    = LV_FTR_ENUM | 0x00580003
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvStrobeDurationMode feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvStrobeDurationMode
{
    /** The maximum strobe duration depends on the maximum frame rate of the camera. 
     *  For very fast sensors the max. strobe duration time, dependent on the specification 
     *  of the LEDs used, cannot be applied in full length, as the recovery time may become 
     *  too short. The calculation is done automatically depending on the LEDs used and the
     *  max. frame rate of the camera in its actual mode of operation. Such calculation also 
     *  includes boosted frame rates e.g. when the camera is in partial scanning and/or binning mode. */
    LvStrobeDurationMode_FrameRateRelated         = LV_FTR_ENUM | 0x00590000,

    /** The maximum strobe duration depends on the maximum allowed ON-time and the minimum required 
     *  recovery time of the LEDs used. The user can program the strobe duration free, according to 
     *  his request, but must be aware himself about the relation of strobe ON-time and recovery time. 
     *  An automatic protection circuit in HW drops a strobe in case the proper relation of ON-time to 
     *  recovery time is not guaranteed. In such case a related error code is returned by the SW 
     *  (frame message or otherwise) */
    LvStrobeDurationMode_Free                     = LV_FTR_ENUM | 0x00590001
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_LvStrobeDropMode feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
 */
enum LvStrobeDropMode
{
    /** Strobe drop mode 'drop' - the strobe is dropped, image is acquired without the strobe. */
    LvStrobeDropMode_DropStrobe                   = LV_FTR_ENUM | 0x005A0000,

    /** Strobe drop mode 'delay' - the frame acquisition is delayed, until the strobe can be issued. */
    LvStrobeDropMode_DelayFrame                   = LV_FTR_ENUM | 0x005A0001
};

//------------------------------------------------------------------------------

/** Enum values for the @ref LvDevice_RegionSelector feature.
 *  @ingroup GroupSynview_EnumEntries
 *  @hideinitializer
*/
enum LvRegionSelector
{
    /** Selects region 0 for configuration. */
    LvRegionSelector_Region0           = LV_FTR_ENUM | 0x005B0000,

    /** Selects region 1 for configuration. */
    LvRegionSelector_Region1           = LV_FTR_ENUM | 0x005B0001,

    /** Selects region 2 for configuration. */
    LvRegionSelector_Region2           = LV_FTR_ENUM | 0x005B0002,

    /** Selects region 3 for configuration. */
    LvRegionSelector_Region3           = LV_FTR_ENUM | 0x005B0003
};

//------------------------------------------------------------------------------
// Internal: Search for *** ADD CODE HERE FOR ENUM VALUES ***
//=== sv.synview.imgproc ======================================================

/** Image attributes. Flags to be used in the Attributes of the @ref LvipImgInfo
 *  structure.
 *  @ingroup GroupImgproc_Defines
 *  @hideinitializer
 */
enum LvipImgAttr
{
    /** Lines in the image buffer are ordered from the bottom line to the top line,
     *  so the image bufer begins with the bottom line.
     */
    LvipImgAttr_BottomUp           = 0x00000002ul,

    /** The line increment is aligned to double word (32 bits).
     *  This is required for example by the Windows Device Independent Bitmap format (DIB, BMP)
     *  This attribute is used only in the LvipInitImgInfo() function (which can be called as a result
     *  of the LvipOption_ReallocateDst attribute).
     */
    LvipImgAttr_DWordAligned       = 0x00000004ul,

    /** The line increment is aligned to quad word (64 bits).
     *  This attribute is used in the LvipInitImgInfo() function (which can be called as a result
     *  of the LvipOption_ReallocateDst attribute).
     */
    LvipImgAttr_QWordAligned       = 0x00000008ul,

    /** The line increment is aligned to SSE words (128 bits).
     *  This attribute is used in the LvipInitImgInfo() function (which can be called as a result
     *  of the LvipOption_ReallocateDst attribute).
     */
    LvipImgAttr_SSEAligned         = 0x00000010ul,

    /** The Image buffer is secured by means of check words and tested for buffer corruption.
     */
    LvipImgAttr_Supervised         = 0x00000020ul,

    /** The @ref LvipImgInfo is not the owner of image data, so the
     *  LvipDeallocateImageData() function will not attempt to deallocate the image data.
     *  This attribute is used when the image data are owned by another @ref LvipImgInfo or belonging
     *  to other code, for example when working directly with the image in the DMA buffer.
     *  Note that LvipDeallocateImageData() may be called from other functions, for example, when you
     *  use the @ref LvipOption_ReallocateDst attribute.
     */
    LvipImgAttr_NotDataOwner       = 0x00000800ul
};


/** Options for image processing functions in the Options parameter.
 *  @ingroup GroupImgproc_Defines
 *  @hideinitializer
 */
enum LvipOption
{
    /** The destination image data can be reallocated if it is needed.
     *  If the function stores a result of the operation to the destination image buffer, it first checks
     *  if the destination @ref LvipImgInfo has appropriate parameters and the buffer(s) allocated.
     *  If not and this attribute is specified, it adapts the parameters of the @ref LvipImgInfo and reallocates the buffer as needed.
     *  If this attribute is not specified, the function returns an error in case of mismatch.
     */
    LvipOption_ReallocateDst       = 0x00001000ul,

    /** The attribute will force conversion of the image to 16-bit mono format, if it is in 9- to 15-bit mono format.
     *  This can be used used when saving mono image to TIFF by the LvipSaveToTiff() function, as many software packages
     *  do not understand mono TIFF if it is in 9- to 15-bit mono format.
     */
    LvipOption_TiffConvertTo16Bit  = 0x00002000ul,

    /** The BMP file will be read to the top-down line layout. This attribute is used in the LvipLoadFromBmp()
     *  and LvipSaveToBmp() functions, as the BMP format can be either in the bottom-up line layout or in the top-down line layout.
     */
    LvipOption_BmpForceTopDown     = 0x00004000ul,

    /** The BMP file will be read to the bottom-up line layout. This attribute is used in the LvipLoadFromBmp()
     *  and LvipSaveToBmp() functions, as the BMP format can be either in the bottom-up line layout or in the top-down line layout.
     */
    LvipOption_BmpForceBottomUp    = 0x00008000ul,

    /** The color JPEG images are stored in RGB format (24-bit). With this option the pixel format will be
     *  reversed to the BGR format in the LvipLoadFromJpg() function.
     */
    LvipOption_JpegConvertToBgr    = 0x00004000ul,
    
    /** The JPEG image data will not be read, only the header will be read. This enables to allocate the image buffer
     *  and then read the full image.
     */
    LvipOption_JpegReadHeaderOnly  = 0x00008000ul,
    
    /** This attribute can be used in the LvipCalcWbFactors() function. If present, it is assumed that the
    *   white balance is calculated from the image to which were applied white balancing factors
    *   passed as input parameters. Thus only a correction is calculated and the existing factors are
    *   modified.
    */
    LvipOption_WbCorrectFactors    = 0x00100000ul
};

/** If the LUT is to be used in @ref GroupImgproc_FunctionsBayer, this attribute is to be OR-ed
 *  to the @ref LvipLutType specification in the LvipAllocateLut() function
 *  Bayer LUT requires bigger size - is needed for the bilinear
 *  interpolation methods and for 10- and 12-bit source formats.
 *  @ingroup GroupImgproc_Defines
 */
#define LVIP_LUT_BAYER    0x00110000ul
/** Bayer16 is a subset of @ref LVIP_LUT_BAYER, suitable for all 10- and 12-bit
    decoding, with the exception of LvipBdBilinearInterpolation() function.
 *  @ingroup GroupImgproc_Defines
*/
#define LVIP_LUT_BAYER_16 0x00010000ul

/** LUT type - to be used in the LvipAllocateLut() function.
 *  @ingroup GroupImgproc_Defines
 */
enum LvipLutType
{
    /** LUT which internally contains 3 LUTs: 8-bit, 10-bit and 12-bit.
     *  All the LUTs are kept synchronized. */
    LvipLutType_Uni           = 0,
    /** 8-bit LUT type, used for images with @ref LvPixelFormat_Mono8. */
    LvipLutType_8Bit          = 8,
    /** 10-bit LUT type, used for images with @ref LvPixelFormat_Mono10. */
    LvipLutType_10Bit         = 10,
    /** 12-bit LUT type, used for images with @ref LvPixelFormat_Mono12 */
    LvipLutType_12Bit         = 12,

    /** LvipLutType_Uni type with the @ref LVIP_LUT_BAYER.  */
    LvipLutType_UniBayer      = 0  | LVIP_LUT_BAYER,
    /** LvipLutType_8Bit type with the @ref LVIP_LUT_BAYER. */
    LvipLutType_8BitBayer     = 8  | LVIP_LUT_BAYER,
    /** LvipLutType_10Bit type with the @ref LVIP_LUT_BAYER. */
    LvipLutType_10BitBayer    = 10 | LVIP_LUT_BAYER,
    /** LvipLutType_12Bit type with the @ref LVIP_LUT_BAYER. */
    LvipLutType_12BitBayer    = 12 | LVIP_LUT_BAYER,

    /** LvipLutType_Uni type with the @ref LVIP_LUT_BAYER_16. */
    LvipLutType_UniBayer16    = 0  | LVIP_LUT_BAYER_16,
    /** LvipLutType_10Bit type with the @ref LVIP_LUT_BAYER_16. */
    LvipLutType_10BitBayer16  = 10 | LVIP_LUT_BAYER_16,
    /** LvipLutType_12Bit type with the @ref LVIP_LUT_BAYER_16. */
    LvipLutType_12BitBayer16  = 12 | LVIP_LUT_BAYER_16
};

/** Color definitions for the Overlay functions.
 *  @ingroup GroupImgproc_Defines
 *  @hideinitializer
 */
enum LvipColor
{
    /** Defines a non-color.
     *  This is useful for the transparent color - specifying
     *  the transparent color as LvipColor_None in LvipSetOverlayTransparentColor()
     *  switches off overlay transparency.
     */
    LvipColor_None = 0xFFFFFFFF
};


/** Text attributes definitions for the Overlay functions.
 *  @ingroup GroupImgproc_Defines
 *  @hideinitializer
 */
enum LvipTextAttr
{
    /** Bold text.
     *  Text attribute for the LvipSetOverlayTextParams() function: Bold text
     */
    LvipTextAttr_Bold           = 0x0001,
    /** Italics text.
     *  Text attribute for the LvipSetOverlayTextParams() function: Italics text
     */
    LvipTextAttr_Italic         = 0x0002,
    /** Underlined text.
     *  Text attribute for the LvipSetOverlayTextParams() function: Underlined text
     */
    LvipTextAttr_Underline      = 0x0004,
    /** Strikeout text.
     *  Text attribute for the LvipSetOverlayTextParams() function: Strikeout text
     */
    LvipTextAttr_Strikeout      = 0x0008,
    /** Text antialiasing off.
     *  Text attribute for the LvipSetOverlayTextParams() function: Text antialiasing will
     *  be switched off - this is useful for text on transparent background, where antialiasing
     *  (like ClearType) can make undesirable effects.
     */
    LvipTextAttr_Nonantialiased = 0x0010,
    /** Text with a 1 pixel shadow.
     *  Text attribute for the LvipSetOverlayTextParams() function: Text with a 1 pixel shadow
     *  at right-bottom direction.
     */
    LvipTextAttr_Shadow         = 0x0100,
    /** Text with a 1 pixel outline.
     *  Text attribute for the LvipSetOverlayTextParams() function: Text with a 1 pixel outline
     *  around the letters. This is useful namely for the text on transparent background - by adding the
     *  outline of different color, then the text is readable even if the background become of the
     *  same color, as the text.
     */
    LvipTextAttr_Outline        = 0xFF00,
    /** Text with a 1 pixel shadow at right-bottom direction.
     *  Text attribute for the LvipSetOverlayTextParams() function: Text with a 1 pixel shadow
     *  at right-bottom direction (equal to @ref LvipTextAttr_Shadow constant). This constant can be
     *  combined with other LvipTextAttr_ShadowXX constants.
     */
    LvipTextAttr_ShadowRB       = 0x0100,
    /** Text with a 1 pixel shadow at right-top direction.
     *  Text attribute for the LvipSetOverlayTextParams() function: Text with a 1 pixel shadow
     *  at right-top direction. This constant can be combined with other LVIP_TEXTATTR_SHADOW_x constants.
     */
    LvipTextAttr_ShadowRT       = 0x0200,
    /** Text with a 1 pixel shadow at left-bottom direction.
     *  Text attribute for the LvipSetOverlayTextParams() function: Text with a 1 pixel shadow
     *  at left-bottom direction. This constant can be combined with other LVIP_TEXTATTR_SHADOW_x constants.
     */
    LvipTextAttr_ShadowLB       = 0x0400,
    /** Text with a 1 pixel shadow at left-top direction.
     *  Text attribute for the LvipSetOverlayTextParams() function: Text with a 1 pixel shadow
     *  at left-top direction. This constant can be combined with other LVIP_TEXTATTR_SHADOW_x constants.
     */
    LvipTextAttr_ShadowLT       = 0x0800,
    /** Text with a 1 pixel shadow at bottom direction.
     *  Text attribute for the LvipSetOverlayTextParams() function: Text with a 1 pixel shadow
     *  at bottom direction. This constant can be combined with other LVIP_TEXTATTR_SHADOW_x constants.
     */
    LvipTextAttr_ShadowB        = 0x1000,
    /** Text with a 1 pixel shadow at top direction.
     *  Text attribute for the LvipSetOverlayTextParams() function: Text with a 1 pixel shadow
     *  at top direction. This constant can be combined with other LVIP_TEXTATTR_SHADOW_x constants.
     */
    LvipTextAttr_ShadowT        = 0x2000,
    /** Text with a 1 pixel shadow at right direction.
     *  Text attribute for the LvipSetOverlayTextParams() function: Text with a 1 pixel shadow
     *  at right direction. This constant can be combined with other LVIP_TEXTATTR_SHADOW_x constants.
     */
    LvipTextAttr_ShadowR        = 0x4000,
    /** Text with a 1 pixel shadow at left direction.
     *  Text attribute for the LvipSetOverlayTextParams() function: Text with a 1 pixel shadow
     *  at left direction. This constant can be combined with other LVIP_TEXTATTR_SHADOW_x constants.
     */
    LvipTextAttr_ShadowL        = 0x8000
};


#endif
