#ifndef SV_SYNVIEW_DISPLAY_H
#define SV_SYNVIEW_DISPLAY_H

#include <sv.synview.defs.h>
#include <sv.synview.enums.h>
#include <sv.synview.status.h>
#include <sv.synview.imgproc.h>

/** Options for image display functions.
 */
enum LvipDisplayOption
{
    /** If the image does not fit to the window client area or the specified rectangle, it is scaled down to fit.
    *   Used in the LvipDisplayImage() and LvipDisplayImageEx() functions.
    */
    LvipDisplayOption_ScaleDown          = 0x00010000ul,

    /** If the image is smaller than the window client area or the specified rectangle, it is scaled up to fit.
    *   Used in the LvipDisplayImage() and LvipDisplayImageEx() functions.
    */
    LvipDisplayOption_ScaleUp            = 0x00020000ul,

    /** If the image is smaller than the window client area or the specified rectangle, it is centered in the area.
    *   Used in the LvipDisplayImage() and LvipDisplayImageEx() functions.
    */
    LvipDisplayOption_Center             = 0x00040000ul,

    /** If the image is scaled down or up, the aspect ratio of the sides is kept.
    *   Used in the LvipDisplayImage() and LvipDisplayImageEx() functions.
    */
    LvipDisplayOption_KeepAspectRatio    = 0x00080000ul,
    
    /** This attribute can be used in the bitmap display functions. When specified, the window background is first wiped with the
    *   background color, then the image is painted.
    */
    LvipDisplayOption_RepaintBackground  = 0x00200000ul,

    /** This attribute can be used in the bitmap display functions. When specified,
    *   a cross hair is shown indicating the middle of the bitmap display.
    */
    LvipDisplayOption_CrossHair          = 0x00400000ul,

    /** internal
    */
    LvipDisplayOption_Persistent         = 0x80000000ul
};

/** The minimum size of the display chache buffer, needed for the LvipDisplayImageEx() function.
*/
#define LVIP_DISPLAY_CACHE_SIZE 1280


// the LV_SIMPLON_SUBSTX11_H is defined in lv.simplon.substx11.h - used only internally
#ifndef LV_SIMPLON_SUBST_X11_H
    #ifndef _WIN32
        #include <X11/Xlib.h>
    #endif
#endif

#ifdef LV_DISPLAY_DLL
    #define LVDSP_DLLENTRY LV_EXTC LV_DLLEXPORT
#else
    #define LVDSP_DLLENTRY LV_EXTC LV_DLLIMPORT
#endif

/** Converts the image to displayable format, if necessary.
 * @ingroup FunctionsDisplay
 * If the pixel format is not compatible
 * for the display, it uses pConvImgInfo for converting the image - mono images to 8-bit
 * color to 24-bit RGB.
 *
 * If pConvImgInfo is NULL, and the image has non-displayable format, the image is not converted.
 * If the pConvImgInfo is not NULL, it is used for conversion. It is the user responsibility
 * to deallocate this image when not needed anymore.
 *
 * Recommended: Create empty @ref LvipImgInfo for pConvImgInfo and in the dwFlags specify the LvipOption_ReallocateDst
 * flag. The function will allocate the appropriate image buffer for the conversion.
 *
 * Next time this function is called it only checks if the buffer is sufficient, if so, it does
 * not reallocate it. Call LvipDeallocateImageData() to deallocate this helper image at the end of your application.
 * - Supported pixel formats: 8-bit to 16-bit mono, 15-bit RGB, 16-bit RGB, 24-bit RGB, 32-bit RGB.
 * 
 * @param pImgInfo pointer to Image info of the image to be converted
 * @param pConvImgInfo If conversion to displayable format has to be done, there is a need to give as param to initialized
 *                     conversion image info structure which will be used for the conversion
 * @param dwFlags Flags - OR-ed combination of @ref ImageFlags
 * @param pConversionNeeded 1 in case the image was converted, 0 in case no conversion was needed.
 * @return LVSTATUS_OK in case of success, or error status in case of failure
 */

LVDSP_DLLENTRY LvStatus LV_STDC LvConvertToDisplayableFormat (LvipImgInfo* pImgInfo, 
                                                              LvipImgInfo* pConvImgInfo, 
                                                              uint32_t dwFlags,
                                                              int32_t* pConversionNeeded);

/** Checks, if the image can be converted to displayable format. This can be used for checking
 *  if this DLL is able to display the image.
 * @ingroup FunctionsDisplay
 * @param SrcPixelFormat Source image pixel format
 * @return 1 in case of the conversion is available, 0 otherwise
 */

LVDSP_DLLENTRY uint32_t LV_STDC LvCanConvertToDisplayableFormat (uint32_t SrcPixelFormat);

/** Displays the image in the window at specified position. It is an extended version of the LvDisplayImage()
 *  function.
 * @ingroup FunctionsDisplay
 * If the pixel format is not compatible
 * for display, it uses pConvImgInfo for converting the image - mono images to 8-bit
 * color to 24-bit RGB.
 *
 * If pConvImgInfo is NULL, and the image has non-displayable format, the image is not converted and displayed.
 * If the pConvImgInfo is not NULL, it is used for conversion. It is the user responsibility
 * to deallocate this image when not needed anymore.
 *
 * Recommended: Create empty @ref LvipImgInfo for pConvImgInfo and in the dwFlags specify the LvipOption_ReallocateDst
 * flag. The function will allocate the appropriate image buffer for the conversion.
 *
 * Next time this function is called it only checks if the buffer is sufficient, if so, it does
 * not reallocate it. Call LvipDeallocateImageData() to deallocate this helper image at the end of your application.
 * - Supported pixel formats: 8-bit to 16-bit mono, 15-bit RGB, 16-bit RGB, 24-bit RGB, 32-bit RGB.
 * 
 * @param pImgInfo pointer to Image info of the image to be painted
 * @param hWnd    Handle to a window where the image has to be painted
 * @param iXOffset X-position of the top left corner, relatively to the top left corner of the window client area. 
 *                 Can be also negative.
 * @param iYOffset Y-position of the top left corner, relatively to the top left corner of the window client area. 
 *                 Can be also negative.
 * @param dwWidth   Width of the display area. If set to 0, the whole window client width is used.
 * @param dwHeight  Height of the display area. If set to 0, the whole window client height is used.
 * @param pConvImgInfo If conversion to displayable format has to be done, there is a need to give as param to initialized
 *                     conversion image info structure which will be used for the conversion
 * @param pCacheParams Buffer for image parameters. This buffer must have a minimum size of LVIP_DISPLAY_CACHE_SIZE.
 *                     At the first use the buffer must be cleared to 0. This function stores to the buffer calculated
 *                     parameters, for example the palette, and if the next displayed image has the same parameters
 *                     (width, height, pixel format), the calculated parameters are reused from this buffer, which is faster
 *                     than calculating them for each image again and again. This also means that you should not change/reallocate
 *                     this buffer during the repetitive display of images (live image).
 * @param dwCacheParamsSize Real size of the buffer passed as pCacheParams.
 * @param dwFlags Flags - OR-ed combination of @ref ImageFlags, note namely the flags:
                  @ref LvipDisplayOption_ScaleDown, @ref LvipDisplayOption_ScaleUp, 
                  @ref LvipDisplayOption_Center, @ref LvipDisplayOption_KeepAspectRatio 
 * @return LVSTATUS_OK in case of success, or error status in case of failure
 */

#ifdef _WIN32

    LVDSP_DLLENTRY LvStatus LV_STDC LvDisplayImageW (LvipImgInfo* pImgInfo, 
                                                      HWND hWnd, 
                                                      int32_t iXOffset,
                                                      int32_t iYOffset,
                                                      uint32_t dwWidth,
                                                      uint32_t dwHeight,
                                                      LvipImgInfo* pConvImgInfo, 
                                                      void* pCacheParams,
                                                      uint32_t dwCacheParamsSize,
                                                      uint32_t dwFlags);

#else /* __UNIX__ */

    //LVDSP_DLLENTRY LvStatus LV_STDC LvDisplayImageL (LvipImgInfo* pImgInfo, 
    //                                                  Display* hDisplay, 
    //                                                  Window *hWnd, 
    //                                                  _XGC* hGc,
    //                                                  int32_t iXOffset,
    //                                                  int32_t iYOffset,
    //                                                  uint32_t dwWidth,
    //                                                  uint32_t dwHeight,
    //                                                  LvipImgInfo* pConvImgInfo, 
    //                                                  void* pCacheParams,
    //                                                  uint32_t dwCacheParamsSize,
    //                                                  uint32_t dwFlags);

#endif


//-----------------------------------------------------------------------------
// New API for Windows and Linux

typedef void* LvHDisplay;

#ifdef _WIN32

    LVDSP_DLLENTRY LvStatus LV_STDC LvDisplayOpen (HWND hWnd, 
                                                   uint32_t dwFlags,
                                                   LvHDisplay* phDisplay);
#else

    LVDSP_DLLENTRY LvStatus LV_STDC LvDisplayOpen (Display *display, 
                                                   Window window,
                                                   uint32_t dwFlags,
                                                   LvHDisplay* phDisplay);
#endif

LVDSP_DLLENTRY LvStatus LV_STDC LvDisplayImage (LvHDisplay hDisplay, 
                                                 LvipImgInfo* pImgInfo, 
                                                 int32_t iXOffset,
                                                 int32_t iYOffset,
                                                 uint32_t dwWidth,
                                                 uint32_t dwHeight,
                                                 uint32_t dwFlags);

LVDSP_DLLENTRY LvStatus LV_STDC LvDisplayClose (LvHDisplay* phDisplay);



// ---- typdefes for dynamic load ------------------------------------------------------

typedef LvStatus (LV_STDC* PLvConvertToDisplayableFormat)(LvipImgInfo* pImgInfo, 
                                                           LvipImgInfo* pConvImgInfo, 
                                                           uint32_t dwFlags,
                                                           int32_t* pConversionNeeded);

typedef LvStatus (LV_STDC* PLvCanConvertToDisplayableFormat) (uint32_t SrcPixelFormat);

#ifdef _WIN32

    typedef LvStatus (LV_STDC* PLvDisplayImageW)(LvipImgInfo* pImgInfo, 
                                                  HWND hWnd, 
                                                  int32_t iXOffset,
                                                  int32_t iYOffset,
                                                  uint32_t dwWidth,
                                                  uint32_t dwHeight,
                                                  LvipImgInfo* pConvImgInfo, 
                                                  void* pCacheParams,
                                                  uint32_t dwCacheParamsSize,
                                                  uint32_t dwFlags);



    typedef LvStatus (LV_STDC* PLvDisplayOpen) (HWND hWnd, 
                                                uint32_t dwFlags,
                                                LvHDisplay* phDisplay);
#else

    //typedef LvStatus (LV_STDC* PLvDisplayImageL) (LvipImgInfo* pImgInfo, 
    //                                               Display* hDisplay, 
    //                                               Window *hWnd, 
    //                                               _XGC* hGc,
    //                                               int32_t iXOffset,
    //                                               int32_t iYOffset,
    //                                               uint32_t dwWidth,
    //                                               uint32_t dwHeight,
    //                                               LvipImgInfo* pConvImgInfo, 
    //                                               void* pCacheParams,
    //                                               uint32_t dwCacheParamsSize,
    //                                               uint32_t dwFlags);

    typedef LvStatus (LV_STDC* PLvDisplayOpen) (Display *display, 
                                                Window window,
                                                uint32_t dwFlags,
                                                LvHDisplay* phDisplay);
#endif

typedef LvStatus (LV_STDC* PLvDisplayImage) (LvHDisplay hDisplay, 
                                              LvipImgInfo* pImgInfo, 
                                              int32_t iXOffset,
                                              int32_t iYOffset,
                                              uint32_t dwWidth,
                                              uint32_t dwHeight,
                                              uint32_t dwFlags);

typedef LvStatus (LV_STDC* PLvDisplayClose) (LvHDisplay* phDisplay);

#endif
