#ifndef SV_SYNVIEW_CLASS_H
#define SV_SYNVIEW_CLASS_H

// This is a C++ class wrapper around the SynView plain C interface.
// If you want to use it, you should add to your project also the 
// sv.synview.class.cpp unit.

#include <sv.synview.defs.h>
#include <sv.synview.enums.h>
#include <sv.synview.imgproc.h>


#define LV_USE_STDLIB
//#define LV_USE_STDEXCEPTION
//#define USE_STATIC_MSG 

#ifdef LV_USE_STDLIB
    #include <string>
#endif
#ifdef LV_USE_STDEXCEPTION
    #include <exception>
#endif

#ifndef LV_USE_STDEXCEPTION

/** Undefine LV_USE_STDLIB in case you do not want to use the standard
 *  template library. If LV_USE_STDLIB is defined, the functions returning
 *  strings are available also overloaded having a std::string& parameter
 *  for returning the string.
 *
 *  Define LV_USE_STDEXCEPTION in case you want to use the exception class
 *  from the standard library instead of LvException.
 *
 *  Call LvLibrary::SetThrowErrorEnable(true) in case you want to use the C++ exceptions
 *  of the @ref LvException type to be thrown when the function returns a status not equal to
 *  LVSTATUS_OK. Then you can use the error handling in the form shown in the example below:
 *
 *       try
 *       {
 *           m_pDevice->AcquisitionStart();
 *           // ... and more SynView API calls, without checking the return value
 *       }
 *       catch (LvException e)
 *       {
 *           DisplayErrorMsg(e.Message(), e.Number());
 *           return;
 *       }
 *  
 */
    class LvException
    {
    public: 
        /** constructor  @ingroup GroupSynview_LvLibrary_Cpp */
        LvException(const char* pMessage, LvStatus Number) throw();
        #ifndef USE_STATIC_MSG
        /** constructor  @ingroup GroupSynview_LvLibrary_Cpp */
        LvException(const LvException& e) throw();
        #endif
        ~LvException() throw();
        /** error message  @ingroup GroupSynview_LvLibrary_Cpp */
        const char* Message() throw();
        /** error status  @ingroup GroupSynview_LvLibrary_Cpp */
        LvStatus Number() throw();
    private:
        #ifdef USE_STATIC_MSG
            char m_szMessage[512];
        #else
            char* m_pMessage;
        #endif
        LvStatus m_Number;
    };

#endif

//------------------------------------------------------------------------------

/** @defgroup GroupSynview_Cpp SynView C++ API functions
 *  @ingroup GroupSynview
 */

/** @defgroup GroupSynview_LvLibrary_Cpp  LvLibrary methods
 *  @ingroup GroupSynview_Cpp
 */

/** @defgroup GroupSynview_LvSystem_Cpp  LvSystem methods
 *  @ingroup GroupSynview_Cpp
 */

/** @defgroup GroupSynview_LvInterface_Cpp  LvInterface methods
 *  @ingroup GroupSynview_Cpp
 */

/** @defgroup GroupSynview_LvDevice_Cpp  LvDevice methods
 *  @ingroup GroupSynview_Cpp
 */

/** @defgroup GroupSynview_FirmwareFunctions_Cpp  LvDevice firmware update methods
 *  @ingroup GroupSynview_LvDevice_Cpp
 */

/** @defgroup GroupSynview_LvStream_Cpp  LvStream methods
 *  @ingroup GroupSynview_Cpp
 */

/** @defgroup GroupSynview_LvBuffer_Cpp  LvBuffer methods
 *  @ingroup GroupSynview_Cpp
 */

/** @defgroup GroupSynview_LvEvent_CPP  LvEvent methods
 *  @ingroup GroupSynview_Cpp
 */

/** @defgroup GroupSynview_LvRenderer_Cpp  LvRenderer methods
 *  @ingroup GroupSynview_Cpp
 */

/** @defgroup GroupSynview_LvModule_Cpp  LvModule methods
 *  @ingroup GroupSynview_Cpp
 */


extern bool g_bThrowErrorEnable;

//##############################################################################
/** The LvLibrary class has all its members static and it is not possible to
 *  create the instance of this class. You can consider its methods as global functions.
 */

class LvLibrary
{
public: 
    //------------------------------------------------------------------------------
    /** Returns SynView version.
     *
     *  @return The returned doubleword contains the build version
     *          in the low word and the high word is the major version in the
     *          upper byte and subversion in the lower byte.
     *  For example:
     *  @code
     *  uint32_t Version = LvLibrary::GetVersion();
     *  printf("SynView %d.%02d.%03d",
     *          ((Version >> 24) & 0xFF),
     *          ((Version >> 16) & 0xFF),
     *          (Version & 0xFFFF));
     *  @endcode
     *  @ingroup GroupSynview_LvLibrary_Cpp
     */
    static uint32_t GetVersion();

    //------------------------------------------------------------------------------
    /** Opens the SynView library. This must be done before you can use any other
     *  SynView function (with the exception of LvLibrary::GetVersion() and LvLibrary::GetErrorMessage()).
     *  If you are using SynView in Windows DLL, avoid calling this in Windows DllMain()
     *  function - for proper functionality this function must be called when the
     *  application or DLL is already fully initialized and there are no restrictions
     *  about synchronization (DllMain has such restrictions).
     *  If you call this function multiple times, you must balance it by the same
     *  number of the LvLibrary::CloseLibrary() calls. Only the first call will actually do
     *  the initialization. IMPORTANT: The library must not be opened again
     *  once it was already uninitialized.
     *
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvLibrary_Cpp
     */
    static LvStatus OpenLibrary();

    //------------------------------------------------------------------------------
    /** Closes the SynView library. This must be performed before you exit your
     *  application. Be sure to close first all dependent modules (System).
     *  If you are using SynView in a Windows DLL, avoid calling this in Windows DllMain()
     *  function - for proper functionality this function must be called when the
     *  application or DLL is still fully functional, which is not the case of
     *  PROCESS_DETACH in the DllMain(). If you have called LvLibrary::OpenLibrary() multiple
     *  times, you must balance it by the same number of calls of this function. Only the last call
     *  actually does the uninitialization. IMPORTANT: The library must not be opened again
     *  once it was already uninitialized.
     *
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvLibrary_Cpp
     */
    static LvStatus CloseLibrary();

    //------------------------------------------------------------------------------
    /** Returns a short description of the error. Note that only some of the errors are
     *  suitable for direct display to the user, many error values indicate states
     *  which are understandable to the programmer, but may not be understandable to
     *  the end user.
     *
     *  @param  Error     The error code (the return value of most SynView functions).
     *  @param  pMessage  Pointer to the text buffer.
     *  @param  Size      Size of the buffer.
     *
     *  @sa @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvLibrary_Cpp
     */
    static void GetErrorMessage (LvStatus Error,
                          char* pMessage,
                          size_t Size);
    #ifdef LV_USE_STDLIB
        //-----------------------------------------------------------------------------
        /** Returns a short description of the error. Note that only some of the errors are
         *  suitable for direct display to the user, many error values indicate states
         *  which are understandable to the programmer, but may not be understandable to
         *  the end user.
         *
         *  @param  Error     The error code (the return value of most SynView functions).
         *  @return Error message in std::string.
         *
         *  @sa @ref GroupSynview_LvStatus.
         *  @ingroup GroupSynview_LvLibrary_Cpp
         */
        static std::string GetErrorMessage(LvStatus Error);
    #endif

    //------------------------------------------------------------------------------
    /** Returns more detailed description of the last error, which happened in the thread
     *  from which this function was called. As the info is recorded inside SynView for each 
     *  error, the desription provides more detailed info, including the name of the function, in which 
     *  the error happened, and possibly more diagnostic info. The difference to LvLibrary::GetErrorMessage() 
     *  is that LvLibrary::GetErrorMessage() returns a static string from a numbered table of errors
     *  while this function returns additionally info recorded at the time the error happened.
     *  If a function returns LVSTATUS_OK, it does not reset this error message (for speed reasons)
     *  so the correct approach is to get the error number as the function return value and if this
     *  return value is not LVSTATUS_OK, then you can get more info about the error using this 
     *  function. be sure to call it from the same thread.
     *  @param  pMessage  Pointer to the text buffer.
     *  @param  Size      Size of the buffer.
     *
     *  @sa @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvLibrary_Cpp
     */
    static void GetLastErrorMessage (char* pMessage, 
                                     size_t Size);

    #ifdef LV_USE_STDLIB
        //-----------------------------------------------------------------------------
        /** Returns more detailed description of the last error, which happened in the thread
         *  from which this function was called. As the info is recorded inside SynView for each 
         *  error, the desription provides more detailed info, including the name of the function, in which 
         *  the error happened, and possibly more diagnostic info. The difference to LvLibrary::GetErrorMessage() 
         *  is that LvLibrary::GetErrorMessage() returns a static string from a numbered table of errors
         *  while this function returns additionally info recorded at the time the error happened.
         *  If a function returns LVSTATUS_OK, it does not reset this error message (for speed reasons)
         *  so the correct approach is to get the error number as the function return value and if this
         *  return value is not LVSTATUS_OK, then you can get more info about the error using this 
         *  function. be sure to call it from the same thread.
         *
         *  @return Error message in std::string.
         *
         *  @sa @ref GroupSynview_LvStatus.
         *  @ingroup GroupSynview_LvLibrary_Cpp
         */
        static std::string GetLastErrorMessage();
    #endif

    //------------------------------------------------------------------------------
    /** Adds a line to the sv.synview.log. The SynView log is a tool for New Electronic
    *  Technology technical support, but in some cases may be useful to put to the log
    *  additional info from your code.
    *
    *  @param   pLogMessage  Pointer to the null teminated string with the message.
    *  @ingroup GroupSynview_LvLibrary_Cpp
    */
    static void Log(const char* pLogMessage);

    //------------------------------------------------------------------------------
    /** Adds a line to the sv.synview.log. The SynView log is a tool for New Electronic
         *  Technology technical support, but in some cases may be useful to put to the log
         *  additional info from your code.
         *
         *  @param   pszFormat  Pointer to the null teminated string with the format string.
         *  @param   ...        variable number of arguments.
         *  @ingroup GroupSynview_LvLibrary_Cpp
         */
    static void Logf(const char* pszFormat, ...);

    //------------------------------------------------------------------------------
    /** Gets a general info in form of a 32-bit integer value.
     *
     *  @param   Info      One of the @ref LvLibInfo values.
     *  @param   pInfo     The value is returned in this parameter.
     *  @param   Param     Additional parameter, required by some types of info.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvLibrary_Cpp
     */
    static LvStatus GetLibInfo(LvEnum Info,
                               int32_t* pInfo,
                               int32_t Param = 0);

    //------------------------------------------------------------------------------
    /** Gets a general info in form of a string value.
     *
     *  @param   Info      One of the @ref LvLibInfo values.
     *  @param   pInfoStr  The string value is returned in this parameter.
     *  @param   Size      Size of the buffer (to which pInfoStr points).
     *  @param   Param     Additional parameter, required by some types of info.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvLibrary_Cpp
     */
    static LvStatus GetLibInfoStr (LvEnum Info,
                                   char* pInfoStr,
                                   size_t Size,
                                   int32_t Param = 0);

    //------------------------------------------------------------------------------
    /** Gets a buffer size needed for a general info in form of a string value.
     *
     *  @param   Info      One of the @ref LvLibInfo values.
     *  @param   pSize     Size of the buffer is returned in this parameter.
     *  @param   Param     Additional parameter, required by some types of info.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvLibrary_Cpp
     */
    static LvStatus GetLibInfoStrSize (LvEnum Info,
                                       size_t* pSize,
                                       int32_t Param = 0);
    #ifdef LV_USE_STDLIB
        //------------------------------------------------------------------------------
        /** Gets a general info in form of a std::string value.
         *
         *  @param   Info      One of the @ref LvLibInfo values.
         *  @param   sInfo     The string value is returned in this parameter.
         *  @param   Param     Additional parameter, required by some types of info.
         *  @return  Returns the @ref LvStatus value indicating the success of the
         *           requested operation. See @ref GroupSynview_LvStatus.
         *  @ingroup GroupSynview_LvLibrary_Cpp
         */
        static LvStatus GetLibInfoStr (LvEnum Info, std::string& sInfo, int32_t Param = 0);
    #endif

    //------------------------------------------------------------------------------
    /** Updates the list of systems available. This function must be called before iterating
     *  through the systems by the LvLibrary::GetNumberOfSystems() and LvLibrary::GetSystemId() functions.
     *  The systems are physically represented by GenTL libraries available in the
     *  operating systems, this call searches for them in standard locations. See also
     *  the description of the sv.synview.ini file in the SynView User's Guide.
     *  Note that this function is seldom needed, most applications will work with the
     *  default system (see LvSystem::Open() for details).
     *
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvLibrary_Cpp
     */
    static LvStatus UpdateSystemList();

    //------------------------------------------------------------------------------
    /** Returns the number of systems found after the LvLibrary::UpdateSystemList() call.
     *  Typical use of this function is in iterating systems using the LvLibrary::GetSystemId()
     *  function.
     *
     *  @param   pNumberOfSystems  The number of systems found.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvLibrary_Cpp
     */
    static LvStatus GetNumberOfSystems(uint32_t* pNumberOfSystems);

    //------------------------------------------------------------------------------
    /** Returns the string ID of the system at given index. This ID is used in the
     *  LvSystem::Open() function for opening the system.
     *
     *  @param   Index      Zero-based index of the system, a value >= 0 and < number of systems,
     *                      returned by the LvLibrary::GetNumberOfSystems() function.
     *  @param   pSystemId  Pointer to a string buffer, where the system ID will be placed.
     *  @param   Size       Size of the buffer.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvLibrary_Cpp
     */
    static LvStatus GetSystemId (uint32_t Index,
                                   char* pSystemId,
                                   size_t Size);

    //------------------------------------------------------------------------------
    /** Returns the size of the string buffer needed to hold the system ID string,
     *  including the terminating zero character.
     *
     *  @param   Index      Zero-based index of the system, a value >= 0 and < number of systems,
     *                      returned by the LvLibrary::GetNumberOfSystems() function.
     *  @param   pSize      Size of the buffer is returned in this parameter.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvLibrary_Cpp
     */
    static LvStatus GetSystemIdSize (uint32_t Index,
                                     size_t* pSize);

    #ifdef LV_USE_STDLIB
        //------------------------------------------------------------------------------
        /** Returns the string ID of the system at given index. This ID is used in the
         *  LvSystem::Open() function for opening the system.
         *
         *  @param   Index      Zero-based index of the system, a value >= 0 and < number of systems,
         *                      returned by the LvLibrary::GetNumberOfSystems() function.
         *  @param   sSystemId  String, where the system ID will be placed.
         *  @return  Returns the @ref LvStatus value indicating the success of the
         *           requested operation. See @ref GroupSynview_LvStatus.
         *  @ingroup GroupSynview_LvLibrary_Cpp
         */
        static LvStatus GetSystemId (uint32_t Index, std::string& sSystemId);
    #endif

    //------------------------------------------------------------------------------
    /** Enables/disables the conversion of @ref LvStatus return values (not equal to @ref LVSTATUS_OK)
     *  to C++ exceptions of the @ref LvException type.
     *  @param   bEnable    Enable/disable the exception throwing.
     *  @ingroup GroupSynview_LvLibrary_Cpp
     */
    static void SetThrowErrorEnable(bool bEnable);

private:
    LvLibrary();
    ~LvLibrary();
};

//##############################################################################
/** The base class for all modules. It provides methods for manipulating the
 *  features, if the module provides any.
 *  This class cannot be instantiated, it only serves as a base class.
 */
class LvModule
{
public:
    /** Returns a number of features for specified group. This is useful for building
     *  a list of all available features (like the tree in lv.explorer).
     *
     *  @param   FtrGroup      One of the @ref LvFtrGroup.
     *  @param   pNumFeatures  The number of features is returned in this parameter.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetNumFeatures (LvFtrGroup FtrGroup, uint32_t* pNumFeatures);
  
    //------------------------------------------------------------------------------
    /** Returns the feature ID at specified position. Can be used to
     *  iterate all the features in a list.
     *
     *  @param   FtrGroup     One of the @ref LvFtrGroup.
     *  @param   Index        Zero based index of the feature in the list.
     *  @param   pFeature     Feature ID is returned in this parameter.
     *  @param   pLevel       Feature Level expressing its position in the tree is returned
     *                        in this parameter. The base level has value 1.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetFeatureAt (LvFtrGroup FtrGroup,
                           uint32_t Index,
                           LvFeature* pFeature,
                           uint32_t* pLevel = NULL);
  
    //------------------------------------------------------------------------------
    /** Returns a feature ID based on the feature name. This function is a substantial
     *  function for the generic approach to the feature - by this function you can
     *  get the ID of any existing feature, that means also for those, for which
     *  a SynView constant is not defined. Be sure to check the success of this function -
     *  if the feature is not mandatory, it may not exist.
     *
     *  @param   FtrGroup     One of the @ref LvFtrGroup.
     *  @param   pName        Name of the feature.
     *  @param   pFeature     Feature ID is returned in this parameter.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetFeatureByName (LvFtrGroup FtrGroup,
                               const char* pName,
                               LvFeature* pFeature);
  
    //------------------------------------------------------------------------------
    /** A helper function, allowing simply to determine, if a feature is implemented.
     *  It is a wrapper around the LvModule::GetAccess() function.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @return  If the feature is implemented, returns true, otherwise false.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    bool IsImplemented (LvFeature Feature);
  
    //------------------------------------------------------------------------------
    /** A helper function, allowing simply to determine, if a feature is implemented.
     *  It is a wrapper around the LvModule::GetAccess() and LvModule::GetFeatureByName() 
     *  functions.
     *
     *  @param   FeatureGroup One of the @ref LvFtrGroup.
     *  @param   pName        Name of the feature.
     *  @return  If the feature is implemented, returns true, otherwise false.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    bool IsImplementedByName (LvEnum FeatureGroup,
                              const char* pName);

    //------------------------------------------------------------------------------
    /** A helper function, allowing simply to determine, if a feature is available.
     *  It is a wrapper around the LvModule::GetAccess() function.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @return  If the feature is available, returns true, otherwise false.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    bool IsAvailable (LvFeature Feature);
  
    //------------------------------------------------------------------------------
    /** A helper function, allowing simply to determine, if a feature is available.
     *  It is a wrapper around the LvModule::GetAccess() and LvModule::GetFeatureByName() 
     *  functions.
     *
     *  @param   FeatureGroup One of the @ref LvFtrGroup.
     *  @param   pName        Name of the feature.
     *  @return  If the feature is available, returns true, otherwise false.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    bool IsAvailableByName (LvEnum FeatureGroup,
                            const char* pName);

    //------------------------------------------------------------------------------
    /** A helper function, allowing simply to determine, if a feature is readable.
     *  It is a wrapper around the LvModule::GetAccess() function.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @return  If the feature is readable, returns true, otherwise false.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    bool IsReadable (LvFeature Feature);
  
    //------------------------------------------------------------------------------
    /** A helper function, allowing simply to determine, if a feature is writable.
     *  It is a wrapper areound the LvModule::GetAccess() function.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @return  If the feature is writable, returns true, otherwise false.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    bool IsWritable (LvFeature Feature);

    //------------------------------------------------------------------------------
    /** A helper function, allowing simply to determine, if an enum entry of an enum
     *  feature is available.
     *
     *  @param   Feature    The feature ID - use a symbolic constant (one of the
     *                      @ref GroupSynview_Features) or an ID obtained by the
     *                      LvModule::GetFeatureByName() function.
     *  @param   EnumEntry  The SynView constant for the enum entry.
     *  @return  If the enum entry is available, returns true, otherwise false.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    bool IsAvailableEnumEntry (LvFeature Feature,
                               LvEnum EnumEntry);
  
    //------------------------------------------------------------------------------
    /** A helper function, allowing simply to determine, if an enum entry of an enum
     *  feature is implemented.
     *
     *  @param   Feature    The feature ID - use a symbolic constant (one of the
     *                      @ref GroupSynview_Features) or an ID obtained by the
     *                      LvModule::GetFeatureByName() function.
     *  @param   EnumEntry  The SynView constant for the enum entry.
     *  @return  If the enum entry is implemented, returns true, otherwise false.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    bool IsImplementedEnumEntry (LvFeature Feature,
                                 LvEnum EnumEntry);

    //------------------------------------------------------------------------------
    /** Returns the feature type, GUI representation and group.
     *
     *  @param   Feature    The feature ID - use a symbolic constant (one of the
     *                      @ref GroupSynview_Features) or an ID obtained by the
     *                      LvModule::GetFeatureByName() function.
     *  @param   pFtrType   The feature type is returned in this parameter.
     *                      The returned value is one of the @ref LvFtrType.
     *                      Can be NULL.
     *  @param   pFtrGui    The feature GUI representation is returned in this parameter.
     *                      The returned value is one of the @ref LvFtrGui.
     *                      Can be NULL.
     *  @param   pFtrGroup  The feature group, to which the feature belongs.
     *                      The returned value is one of the @ref LvFtrGroup.
     *                      Can be NULL.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetType (LvFeature Feature,
                      LvFtrType* pFtrType,
                      LvFtrGui* pFtrGui = NULL,
                      LvFtrGroup* pFtrGroup = NULL);
  
    //------------------------------------------------------------------------------
    /** Gets a Boolean value.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   pValue    The bool value is returned in this parameter.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetBool  (LvFeature Feature,
                       bool* pValue);
  
    //------------------------------------------------------------------------------
    /** Sets a Boolean value.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   Value     Value to be set.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus SetBool  (LvFeature Feature,
                       bool Value);
  
    //------------------------------------------------------------------------------
    /** Gets a 32-bit integer value.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   pValue    The integer value is returned in this parameter.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @note    The value is internally kept always as a 64-bit value; the
     *           functions for setting and getting a 32-bit value are provided
     *           just for convenience.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetInt32 (LvFeature Feature,
                       int32_t* pValue);
  
    //------------------------------------------------------------------------------
    /** Sets a 32-bit value.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   Value     Value to be set.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @note    The value is internally kept always as a 64-bit value; the
     *           functions for setting and getting a 32-bit value are provided
     *           just for convenience.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus SetInt32 (LvFeature Feature,
                       int32_t Value);
  
    //------------------------------------------------------------------------------
    /** Returns a range and increment of an 32-bit integer feature.
     *
     *  @param   Feature     The feature ID - use a symbolic constant (one of the
     *                       @ref GroupSynview_Features) or an ID obtained by the
     *                       LvModule::GetFeatureByName() function.
     *  @param   pMinValue   The minimum value is returned in this parameter. Can be NULL.
     *  @param   pMaxValue   The maximum value is returned in this parameter. Can be NULL.
     *  @param   pIncrement  The increment value is returned in this parameter. Can be NULL.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @note    The value is internally kept always as a 64-bit value; the
     *           functions for setting and getting a 32-bit value are provided
     *           just for convenience.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetInt32Range (LvFeature Feature,
                            int32_t* pMinValue,
                            int32_t* pMaxValue,
                            int32_t* pIncrement);
  
    //------------------------------------------------------------------------------
    /** Gets a 64-bit integer value.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   pValue    The integer value is returned in this parameter.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @note    This function is equal to the LvModule::GetInt() function.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetInt64 (LvFeature Feature,
                       int64_t* pValue);
  
    //------------------------------------------------------------------------------
    /** Sets a 64-bit integer value.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   Value     Value to be set.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @note    This function is equal to the LvModule::SetInt() function.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus SetInt64 (LvFeature Feature,
                       int64_t Value);
  
    //------------------------------------------------------------------------------
    /** Returns a range and increment of an 64-bit integer feature.
     *
     *  @param   Feature     The feature ID - use a symbolic constant (one of the
     *                       @ref GroupSynview_Features) or an ID obtained by the
     *                       LvModule::GetFeatureByName() function.
     *  @param   pMinValue   The minimum value is returned in this parameter. Can be NULL.
     *  @param   pMaxValue   The maximum value is returned in this parameter. Can be NULL.
     *  @param   pIncrement  The increment value is returned in this parameter. Can be NULL.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @note    This function is equal to the LvModule::GetIntRange() function.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetInt64Range (LvFeature Feature,
                            int64_t* pMinValue,
                            int64_t* pMaxValue,
                            int64_t* pIncrement);
  
    //------------------------------------------------------------------------------
    /** Gets a 64-bit integer value.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   pValue    The integer value is returned in this parameter.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @note    This function is equal to the LvModule::GetInt64() function.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetInt (LvFeature Feature,
                     int64_t* pValue);
  
    //------------------------------------------------------------------------------
    /** Sets a 64-bit integer value.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   Value     Value to be set.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @note    This function is equal to the LvModule::SetInt64() function.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus SetInt (LvFeature Feature,
                     int64_t Value);
  
    //------------------------------------------------------------------------------
    /** Returns a range and increment of an 64-bit integer feature.
     *
     *  @param   Feature     The feature ID - use a symbolic constant (one of the
     *                       @ref GroupSynview_Features) or an ID obtained by the
     *                       LvModule::GetFeatureByName() function.
     *  @param   pMinValue   The minimum value is returned in this parameter. Can be NULL.
     *  @param   pMaxValue   The maximum value is returned in this parameter. Can be NULL.
     *  @param   pIncrement  The increment value is returned in this parameter. Can be NULL.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @note    This function is equal to the LvModule::GetInt64Range() function.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetIntRange (LvFeature Feature,
                          int64_t* pMinValue,
                          int64_t* pMaxValue,
                          int64_t* pIncrement);
  
    //------------------------------------------------------------------------------
    /** Gets a float value.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   pValue    The float value is returned in this parameter.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetFloat (LvFeature Feature,
                       double* pValue);
  
    //------------------------------------------------------------------------------
    /** Sets a float value.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   Value     The value to be set.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus SetFloat (LvFeature Feature,
                       double Value);
  
    //------------------------------------------------------------------------------
    /** Returns a range of a float feature.
     *
     *  @param   Feature    The feature ID - use a symbolic constant (one of the
     *                      @ref GroupSynview_Features) or an ID obtained by the
     *                      LvModule::GetFeatureByName() function.
     *  @param   pMinValue  The minimum value is returned in this parameter. Can be NULL.
     *  @param   pMaxValue  The maximum value is returned in this parameter. Can be NULL.
     *  @param   pIncrement The increment value is returned in this parameter. If the increment is
     *                      not defined, 0 is returned. Can be NULL.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetFloatRange (LvFeature Feature,
                            double* pMinValue,
                            double* pMaxValue,
                            double* pIncrement = NULL);
  
    //------------------------------------------------------------------------------
    /** Gets a string value. If you need first to get the string size, use the
     *  LvModule::GetStringSize() function.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   pValue    Pointer to a null-terminated string buffer.
     *  @param   Size      Size of the buffer.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetString (LvFeature Feature,
                        char* pValue,
                        size_t Size);
  
    //------------------------------------------------------------------------------
    /** Gets a buffer size needed for a string.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   pSize     Size of the buffer (including space for terminating zero)
     *                     is returned in this parameter.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetStringSize (LvFeature Feature,
                            size_t* pSize);
  
    //------------------------------------------------------------------------------

    #ifdef LV_USE_STDLIB
        /** Gets a string value as std::string.
         *
         *  @param   Feature   The feature ID - use a symbolic constant (one of the
         *                     @ref GroupSynview_Features) or an ID obtained by the
         *                     LvModule::GetFeatureByName() function.
         *  @param   sValue    In this parametr the string value is returned.
         *  @return  Returns the @ref LvStatus value indicating the success of the
         *           requested operation. See @ref GroupSynview_LvStatus.
         *  @ingroup GroupSynview_LvModule_Cpp
         */
        LvStatus GetString (LvFeature Feature,
                            std::string& sValue);
    #endif

    //------------------------------------------------------------------------------
    /** Sets a string value.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   pValue    The string value (null-terminated).
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus SetString (LvFeature Feature,
                        const char* pValue);
  
    //------------------------------------------------------------------------------
    /** Gets a block of data.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   pBuffer   Pointer to a buffer, to which the data will be stored.
     *  @param   Size      Size of the buffer.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetBuffer (LvFeature Feature,
                        void* pBuffer, 
                        size_t Size);
  
    //------------------------------------------------------------------------------
    /** Gets the block data size.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   pSize     The needed size of the buffer is returned in this parameter.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetBufferSize (LvFeature Feature,
                            size_t* pSize);
  
    //------------------------------------------------------------------------------
    /** Sets a block of data.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   pBuffer   Pointer to the data.
     *  @param   Size      Size of the data.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus SetBuffer (LvFeature Feature,
                        void* pBuffer,
                        size_t Size);
  
    //------------------------------------------------------------------------------
    /** Gets a pointer.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   ppValue   The pointer is returned in this parameter.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetPtr (LvFeature Feature,
                     void** ppValue);
  
    //------------------------------------------------------------------------------
    /** Sets a pointer.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   pValue    The pointer to be set.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus SetPtr (LvFeature Feature,
                     void* pValue);
  
    //------------------------------------------------------------------------------
    /** Gets the SynView constant for the enumeration entry, if exists. If does not exist,
     *  you must work with the string enumeration entry value.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   pValue    SynView constant for the enum entry is returned in this parameter.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetEnum (LvFeature Feature,
                      LvEnum* pValue);
  
    //------------------------------------------------------------------------------
    /** Sets the enumeration entry by the SynView constant. If the SynView constant is not
     *  defined for the feature, then use LvModule::SetEnumStr() to set the enum entry by
     *  a string.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   Value     SynView constant for the requested enumeration entry.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus SetEnum (LvFeature Feature,
                      LvEnum Value);
  
    //------------------------------------------------------------------------------
    /** Gets the enumeration entry as a string (symbolic name). It is not possible to
     *  get the needed size for this single feature, instead, it is possible to get the
     *  maximum size of the all enum values of this feature, by the
     *  LvModule::GetInfo(LvFtrInfo_EnumEntryNameMaxSize) function.
     *
     *  @param   Feature        The feature ID - use a symbolic constant (one of the
     *                          @ref GroupSynview_Features) or an ID obtained by the
     *                          LvModule::GetFeatureByName() function.
     *  @param   pSymbolicName  A pointer to a string buffer, where the symbolic name will be returned.
     *  @param   Size           Size of the buffer.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetEnumStr (LvFeature Feature,
                         char* pSymbolicName,
                         size_t Size);

    //------------------------------------------------------------------------------
    #ifdef LV_USE_STDLIB
        /** Gets the enumeration entry as a standard string (symbolic name).
         *
         *  @param   Feature        The feature ID - use a symbolic constant (one of the
         *                          @ref GroupSynview_Features) or an ID obtained by the
         *                          LvModule::GetFeatureByName() function.
         *  @param   sSymbolicName  A string, where the symbolic name will be returned.
         *  @return  Returns the @ref LvStatus value indicating the success of the
         *           requested operation. See @ref GroupSynview_LvStatus.
         *  @ingroup GroupSynview_LvModule_Cpp
         */
        LvStatus GetEnumStr (LvFeature Feature,
                             std::string& sSymbolicName);
    #endif

    //------------------------------------------------------------------------------
    /** Sets enumeration entry by its string symbolic name.
     *
     *  @param   Feature        The feature ID - use a symbolic constant (one of the
     *                          @ref GroupSynview_Features) or an ID obtained by the
     *                          LvModule::GetFeatureByName() function.
     *  @param   pSymbolicName  A string with the symbolic name of the
     *                          enumeration entry.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus SetEnumStr (LvFeature Feature,
                         const char* pSymbolicName);
  
  
    //------------------------------------------------------------------------------
    /** Gets the SynView constant for the enumeration entry, if exists.
     *
     *  @param   Feature        The feature ID - use a symbolic constant (one of the
     *                          @ref GroupSynview_Features) or an ID obtained by the
     *                          LvModule::GetFeatureByName() function.
     *  @param   pSymbolicName  A string with symbolic name of the enum entry.
     *  @param   pValue         The SynView constant for the enum entry is returned
     *                          in this parameter. If the SynView constant does not exist for 
     *                          this enumeration entry, 0 is returned (no error in indicated).
     *  @param   pFtrAccess     The feature access is returned in this parameter - one of
     *                          @ref LvFtrAccess. Can be NULL.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetEnumValByStr (LvFeature Feature,
                              const char* pSymbolicName,
                              LvEnum* pValue,
                              LvFtrAccess* pFtrAccess = NULL);
  
    //------------------------------------------------------------------------------
    /** Returns a string symbolic name of the enum entry for the SynView constant.
     *
     *  @param   Feature           The feature ID - use a symbolic constant (one of the
     *                             @ref GroupSynview_Features) or an ID obtained by the
     *                             LvModule::GetFeatureByName() function.
     *  @param   Value             The SynView constant for the enum entry.
     *  @param   pSymbolicName     Pointer to string buffer, where the symbolic name is returned.
     *                             Can be NULL.
     *  @param   SymbolicNameSize  Size of pSymbolicName buffer.
     *  @param   pFtrAccess        The access mode of the enum entry is returned in this
     *                             parameter - one of @ref LvFtrAccess. Can be NULL.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetEnumStrByVal (LvFeature Feature,
                              LvEnum Value,
                              char* pSymbolicName,
                              size_t SymbolicNameSize,
                              LvFtrAccess* pFtrAccess = NULL);
  
    //------------------------------------------------------------------------------
    #ifdef LV_USE_STDLIB
        /** Returns a string symbolic name of the enum entry for the SynView constant.
         *
         *  @param   Feature           The feature ID - use a symbolic constant (one of the
         *                             @ref GroupSynview_Features) or an ID obtained by the
         *                             LvModule::GetFeatureByName() function.
         *  @param   Value             The SynView constant for the enum entry.
         *  @param   sSymbolicName     In this parameter the symbolic name is returned.
         *  @param   pFtrAccess        The access mode of the enum entry is returned in this
         *                             parameter - one of @ref LvFtrAccess. Can be NULL.
         *  @return  Returns the @ref LvStatus value indicating the success of the
         *           requested operation. See @ref GroupSynview_LvStatus.
         *  @ingroup GroupSynview_LvModule_Cpp
         */
        LvStatus GetEnumStrByVal (LvFeature Feature,
                                  LvEnum Value,
                                  std::string& sSymbolicName,
                                  LvFtrAccess* pFtrAccess = NULL);
    #endif

    //------------------------------------------------------------------------------
    /** Executes a command.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   Timeout   If greater than 0, the LvModule::CmdIsDone() is called in a loop to wait
     *                     for the command completion, until the LvModule::CmdIsDone() returns true or
     *                     the Timeout (in milliseconds) expires. If set to 0, no wait is done.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus CmdExecute (LvFeature Feature,
                         uint32_t Timeout = 0);
  
    //------------------------------------------------------------------------------
    /** Checks if the command execution has completed.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   pIsDone   In this parameter is returned true, if the command is
     *                     completed, otherwise false.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus CmdIsDone (LvFeature Feature,
                        bool* pIsDone);
  
    //------------------------------------------------------------------------------
    /** Gets the access mode of the feature.
     *
     *  @param   Feature      The feature ID - use a symbolic constant (one of the
     *                        @ref GroupSynview_Features) or an ID obtained by the
     *                        LvModule::GetFeatureByName() function.
     *  @param   pFtrAccess   The access is returned in this parameter. One of the
     *                        @ref LvFtrAccess.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetAccess (LvFeature Feature,
                        LvFtrAccess* pFtrAccess);
  
    //------------------------------------------------------------------------------
    /** Gets the feature visibility (beginner-expert-guru).
     *
     *  @param   Feature          The feature ID - use a symbolic constant (one of the
     *                            @ref GroupSynview_Features) or an ID obtained by the
     *                            LvModule::GetFeatureByName() function.
     *  @param   pFtrVisibility   The visibility is returned in this parameter. One of the
     *                            @ref LvFtrVisibility.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetVisibility (LvFeature Feature,
                            LvFtrVisibility* pFtrVisibility);
  
    //------------------------------------------------------------------------------
    /** Gets an info in form of a 32-bit integer value.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   FtrInfo   One of the @ref LvFtrInfo.
     *  @param   pInfo     The value is returned in this parameter.
     *  @param   Param     Additional parameter, required by some types of info.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetInfo (LvFeature Feature,
                      LvFtrInfo FtrInfo,
                      int32_t* pInfo, 
                      int32_t Param = 0);
  
    //------------------------------------------------------------------------------
    /** Gets an info in form of a string value.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   FtrInfo   One of the @ref LvFtrInfo.
     *  @param   pInfoStr  The string value is returned in this parameter.
     *  @param   Size      Size of the buffer (to which pInfoStr points).
     *  @param   Param     Additional parameter, required by some types of info.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetInfoStr (LvFeature Feature,
                         LvFtrInfo FtrInfo,
                         char* pInfoStr,
                         size_t Size,
                         int32_t Param = 0);
  
    //------------------------------------------------------------------------------
    /** Gets a buffer size needed for an info in form of a string value.
     *
     *  @param   Feature   The feature ID - use a symbolic constant (one of the
     *                     @ref GroupSynview_Features) or an ID obtained by the
     *                     LvModule::GetFeatureByName() function.
     *  @param   FtrInfo  One of the @ref LvFtrInfo.
     *  @param   pSize     Size of the buffer is returned in this parameter.
     *  @param   Param     Additional parameter, required by some types of info.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus GetInfoStrSize (LvFeature Feature,
                             LvFtrInfo FtrInfo,
                             size_t* pSize,
                             int32_t Param = 0);
  
    //------------------------------------------------------------------------------
    #ifdef LV_USE_STDLIB
        /** Gets an info in form of a string value.
         *
         *  @param   Feature   The feature ID - use a symbolic constant (one of the
         *                     @ref GroupSynview_Features) or an ID obtained by the
         *                     LvModule::GetFeatureByName() function.
         *  @param   FtrInfo   One of the @ref LvFtrInfo.
         *  @param   sInfoStr  The string value is returned in this parameter.
         *  @param   Param     Additional parameter, required by some types of info.
         *  @return  Returns the @ref LvStatus value indicating the success of the
         *           requested operation. See @ref GroupSynview_LvStatus.
         *  @ingroup GroupSynview_LvModule_Cpp
         */
        LvStatus GetInfoStr (LvFeature Feature,
                             LvFtrInfo FtrInfo,
                             std::string& sInfoStr,
                             int32_t Param = 0);
    #endif

    //------------------------------------------------------------------------------
    /** Registers or unregisters a callback function for the feature.
     *  This callback is produced by GenApi when a feature changes its value or status.
     *  The application should process this callback fast. Note that the callback can be
     *  called also from another thread - see @ref LvEventType_FeatureDevEvent.
     *  Important note: The feature callback function should never set any other feature.
     *  Doing so can lead to recursions, which would be probably hard to diagnose and
     *  could cause unexpected behavior.
     *
     *  @param   Feature       The feature ID - use a symbolic constant (one of the
     *                         @ref GroupSynview_Features) or an ID obtained by the
     *                         LvModule::GetFeatureByName() function.
     *  @param   pFunction     The callback function in the form of LvFeatureCallbackFunct.
     *                         If you want to unregister the function, use NULL at this parameter.
     *  @param   pUserParam    User parameter, which will be passed to each callback call.
     *  @param   pFeatureParam Second user parameter, which will be passed to each callback call.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus RegisterFeatureCallback(LvFeature Feature,
                                     LvFeatureCallbackFunct pFunction,
                                     void* pUserParam = NULL, 
                                     void* pFeatureParam = NULL);

    //------------------------------------------------------------------------------
    /** Starts a thread, which in a loop polls the non-cached features. If the feature
     *  polling interval expires, the value is read and the feature callback is called.
     *
     *  @param   PollingTime      A time in milliseconds between 2 calls to poll the features.
     *  @param   PollChildren     If set to true, also the features in all children modules
     *                            are polled. For example, if your application uses only one System
     *                            module, then it is a parent of all other modules, so the polling
     *                            will be propagated to all modules from a single thread.
     *                            If a module has started own polling thread, then it is excluded
     *                            from the propagating.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus StartPollingThread(uint32_t PollingTime = 1000,
                                bool PollChildren = false);

    //------------------------------------------------------------------------------
    /** Stops the polling thread. See LvModule::StartPollingThread() for details.
     *
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus StopPollingThread();

    //------------------------------------------------------------------------------
    /** Polls all the non-cached features of the module. If the feature
     *  polling interval expires, the value is read and the feature callback is called.
     *
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvStatus Poll();

protected:
	//------------------------------------------------------------------------------
    /** The base class handle.
     *  @ingroup GroupSynview_LvModule_Cpp
     */
    LvHModule m_hModule;
};


class LvDevice;
class LvInterface;
class LvStream;
class LvBuffer;
class LvEvent;
class LvRenderer;

#ifdef LVAVISAVER
    class LvAviSaver;
#endif

//##############################################################################
/** The LvSystem class.
 *  @note   For all the SynView module classes you cannot use the new and delete
 *           operators directly (the constructor and destructor are private).
 *           Instead, the static methods for opening and closing the class instance
 *           assure that if the opening is successful, you get a valid pointer, otherwise
 *           you get a NULL pointer. Also, the closing functions set the pointer back to
 *           NULL. Another advantage is that these functions return a status value, which
 *           can clarify the error nature, if the opening or closing fails.
 */
class LvSystem: public LvModule
{
public:
    //------------------------------------------------------------------------------
    /** Creates the LvSystem class instance. Opening the system actually means loading the
     *  corresponding GenTL library. Note that before you can open the System, the
     *  LvOpenLibrary() must be called. The same system can be open multiple times
     *  (there is a reference counter inside); in such case there must be also
     *  the same number of LvSystem::Close() calls used (every open increase the
     *  reference count and every close decreases it).
     *
     *  @param   pSystemId  A string ID of the system. This can be either an empty string - then
     *                      the default system is opened, or it can be a string obtained from the
     *                      LvGetSystemId() function.
     *  @param   pSystem    Pointer to the opened LvSystem instance is returned here
     *                      in case the opening succeeds, NULL if fails
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvSystem_Cpp
     */
    static LvStatus Open (const char* pSystemId, LvSystem*& pSystem);

    //------------------------------------------------------------------------------
    /** Deletes the LvSystem class instance. Actually it means freeing the coresponding GenTL
     *  library. Be sure you first close all dependent modules (LvInterface, LvEvent etc.).
     *  If the System was opened multiple times, it only decreases the reference counter
     *  (see the note by the LvSystem::Open()).
     *
     *  @param   pSystem   Pointer to LvSystem instance, obtained from the
     *                     LvSystem::Open() function. The pointer is assigned NULL after
     *                     the operation.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvSystem_Cpp
     */
    static LvStatus Close (LvSystem*& pSystem);
  
    //------------------------------------------------------------------------------
    /** Updates the internal list of available interfaces. You can then iterate
     *  through them by LvSystem::GetNumberOfInterfaces() and LvSystem::GetInterfaceId().
     *
     *  @param   Timeout   Specifies a timeout in ms for searching the interfaces. This applies
     *                     only to special cases of interfaces, where some delay can happen;
     *                     common interfaces are detected without any significant delays.
     *  @return  If the timeout has expired while waiting for the completion, the function returns
     *           @ref LVSTATUS_TIMEOUT, otherwise it returns the @ref LvStatus value indicating the
     *           success of the requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvSystem_Cpp
     */
    LvStatus UpdateInterfaceList (uint32_t Timeout = 0xFFFFFFFF);
  
    //------------------------------------------------------------------------------
    /** Returns the number of found interfaces, after the LvSystem::UpdateInterfaceList()
     *  call.
     *
     *  @param   pNumberOfInterfaces  Number of interfaces found.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvSystem_Cpp
     */
    LvStatus GetNumberOfInterfaces (uint32_t* pNumberOfInterfaces);
  
    //------------------------------------------------------------------------------
    /** Returns a string ID of the interface, which is used by the LvInterface::Open()
     *  function.
     *
     *  @param   Index        Zero-based index of the interface, a value >= 0 and < number of interfaces,
     *                        returned by the LvSystem::GetNumberOfInterfaces() function.
     *  @param   pInterfaceId Pointer to a string buffer, where the interface ID will be placed.
     *  @param   Size         Size of the string buffer.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvSystem_Cpp
     */
    LvStatus GetInterfaceId (uint32_t Index,
                             char* pInterfaceId,
                             size_t Size);
  
    //------------------------------------------------------------------------------
    /** Returns the size of the string buffer needed to hold the Interface ID string,
     *  including the terminating zero character.
     *
     *  @param   Index     Zero-based index of the interface, a value >= 0 and < number of interfaces,
     *                     returned by the LvSystem::GetNumberOfInterfaces() function.
     *  @param   pSize     Size of the buffer is returned in this parameter.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvSystem_Cpp
     */
    LvStatus GetInterfaceIdSize (uint32_t Index,
                                 size_t* pSize);
    
    //------------------------------------------------------------------------------
    #ifdef LV_USE_STDLIB
        /** Returns a string ID of the interface, which is used by the LvInterface::Open()
         *  function.
         *
         *  @param   Index        Zero-based index of the interface, a value >= 0 and < number of interfaces,
         *                        returned by the LvSystem::GetNumberOfInterfaces() function.
         *  @param   sInterfaceId String, where the interface ID will be placed.
         *  @return  Returns the @ref LvStatus value indicating the success of the
         *           requested operation. See @ref GroupSynview_LvStatus.
         *  @ingroup GroupSynview_LvSystem_Cpp
         */
        LvStatus GetInterfaceId (uint32_t Index,
                                 std::string& sInterfaceId);
    #endif

    //------------------------------------------------------------------------------
    /** Finds the interface according specified criteria and returns a string ID of the
     *  interface, which is used by the LvInterface::Open() function. This function does not
     *  update the interface list - if you need to do so, call the LvSystem::UpdateInterfaceList()
     *  function before calling this function.
     *
     *  @param   FindBy       Specifies by which criteria to find the interface. Use one of the
     *                        @ref LvFindBy constants.
     *  @param   pFindStr     Specifies the find string, the meaning of which is determined by the
     *                        FindBy parameter, for example when using the LvFindBy_IPAddress,
     *                        this string should contain the IP address searched for. The searched
     *                        string is not case sensitive and need not be complete (is searched as
     *                        a substring).
     *  @param   pInterfaceId Pointer to a string buffer, where the interface ID will be placed.
     *  @param   Size         Size of the string buffer.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus. If the Interface is found,
     *           the returned status is LVSTATUS_OK.
     *  @ingroup GroupSynview_LvSystem_Cpp
     */
    LvStatus FindInterface (LvFindBy FindBy,
                            const char* pFindStr,
                            char* pInterfaceId,
                            size_t Size);
    
    //------------------------------------------------------------------------------
    #ifdef LV_USE_STDLIB
        /** Finds the interface according specified criteria and returns a string ID of the
         *  interface, which is used by the LvInterface::Open() function. This function does not
         *  update the interface list - if you need to do so, call the LvSystem::UpdateInterfaceList()
         *  function before calling this function.
         *
         *  @param   FindBy       Specifies by which criteria to find the interface. Use one of the
         *                        @ref LvFindBy constants.
         *  @param   pFindStr     Specifies the find string, the meaning of which is determined by the
         *                        FindBy parameter, for example when using the LvFindBy_IPAddress,
         *                        this string should contain the IP address searched for. The searched
         *                        string is not case sensitive and need not be complete (is searched as
         *                        a substring).
         *  @param   sInterfaceId String to which the interface ID will be placed.
         *  @return  Returns the @ref LvStatus value indicating the success of the
         *           requested operation. See @ref GroupSynview_LvStatus. If the Interface is found,
         *           the returned status is LVSTATUS_OK.
         *  @ingroup GroupSynview_LvSystem_Cpp
         */
        LvStatus FindInterface (LvFindBy FindBy,
                                const char* pFindStr,
                                std::string& sInterfaceId);
    #endif

    //------------------------------------------------------------------------------
    /** Returns a handle of the System (used in the Plain C API), associated
     *  with this class.
     *
     *  @return  The Plain C API handle.
     *  @ingroup GroupSynview_LvSystem_Cpp
     */
    LvHSystem GetHandle();
  
    //------------------------------------------------------------------------------
    /** Creates the LvInterface class instance. This method is provided just for convenience,
     *  it has the same functionality as the LvInterface::Open() static method.
     *  The same Interface can be open multiple times (there is a reference counter
     *  inside); in such case there must be also the same number of LvInterface::Close()
     *  or LvSystem::InterfaceClose() calls used (every open increase the reference
     *  count and every close decreases it).
     *
     *  @param   pInterfaceId  A string interface ID, obtained by the LvSystem::GetInterfaceId().
     *  @param   pInterface    In this parameter the pointer to the LvInterface instance is
     *                         returned.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvInterface::Open().
     *  @ingroup GroupSynview_LvSystem_Cpp
     */
    LvStatus OpenInterface (const char* pInterfaceId,
                            LvInterface*& pInterface);

    //------------------------------------------------------------------------------
    /** Deletes the LvInterface class instance. This method is provided just for convenience,
     *  it has the same functionality as the LvInterface::Close() static method.
     *  If the Interface was opened multiple times, it only decreases the reference
     *  counter (see a note by the LvInterface::Open()).
     *  Be sure you first close all dependent modules (LvDevice, LvEvent etc.).
     *
     *  @param   pInterface   Pointer to the LvInterface instance, obtained from the
     *                        LvInterface::Open() function. The pointer is assigned NULL after
     *                        the operation.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvInterface::Close().
     *  @ingroup GroupSynview_LvSystem_Cpp
     */
    LvStatus CloseInterface (LvInterface*& pInterface);
  
    //------------------------------------------------------------------------------
    /** Creates the LvEvent class instance, owned by the System.
     *  This method is provided just for convenience, it has the same functionality
     *  as the LvEvent::Open() static method.
     *
     *  @param   EventType  One of the @ref LvEventType.
     *  @param   pEvent     To this parameter the Event class instance is stored.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvEvent::Open().
     *  @ingroup GroupSynview_LvSystem_Cpp
     */
    LvStatus OpenEvent (LvEventType EventType,
                        LvEvent*& pEvent);
  
    //------------------------------------------------------------------------------
    /** Deletes the LvEvent class instance.
     *  This method is provided just for convenience, it has the same functionality
     *  as the LvEvent::Close() static method.
     *
     *  @param   pEvent     Pointer the Event class instance, is assigned NULL after
     *                      the closing is done.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvEvent::Close().
     *  @ingroup GroupSynview_LvSystem_Cpp
     */
    LvStatus CloseEvent (LvEvent*& pEvent);
  
    //------------------------------------------------------------------------------

private:
    LvSystem();
    ~LvSystem();
    LvHSystem m_hSystem;
};

//##############################################################################
/** The LvInterface class.
 *  @note   For all the SynView module classes you cannot use the new and delete
 *           operators directly (the constructor and destructor are private).
 *           Instead, the static methods for opening and closing the class instance
 *           assure that if the opening is successful, you get a valid pointer, otherwise
 *           you get a NULL pointer. Also, the closing functions set the pointer back to
 *           NULL. Another advantage is that these functions return a status value, which
 *           can clarify the error nature, if the opening or closing fails.
 */
class LvInterface: public LvModule
{
public:

    //------------------------------------------------------------------------------
    /** Creates the LvInterface class instance.
     *  The same Interface can be open multiple times (there is a reference counter
     *  inside); in such case there must be also the same number of LvInterface::Close()
     *  or LvSystem::InterfaceClose() calls used (every open increase the reference
     *  count and every close decreases it) .
     *
     *  @param   pSystem       A pointer to the LvSystem instance, obtained from the
     *                         LvSystem::Open() function.
     *  @param   pInterfaceId  A string interface ID, obtained by the LvSystem::GetInterfaceId().
     *  @param   pInterface    In this parameter the pointer to the LvInterface instance is
     *                         returned.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvSystem::OpenInterface().
     *  @ingroup GroupSynview_LvInterface_Cpp
     */
    static LvStatus Open (LvSystem* pSystem, const char* pInterfaceId, LvInterface*& pInterface);
  
    //------------------------------------------------------------------------------
    /** Deletes the LvInterface class instance.
     *  If the Interface was opened multiple times, it only decreases the reference
     *  counter (see a note by the LvInterface::Open()).
     *  Be sure you first close all dependent modules (LvDevice, LvEvent etc.).
     *
     *  @param   pInterface   Pointer to the LvInterface instance, obtained from the
     *                        LvInterface::Open() function. The pointer is assigned NULL after
     *                        the operation.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvSystem::CloseInterface().
     *  @ingroup GroupSynview_LvInterface_Cpp
     */
    static LvStatus Close (LvInterface*& pInterface);
  
    //------------------------------------------------------------------------------
    /** Updates the Device list. The available devices are searched.
     *
     *  @param   Timeout     Specifies a timeout in ms for searching the devices.
     *  @return  If the timeout has expired while waiting for the completion, the function returns
     *           @ref LVSTATUS_TIMEOUT, otherwise it returns the @ref LvStatus value indicating the
     *           success of the requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvInterface_Cpp
     */
    LvStatus UpdateDeviceList (uint32_t Timeout = 0xFFFFFFFF);
  
    //------------------------------------------------------------------------------
    /** Returns the number of devices found by the LvInterface::UpdateDeviceList()
     *  function.
     *
     *  @param   pDevices    Number of devices found.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvInterface_Cpp
     */
    LvStatus GetNumberOfDevices (uint32_t* pDevices);
  
    //------------------------------------------------------------------------------
    /** Returns a string ID of the device at specified position in the list.
     *  Note that this device ID is stable (the same physical device has always the
     *  same ID) and it is unique (no other physical device can have the same ID).
     *  To hardcode directly the device ID in your application is not recommended, as
     *  the application would not be usable, when a deffective device needs to be
     *  replaced. The SynView User's Guide discuss the ways, how to solve such
     *  maintainability demands.
     *
     *  @param   Index       Zero-based index of the device, a value >= 0 and < number of devices,
     *                       returned by the LvInterface::GetNumberOfDevices() function.
     *  @param   pDeviceId   Pointer to a string buffer, where the device ID will be placed.
     *  @param   Size        Size of the buffer.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvInterface_Cpp
     */
    LvStatus GetDeviceId (uint32_t Index,
                          char* pDeviceId,
                          size_t Size);
  
    //------------------------------------------------------------------------------
    /** Returns the size of the string buffer needed to hold the Device ID string,
     *  including the terminating zero character.
     *
     *  @param   Index       Zero-based index of the device, a value >= 0 and < number of devices,
     *                       returned by the LvInterface::GetNumberOfDevices() function.
     *  @param   pSize       Size of the buffer is returned in this parameter.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvInterface_Cpp
     */
    LvStatus GetDeviceIdSize (uint32_t Index,
                              size_t* pSize);

    //------------------------------------------------------------------------------
    #ifdef LV_USE_STDLIB
        /** Returns a string ID of the device at specified position in the list.
         *  Note that this device ID is stable (the same physical device has always the
         *  same ID) and it is unique (no other physical device can have the same ID).
         *  To hardcode directly the device ID in your application is not recommended, as
         *  the application would not be usable, when a deffective device needs to be
         *  replaced. The SynView User's Guide discuss the ways, how to solve such
         *  maintainability demands.
         *
         *  @param   Index       Zero-based index of the device, a value >= 0 and < number of devices,
         *                       returned by the LvInterface::GetNumberOfDevices() function.
         *  @param   sDeviceId   String to which the device ID will be placed.
         *  @return  Returns the @ref LvStatus value indicating the success of the
         *           requested operation. See @ref GroupSynview_LvStatus.
         *  @ingroup GroupSynview_LvInterface_Cpp
         */
        LvStatus GetDeviceId (uint32_t Index,
                              std::string& sDeviceId);
    #endif

    //------------------------------------------------------------------------------
    /** Finds the device according specified criteria and returns a string ID of the
     *  device, which can be used by the LvDevice::Open() function. This function does not
     *  update the device list - if you need to do so, call the LvInterface::UpdateDeviceList()
     *  function before calling this function.
     *
     *  @param   FindBy      Specifies by which criteria to find the interface. Use one of the
     *                       @ref LvFindBy constants.
     *  @param   pFindStr    Specifies the find string, the meaning of which is determined by the
     *                       FindBy parameter, for example when using the LvFindBy_IPAddress,
     *                       this string should contain the IP address searched for. The searched
     *                       string is not case sensitive and need not be complete (is searched as
     *                       a substring).
     *  @param   pDeviceId   Pointer to a string buffer, where the device ID will be placed.
     *  @param   Size        Size of the buffer.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus. If the Device is found,
     *           the returned status is LVSTATUS_OK.
     *  @ingroup GroupSynview_LvInterface_Cpp
     */
    LvStatus FindDevice (LvFindBy FindBy,
                         const char* pFindStr,
                         char* pDeviceId,
                         size_t Size);

    //------------------------------------------------------------------------------
    #ifdef LV_USE_STDLIB
        /** Finds the device according specified criteria and returns a string ID of the
         *  device, which can be used by the LvDevice::Open() function. This function does not
         *  update the device list - if you need to do so, call the LvInterface::UpdateDeviceList()
         *  function before calling this function.
         *
         *  @param   FindBy      Specifies by which criteria to find the interface. Use one of the
         *                       @ref LvFindBy constants.
         *  @param   pFindStr    Specifies the find string, the meaning of which is determined by the
         *                       FindBy parameter, for example when using the LvFindBy_IPAddress,
         *                       this string should contain the IP address searched for. The searched
         *                       string is not case sensitive and need not be complete (is searched as
         *                       a substring).
         *  @param   sDeviceId   To this string the found device ID is placed.
         *  @return  Returns the @ref LvStatus value indicating the success of the
         *           requested operation. See @ref GroupSynview_LvStatus. If the Device is found,
         *           the returned status is LVSTATUS_OK.
         *  @ingroup GroupSynview_LvInterface_Cpp
         */
        LvStatus FindDevice (LvFindBy FindBy,
                             const char* pFindStr,
                             std::string& sDeviceId);
    #endif

    //------------------------------------------------------------------------------
    /** Returns a handle of the Interface (used in the Plain C API), associated
     *  with this class.
     *
     *  @return  The Plain C API handle.
     *  @ingroup GroupSynview_LvInterface_Cpp
     */
    LvHInterface GetHandle();
  
    //------------------------------------------------------------------------------
    /** Creates the LvDevice class instance. This method is provided just for convenience,
     *  it has the same functionality as the LvDevice::Open() static method.
     *  This physically means opening a connection with the device and retrieving
     *  a list of device remote features. Always check the success of this function
     *  call; the opening may fail for example when you request an exclusive
     *  access and the device is already open by some other application.
     *
     *  @param   pDeviceId   A string ID of the device, obtained by LvInterface::GetDeviceId()
     *                       function.
     *  @param   pDevice     In this parameter the pointer to the LvDevice instance is
     *                       returned.
     *  @param   Access      Desired device access, one of the @ref LvDeviceAccess constants.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvDevice::Open().
     *  @ingroup GroupSynview_LvInterface_Cpp
     */
    LvStatus OpenDevice (const char* pDeviceId,
                         LvDevice*& pDevice,
                         LvDeviceAccess Access = LvDeviceAccess_Exclusive);

    //------------------------------------------------------------------------------
    /** Deletes the LvDevice class instance. This method is provided just for convenience,
     *  it has the same functionality as the LvDevice::Close() static method.
     *  Be sure you first close all dependent modules (LvStream, LvEvent etc.).
     *
     *  @param   pDevice   Pointer to the LvDevice instance, obtained from the
     *                     LvDevice::Open() function. This pointer is assigned NULL after
     *                     the operation.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvDevice::Close().
     *  @ingroup GroupSynview_LvInterface_Cpp
     */
    LvStatus CloseDevice (LvDevice*& pDevice);
  
private:
    LvInterface();
    ~LvInterface();
    LvHInterface m_hInterface;
};

//##############################################################################
/** The LvDevice class.
 *  @note   For all the SynView module classes you cannot use the new and delete
 *           operators directly (the constructor and destructor are private).
 *           Instead, the static methods for opening and closing the class instance
 *           assure that if the opening is successful, you get a valid pointer, otherwise
 *           you get a NULL pointer. Also, the closing functions set the pointer back to
 *           NULL. Another advantage is that these functions return a status value, which
 *           can clarify the error nature, if the opening or closing fails.
 */
class LvDevice: public LvModule
{
public:

    //------------------------------------------------------------------------------
    /** Creates the LvDevice class instance.
     *
     *  This physically means opening a connection with the device and retrieving
     *  a list of device remote features. Always check the success of this function
     *  call; the opening may fail for example when you request an exclusive
     *  access and the device is already open by some other application.
     *
     *  @param   pInterface  A pointer to the LvInterface instance, obtained from the
     *                       LvInterface::Open() function.
     *  @param   pDeviceId   A string ID of the device, obtained by LvInterface::GetDeviceId()
     *                       function.
     *  @param   pDevice     In this parameter the pointer to the LvDevice instance is
     *                       returned.
     *  @param   Access      Desired device access, one of the @ref LvDeviceAccess constants.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvInterface::OpenDevice().
     *  @ingroup GroupSynview_LvDevice_Cpp
     */
    static LvStatus Open (LvInterface* pInterface,
                          const char* pDeviceId,
                          LvDevice*& pDevice,
                          LvDeviceAccess Access = LvDeviceAccess_Exclusive);
  
    //------------------------------------------------------------------------------
    /** Deletes the LvDevice class instance. Be sure you first close all dependent
     *  modules (LvStream, LvEvent etc.).
     *
     *  @param   pDevice   Pointer to the LvDevice instance, obtained from the
     *                     LvDevice::Open() function. This pointer is assigned NULL after
     *                     the operation.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvInterface::CloseDevice().
     *  @ingroup GroupSynview_LvDevice_Cpp
     */
    static LvStatus Close (LvDevice*& pDevice);
  
    //------------------------------------------------------------------------------
    /** Returns the number of available stream types for this device. You can then
     *  iterate the streams by the LvDevice::GetStreamId() function.
     *
     *  @param   pNumberOfStreams  The number of streams is returned here.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvDevice_Cpp
     */
    LvStatus GetNumberOfStreams (uint32_t* pNumberOfStreams);
  
    //------------------------------------------------------------------------------
    /** Returns a string Stream ID, needed for opening the stream.
     *
     *  @param   Index      Zero-based index of the stream type, a value >= 0 and < number of streams,
     *                      returned by the LvDevice::GetNumberOfStreams() function.
     *  @param   pStreamId  Pointer to a string buffer, where the stream ID will be placed.
     *  @param   Size       Size of the buffer.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvDevice_Cpp
     */
    LvStatus GetStreamId (uint32_t Index,
                          char* pStreamId,
                          size_t Size);
  
    //------------------------------------------------------------------------------
    /** Returns the size of the string buffer needed to hold the stream ID at given
     *  index, including the space for the terminating zero character.
     *
     *  @param   Index    Zero-based index of the stream type, a value >= 0 and < number of streams,
     *                    returned by the LvDevice::GetNumberOfStreams() function.
     *  @param   pSize    Size of the buffer is returned in this parameter.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvDevice_Cpp
     */
    LvStatus GetStreamIdSize (uint32_t Index,
                              size_t* pSize);
  
    //------------------------------------------------------------------------------
    #ifdef LV_USE_STDLIB
        /** Returns a string Stream ID, needed for opening the stream.
         *
         *  @param   Index      Zero-based index of the stream type, a value >= 0 and < number of streams,
         *                      returned by the LvDevice::GetNumberOfStreams() function.
         *  @param   sStreamId  A string to which the stream ID is placed.
         *  @return  Returns the @ref LvStatus value indicating the success of the
         *           requested operation. See @ref GroupSynview_LvStatus.
         *  @ingroup GroupSynview_LvDevice_Cpp
         */
        LvStatus GetStreamId (uint32_t Index,
                              std::string& sStreamId);
    #endif

    //------------------------------------------------------------------------------
    /** Starts the acquisition. This function includes more than just calling the
     *  remote AcquisitionStart command on the device - it checks the size of the buffers,
     *  prepares the streams for the start, locks GenTL params and then starts the
     *  acquisition on the device itself.
     *  Always check the success of this function call.
     *
     *  @param   Options  Reserved for future use, must be 0 or ommitted.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvDevice_Cpp
     */
    LvStatus AcquisitionStart (uint32_t Options = 0);
  
    //------------------------------------------------------------------------------
    /** Stops the acquisition.
     *
     *  @param   Options  Reserved for future use, must be 0 or ommitted.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvDevice_Cpp
     */
    LvStatus AcquisitionStop (uint32_t Options = 0);
    
    //------------------------------------------------------------------------------
    /** Aborts athe acquisition immediately, without completing the current Frame
     *  or waiting on a trigger.
     *
     *  @param   Options  Reserved for future use, must be 0 or ommitted.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvDevice_Cpp
     */
    LvStatus AcquisitionAbort (uint32_t Options = 0);
    
    //------------------------------------------------------------------------------
    /** Prepares the device for acquisition, so that the acquisition using the
     *  LvDevice::AcquisitionStart() function then can start fast. If it is not called
     *  before LvDevice::AcquisitionStart(), it is called automatically inside the
     *  LvDevice::AcquisitionStart().
     *
     *  @param   Options  Reserved for future use, must be 0 or ommitted.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvDevice_Cpp
     */
    LvStatus AcquisitionArm (uint32_t Options = 0);
  
    #ifdef SIMPLON_INC_OBSOLETE
        LvStatus SetHwLUT (LvLUTSelector Selector,
                           void* pLUT, 
                           size_t Size);
      
        LvStatus GetHwLUT (LvLUTSelector Selector,
                           void* pLUT, 
                           size_t Size);
    #endif
  
    //------------------------------------------------------------------------------
    /** Saves the device settings to a file. In the Options can be specified which
     *  parts of the device configuration are to be saved.
     *
     *  @param   pId        A string ID enabling to distinguish more configurations
     *                      in one file. If empty, the "Default" will be used.
     *  @param   pFileName  The file specification, to which the configuration is
     *                      stored. It is a text file.
     *  @param   Options    One or or-ed combination of @ref GroupSynview_SaveFlags.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvDevice_Cpp
     */
    LvStatus SaveSettings (const char* pId,
                           const char* pFileName,
                           uint32_t Options);
  
    //------------------------------------------------------------------------------
    /** Loads the device settings from a file. In the Options can be specified which
     *  parts of the device configuration are to be loaded.
     *
     *  @param   pId        A string ID enabling to distinguish more configurations
     *                      in one file. If empty, the "Default" will be used.
     *  @param   pFileName  The file specification, where the configuration is
     *                      stored. It is a text file.
     *  @param   Options    One or or-ed combination of @ref GroupSynview_SaveFlags.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvDevice_Cpp
     */
    LvStatus LoadSettings (const char* pId,
                           const char* pFileName,
                           uint32_t Options);

    //------------------------------------------------------------------------------
    /** Loads device batch commands from a file.
    *  @param   pFileName  The file specification, where the configuration is
    *                      stored. It is a text file.
    *  @return  Returns the @ref LvStatus value indicating the success of the
    *           requested operation. See @ref GroupSynview_LvStatus.
    *  @ingroup GroupSynview_DeviceFunctions
    */
    LvStatus LoadBatch (const char* pFileName);

    //------------------------------------------------------------------------------
    /** Sets the lookup table. If the hardware lookup table is available, it is used,
     *  otherwise a software lookup table is set.
     *  This function belongs to a set of functions, which unify the functionality
     *  of devices with real-time processing embedded in hardware (RTF) and devices
     *  without real-time processing, for which the processing is made by software.
     *  The LUT is automatically recalculated to appropriate type, if you use different
     *  LUT bit depth than is the actually used for the current pixel format.
     *
     *  @param   Selector  Lookup table selector, see @ref LvLUTSelector.
     *  @param   pLUT      Pointer to the lookup table.
     *  @param   Size      Size of the lookup table. The only valid values are
     *                     - 256 for 8-bit LUT
     *                     - 2048 for 10-bit LUT
     *                     - 8192 for 12-bit LUT
     *  @param   Options   The LvUniLutFlags_HwLut option can be used to apply to function directly on HW LUT.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvDevice_Cpp
     */
    LvStatus UniSetLut (LvLUTSelector Selector,
                        void* pLUT, 
                        size_t Size,
                        uint32_t Options = 0);

    //------------------------------------------------------------------------------
    /** Gets the lookup table. See LvDevice::UniSetLut() for details.
     *  The LUT is automatically recalculated to appropriate type, if you use different
     *  LUT bit depth than is the actually used for the current pixel format. So you can
     *  for example read the 12-bit LUT to 8-bit LUT array.
     *
     *  @param   Selector  Lookup table selector, see @ref LvLUTSelector.
     *  @param   pLUT      Pointer to the lookup table.
     *  @param   Size      Size of the lookup table. The only valid values are
     *                     - 256 for 8-bit LUT
     *                     - 2048 for 10-bit LUT
     *                     - 8192 for 12-bit LUT
     *  @param   Options   The LvUniLutFlags_HwLut option can be used to apply to function directly on HW LUT.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvDevice_Cpp
     */
    LvStatus UniGetLut (LvLUTSelector Selector,
                        void* pLUT, 
                        size_t Size,
                        uint32_t Options = 0);
                        
    //=== Firmware update functions ===============================================

    /** Returns the file name mask (with wildcard characters), for searching the
     *  file with the appropriate firmware update. The files with the FW update
     *  have in their names coded the hardware IDs, so using this mask (for example
     *  in a filter in a file open dialog box) assures the file appropriate for this
     *  device is used.
     *
     *  @param   Which         An ID specific for a hardware. Discussed in the SynView
     *                         User's Guide.
     *  @param   pFilePattern  In this parameter the file pattern is returned.
     *  @param   Size          Size of the buffer (to which the pFilePattern points).
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_FirmwareFunctions_Cpp
     */
    LvStatus FwGetFilePattern (uint32_t Which,
                               char* pFilePattern,
                               size_t Size);

    //------------------------------------------------------------------------------
    /** Loads the firmware from a file to the hardware. It can be very long process
     *  (taking minutes) and this functions blocks the thread druting this process.
     *  It is recommended to check the load status from another thread using the
     *  LvFwGetLoadStatus() function.
     *
     *  @param   Which       An ID specific for a hardware. Discussed in the SynView
     *                       User's Guide.
     *  @param   pFilePath   File specification, with full path.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_FirmwareFunctions_Cpp
     */
    LvStatus FwLoad (uint32_t Which,
                     const char* pFilePath);

    //------------------------------------------------------------------------------
    /** Returns the byte count and whether the loading is still in progress.
     *
     *  @param   Which              An ID specific for a hardware. Discussed in the SynView
     *                              User's Guide.
     *  @param   pCurrentByteCount  Returns number of bytes transferred so far.
     *  @param   pIsLoading         Returns true if the loading is still in progress.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_FirmwareFunctions_Cpp
     */
    LvStatus FwGetLoadStatus (uint32_t Which,
                              uint32_t* pCurrentByteCount,
                              bool* pIsLoading);

    //------------------------------------------------------------------------------
    /** Creates the LvStream class instance associated with the device.
     *  This method is provided just for convenience,
     *  it has the same functionality as the LvStream::Open() static method.
     *
     *  @param   pStreamId   A string ID of the stream, obtained from LvDevice::GetStreamId().
     *                       If an empty string is used, the first found stream is
     *                       opened. This is usually the image data stream.
     *  @param   pStream     In this parameter the pointer to the LvStream instance is
     *                       returned.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvStream::Open().
     *  @ingroup GroupSynview_LvDevice_Cpp
     */
    LvStatus OpenStream (const char* pStreamId,
                         LvStream*& pStream);
  
    //------------------------------------------------------------------------------
    /** Deletes the LvStream class instance. This method is provided just for convenience,
     *  it has the same functionality as the LvStream::Close() static method.
     *  Be sure you first close all dependent modules
     *  (LvBuffers, LvEvent, LvRenderer etc.).
     *
     *  @param   pStream   Pointer to the LvStream instance, obtained from the
     *                     LvStream::Open() or LvDevice::OpenStream() function.
     *                     This pointer is assigned NULL after the operation.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvStream::Close().
     *  @ingroup GroupSynview_LvDevice_Cpp
     */
    LvStatus CloseStream (LvStream*& pStream);
  
    //------------------------------------------------------------------------------
    /** Creates the LvEvent class instance for specified owner module.
     *  This method is provided just for convenience,
     *  it has the same functionality as the LvEvent::Open() static method.
     *
     *  @param   EventType  One of the @ref LvEventType.
     *  @param   pEvent     To this parameter the pointer LvEvent instance is stored.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvEvent::Open().
     *  @ingroup GroupSynview_LvDevice_Cpp
     */
    LvStatus OpenEvent (LvEventType EventType,
                        LvEvent*& pEvent);
  
    //------------------------------------------------------------------------------
    /** Deletes the LvEvent class instance.
     *  This method is provided just for convenience,
     *  it has the same functionality as the LvEvent::Close() static method.
     *
     *  @param   pEvent  A pointer to the LvEvent class instance, obtained from the
     *                   LvEvent::Open() or LvDevice::OpenEvent() function.
     *                   This pointer is assigned NULL after the operation.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvEvent::Close().
     *  @ingroup GroupSynview_LvDevice_Cpp
     */
    LvStatus CloseEvent (LvEvent*& pEvent);
  
    //------------------------------------------------------------------------------
    /** Returns a handle of the Device (used in the Plain C API), associated
     *  with this class.
     *
     *  @return  The Plain C API handle.
     *  @ingroup GroupSynview_LvDevice_Cpp
     */
    LvHDevice GetHandle();
  
private:
    LvDevice();
    ~LvDevice();
    LvHDevice m_hDevice;
};

//##############################################################################
/** The LvStream class.
 *  @note   For all the SynView module classes you cannot use the new and delete
 *           operators directly (the constructor and destructor are private).
 *           Instead, the static methods for opening and closing the class instance
 *           assure that if the opening is successful, you get a valid pointer, otherwise
 *           you get a NULL pointer. Also, the closing functions set the pointer back to
 *           NULL. Another advantage is that these functions return a status value, which
 *           can clarify the error nature, if the opening or closing fails.
 */

class LvStream: public LvModule
{
public:
  
    /** Creates the LvStream class instance, associated with the device.
     *
     *  @param   pDevice     A pointer to the LvDevice instance, obtained from the
     *                       LvDevice::Open() function.
     *  @param   pStreamId   A string ID of the stream, obtained from LvDevice::GetStreamId().
     *                       If an empty string is used, the first found stream is
     *                       opened. This is usually the image data stream.
     *  @param   pStream     In this parameter the pointer to the LvStream instance is
     *                       returned.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvDevice::OpenStream().
     *  @ingroup GroupSynview_LvStream_Cpp
     */
    static LvStatus Open (LvDevice* pDevice,
                          const char* pStreamId,
                          LvStream*& pStream);
  
    //------------------------------------------------------------------------------
    /** Deletes the LvStream class instance. Be sure you first close all dependent modules
     *  (LvBuffers, LvEvent, LvRenderer etc.).
     *
     *  @param   pStream   Pointer to the LvStream instance, obtained from the
     *                     LvStream::Open() function. This pointer is assigned NULL after
     *                     the operation.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvDevice::CloseStream().
     *  @ingroup GroupSynview_LvStream_Cpp
     */
    static LvStatus Close (LvStream*& pStream);
  
    //------------------------------------------------------------------------------
    /** Returns the buffer instance at given index.
     *
     *  @param   BufferIndex  Zero-based index.
     *  @param   pBuffer      In this parameter the pointer to LvBuffer instance is returned.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvStream_Cpp
     */
    LvStatus GetBufferAt (uint32_t BufferIndex,
                          LvBuffer*& pBuffer);
  
    //------------------------------------------------------------------------------
    /** Moves the buffers according to the LvQueueOperation specified.
     *
     *  @param   Operation  One of the @ref LvQueueOperation.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvStream_Cpp
     */
    LvStatus FlushQueue (LvQueueOperation Operation);
  
    //------------------------------------------------------------------------------
    /** Starts the stream. This function need not be used on the image stream, where
     *  it is called automatically in the LvDevice::AcquisitionStart() function.
     *
     *  @param   StartFlags      One of the GroupSynview_StreamStartFlags.
     *  @param   ImagesToAcquire Number of images to acquire.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvStream_Cpp
     */
    LvStatus Start (uint32_t StartFlags = LvStreamStartFlags_Default,
                    uint32_t ImagesToAcquire = 0xFFFFFFFF);
  
    //------------------------------------------------------------------------------
    /** Stops the stream. This function need not be used on the image stream, where
     *  it is called automatically in the LvDevice::AcquisitionStop() function.
     *
     *  @param   StopFlags  One of the GroupSynview_StreamStopFlags.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvStream_Cpp
     */
    LvStatus Stop (uint32_t StopFlags = LvStreamStopFlags_Default);
  
    //------------------------------------------------------------------------------
    /** Returns a handle of the Stream (used in the Plain C API), associated
     *  with this class.
     *
     *  @return  The Plain C API handle.
     *  @ingroup GroupSynview_LvStream_Cpp
     */
    LvHStream GetHandle();
  
    /** Creates the LvBuffer class instance. On the GenTL level it corresponds to DSAnnounceBuffer() or
     *  DSAllocAndAnnounceBuffer().
     *  This method is provided just for convenience, it has the same functionality
     *  as the LvBuffer::Open() static method.
     *
     *  @param   pDataPointer  Pointer to image data buffer. This can be supplied by
     *                         the application (in such case the DataSize must be set
     *                         to the actual size of the buffer), or can be left NULL -
     *                         in such case the buffer is allocated by SynView.
     *  @param   DataSize      Size of the buffer supplied, or 0 if the pDataPointer
     *                         is NULL.
     *  @param   pUserPointer  A user pointer, which is then passed back in the
     *                         LvEventCallbackNewBufferFunct(). It enables the application
     *                         to reference some own data structure associated with the
     *                         buffer.
     *  @param   Options       Currently unused, must be 0.
     *  @param   pBuffer       To this parameter the pointer to the LvBuffer instance is returned.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvBuffer::Open().
     *  @ingroup GroupSynview_LvStream_Cpp
     */
    LvStatus OpenBuffer (void* pDataPointer,
                         size_t DataSize, 
                         void* pUserPointer, 
                         uint32_t Options,
                         LvBuffer*& pBuffer);
  
    //------------------------------------------------------------------------------
    /** Deletes the LvBuffer class instance. On the GenTL level it corresponds to the DSRevokeBuffer()
     *  function.
     *  This method is provided just for convenience, it has the same functionality
     *  as the LvBuffer::Close() static method.
     *
     *  @param   pBuffer   A pointer to the LvBuffer instance, obtained from the
     *                     LvBuffer::Open() function. This pointer is assigned NULL after
                           the operation.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvBuffer::Close().
     *  @ingroup GroupSynview_LvStream_Cpp
     */
    LvStatus CloseBuffer (LvBuffer*& pBuffer);
  
    //------------------------------------------------------------------------------
    /** Creates the LvEvent class instance, owned by the Stream.
     *  This method is provided just for convenience, it has the same functionality
     *  as the LvEvent::Open() static method.
     *
     *  @param   EventType  One of the @ref LvEventType.
     *  @param   pEvent     To this parameter the Event class instance is stored.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvEvent::Open().
     *  @ingroup GroupSynview_LvStream_Cpp
     */
    LvStatus OpenEvent (LvEventType EventType,
                        LvEvent*& pEvent);

    //------------------------------------------------------------------------------
    /** Deletes the LvEvent class instance.
     *  This method is provided just for convenience, it has the same functionality
     *  as the LvEvent::Close() static method.
     *
     *  @param   pEvent     Pointer the Event class instance, is assigned NULL after
     *                      the closing is done.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvEvent::Close().
     *  @ingroup GroupSynview_LvStream_Cpp
     */
    LvStatus CloseEvent (LvEvent*& pEvent);

    //------------------------------------------------------------------------------
    /** Creates the LvRenderer class instance for image display. The renderer attepts to load
     *  the sv.synview.display library. In case of SynView installation in an operating
     *  system without possibility to graphically display (for example Linux without
     *  XWindows), the load of this library fails and the calls to Renderer functions
     *  will return errors.
     *  This method is provided just for convenience, it has the same functionality
     *  as the LvRenderer::Open() static method.
     *
     *  @param   pRenderer    In this parameter the pointer to the LvRenderer is returned.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvRenderer::Open().
     *  @ingroup GroupSynview_LvStream_Cpp
     */
    LvStatus OpenRenderer (LvRenderer*& pRenderer);
  
    //------------------------------------------------------------------------------
    /** Deletes the LvRenderer class instance.
     *  This method is provided just for convenience, it has the same functionality
     *  as the LvRenderer::Close() static method.
     *
     *  @param   pRenderer  A pointer to the LvRenderer instance, obtained from the
     *                      LvRenderer::Open() function. This pointer is set to NULL after close.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvRenderer::Close().
     *  @ingroup GroupSynview_LvStream_Cpp
     */
    LvStatus CloseRenderer (LvRenderer*& pRenderer);


    #ifdef LVAVISAVER
        LvStatus OpenAviSaver (LvAviSaver*& pAviSaver);
        LvStatus CloseAviSaver (LvAviSaver*& pAviSaver);
    #endif

private:
    LvStream();
    ~LvStream();
    LvHStream m_hStream;
};

//##############################################################################
/** The LvBuffer class.
 *  @note   For all the SynView module classes you cannot use the new and delete
 *           operators directly (the constructor and destructor are private).
 *           Instead, the static methods for opening and closing the class instance
 *           assure that if the opening is successful, you get a valid pointer, otherwise
 *           you get a NULL pointer. Also, the closing functions set the pointer back to
 *           NULL. Another advantage is that these functions return a status value, which
 *           can clarify the error nature, if the opening or closing fails.
 */

class LvBuffer: public LvModule
{
public:
    /** Creates the LvBuffer class instance. On the GenTL level it corresponds to DSAnnounceBuffer() or
     *  DSAllocAndAnnounceBuffer().
     *
     *  @param   pStream       A pointer to the LvStream instance, obtained from the
     *                         LvStream::Open() function.
     *  @param   pDataPointer  Pointer to image data buffer. This can be supplied by
     *                         the application (in such case the DataSize must be set
     *                         to the actual size of the buffer), or can be left NULL -
     *                         in such case the buffer is allocated by SynView.
     *  @param   DataSize      Size of the buffer supplied, or 0 if the pDataPointer
     *                         is NULL.
     *  @param   pUserPointer  A user pointer, which is then passed back in the
     *                         LvEventCallbackNewBufferFunct(). It enables the application
     *                         to reference some own data structure associated with the
     *                         buffer.
     *  @param   Options       Currently unused, must be 0.
     *  @param   pBuffer       To this parameter the pointer to the LvBuffer instance is returned.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvStream::OpenBuffer().
     *  @ingroup GroupSynview_LvBuffer_Cpp
     */
    static LvStatus Open (LvStream* pStream,
                          void* pDataPointer, 
                          size_t DataSize, 
                          void* pUserPointer, 
                          uint32_t Options,
                          LvBuffer*& pBuffer);
  
    //------------------------------------------------------------------------------
    /** Deletes the LvBuffer class instance. On the GenTL level it corresponds to the DSRevokeBuffer()
     *  function.
     *
     *  @param   pBuffer   A pointer to the LvBuffer instance, obtained from the
     *                     LvBuffer::Open() function. This pointer is assigned NULL after
                           the operation.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvStream::CloseBuffer().
     *  @ingroup GroupSynview_LvBuffer_Cpp
     */
    static LvStatus Close (LvBuffer*& pBuffer);

    //------------------------------------------------------------------------------
    /** Attaches a process buffer to a buffer. The process buffer may be needed for
     *  software processing, for example Bayer decoding, if the device hardware is
     *  not capable of it. The process buffer can be either supplied by the application
     *  by this function, or allocated automatically by SynView, upon need.
     *
     *  @param   pDataPointer  Pointer to the supplied buffer.
     *  @param   DataSize      Size of the buffer.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvBuffer_Cpp
     */
    LvStatus AttachProcessBuffer (void* pDataPointer, size_t DataSize);

    //------------------------------------------------------------------------------
    /** Puts the buffer to the input buffer pool. This is an important part of the
     *  image handling loop: after the buffer with the acquired image is passed
     *  to the application, the application must return it to the input buffer
     *  pool by this function after processing.
     *
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvBuffer_Cpp
     */
    LvStatus Queue ();
  
    //------------------------------------------------------------------------------
    /** Parses the chunk data of the image. The chunk data are then accessible as
     *  device remote features (for example @ref LvDevice_ChunkTimestamp).
     *
     *  @param   UpdateLayout  If set to true, the layout of chunk data is decoded. If set
     *                         to false, the data are only read from already decoded layout -
     *                         this is faster. Usually, the layout of the chunk data is
     *                         constant, so it needs to be decoded only at first call
     *                         of this function.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvBuffer_Cpp
     */
    LvStatus ParseChunkData(bool UpdateLayout = false);
  
    //------------------------------------------------------------------------------
    /** Saves the image to a file in Windows BMP format. If the image is in the
     *  pixel format not compatible with the BMP format, it is automatically converted.
     *
     *  @param   pFileName  The file name. Be sure to specify it with the full path.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvBuffer_Cpp
     */
    LvStatus SaveImageToBmpFile (const char* pFileName);
  
    //------------------------------------------------------------------------------
    /** Saves the image to a file in JPEG format. If the image is in the
     *  pixel format not compatible with the JPEG format, it is automatically converted.
     *
     *  @param   pFileName  The file name. Be sure to specify it with the full path.
     *  @param   Quality    The quality factor in range from from 1 to 100. The higher
     *                      is the factor, the higher is the quality and lower the
     *                      compression.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvBuffer_Cpp
     */
    LvStatus SaveImageToJpgFile (const char* pFileName,
                                 uint32_t Quality);
  
    //------------------------------------------------------------------------------
    /** Saves the image to a file in the TIFF format. If the image is in the
     *  pixel format not compatible with the TIF format, it is automatically converted.
     *
     *  @param   pFileName  The file name. Be sure to specify it with the full path.
     *  @param   Options    Options for saved pixel format. The @ref LvipOption_TiffConvertTo16Bit
     *                      flag can be used there.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvBuffer_Cpp
     */
    LvStatus SaveImageToTifFile (const char* pFileName,
                                 uint32_t Options = 0);
  
    //------------------------------------------------------------------------------
    /** Fills the @ref LvipImgInfo structure for the image in the buffer. This simplifies
     *  a direct use of the @ref GroupImgproc. If the image is processed, the image info
     *  points to the processed image, othewise it points to the original image.
     *
     *  @param   ImgInfo  The ImgInfo structure, to which are the image
     *                    parameters stored.
     *  @param   Options  Currently unused, must be 0.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvBuffer_Cpp
     */
    LvStatus GetImgInfo (LvipImgInfo& ImgInfo, uint32_t Options = 0);

    //------------------------------------------------------------------------------
    /** Returns the rectangle to which the buffer was last painted. This is useful
     *  namely in case you have a tile mode and want to identify the buffer according
     *  a mouse click location. If the buffer was not yet painted by the renderer, the 
     *  returned values are 0.
     *  @param   pX       Pointer to X offset in pixels.
     *  @param   pY       Pointer to Y offset in pixels.
     *  @param   pWidth   Pointer to Width in pixels.
     *  @param   pHeight  Pointer to Height in pixels.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvBuffer_Cpp
     */
    LvStatus GetLastPaintRect (int32_t* pX, 
                               int32_t* pY, 
                               int32_t* pWidth, 
                               int32_t* pHeight);
  
    //------------------------------------------------------------------------------
    /** Calculates white balance factors from the current image.
     *
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvBuffer_Cpp
     */
    LvStatus UniCalculateWhiteBalance ();

    //------------------------------------------------------------------------------
    /** Returns a handle of the Buffer (used in the Plain C API), associated
     *  with this class.
     *
     *  @return  The Plain C API handle.
     *  @ingroup GroupSynview_LvBuffer_Cpp
     */
    LvHBuffer GetHandle();

	//------------------------------------------------------------------------------
    /** Returns a pointer to the Buffer (used in the Plain C API), associated
     *  with this class.
     *
     *  @return  void pointer.
     *  @ingroup GroupSynview_LvBuffer_Cpp
     */
    void* GetUserPtr();

private:
    LvBuffer();
    ~LvBuffer();
    LvHBuffer m_hBuffer;
    void* m_pUserPointer;
};

//##############################################################################
/** The LvEvent class.
 *  @note   For all the SynView module classes you cannot use the new and delete
 *           operators directly (the constructor and destructor are private).
 *           Instead, the static methods for opening and closing the class instance
 *           assure that if the opening is successful, you get a valid pointer, otherwise
 *           you get a NULL pointer. Also, the closing functions set the pointer back to
 *           NULL. Another advantage is that these functions return a status value, which
 *           can clarify the error nature, if the opening or closing fails.
 */

class LvEvent: public LvModule
{
public:
  
    //------------------------------------------------------------------------------
    /** Creates the LvEvent class instance for specified LvSystem module.
     *
     *  @param   pSystem    Owner LvSystem instance.
     *  @param   EventType  One of the @ref LvEventType.
     *  @param   pEvent     In this parameter a pointer to LvEvent instance is returned.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvSystem::OpenEvent().
     *  @ingroup GroupSynview_LvEvent_CPP
     */
    static LvStatus Open (LvSystem* pSystem,
                          LvEventType EventType,
                          LvEvent*& pEvent);
  
    //------------------------------------------------------------------------------
    /** Creates the LvEvent class instance for specified LvDevice module.
     *
     *  @param   pDevice    Owner LvDevice instance.
     *  @param   EventType  One of the @ref LvEventType.
     *  @param   pEvent     In this parameter a pointer to LvEvent instance is returned.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvDevice::OpenEvent().
     *  @ingroup GroupSynview_LvEvent_CPP
     */
    static LvStatus Open (LvDevice* pDevice,
                          LvEventType EventType,
                          LvEvent*& pEvent);
  
    //------------------------------------------------------------------------------
    /** Creates the LvEvent class instance for specified LvStream module.
     *
     *  @param   pStream    Owner LvStream instance.
     *  @param   EventType  One of the @ref LvEventType.
     *  @param   pEvent     In this parameter a pointer to LvEvent instance is returned.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvStream::OpenEvent().
     *  @ingroup GroupSynview_LvEvent_CPP
     */
    static LvStatus Open (LvStream* pStream,
                          LvEventType EventType,
                          LvEvent*& pEvent);
  
    //------------------------------------------------------------------------------
    /** Deletes the LvEvent class instance.
     *
     *  @param   pEvent  A pointer to the LvEvent instance, obtained from the
     *                   LvEvent::Open() function. This pointer is assigned NULL after
                         the operation.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvSystem::CloseEvent(), LvDevice::CloseEvent(), LvStream::CloseEvent().
     *  @ingroup GroupSynview_LvEvent_CPP
     */
    static LvStatus Close (LvEvent*& pEvent);
  
    //------------------------------------------------------------------------------
    /** Terminates a single wait in the LvEvent::WaitAndGetData() function.
     *
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvEvent_CPP
     */
    LvStatus Kill ();
  
    //------------------------------------------------------------------------------
    /** Discards all buffers in the output buffer queue (waiting to be delivered
     *  to the application).
     *
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvEvent_CPP
     */
    LvStatus Flush ();
  
    //------------------------------------------------------------------------------
    /** Waits for the event and gets its data in one atomic operation. Use this function
     *  only for events other than LvEventType_NewBuffer, for the the LvEventType_NewBuffer
     *  event type use the LvEvent::WaitAndGetNewBuffer() function instead. Do not use
     *  this function if you use the callback - see @ref LvEvent::SetCallback() or
     *  @ref LvEvent::SetCallbackNewBuffer().
     *
     *  @param   pBuffer Pointer to a user allocated buffer to receive the event data.
     *                   The buffer can be parsed by the LvEvent::GetDataInfo() function.
     *  @param   pSize   Size of the buffer must be specified in this parameter and
     *                   after the function returns, the actual size is returned in
     *                   this parameter.
     *  @param   Timeout The wait timeout in milliseconds. The value 0xFFFFFFFF is
     *                   considered as infinite. Note that you can also kill waiting
     *                   from another thread using the LvEvent::Kill() function.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvEvent_CPP
     */
    LvStatus WaitAndGetData (void* pBuffer,
                             size_t* pSize,
                             uint32_t Timeout=0xFFFFFFFF);
  
    //------------------------------------------------------------------------------
    /** Waits for the event and gets its data in one atomic operation. Use this function
     *  only for events of the LvEventType_NewBuffer type. Do not use
     *  this function if you use the callback - see @ref LvEvent::SetCallback() or
     *  @ref LvEvent::SetCallbackNewBuffer().
     *
     *  @param   pBuffer  The pointer to the received LvBuffer instance is returned in this parameter.
     *  @param   Timeout  The wait timeout in milliseconds. The value 0xFFFFFFFF is
     *                    considered as infinite. Note that you can also kill waiting
     *                    from another thread using the LvEvent::Kill() function.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvEvent_CPP
     */
    LvStatus WaitAndGetNewBuffer (LvBuffer*& pBuffer,
                                  uint32_t Timeout=0xFFFFFFFF);
  
    //------------------------------------------------------------------------------
    /** Enables to parse the buffer from LvEvent::WaitAndGetData.
     *
     *  @param   pInBuffer  Pointer to a buffer containing event data. This value must
     *                      not be NULL.
     *  @param   InSize     Size of the provided pInBuffer in bytes.
     *  @param   Info       One of the @ref LvEventDataInfo.
     *  @param   pBuffer    Pointer to a user allocated buffer to receive the requested
     *                      information. If this parameter is NULL, pSize will contain
     *                      the minimal size of pBuffer in bytes. If the pType is
     *                      a string, the size includes the terminating 0.
     *  @param   pSize      Size of the buffer is returned in this parameter.
     *  @param   pType      One of the @ref LvInfoDataType.
     *  @param   Param      Additional parameter, if used, its role is explained by the
     *                      @ref LvEventDataInfo.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvEvent_CPP
     */
    LvStatus GetDataInfo (void* pInBuffer,
                          size_t InSize,
                          LvEventDataInfo Info,
                          void* pBuffer, 
                          size_t* pSize,
                          LvInfoDataType* pType = NULL,
                          int32_t Param = 0);
  
    //------------------------------------------------------------------------------
    /** Puts a new event to Event ouptut queue. This function can be used only for
     *  user-defined events.
     *
     *  @param   pBuffer  Pointer to event data.
     *  @param   Size     Size of the event data.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvEvent_CPP
     */
    LvStatus PutData (void* pBuffer,
                      size_t Size);
  
    //------------------------------------------------------------------------------
    /** Specifies a callback function for the event thread. Note that the callback function
     *  cannot be a method of a class.
     *
     *  @param   pFunction  The callback function in the forms of @ref LvEventCallbackFunct.
     *  @param   pUserParam User parameter, which will be passed to each callback call.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvEvent_CPP
     */
    LvStatus SetCallback (LvEventCallbackFunct pFunction,
                          void* pUserParam);
  
    //------------------------------------------------------------------------------
    /** Specifies a callback function for the thread of the Event of the @ref LvEventType_NewBuffer.
     *  Once the application specifies this callback, it becomes responsible for returning
     *  the image buffers to the input buffer pool. Note that the callback function
     *  cannot be a method of a class.
     *
     *  @param   pFunction  The callback function in the forms of @ref LvEventCallbackNewBufferFunct.
     *  @param   pUserParam User parameter, which will be passed to each callback call.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvEvent_CPP
     */
    LvStatus SetCallbackNewBuffer (LvEventCallbackNewBufferFunct pFunction,
                                   void* pUserParam);
  
    //------------------------------------------------------------------------------
    /** Starts an internal thread, which waits for events and passes them to specified
     *  callback function. When the thread is started, the application must no longer
     *  call the LvEvent::WaitAndGetData() or LvEvent::WaitAndGetNewBufer() functions -
     *  this is called internally in the thread and upon return from this function a
     *  callback function is called.
     *
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvEvent_CPP
     */
    LvStatus StartThread ();
  
    //------------------------------------------------------------------------------
    /** Stops the event internal thread.
     *
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvEvent_CPP
     */
    LvStatus StopThread  ();

    //------------------------------------------------------------------------------
    /** Indicates that the callback function for LvEventSetCallback or 
    *  LvEventSetCallbackNewBuffer needs to exit as soon as possible
    *  This can be used in user callback functions to speed up closing
    *
    *  @return  Returns the must exit status of the callback function of LvEventType_NewBuffer
    *  @ingroup GroupSynview_LvEvent_CPP
    */
    bool CallbackMustExit();
  
    //------------------------------------------------------------------------------
    /** Returns a handle of the Event (used in the Plain C API), associated
     *  with this class.
     *
     *  @return  The Plain C API handle.
     *  @ingroup GroupSynview_LvEvent_CPP
     */
    LvHEvent GetHandle();
  

private:
    LvEvent();
    ~LvEvent();
    LvHEvent m_hEvent;
};

//##############################################################################
/** The LvRenderer class.
 *  @note   For all the SynView module classes you cannot use the new and delete
 *           operators directly (the constructor and destructor are private).
 *           Instead, the static methods for opening and closing the class instance
 *           assure that if the opening is successful, you get a valid pointer, otherwise
 *           you get a NULL pointer. Also, the closing functions set the pointer back to
 *           NULL. Another advantage is that these functions return a status value, which
 *           can clarify the error nature, if the opening or closing fails.
 */

class LvRenderer: public LvModule
{
public:
    /** Creates the LvRenderer class instance for image display. The renderer attepts to load
     *  the sv.synview.display library. In case of SynView installation in an operating
     *  system without possibility to graphically display (for example Linux without
     *  XWindows), the load of this library fails and the calls to Renderer functions
     *  will return errors.
     *
     *  @param   pStream      A pointer to the LvStream instance, obtained from the
     *                        LvStream::Open() function.
     *  @param   pRenderer    In this parameter the pointer to the LvRenderer is returned.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvStream::OpenRenderer().
     *  @ingroup GroupSynview_LvRenderer_Cpp
     */
    static LvStatus Open (LvStream* pStream, LvRenderer*& pRenderer);
  
    //------------------------------------------------------------------------------
    /** Starts the LvRenderer class instance.
     *
     *  @param   pRenderer  A pointer to the LvRenderer instance, obtained from the
     *                      LvRenderer::Open() function. This pointer is set to NULL after close.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvStream::StartRenderer().
     *  @ingroup GroupSynview_LvRenderer_Cpp
     */
    static LvStatus Start (LvRenderer*& pRenderer);

    //------------------------------------------------------------------------------
    /** Stops the LvRenderer class instance.
     *
     *  @param   pRenderer  A pointer to the LvRenderer instance, obtained from the
     *                      LvRenderer::Open() function. This pointer is set to NULL after close.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvStream::StopRenderer().
     *  @ingroup GroupSynview_LvRenderer_Cpp
     */
    static LvStatus Stop (LvRenderer*& pRenderer);

    //------------------------------------------------------------------------------
    /** Deletes the LvRenderer class instance.
     *
     *  @param   pRenderer  A pointer to the LvRenderer instance, obtained from the
     *                      LvRenderer::Open() function. This pointer is set to NULL after close.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @sa      LvStream::CloseRenderer().
     *  @ingroup GroupSynview_LvRenderer_Cpp
     */
    static LvStatus Close (LvRenderer*& pRenderer);
  
    #ifdef _WIN32
        //------------------------------------------------------------------------------
        /** Sets the target window, in which the renderer has to display. Note that
         *  the application itself assure any repainting (when the window need to be
         *  repainted due to a movement of overlapping) - use LvRenderer::Repaint()
         *  in such case.
         *
         *  @param   hWnd       Handle to the window.
         *  @return  Returns the @ref LvStatus value indicating the success of the
         *           requested operation. See @ref GroupSynview_LvStatus.
         *  @ingroup GroupSynview_LvRenderer_Cpp
         */
        LvStatus SetWindow (HWND hWnd);
    #else
        //------------------------------------------------------------------------------
        /** Sets the target window, in which the renderer has to display. Note that
         *  the application itself assure any repainting (when the window need to be
         *  repainted due to a movement of overlapping) - use LvRenderer::Repaint()
         *  in such case.
         *
         *  @param   pDisplay   Pointer to the display. It is defined as void* in order
         *                      to make SynView header files independent on XWindows.
         *  @param   hWindow    Handle to the window.
         *  @return  Returns the @ref LvStatus value indicating the success of the
         *           requested operation. See @ref GroupSynview_LvStatus.
         *  @ingroup GroupSynview_LvRenderer_Cpp
         */
        LvStatus SetWindow (void* pDisplay,
                            int64_t hWindow);
    #endif

    //------------------------------------------------------------------------------
    /** Displays the image. The image display mode is set by Renderer features,
     *  see @ref LvRendererFtr.
     *
     *  @param   pBuffer      Pointer to the LvBuffer to be displayed.
     *  @param   RenderFlags  Zero or a combination of @ref LvRenderFlags.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvRenderer_Cpp
     */
    LvStatus DisplayImage (LvBuffer* pBuffer,
                           uint32_t RenderFlags=0);
  
    //------------------------------------------------------------------------------
    /** Repaints the contents of the display window. In order to be able to repaint,
     *  all images to be displayed must be still held by the application,
     *  i.e. must not be returned to the input buffer pool. See also
     *  @ref LvStream_LvPostponeQueueBuffers feature.
     *  A typical usage of this function is in the WM_PAINT handler in a Windows
     *  application.
     *
     *  @param   RenderFlags  Zero or a combination of @ref LvRenderFlags.
     *  @return  Returns the @ref LvStatus value indicating the success of the
     *           requested operation. See @ref GroupSynview_LvStatus.
     *  @ingroup GroupSynview_LvRenderer_Cpp
     */
    LvStatus Repaint (uint32_t RenderFlags=0);
    
    //------------------------------------------------------------------------------
    /** Returns a handle of the Renderer (used in the Plain C API), associated
     *  with this class.
     *
     *  @return  The Plain C API handle.
     *  @ingroup GroupSynview_LvRenderer_Cpp
     */
    LvHRenderer GetHandle();
  
private:
    LvRenderer();
    ~LvRenderer();
    LvHRenderer m_hRenderer;
};

//##############################################################################
#ifdef LVAVISAVER
    class LvAviSaver: public LvModule
    {
    public:
        static LvStatus Open (LvStream* pStream, LvAviSaver*& pAviSaver);
        static LvStatus Close (LvAviSaver*& pAviSaver);
        LvStatus StoreImage (LvBuffer* pBuffer,
                             uint32_t Options);
        LvStatus OpenFile (const char* pName,
                           uint32_t Options);

        LvStatus CloseFile ();
        LvStatus SaveSettings (const char* pId,
                               const char* pFileName,
                               uint32_t Options);
        LvStatus LoadSettings (const char* pId,
                               const char* pFileName,
                               uint32_t Options);
        LvHAviSaver GetHandle();

    private:
        LvAviSaver();
        ~LvAviSaver();
        LvHAviSaver m_hAviSaver;
    };
    
#endif


// for backward compatibility only, for new projects use LvLibrary methods
LV_DLLENTRY uint32_t LV_STDC LvGetVersion();
LV_DLLENTRY LvStatus LV_STDC LvOpenLibrary();
LV_DLLENTRY LvStatus LV_STDC LvCloseLibrary();
LV_DLLENTRY void LV_STDC LvGetErrorMessage (LvStatus Error, char* pMessage, size_t Size);
LV_DLLENTRY void LV_STDC LvGetLastErrorMessage (char* pMessage, size_t Size);
LV_DLLENTRY void LV_STDC LvLog (const char* pLogMessage);
#ifndef SIMPLON_CLASS_IMPLEMENTATION
    LV_DLLENTRY LvStatus LV_STDC LvGetLibInfo (LvEnum Info, int32_t* pInfo, int32_t Param = 0);
    LV_DLLENTRY LvStatus LV_STDC LvGetLibInfoStr (LvEnum Info, char* pInfoStr, size_t Size, int32_t Param = 0);
    LV_DLLENTRY LvStatus LV_STDC LvGetLibInfoStrSize (LvEnum Info, size_t* pSize, int32_t Param = 0);
#endif
LV_DLLENTRY LvStatus LV_STDC LvUpdateSystemList ();
LV_DLLENTRY LvStatus LV_STDC LvGetNumberOfSystems (uint32_t* pNumberOfSystems);
LV_DLLENTRY LvStatus LV_STDC LvGetSystemId (uint32_t Index, char* pSystemId, size_t Size);
LV_DLLENTRY LvStatus LV_STDC LvGetSystemIdSize (uint32_t Index, size_t* pSize);
#ifdef LV_USE_STDLIB
    LvStatus LvGetLibInfoStr (LvEnum Info, std::string& sInfo, int32_t Param = 0);
    std::string LvGetErrorMessage (LvStatus Error);
    std::string LvGetLastErrorMessage ();
    LvStatus LvGetSystemId (uint32_t Index, std::string& sSystemId);
#endif

#endif
