/***************************************************************************
 *   Copyright (C) 2021                                                  *
 *   NET GmbH                                                      		   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <iostream>
#include <string>
#include <sstream>
#include <unistd.h>

#include "ICubeHID_Defines.h"  
#include "ICubeHID.h"


using namespace std;


//---------------------------------------------------------------------------
// DEFINES

#define VENDOR_ID   0x20F1
#define PRODUCT_ID  0x0203


#define FREERUN 					0
#define TRIGGER_SW 					1
#define TRIGGER_HW 					2
#define DO_SOFTWARE_TRIGGER 		3
#define GETMODE						4
#define SET_TRIGGER_DELAY			5
#define GET_TRIGGER_DELAY			6
#define SET_WHITE_BALANCE_ONCE		7
#define SET_WHITE_BALANCE_CONTINOUS	8
#define SET_WHITE_BALANCE_OFF		9

//---------------------------------------------------------------------------
// GLOBALS

int cam_index = 0;

ICube_HID *icube_hid;

//---------------------------------------------------------------------------
// PROTOTYPES

void DecodeError(int nErrCode);
void Show_Usage(std::string name);
void ExecuteCMD(int cmd,std::string value);
void ShowMissing(std::string name);
//---------------------------------------------------------------------------
//
int main(int argc, char* argv[]) {
	    
    std::string name = argv[0];
	std::string val = " ";
	   	   
    // we need at least one command line parameter   
    if (argc < 2) {
        ShowMissing(name);
        return 1;
    }
    
    if(argv[2] != NULL){
		val = argv[2];
	}

    int n_mode = 0xff;
	
    while ((++argv)[0])
    {
	if (argv[0][0] == '-' )
	{
	    switch (argv[0][1])  {
			case 'c':
					cout << "Get camera trigger delay \n";
				    n_mode = GET_TRIGGER_DELAY;
				    break;
			case 'e':
				     if (argc < 3) { // additional parameter val needed
						ShowMissing(name);
						return 1;
					}
					cout << "Set camera trigger delay to: " << val << "\n";
				    n_mode = SET_TRIGGER_DELAY;
				    break;
		    case 'g':
				    cout << "Get current camera mode\n";
				    n_mode = GETMODE;
				    break;
		    case 'f':
				    cout << "Change camera to freerun mode\n";
				    n_mode = FREERUN;
				    break;
		    case 's':
				    cout << "Change camera to software trigger mode\n"; 
				    n_mode = TRIGGER_SW;
				    break;
		    case 't':
				    cout << "Change camera to hardware trigger mode\n";
				    n_mode = TRIGGER_HW;
				    break;
		    case 'd':
				    cout << "Execute software trigger command\n";
				    n_mode = DO_SOFTWARE_TRIGGER;
				    break;	
		    case 'v':
				    cout << "Version is 1.0.0.2 \n";
				    return 0;	
			case 'w':
				    cout << "Execute white balance command once\n";
				    n_mode = SET_WHITE_BALANCE_ONCE;
				    break;	
			case 'x':
				    cout << "Execute white balance command continous\n";
				    n_mode = SET_WHITE_BALANCE_CONTINOUS;
				    break;	
			case 'y':
				    cout << "Execute white balance command off\n";
				    n_mode = SET_WHITE_BALANCE_OFF;
				    break;    		
		    case 'h':
		    default:
				    Show_Usage(name);
				    return 0;				
		}
	    }
		
	}

	if(n_mode == 0xff){
	    cout << name << ": unrecognized option\n";
	    cout << "Try " << name << " -h for more information.\n";
	    return 1;
	}
	
	ExecuteCMD(n_mode,val);

    return 0;
}
//------------------------------------------------------------------------
//
void ShowMissing(std::string name)
{
	cout << name << ": missing file arguments\n";
	cout << "Try " << name << " -h for more information.\n";
}
//------------------------------------------------------------------------
//
void ExecuteCMD(int cmd, std::string value)
{
    // open hid interface
    icube_hid = new ICube_HID();

    uint8_t nNumberOfCameras;
    uint8_t nStartupError = 0;

    int status = icube_hid->Init(VENDOR_ID,PRODUCT_ID,&nNumberOfCameras);
    if(IC_UVC_STATUS_SUCCESS != status)
            nStartupError = 1;

    if(nNumberOfCameras == 0)
           nStartupError = 2;

    if(nStartupError == 0){
        status = icube_hid->Open(cam_index);
        if(IC_UVC_STATUS_SUCCESS != status)
               nStartupError = 3;
    }

    if(nStartupError != 0)
    {
        switch(nStartupError)
        {
        case 1:  cout << "Error:: Init Hid Interface\n"; break;
        case 2:  cout << "Error:: No device connected\n"; break;
        case 3:  cout << "Error:: Open Hid Interface\n";break;
        default: cout << "Error:: unknown";break;
        }
    }
else
{
    int status = 0;
    
    uint32_t nEnVal=0,nTrigVal=0;
    
    int nId,nnId;
    float fValue = 0,f_min = 0,f_max = 0;
    std::stringstream ss;
    
    switch(cmd)
    {
	case SET_WHITE_BALANCE_OFF:	
		// use this mode to switch back from continous mode
		nId = UVC_CAM_REG_WHITE_BALANCE;
		status = icube_hid->SetInt32(cam_index,nId,0);
		if(status  != 0)
			DecodeError(status);
		status = icube_hid->SetFunc(cam_index,IC_UVC_SAVE_USER_MODE,NULL,0);    // save the mode permanently	
		break;		
	case SET_WHITE_BALANCE_ONCE:	
		// once mode falls back to off, when finished
		nId = UVC_CAM_REG_WHITE_BALANCE;
		status = icube_hid->SetInt32(cam_index,nId,1);
		if(status  != 0)
			DecodeError(status);
			
		nEnVal = 1;	
		while(1)
		{
			usleep(1000);
			cout << "Wait for white balance once to be finished...\n";
			icube_hid->GetInt32(cam_index,nId,&nEnVal);
			if(nEnVal == 0)
				break;
		}
		break;	
	case SET_WHITE_BALANCE_CONTINOUS:	
		nId = UVC_CAM_REG_WHITE_BALANCE;
		status = icube_hid->SetInt32(cam_index,nId,2);
		if(status  != 0)
			DecodeError(status);
		status = icube_hid->SetFunc(cam_index,IC_UVC_SAVE_USER_MODE,NULL,0);    // save the mode permanently
		break;		
	case GET_TRIGGER_DELAY:	
		nId = UVC_CAM_REG_TRIGGER_DELAY;
		status = icube_hid->GetFloat(cam_index,nId,&fValue);
		if(status  == 0)
			cout << "Current trigger delay: " << fValue <<"\n";
		break;
	case SET_TRIGGER_DELAY:
		
		// convert to float
		ss.str(value);
		ss >> fValue;
		
		nId = UVC_CAM_REG_TRIGGER_DELAY;
		// get parameter range
		nnId = nId | 0x2000;  // get min
		status = icube_hid->GetFloat(cam_index,nnId,&f_min);
		if(status  != 0)
			DecodeError(status);

		nnId = nId | 0x3000;  // get max
		status = icube_hid->GetFloat(cam_index,nnId,&f_max);
		if(status  != 0)
			DecodeError(status);
		
		if(fValue >= f_max){
			cout << "value out of range (min)" << f_min << " (max)" << f_max <<"\n";
		}
		else
		{
			status = icube_hid->SetFloat(cam_index,nId,fValue);
			if(status  != 0)
				DecodeError(status);
				
			status = icube_hid->SetFunc(cam_index,IC_UVC_SAVE_USER_MODE,NULL,0);    // save the mode permanently	
		}
		
		break;
	case GETMODE:
	    
	    status = icube_hid->GetInt32(cam_index,UVC_CAM_REG_TRIG_ENABLE,&nEnVal);
	    if(status  != 0)
		DecodeError(status);

	    if(nEnVal != 0){
		status = icube_hid->GetInt32(cam_index,UVC_CAM_REG_TRIG_MODE,&nTrigVal);
		if(status  != 0)
		    DecodeError(status);

		if(nTrigVal == 0) cout << "Current mode:: software trigger\n";
		else              cout << "Current mode:: hardware trigger\n";
	    }
	    else                    // free run
	    {
		cout << "Current mode:: free run\n";
	    }
		break;
	case FREERUN:  
		status = icube_hid->SetInt32(cam_index,UVC_CAM_REG_TRIG_ENABLE,0);
		 if(status  != 0)
			DecodeError(status);
		status = icube_hid->SetFunc(cam_index,IC_UVC_SAVE_USER_MODE,NULL,0);    // save the mode permanently
		break;
	case TRIGGER_SW:  
		 status = icube_hid->SetInt32(cam_index,UVC_CAM_REG_TRIG_MODE,0); 
		 if(status  != 0)
			DecodeError(status);
		status = icube_hid->SetInt32(cam_index,UVC_CAM_REG_TRIG_ENABLE,1);
		 if(status  != 0)
			DecodeError(status);
		status = icube_hid->SetFunc(cam_index,IC_UVC_SAVE_USER_MODE,NULL,0);    // save the mode permanently
		break;
	case TRIGGER_HW:  
		 status = icube_hid->SetInt32(cam_index,UVC_CAM_REG_TRIG_MODE,1);
		if(status  != 0)
			DecodeError(status);
		status = icube_hid->SetInt32(cam_index,UVC_CAM_REG_TRIG_ENABLE,1);
		 if(status  != 0)
			DecodeError(status);
		status = icube_hid->SetFunc(cam_index,IC_UVC_SAVE_USER_MODE,NULL,0);    // save the node permanently
		break;
	case DO_SOFTWARE_TRIGGER:  
		status = icube_hid->SetGlobal(cam_index,UVC_CAM_REG_TRIG_SW_DO,0);
		break;
	default: cout << "Error:: unknown\n";
		break;
	}
	
	 if(status  != 0)
			DecodeError(status);
	 else
		cout << "Success\n";
    }
    
    icube_hid->Close(cam_index);
}
//------------------------------------------------------------------------
//
void Show_Usage(std::string name)
{
    std::cerr << "Usage: " << name << " [OPTION] \n"
              << "\t-h		print this help and exit\n"
              << "\t-v		print version information and exit\n"
              << "\t-c 		get trigger delay\n"
              << "\t-e [value]      set trigger delay\n"
	          << "\t-g		get current camera mode\n"
              << "\t-f		switch camera to freerun mode\n"
              << "\t-s		switch camera to software trigger mode\n"
              << "\t-t		switch camera to hardware trigger mode\n"
              << "\t-d		execute a software trigger command\n"
              << "\t-w		set white balance to once mode (color models only)\n"
              << "\t-x		set white balance to continous mode (color models only)\n"
              << "\t-y	set white balance to off mode (color models only)\n"
              << " \nOnly change trigger modes, when camera is NOT streaming.\n"
              << "Change white balance modes, when camera is streaming.\n"
              << "Modes, values are saved permanently.\n"
              << std::endl;
	      
}
//------------------------------------------------------------------------
//
void DecodeError(int nErrCode)
{
    switch(nErrCode)
    {
    case IC_UVC_STATUS_UNKNOWN_ERROR:          cout << "Error:: IC_UVC_STATUS_UNKNOWN_ERROR\n"; break;
    case IC_UVC_STATUS_BAD_ARGUMENT:           cout << "Error:: IC_UVC_STATUS_BAD_ARGUMENT\n"; break;
    case IC_UVC_STATUS_INVALID_CMD_ID:         cout << "Error:: IC_UVC_STATUS_INVALID_CMD_ID\n"; break;
    case IC_UVC_STATUS_NOT_IMPLEMENTED:        cout << "Error:: IC_UVC_STATUS_NOT_IMPLEMENTED\n"; break;
    case IC_UVC_STATUS_NOT_AVAILABLE:          cout << "Error:: IC_UVC_STATUS_NOT_AVAILABLE\n"; break;

    case IC_UVC_STATUS_INVALID_DATA_BLOCK:     cout << "Error:: IC_UVC_STATUS_INVALID_DATA_BLOCK\n"; break;

    case IC_UVC_STATUS_INVALID_CHUNK_ID :      cout << "Error:: IC_UVC_STATUS_INVALID_CHUNK_ID\n"; break;
    case IC_UVC_STATUS_DATA_BLOCK_TOO_LARGE:   cout << "Error:: IC_UVC_STATUS_DATA_BLOCK_TOO_LARGE\n"; break;
    case IC_UVC_STATUS_MISSSING_DATA_CHUNKS:   cout << "Error:: IC_UVC_STATUS_MISSSING_DATA_CHUNKS\n"; break;
    case IC_UVC_STATUS_SENSOR_DISABLED:        cout << "Error:: IC_UVC_STATUS_SENSOR_DISABLED\n"; break;

    case IC_UVC_STATUS_USB_RD_ERR:             cout << "Error:: IC_UVC_STATUS_USB_RD_ERR\n"; break;
    case IC_UVC_STATUS_USB_WR_ERR:             cout << "Error:: IC_UVC_STATUS_USB_WR_ERR\n"; break;
    case IC_UVC_STATUS_FLASH_RD_ERR:           cout << "Error:: IC_UVC_STATUS_FLASH_RD_ERR\n"; break;
    case IC_UVC_STATUS_FLASH_WR_ERR:           cout << "Error:: IC_UVC_STATUS_FLASH_WR_ERR\n"; break;

    case IC_UVC_STATUS_OUT_OF_RANGE:           cout << "Error:: IC_UVC_STATUS_OUT_OF_RANGE\n"; break;
    case IC_UVC_STATUS_HW_ERR:                 cout << "Error:: IC_UVC_STATUS_HW_ERR\n"; break;
    case IC_UVC_STATUS_HW_ERR_FPGA:            cout << "Error:: DC2_STATUS_HW_ERR_FPGA\n"; break;

    case IC_UVC_STATUS_NOT_IF_STREAMING_ERROR: cout << "Error:: IC_UVC_STATUS_NOT_IF_STREAMING_ERROR\n"; break;

    case HID_ERROR_TIMEOUT:                    cout << "Error:: HID_ERROR_TIMEOUT\n"; break;
    case HID_ERROR_PROTOCOL_WRONG_ID:          cout << "Error:: HID_ERROR_PROTOCOL_WRONG_ID\n"; break;

    default:                                   cout << "Error:: unknown error code\n"; break;
    }

    
}
