/***************************************************************************
 *   Copyright (C) 2019                                                    *
 *   NET GmbH                                                              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "ICubeHID.h"
#include "ICubeHID_Defines.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define HID_REPORT_ID   1
#define HID_EVENT_ID    2

#define HID_WAIT_TIME	5000	// timeout for hid communication


//------------------------------------------------------------------------
// 
int32_t CALLBACK ICube_HID::ReceiveCallbackHid(uint8_t buffer[], void *context) {

    ICube_HID* pICube=(ICube_HID*)context;

    if(buffer[0] == HID_REPORT_ID)
    {
       if(buffer[1] != pICube->n_id){
           #ifdef DBG_UVC_INTERFACE
           printf("cmd rd --> wrong id %d %d \n",buffer[1],pICube->n_id);
           #endif
           pICube->n_cmd_error = HID_ERROR_PROTOCOL_WRONG_ID;
       }
       else if(buffer[2] != 0){
           #ifdef DBG_UVC_INTERFACE
           printf("cmd rd --> protocol error %d \n",buffer[2]);
           #endif
           pICube->n_cmd_error = buffer[2];
       }
       else
       {
           pICube->n_cmd_error = 0;
           memcpy(pICube->rcv_buf,&buffer[pICube->n_data_offset],64);
           #ifdef DBG_UVC_INTERFACE
           printf("cmd rd ok\n");
           #endif
       }
       #ifdef __linux__
       pthread_cond_signal(&pICube->condition);
       #endif
       #ifdef __WIN32__
       SetEvent(pICube->m_hndEvent);
       #endif
    }
	
    else if(buffer[0] == HID_EVENT_ID)
    {
        //printf("event rd %x %x %x %x \n",buffer[0],buffer[1],buffer[2],buffer[3]);
        memcpy(&pICube->n_event[0],&buffer[0],12);
    }
    else
    {
        printf("unknown id \n");
    }

return 0;
}
//------------------------------------------------------------------------
// ctor
ICube_HID::ICube_HID()
{
    n_mode = 0;
    
    n_data_offset = 4;
    n_id = 0;
    n_cmd_error = 0;

    memset(n_event,0,2);
    #ifdef __linux__
    pthread_cond_init(&condition, NULL);
    pthread_mutex_init(&signalMutex, NULL); 
    #endif
    #ifdef __WIN32__
    m_hndEvent = CreateEvent(NULL, TRUE, FALSE, NULL);
    #endif
}
//------------------------------------------------------------------------
// dtor
ICube_HID::~ICube_HID()
{
   NET_HID_DeInit();
   #ifdef __linux__
   pthread_cond_destroy(&condition);
   pthread_mutex_destroy(&signalMutex);
   #endif
   #ifdef __WIN32__
   CloseHandle(m_hndEvent);
   #endif
}
//------------------------------------------------------------------------
//
int ICube_HID::Init(uint16_t vid,uint16_t pid,uint8_t *n_num_of_cams)
{
   uint8_t stat = NET_HID_Init();

   if(stat == IC_UVC_STATUS_SUCCESS)
   {
        *n_num_of_cams = NET_HID_Enumerate(vid,pid);
   }
   return stat;
}
//------------------------------------------------------------------------
//
int ICube_HID::GetVersion(uint8_t cam_index,unsigned char nID,char* version, unsigned char length)
{
    if(nID == 0)
    {
        NET_HID_GetVersion(version, length);
        return 0;
    }
    else if(nID == IC_UVC_GET_HID_PROT_VERSION)
    {
        unsigned char data[64];
        n_data_offset = 3;
        GetFunc(cam_index,IC_UVC_GET_HID_PROT_VERSION,data,2);
        snprintf(version, (length-1), "%d.%d",data[0],data[1]);
    }
    else
    {
        unsigned char data[64];
        data[0] = 0;
        data[1] = (unsigned char) UVC_CAM_REG_FIRMWARE_VERS;
        data[2] = (unsigned char)(UVC_CAM_REG_FIRMWARE_VERS>> 8);
        n_data_offset = 4;
        GetFunc(cam_index,IC_UVC_GET_VALUE,data,5);
        snprintf(version, (length-1), "%d.%d.%d.%d",data[0],data[1],data[2],(data[3]<<8) | data[4]);
    }
 return 0;   
}
//------------------------------------------------------------------------
//
int ICube_HID::Open(uint8_t cam_index)
{
   int status = NET_HID_Open(cam_index);
   if(status  != 0)
      return 1;
       
   return NET_HID_SetCallbackHid(cam_index,&ReceiveCallbackHid, this);
}
//------------------------------------------------------------------------
//
int ICube_HID::Close(uint8_t cam_index)
{
    return NET_HID_Close(cam_index);
}
//------------------------------------------------------------------------
//
int ICube_HID::SetGlobal(uint8_t cam_index,uint16_t nID,uint32_t nValue)
{
    unsigned char data[64];
    data[0] = 0;  // mode not used
    data[1] = nID;
    data[2] = nID >> 8;
    data[3] = nValue;
    data[4] = nValue>>8;
    data[5] = nValue>>16;
    data[6] = nValue>>24;

    n_id = IC_UVC_SET_VALUE; // save global
    
    int status = NET_HID_Report(cam_index,HID_REPORT_ID,IC_UVC_SET_VALUE,data,8);

    #ifdef __linux__
    struct timespec timeToWait;
    struct timeval tp;
    gettimeofday(&tp,NULL);
    timeToWait.tv_sec  = tp.tv_sec;
    timeToWait.tv_nsec = tp.tv_usec * 1000;
    timeToWait.tv_sec += (HID_WAIT_TIME/1000);
    int erg = pthread_cond_timedwait(&condition, &signalMutex, &timeToWait);
    if(erg != 0) return HID_ERROR_TIMEOUT;  // timeout
    #endif
    #ifdef __WIN32__
    if (WaitForSingleObject(m_hndEvent, HID_WAIT_TIME) != WAIT_TIMEOUT){
        ResetEvent(m_hndEvent);
    } else return HID_ERROR_TIMEOUT;  // timeout
    #endif

    if(n_cmd_error != 0)
       return n_cmd_error;

    return status;
}
//------------------------------------------------------------------------
//
int ICube_HID::GetGlobal(uint8_t cam_index,uint16_t nID,uint32_t* nValue)
{
    unsigned char data[64];
    data[0] = 0;  // mode not used
    data[1] = nID;
    data[2] = nID >> 8;

    n_id = IC_UVC_GET_VALUE; // save global
    n_data_offset = 4;

    int status = NET_HID_Report(cam_index,HID_REPORT_ID,IC_UVC_GET_VALUE,data, 5);

    #ifdef __linux__
    struct timespec timeToWait;
    struct timeval tp;
    gettimeofday(&tp,NULL);
    timeToWait.tv_sec  = tp.tv_sec;
    timeToWait.tv_nsec = tp.tv_usec * 1000;
    timeToWait.tv_sec += (HID_WAIT_TIME/1000);
    int erg = pthread_cond_timedwait(&condition, &signalMutex, &timeToWait);
    if(erg != 0) return HID_ERROR_TIMEOUT;  // timeout
    #endif
    #ifdef __WIN32__
    if (WaitForSingleObject(m_hndEvent, HID_WAIT_TIME) != WAIT_TIMEOUT){
        ResetEvent(m_hndEvent);
    } else return HID_ERROR_TIMEOUT;  // timeout
    #endif

    if(n_cmd_error != 0)
       return n_cmd_error;

    if(status  == 0)
    {
        *nValue = (rcv_buf[0]) | (rcv_buf[1]<<8) | (rcv_buf[2]<<16) | (rcv_buf[3]<<24);
    }
    return status;
}
//------------------------------------------------------------------------
//
int ICube_HID::SetInt32(uint8_t cam_index,uint16_t nID,uint32_t nValue)
{
    unsigned char data[64];
    data[0] = n_mode;
    data[1] = nID;
    data[2] = nID >> 8;
    data[3] = nValue;
    data[4] = nValue>>8;
    data[5] = nValue>>16;
    data[6] = nValue>>24;

    n_id = IC_UVC_SET_VALUE; // save global

    int status = NET_HID_Report(cam_index,HID_REPORT_ID,IC_UVC_SET_VALUE,data, 8);

    #ifdef __linux__
    struct timespec timeToWait;
    struct timeval tp;
    gettimeofday(&tp,NULL);
    timeToWait.tv_sec  = tp.tv_sec;
    timeToWait.tv_nsec = tp.tv_usec * 1000;
    timeToWait.tv_sec += (HID_WAIT_TIME/1000);
    int erg = pthread_cond_timedwait(&condition, &signalMutex, &timeToWait);
    if(erg != 0) return HID_ERROR_TIMEOUT;  // timeout
    #endif
    #ifdef __WIN32__
    if (WaitForSingleObject(m_hndEvent, HID_WAIT_TIME) != WAIT_TIMEOUT){
        ResetEvent(m_hndEvent);
    } else return HID_ERROR_TIMEOUT;  // timeout
    #endif

    if(n_cmd_error != 0)
       return n_cmd_error;

    return status;
}
//------------------------------------------------------------------------
//
int ICube_HID::GetInt32(uint8_t cam_index,uint16_t nID,uint32_t* nValue)
{
    unsigned char data[64];
    data[0] = n_mode;
    data[1] = nID;
    data[2] = nID >> 8;

    n_id = IC_UVC_GET_VALUE; // save global
    n_data_offset = 4;
    
    int status = NET_HID_Report(cam_index,HID_REPORT_ID,IC_UVC_GET_VALUE,data, 6);

    #ifdef __linux__
    struct timespec timeToWait;
    struct timeval tp;
    gettimeofday(&tp,NULL);
    timeToWait.tv_sec  = tp.tv_sec;
    timeToWait.tv_nsec = tp.tv_usec * 1000;
    timeToWait.tv_sec += (HID_WAIT_TIME/1000);
    int erg = pthread_cond_timedwait(&condition, &signalMutex, &timeToWait);
    if(erg != 0) return HID_ERROR_TIMEOUT;  // timeout
    #endif
    #ifdef __WIN32__
    if (WaitForSingleObject(m_hndEvent, HID_WAIT_TIME) != WAIT_TIMEOUT){
        ResetEvent(m_hndEvent);
    } else return HID_ERROR_TIMEOUT;  // timeout
    #endif

    if(n_cmd_error != 0)
       return n_cmd_error;

    if(status  == 0)
    {
        *nValue = (rcv_buf[1]) | (rcv_buf[2]<<8) | (rcv_buf[3]<<16) | (rcv_buf[4]<<24);
    }
    return status;
}
//------------------------------------------------------------------------
//
int ICube_HID::SetFloat(uint8_t cam_index,uint16_t nID,float fValue)
{
    unsigned char data[64];
    data[0] = n_mode;
    data[1] = nID;
    data[2] = nID >> 8;

    memcpy(&data[3],&fValue,4);

    n_id = IC_UVC_SET_VALUE; // save global

    int status = NET_HID_Report(cam_index,HID_REPORT_ID,IC_UVC_SET_VALUE,data, 8);

    #ifdef __linux__
    struct timespec timeToWait;
    struct timeval tp;
    gettimeofday(&tp,NULL);
    timeToWait.tv_sec  = tp.tv_sec;
    timeToWait.tv_nsec = tp.tv_usec * 1000;
    timeToWait.tv_sec += (HID_WAIT_TIME/1000);
    int erg = pthread_cond_timedwait(&condition, &signalMutex, &timeToWait);
    if(erg != 0) return HID_ERROR_TIMEOUT;  // timeout
    #endif
    #ifdef __WIN32__
    if (WaitForSingleObject(m_hndEvent, HID_WAIT_TIME) != WAIT_TIMEOUT){
        ResetEvent(m_hndEvent);
    } else return HID_ERROR_TIMEOUT;  // timeout
    #endif

    if(n_cmd_error != 0)
       return n_cmd_error;

    return status;
}
//------------------------------------------------------------------------
//
int ICube_HID::GetFloat(uint8_t cam_index,uint16_t nID,float* fValue)
{
    unsigned char data[64];

    data[0] = n_mode;
    data[1] = nID;
    data[2] = nID >> 8;

    n_id = IC_UVC_GET_VALUE; // save global
    n_data_offset = 4;

    int status = NET_HID_Report(cam_index,HID_REPORT_ID,IC_UVC_GET_VALUE,data, 8);

    #ifdef __linux__
    struct timespec timeToWait;
    struct timeval tp;
    gettimeofday(&tp,NULL);
    timeToWait.tv_sec  = tp.tv_sec;
    timeToWait.tv_nsec = tp.tv_usec * 1000;
    timeToWait.tv_sec += (HID_WAIT_TIME/1000);
    int erg = pthread_cond_timedwait(&condition, &signalMutex, &timeToWait);
    if(erg != 0) return HID_ERROR_TIMEOUT;  // timeout
    #endif
    #ifdef __WIN32__
    if (WaitForSingleObject(m_hndEvent, HID_WAIT_TIME) != WAIT_TIMEOUT){
        ResetEvent(m_hndEvent);
    } else return HID_ERROR_TIMEOUT;  // timeout
    #endif

    if(n_cmd_error != 0)
       return n_cmd_error;

    if(status  == 0)
    {
       memcpy(fValue,&rcv_buf[1],4);
    }
    return status;
}
//------------------------------------------------------------------------
//
int ICube_HID::SetFunc(uint8_t cam_index,uint16_t nFuncID,unsigned char* ndata,int len)
{
   unsigned char data[64];
   for(int i=0;i<len;i++){
      data[i]=ndata[i];
   }
     
   n_id = nFuncID; // save global
   
   int status = NET_HID_Report(cam_index,HID_REPORT_ID,nFuncID,data, 4);

    #ifdef __linux__
    struct timespec timeToWait;
    struct timeval tp;
    gettimeofday(&tp,NULL);
    timeToWait.tv_sec  = tp.tv_sec;
    timeToWait.tv_nsec = tp.tv_usec * 1000;
    timeToWait.tv_sec += (HID_WAIT_TIME/1000);
    int erg = pthread_cond_timedwait(&condition, &signalMutex, &timeToWait);
    if(erg != 0) return HID_ERROR_TIMEOUT;  // timeout
    #endif
    #ifdef __WIN32__
    if (WaitForSingleObject(m_hndEvent, HID_WAIT_TIME) != WAIT_TIMEOUT){
        ResetEvent(m_hndEvent);
    } else return HID_ERROR_TIMEOUT;  // timeout
    #endif

   if(n_cmd_error != 0)
      return n_cmd_error;

return status;
}
//------------------------------------------------------------------------
//
int ICube_HID::GetFunc(uint8_t cam_index,uint16_t nFuncID,unsigned char* ndata,int len)
{
   unsigned char data[64];

   n_id = nFuncID; // save global
   n_data_offset = 3;

   data[0] = ndata[0];
   data[1] = ndata[1];
   data[2] = ndata[2];

   int status = NET_HID_Report(cam_index,HID_REPORT_ID,nFuncID, data, 4);

    #ifdef __linux__    
    struct timespec timeToWait;
    struct timeval tp;
    gettimeofday(&tp,NULL);
    timeToWait.tv_sec  = tp.tv_sec;
    timeToWait.tv_nsec = tp.tv_usec * 1000;
    timeToWait.tv_sec += (HID_WAIT_TIME/1000);
    int erg = pthread_cond_timedwait(&condition, &signalMutex, &timeToWait);
    if(erg != 0) return HID_ERROR_TIMEOUT;  // timeout
    #endif
    #ifdef __WIN32__
    if (WaitForSingleObject(m_hndEvent, HID_WAIT_TIME) != WAIT_TIMEOUT){
        ResetEvent(m_hndEvent);
    } else return HID_ERROR_TIMEOUT;  // timeout
    #endif

   if(n_cmd_error != 0)
      return n_cmd_error;

   if(status  == 0)
   {
      for(int i=0;i<len;i++)
      {
         ndata[i]=rcv_buf[i];
      }
   }
    
return status;
}
//------------------------------------------------------------------------
// [EOF]
