#include "osdep.h"

// This unit provides some system functions unified for Windows and Linux

#ifdef _WIN32

    #include <windows.h>

    uint32_t OsGetTickCount()
    {
        return GetTickCount();
    }

    void OsSleep(uint32_t ms)
    {
        Sleep(ms);
    }

    void OsInitializeCriticalSection (OS_CRITICAL_SECTION *pCriticalSection)
    {
        InitializeCriticalSection(pCriticalSection);
    }

    void OsDeleteCriticalSection (OS_CRITICAL_SECTION *pCriticalSection)
    {
        DeleteCriticalSection(pCriticalSection);
    }

    void OsEnterCriticalSection (OS_CRITICAL_SECTION *pCriticalSection)
    {
        EnterCriticalSection(pCriticalSection);
    }

    void OsLeaveCriticalSection (OS_CRITICAL_SECTION *pCriticalSection)
    {
        LeaveCriticalSection(pCriticalSection);
    }

    int OsStricmp(const char *str1, const char *str2)
    {
        return stricmp (str1, str2);
    }


#else
    #include <string.h>

    uint32_t OsGetTickCount(void) 
    {
        struct timespec tval;
        if (0==clock_gettime(CLOCK_MONOTONIC, &tval)) 
        {
            return tval.tv_sec*1000 + tval.tv_nsec/1000000;
        }
        else 
        {
            return 0; // what else?
        }
    }

    void OsSleep(uint32_t ms) 
    {
        usleep( 1000 * ms );
    }

    void OsInitializeCriticalSection (OS_CRITICAL_SECTION *pCriticalSection) 
    {
        // Set the mutex as a recursive mutex
        pthread_mutexattr_t mutexattr;
        pthread_mutexattr_init(&mutexattr);
        pthread_mutexattr_settype(&mutexattr, PTHREAD_MUTEX_RECURSIVE_NP);
        pthread_mutex_init(pCriticalSection, &mutexattr);
        pthread_mutexattr_destroy(&mutexattr);
    }

    void OsDeleteCriticalSection (OS_CRITICAL_SECTION *pCriticalSection) 
    {
        pthread_mutex_destroy (pCriticalSection);
    }

    void OsEnterCriticalSection (OS_CRITICAL_SECTION *pCriticalSection) 
    {
        pthread_mutex_lock (pCriticalSection);
    }

    void OsLeaveCriticalSection (OS_CRITICAL_SECTION *pCriticalSection) 
    {
        pthread_mutex_unlock (pCriticalSection);
    }

    int OsStricmp(const char *str1, const char *str2)
    {
        return strcasecmp (str1, str2);
    }

#endif


bool OsFileExists(const char *FileName)
{
    struct stat statinfo;
    return stat (FileName, &statinfo) == 0;
}

bool OsDirExists(const char *FileName)
{
    struct stat statinfo;
    return stat (FileName, &statinfo)==0 && statinfo.st_mode & S_IFDIR;;
}
