// This application has simple GUI with a menu for Open/Close camera and
// Start/Stop the acquisition. The acquired images are processed with the
// OpenCV library - on a ROI of each image is applied EITHER the Canny
// operator - in case the image is acquired or converted to 8-bit pixel format,
// OR the Smooth operator, in case the image is in different that 8-bit
// pixel format.

// To run this sample, you must have the OpenCV library installed,
// at least the version 2.3.1 (see http://en.wikipedia.org/wiki/OpenCV)
// For compilation, the OPENCV environment variable must be created,
// containing the path to the installation folder, for example
// OPENCV=C:\tools\opencv\build
// For execution, the PATH environment variable must include a path
// to the installed OpenCV binaries, for example
// $PATH=$PATH;C:\tools\opencv\build\x64\vc10\bin

// The sv.synview.class.h and sv.synview.class.cpp files are copied to the
// project folder before the compilation.

// All the parts where the OpenCV is used are bracketed by
// the #ifdef USE_OPENCV_PROCESSING macro. If this macro is undefined
// a simple pixel data inversion is done instead of OpenCV processing.

// Important note: If the 10-bit or 12-bit monochrome pixel format is
// processed by OpenCV, the resulting pixel values can be outside the permitted
// pixel range, as OpenCV treats such images as 16-bit. If you copy such image
// back to SynView buffer and apply on such image the LUT from NET Image
// Processing library, you can get a crash caused by the fact that the pixel
// value points outside the LUT.

#define _CRT_SECURE_NO_DEPRECATE // for Visual Studio 8.0 and later

#include <windows.h>
#include <string>
#include <vector>
#include <sv.synview.class.h>

#define USE_OPENCV_PROCESSING

#ifdef USE_OPENCV_PROCESSING
    #include <opencv2/imgproc/imgproc_c.h>
    #include <opencv2/imgproc/imgproc.hpp>
    #include <opencv2/highgui/highgui.hpp>
    // add more OpenCV modules if needed ...
#endif

//=============================================================================
// The CCamera class implements access to one camera. In this simple sample
// it is not necessary; we could use global variables and functions instead.
// However, the CCamera class becomes practical when working with multiple 
// cameras (not demonstrated in this sample).

#define NUMBER_OF_BUFFERS %%Buffers%%

class CCamera
{
public:
    CCamera();
    ~CCamera();
    void OpenCamera(HWND hDisplayWnd, LvSystem* pSystem);
    void CloseCamera();
    void StartAcquisition();
    void StopAcquisition();
    void SetProcessing(bool bDoProcessing);
    bool IsOpen();
    bool IsAcquiring();
    void CallbackNewBuffer(LvBuffer* pBuffer);
    --- %%IF IncUniImgPreprocess=1 ------------------
    void SetOptimalUniPixelFormat();
    --- %%ENDIF -------------------------------------

private:
    #ifdef USE_OPENCV_PROCESSING
        void OpenCvProcessingOpen();
        void OpenCvCreateImages();
        void OpenCvProcessImage(void* pSrcData);
        void OpenCvProcessingClose();

        IplImage* m_pSrcImg;
        IplImage* m_pDstImg;
        CvFont    m_TextFont;
        CvScalar  m_DrawColorBlack;
        CvScalar  m_DrawColorWhite;
        int       m_SizeROI;    // initial size of ROI in percentage of window size
    #endif

    LvSystem*    m_pSystem;
    LvInterface* m_pInterface;
    LvDevice*    m_pDevice;
    LvStream*    m_pStream;
    LvBuffer*    m_Buffers[NUMBER_OF_BUFFERS];
    --- %%IF IncImgDisplay=1 ------------------------
    LvRenderer*  m_pRenderer;
    HWND         m_hDisplayWnd;
    --- %%ENDIF -------------------------------------
    LvEvent*     m_pEvent;
    bool         m_bDoProcessing;
};

// global variables
LvSystem*    g_pSystem         = 0;
CCamera*     g_pCamera         = NULL;
bool         g_bProcessImage   = true;
HWND         g_hMainWnd        = NULL;

//-----------------------------------------------------------------------------
// Error check with a message box displayed, if it is error

bool ErrorOccurred(LvStatus ErrorStatus)
{
    if (ErrorStatus == LVSTATUS_OK) return false;
    MessageBox(GetActiveWindow(), LvLibrary::GetLastErrorMessage().c_str(), "Error", MB_OK | MB_ICONEXCLAMATION);
    return true;
}

//-----------------------------------------------------------------------------
// Callback function for the delivered image. Cannot be a method of a class,
// thus we use the pUserParam to set the pointer to the CCamera class instance
// when registering the callback, so that we can then pass the callback to the 
// CallbackNewBuffer() method of CCamera class.

void LV_STDC CallbackNewBufferFunction(LvHBuffer hBuffer,
                                       void* pUserPointer,
                                       void* pUserParam)
{
    CCamera* pCamera = (CCamera*) pUserParam;
    // in UserPointer we hold pointer to buffer
    pCamera->CallbackNewBuffer((LvBuffer*) pUserPointer);
}

//-----------------------------------------------------------------------------
// CCamera constructor

CCamera::CCamera()
{
    m_pSystem     = NULL;
    m_pInterface  = NULL;
    m_pDevice     = NULL;
    m_pStream     = NULL;
    --- %%IF IncImgDisplay=1 ------------------
    m_pRenderer   = NULL;
    m_hDisplayWnd = NULL;
    --- %%ENDIF -------------------------------
    m_pEvent      = NULL;
    m_bDoProcessing = g_bProcessImage;
    memset(m_Buffers, 0, sizeof(m_Buffers));
}

//-----------------------------------------------------------------------------
// CCamera destructor

CCamera::~CCamera()
{
    if (m_pDevice != NULL) CloseCamera();
}

//-----------------------------------------------------------------------------

void CCamera::OpenCamera(HWND hDisplayWnd, LvSystem* pSystem)
{
    if (m_pDevice != NULL) CloseCamera();

    #ifdef USE_OPENCV_PROCESSING
        OpenCvProcessingOpen();
    #endif

    m_pSystem = pSystem;
    --- %%IF IncImgDisplay=1 -------------------
    m_hDisplayWnd = hDisplayWnd;
    --- %%ENDIF --------------------------------

    %%REM The SynViewStatus is used instead of ErrorStatus, because the SynViewGen inserts ErrorStatus in CameraRemoteFeaturesCode
    LvStatus SynViewStatus;
    LvInterface* pInterface = NULL;
    LvDevice* pDevice = NULL;
    pSystem->UpdateInterfaceList();
    --- %%IF OpenInterface=ByInterfaceId -------
    SynViewStatus = pSystem->OpenInterface("%%INTERFACE_ID%%", pInterface);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF --------------------------------
    --- %%IF OpenInterface=ByInterfaceTlType ---
    char InterfaceId[256];
    SynViewStatus = pSystem->FindInterface(LvFindBy_TLType, "%%INTERFACE_TLTYPE%%", InterfaceId, sizeof(InterfaceId));
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = pSystem->OpenInterface(InterfaceId, pInterface);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF --------------------------------

    pInterface->UpdateDeviceList();
    --- %%IF OpenDevice=ByDeviceId -------------
    SynViewStatus = pInterface->OpenDevice("%%DEVICE_ID%%", pDevice, LvDeviceAccess_Control);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF --------------------------------
    --- %%IF OpenDevice=ByDeviceModelName ------
    char DeviceId[256];
    SynViewStatus = pInterface->FindDevice(LvFindBy_ModelName, "%%DEVICE_MODELNAME%%", DeviceId, sizeof(DeviceId));
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = pInterface->OpenDevice(DeviceId, pDevice, LvDeviceAccess_Control);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF --------------------------------
    --- %%IF OpenDevice=ByDeviceIpAddress ------
    char DeviceId[256];
    SynViewStatus = pInterface->FindDevice(LvFindBy_GevIPAddress, "%%DEVICE_IPADDRESS%%", DeviceId, sizeof(DeviceId));
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = pInterface->OpenDevice(DeviceId, pDevice, LvDeviceAccess_Control);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF --------------------------------
    --- %%IF OpenDevice=ByDeviceUserId ---------
    char DeviceId[256];
    SynViewStatus = pInterface->FindDevice(LvFindBy_UserID, "%%DEVICE_USERID%%", DeviceId, sizeof(DeviceId));
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = pInterface->OpenDevice(DeviceId, pDevice, LvDeviceAccess_Control);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF --------------------------------
    --- %%IF Sys.AutoLoadedCfg=1 ---------------
    // Note: When the code was generated, in SynView Explorer was switched ON the option
    // to autoload the camera settings after opening the camera. For this reason the following
    // line loads the camera settings from the file to assure the same starting status.
    --- %%ENDIF --------------------------------
    --- %%IF Sys.SnippetFile=1 -----------------
    SynViewStatus = pDevice->LoadSettings("", "%%Sys.CameraCfgFile%%", LvSaveFlag_All);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF --------------------------------

    --- %%IF Snippet=All ---------------------
    // The #error line below is intentionally inserted to the code in case you
    // generate the code from streamable or all writable features.
    #error Review the feature settings code and remove the unnecessary items!
    // Before removing this line from the code, go carefully through all the feature
    // settings below and leave there only those, which really need to be set.

    --- %%ENDIF ------------------------------
    --- %%IF Snippet=Streamable --------------
    // The #error line below is intentionally inserted to the code in case you
    // generate the code from streamable or all writable features.
    #error Review the feature settings code and remove the unnecessary items!
    // Before removing this line from the code, go carefully through all the feature
    // settings below and leave there only those, which really need to be set.

    --- %%ENDIF ------------------------------
    %%CameraRemoteFeaturesCode%%

    m_pInterface = pInterface;
    m_pDevice = pDevice;

    --- %%IF IncUniImgPreprocess=1 ------------------
    SynViewStatus = m_pDevice->SetEnum(LvDevice_LvUniProcessMode, LvUniProcessMode_Auto);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%IF ForceUniConvertTo8bit=1 ----------------
    // OpenCV enables to use the Canny operator only on 8-bit monochrome image
    // so we set UniProcessing to this pixel format.
    SynViewStatus = m_pDevice->SetEnum(LvDevice_LvUniPixelFormat, LvPixelFormat_Mono8);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ELSE  --------------------------------
    SetOptimalUniPixelFormat();
    --- %%ENDIF --------------------------------
    --- %%ENDIF --------------------------------

    --- %%IF SwTriggered=1 -------------------
    // Set the camera to software triggered mode
    SynViewStatus = pDevice->SetEnum(LvDevice_TriggerSelector, LvTriggerSelector_FrameStart);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = pDevice->SetEnum(LvDevice_TriggerMode, LvTriggerMode_On);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = pDevice->SetEnum(LvDevice_TriggerSource, LvTriggerSource_Software);
    if (ErrorOccurred(SynViewStatus)) return;

    --- %%ENDIF ------------------------------
    SynViewStatus = m_pDevice->OpenStream("", m_pStream);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pStream->OpenEvent(LvEventType_NewBuffer, m_pEvent);
    if (ErrorOccurred(SynViewStatus)) return;
    for (int i=0; i<NUMBER_OF_BUFFERS; i++)
    {
        SynViewStatus = m_pStream->OpenBuffer(NULL, 0, NULL, 0, m_Buffers[i]);
        if (ErrorOccurred(SynViewStatus)) return;
        SynViewStatus = m_Buffers[i]->Queue();
        if (ErrorOccurred(SynViewStatus)) return;
    }
    --- %%IF IncImgDisplay=1 ------------------
    SynViewStatus = m_pStream->OpenRenderer(m_pRenderer);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pRenderer->SetWindow(m_hDisplayWnd);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF -------------------------------
    SynViewStatus = m_pEvent->SetCallbackNewBuffer(CallbackNewBufferFunction, this);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pEvent->StartThread();
    if (ErrorOccurred(SynViewStatus)) return;

    #ifdef USE_OPENCV_PROCESSING
        // Initialize OpenCV objects --------------------------------
        // create an image descriptor for source image with full sensor resolution
        // and the destination image buffer
        OpenCvCreateImages();
    #endif
}

//-----------------------------------------------------------------------------
// Starts acquisition

void CCamera::StartAcquisition()
{
    if (m_pDevice == NULL) return;
    LvStatus SynViewStatus;
    SynViewStatus = m_pDevice->AcquisitionStart();
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%IF SwTriggered=1 -------------------
    // and issue the first software trigger
    SynViewStatus = m_pDevice->CmdExecute(LvDevice_TriggerSoftware);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF ------------------------------
}

//-----------------------------------------------------------------------------
// Stops acquisition

void CCamera::StopAcquisition()
{
    if (!IsAcquiring()) return;
    LvStatus SynViewStatus;
    SynViewStatus = m_pDevice->AcquisitionStop();
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pStream->FlushQueue(LvQueueOperation_AllToInput);
    if (ErrorOccurred(SynViewStatus)) return;
}

//-----------------------------------------------------------------------------
// Closes the cameras

void CCamera::CloseCamera()
{
    if (m_pDevice == NULL) return;
    if (IsAcquiring()) StopAcquisition();
    LvStatus SynViewStatus;
    SynViewStatus = m_pEvent->StopThread();
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pStream->CloseEvent(m_pEvent);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%IF IncImgDisplay=1 ------------------
    SynViewStatus = m_pStream->CloseRenderer(m_pRenderer);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%ENDIF -------------------------------
    SynViewStatus = m_pStream->FlushQueue(LvQueueOperation_AllDiscard);
    if (ErrorOccurred(SynViewStatus)) return;
    for (int i=0; i<NUMBER_OF_BUFFERS; i++)
        if (m_Buffers[i] != NULL)
        {
            SynViewStatus = m_pStream->CloseBuffer(m_Buffers[i]);
            if (ErrorOccurred(SynViewStatus)) return;
        }
    SynViewStatus = m_pDevice->CloseStream(m_pStream);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pInterface->CloseDevice(m_pDevice);
    if (ErrorOccurred(SynViewStatus)) return;
    SynViewStatus = m_pSystem->CloseInterface(m_pInterface);
    if (ErrorOccurred(SynViewStatus)) return;
    --- %%IF IncImgDisplay=1 ------------------
    InvalidateRect(m_hDisplayWnd, NULL, TRUE);
    --- %%ENDIF -------------------------------
    #ifdef USE_OPENCV_PROCESSING
        OpenCvProcessingClose();
    #endif
}

//-----------------------------------------------------------------------------
// Utility function for enabling/disabling menu items

bool CCamera::IsOpen()
{
    return m_pDevice != NULL;
}

//-----------------------------------------------------------------------------
// Utility function for enabling/disabling menu items

bool CCamera::IsAcquiring()
{
    if (m_pDevice == NULL) return false;
    int32_t iIsAcquiring;
    m_pDevice->GetInt32(LvDevice_LvDeviceIsAcquiring, &iIsAcquiring);
    return iIsAcquiring != 0;
}

//-----------------------------------------------------------------------------
// Switches ON/OFF image processing done in the callback

void CCamera::SetProcessing(bool bDoProcessing)
{
    m_bDoProcessing = bDoProcessing;
}

//-----------------------------------------------------------------------------
// Callback function called for each delivered image

void CCamera::CallbackNewBuffer(LvBuffer* pBuffer)
{
    // get the pointer to the image data
    void* pData = NULL;
    int32_t iImageOffset = 0;
    --- %%IF IncUniImgPreprocess=1 ------------------
    pBuffer->GetPtr (LvBuffer_UniBase, &pData);
    pBuffer->GetInt32 (LvBuffer_UniImageOffset, &iImageOffset);
    --- %%ELSE --------------------------------------
    pBuffer->GetPtr (LvBuffer_Base, &pData);
    pBuffer->GetInt32 (LvBuffer_ImageOffset, &iImageOffset);
    --- %%ENDIF -------------------------------------
    pData = (uint8_t*)pData + iImageOffset;
    if (m_bDoProcessing && pData != NULL)
    {
        #ifdef USE_OPENCV_PROCESSING
            OpenCvProcessImage(pData);
        #else
            // we will do some easy processing - invert the pixel values in an area
            int32_t iWidth;
            int32_t iHeight;
            int32_t iLinePitch;
            LvEnum iPixelFormat;    // LvPixelFormat enumeration value
            m_pDevice->GetInt32(LvDevice_Width, &iWidth);
            m_pDevice->GetInt32(LvDevice_Height, &iHeight);
            --- %%IF IncUniImgPreprocess=1 ------------------
            m_pDevice->GetEnum(LvDevice_LvUniPixelFormat, &iPixelFormat);
            m_pDevice->GetInt32(LvDevice_LvUniLinePitch, &iLinePitch);
            int32_t iBytesPerPixel = ((iPixelFormat & LV_PIX_EFFECTIVE_PIXEL_SIZE_MASK) >> LV_PIX_EFFECTIVE_PIXEL_SIZE_SHIFT) / 8;
            --- %%ELSE --------------------------------------
            m_pDevice->GetEnum(LvDevice_PixelFormat, &iPixelFormat);
            int iBitsPerPixel = (iPixelFormat & LV_PIX_EFFECTIVE_PIXEL_SIZE_MASK) >> LV_PIX_EFFECTIVE_PIXEL_SIZE_SHIFT;
            iLinePitch = ((iWidth * iBitsPerPixel)+7)/8;
            int32_t iBytesPerPixel = iBitsPerPixel / 8;
            --- %%ENDIF -------------------------------------

            for (int32_t j=0; j<(iHeight/2); j++)
            {
                uint8_t* pPixel = ((uint8_t*)pData) + (iHeight/4 + j)*iLinePitch + (iWidth/4)*iBytesPerPixel;
                for (int32_t i=0; i<(iWidth/2); i++)
                {
                    for (int32_t k=0; k<iBytesPerPixel; k++)
                    {
                        *pPixel = ~*pPixel;
                        pPixel++;
                    }
                }
            }
        #endif
    }

    --- %%IF IncImgDisplay=1 ------------------------
    m_pRenderer->DisplayImage(pBuffer);
    --- %%ELSE --------------------------------------
    // no image display demonstrated - switch it ON in the code generation Wizard if you need it
    --- %%ENDIF -------------------------------------
    pBuffer->Queue();
    --- %%IF SwTriggered=1 --------------------------
    // finished, trigger another image acquisition
    if (IsAcquiring())
        m_pDevice->CmdExecute(LvDevice_TriggerSoftware);
    --- %%ENDIF -------------------------------------
}

--- %%IF IncUniImgPreprocess=1 ------------------
//-----------------------------------------------------------------------------
// Determines optimal pixel format for the uni-process

void CCamera::SetOptimalUniPixelFormat()
{
    LvEnum PixelFormat;
    m_pDevice->GetEnum(LvDevice_PixelFormat, &PixelFormat);
    LvEnum UniPixelFormat = PixelFormat;
    switch (PixelFormat)
    {
        case LvPixelFormat_Mono8:
        case LvPixelFormat_Mono10:
        case LvPixelFormat_Mono12:
        case LvPixelFormat_Mono16:
            UniPixelFormat = LvPixelFormat_Mono8;
            break;

        case LvPixelFormat_BayerGR8:
        case LvPixelFormat_BayerRG8:
        case LvPixelFormat_BayerGB8:
        case LvPixelFormat_BayerBG8:
        case LvPixelFormat_BayerGR10:
        case LvPixelFormat_BayerRG10:
        case LvPixelFormat_BayerGB10:
        case LvPixelFormat_BayerBG10:
        case LvPixelFormat_BayerGR12:
        case LvPixelFormat_BayerRG12:
        case LvPixelFormat_BayerGB12:
        case LvPixelFormat_BayerBG12:
        case LvPixelFormat_RGB8Packed:
        case LvPixelFormat_RGBA8Packed:
            UniPixelFormat = LvPixelFormat_BGRA8Packed;
            break;
    }
    m_pDevice->SetEnum(LvDevice_LvUniPixelFormat, UniPixelFormat);
}

--- %%ENDIF -------------------------------------
//-----------------------------------------------------------------------------

#ifdef USE_OPENCV_PROCESSING

    void CCamera::OpenCvProcessingOpen()
    {
        m_pSrcImg = NULL;
        m_pDstImg = NULL;
        m_DrawColorBlack = CV_RGB (0, 0, 0);
        m_DrawColorWhite = CV_RGB (0xFF, 0xFF, 0xFF);
        cvInitFont (&m_TextFont, CV_FONT_HERSHEY_SIMPLEX, 0.6, 0.6);
        m_SizeROI = 50; // initial size of ROI in percentage of window size
    }

//----------------------------------------------------------------------------
// Create an OpenCV image header for full sensor resolution

    void CCamera::OpenCvCreateImages()
    {
        m_pSrcImg = NULL;
        int32_t Val_Width;
        int32_t Val_Height;
        LvEnum  Val_PixelFormat;

        m_pDevice->GetInt32(LvDevice_Width , &Val_Width );
        m_pDevice->GetInt32(LvDevice_Height, &Val_Height);
        --- %%IF IncUniImgPreprocess=1 ------------------
        m_pDevice->GetEnum(LvDevice_LvUniPixelFormat, &Val_PixelFormat);
        --- %%ELSE --------------------------------------
        m_pDevice->GetEnum(LvDevice_PixelFormat, &Val_PixelFormat);
        --- %%ENDIF -------------------------------------

        try
        {
            CvSize ImgSize = cvSize (Val_Width, Val_Height);
            m_DrawColorWhite = cvScalar (0xFF, 0xFF, 0xFF, 0xFF);

            switch (Val_PixelFormat)
            {
                case LvPixelFormat_Mono8:
                case LvPixelFormat_BayerGR8:
                case LvPixelFormat_BayerRG8:
                case LvPixelFormat_BayerGB8:
                case LvPixelFormat_BayerBG8:
                    m_pSrcImg = cvCreateImageHeader (ImgSize, IPL_DEPTH_8U, 1);
                    m_pDstImg = cvCreateImage       (ImgSize, IPL_DEPTH_8U, 1);
                    break;

		        case LvPixelFormat_Mono8Signed: // unsupported on NET HW
			        m_pSrcImg = cvCreateImageHeader (ImgSize, IPL_DEPTH_8S, 1);
			        break;

                case LvPixelFormat_Mono10:
		        case LvPixelFormat_Mono12:
		        case LvPixelFormat_Mono14:
		        case LvPixelFormat_Mono16:
                case LvPixelFormat_BayerGR10:
		        case LvPixelFormat_BayerRG10:
		        case LvPixelFormat_BayerGB10:
		        case LvPixelFormat_BayerBG10:
                case LvPixelFormat_BayerGR12:
		        case LvPixelFormat_BayerRG12:
		        case LvPixelFormat_BayerGB12:
		        case LvPixelFormat_BayerBG12:
			        m_pSrcImg = cvCreateImageHeader (ImgSize, IPL_DEPTH_16U, 1);
                    m_pDstImg = cvCreateImage       (ImgSize, IPL_DEPTH_16U, 1);
                    m_DrawColorWhite = CV_RGB (0xFFFF, 0xFFFF, 0xFFFF);
			        break;

		        case LvPixelFormat_BGR8Packed:
		        case LvPixelFormat_RGB8Packed:
			        m_pSrcImg = cvCreateImageHeader (ImgSize, IPL_DEPTH_8U, 3);
                    m_pDstImg = cvCreateImage       (ImgSize, IPL_DEPTH_8U, 3);
			        break;

		        case LvPixelFormat_BGRA8Packed:
		        case LvPixelFormat_RGBA8Packed:
			        m_pSrcImg = cvCreateImageHeader (ImgSize, IPL_DEPTH_8U, 4);
                    m_pDstImg = cvCreateImage       (ImgSize, IPL_DEPTH_8U, 4);
			        break;

                default:
                {
                    char msg[256];
                    sprintf (msg, "Unsupported pixel format 0x%X. Cannot create the OpenCV Image Header", Val_PixelFormat);
                    MessageBox(GetActiveWindow(), msg, "Error", MB_OK | MB_ICONEXCLAMATION);
                    break;
                }
            }
        }
        catch (cv::Exception e)
        {
            MessageBox(GetActiveWindow(), e.what(), "Error", MB_OK | MB_ICONEXCLAMATION);
        }
    }

//-----------------------------------------------------------------------------

    void CCamera::OpenCvProcessImage(void* pSrcData)
    {
        LvEnum Val_PixelFormat;
        --- %%IF IncUniImgPreprocess=1 ------------------
        m_pDevice->GetEnum(LvDevice_LvUniPixelFormat, &Val_PixelFormat);
        --- %%ELSE --------------------------------------
        m_pDevice->GetEnum(LvDevice_PixelFormat, &Val_PixelFormat);
        --- %%ENDIF -------------------------------------
        int iBitsPerPixel = (Val_PixelFormat & LV_PIX_EFFECTIVE_PIXEL_SIZE_MASK) >> LV_PIX_EFFECTIVE_PIXEL_SIZE_SHIFT;
        if (m_pSrcImg != NULL && m_pDstImg != NULL)
        {
            try
            {
                char Txt[128];

                // Adjust the address for this buffer
                m_pSrcImg->imageData= (char*) pSrcData;

                //define ROI for image processing
                CvRect rectTmp;
                rectTmp = cvRect (m_pSrcImg->width*(100-m_SizeROI)/2/100, m_pSrcImg->height*(100-m_SizeROI)/2/100,
                                  m_pSrcImg->width*m_SizeROI/100, m_pSrcImg->height*m_SizeROI/100);
                cvSetImageROI(m_pSrcImg, rectTmp);
                cvSetImageROI(m_pDstImg, rectTmp);

                // Call the cv processing function
                if (iBitsPerPixel == 8)
                {
                    cvCanny (m_pSrcImg, m_pDstImg, 50, 150, 3);
                    strcpy(Txt, "Image processed using the OpenCV cvCanny operator");
                }
                else
                {
	                cvSmooth(m_pSrcImg, m_pDstImg, CV_GAUSSIAN, 11);
                    strcpy(Txt, "Image processed using the OpenCV cvSmooth operator");
                }
                // Copy back the result to source image for display
                cvCopy (m_pDstImg, m_pSrcImg);

                // draw a frame around ROI
                cvRectangle (m_pSrcImg, cvPoint(2,2), cvPoint(m_pSrcImg->roi->width-3,
                             m_pSrcImg->roi->height-3), m_DrawColorWhite);

                cvResetImageROI (m_pSrcImg);
                cvResetImageROI (m_pDstImg);

                // Draw message to image.
                cvPutText(m_pSrcImg, Txt, cvPoint(49,49), &m_TextFont, m_DrawColorBlack);
                cvPutText(m_pSrcImg, Txt, cvPoint(49,51), &m_TextFont, m_DrawColorBlack);
                cvPutText(m_pSrcImg, Txt, cvPoint(51,49), &m_TextFont, m_DrawColorBlack);
                cvPutText(m_pSrcImg, Txt, cvPoint(51,51), &m_TextFont, m_DrawColorBlack);
                cvPutText(m_pSrcImg, Txt, cvPoint(50,50), &m_TextFont, m_DrawColorWhite);
            }
            catch (cv::Exception e)
            {
                // do nothing, we are in the callback, so no MsgBox possible
            }
        }
    }

//-----------------------------------------------------------------------------

    void CCamera::OpenCvProcessingClose()
    {
        try
        {
            if (m_pSrcImg != NULL)
            {
                cvReleaseImageHeader (&m_pSrcImg);
                m_pSrcImg = NULL;
            }
            if (m_pDstImg != NULL)
            {
                cvReleaseImage (&m_pDstImg);
                m_pDstImg = NULL;
            }
        }
        catch (cv::Exception e)
        {
            MessageBox(GetActiveWindow(), e.what(), "Error", MB_OK | MB_ICONEXCLAMATION);
        }
    }

#endif

//=============================================================================
//=== Windows application skeleton ============================================
//=============================================================================

HINSTANCE    g_hInstance;              
TCHAR*       g_pszTitle        = "SynView Sample Skeleton";
TCHAR*       g_pszMainWndClass = "SynViewSampleMainWnd";
HMENU        g_hMenu           = NULL;    // Main window menu

// Menu definitions         
#define IDM_OPEN_CAMERA           101
#define IDM_CLOSE_CAMERA          102
#define IDM_ACQUISITION_START     103
#define IDM_ACQUISITION_STOP      104
#define IDM_PROCESS_IMAGE         105
#define IDM_EXIT                  199

#define CHECK_MENU(id,check)   CheckMenuItem (g_hMenu, id, ((check) ? MF_CHECKED : MF_UNCHECKED) | MF_BYCOMMAND)
#define ENABLE_MENU(id,enable) EnableMenuItem(g_hMenu, id, ((enable) ?  MF_ENABLED : MF_GRAYED) | MF_BYCOMMAND)

//-----------------------------------------------------------------------------

void UpdateControls()
{
    ENABLE_MENU(IDM_OPEN_CAMERA      ,  !g_pCamera->IsOpen());
    ENABLE_MENU(IDM_CLOSE_CAMERA     ,  g_pCamera->IsOpen() && !g_pCamera->IsAcquiring());
    ENABLE_MENU(IDM_ACQUISITION_START,  g_pCamera->IsOpen() && !g_pCamera->IsAcquiring());
    ENABLE_MENU(IDM_ACQUISITION_STOP ,  g_pCamera->IsOpen() && g_pCamera->IsAcquiring());
    CHECK_MENU(IDM_PROCESS_IMAGE     ,  g_bProcessImage);
    char szTitle[256];
    sprintf(szTitle, "Camera %s", g_pCamera->IsOpen() ? "open" : "not open");
    if (g_pCamera->IsOpen())
    {
        if (g_pCamera->IsAcquiring())
            strcat(szTitle, ", acquiring.");
        else
            strcat(szTitle, ", not acquiring.");
    }
    SetWindowText(g_hMainWnd, szTitle);
}

//-----------------------------------------------------------------------------

long CALLBACK MainWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    switch (message) 
    {
        case WM_COMMAND: 
        {
            WORD Id = LOWORD(wParam);
            switch (Id) 
            {
                case IDM_OPEN_CAMERA:  
                    g_pCamera->OpenCamera(g_hMainWnd, g_pSystem);
                    UpdateControls();
                    break;
                case IDM_ACQUISITION_START:
                    g_pCamera->StartAcquisition();
                    UpdateControls();
                    break;
                case IDM_ACQUISITION_STOP: 
                    g_pCamera->StopAcquisition();
                    UpdateControls();
                    break;
                case IDM_PROCESS_IMAGE:
                    g_bProcessImage = !g_bProcessImage;
                    g_pCamera->SetProcessing(g_bProcessImage);
                    UpdateControls();
                    break;
                case IDM_CLOSE_CAMERA:
                    g_pCamera->CloseCamera();
                    UpdateControls();
                    break;
                case IDM_EXIT:  
                    DestroyWindow(hWnd);                     
                    break;
                default:
                    return (long) DefWindowProc(hWnd, message, wParam, lParam);
            }
            break;
        }
    
        case WM_DESTROY:
            g_pCamera->CloseCamera();
            PostQuitMessage(0);
            break;
    
        default:
            return (long) DefWindowProc(hWnd, message, wParam, lParam);
    }
    return 0;
}

//-----------------------------------------------------------------------------

BOOL InitApplication(HINSTANCE hInstance)
{
      WNDCLASSEX wcex;
      wcex.cbSize = sizeof(WNDCLASSEX);
  
      wcex.style            = CS_HREDRAW | CS_VREDRAW;
      wcex.lpfnWndProc      = (WNDPROC)MainWndProc;
      wcex.cbClsExtra       = 0;
      wcex.cbWndExtra       = 0;
      wcex.hInstance        = hInstance;
      wcex.hIcon            = LoadIcon(NULL, IDI_APPLICATION);
      wcex.hCursor          = LoadCursor(NULL, IDC_ARROW);
      wcex.hbrBackground    = (HBRUSH)(COLOR_WINDOW+1);
      wcex.lpszMenuName     = NULL;
      wcex.lpszClassName    = g_pszMainWndClass;
      wcex.hIconSm          = LoadIcon(NULL, IDI_APPLICATION);
      RegisterClassEx(&wcex);
  
    return TRUE;
}

//-----------------------------------------------------------------------------

HMENU BuildMenu(void)
{
    HMENU hMenu=CreateMenu();
    HMENU hSubMenu=CreatePopupMenu();
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_EXIT, "Exit");
    AppendMenu(hMenu, MF_POPUP | MF_ENABLED, (UINT_PTR)hSubMenu, "File");
    hSubMenu=CreatePopupMenu();
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_OPEN_CAMERA,       "Open camera");  
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_CLOSE_CAMERA,      "Close camera");  
    AppendMenu(hSubMenu, MF_SEPARATOR, -1, "");
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_ACQUISITION_START, "Acquisition Start");  
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_UNCHECKED, IDM_ACQUISITION_STOP,  "Acquisition Stop");  
    AppendMenu(hSubMenu, MF_SEPARATOR, -1, "");
    AppendMenu(hSubMenu, MF_STRING | MF_ENABLED | MF_CHECKED, IDM_PROCESS_IMAGE,       "Process image");
    AppendMenu(hMenu, MF_POPUP | MF_ENABLED, (UINT_PTR)hSubMenu, "Camera");
    return hMenu;
}

//-----------------------------------------------------------------------------

BOOL InitInstance(HINSTANCE hInstance, int nCmdShow)
  {
    // Create the main window menu
    g_hMenu=BuildMenu();
  
    int iScreenWidth  = GetSystemMetrics(SM_CXSCREEN); 
    int iScreenHeight = GetSystemMetrics(SM_CYSCREEN); 
    int iAppWidth = (iScreenWidth * 9)/10; // 90% of width 
    int iAppHeight = (iScreenHeight * 9)/10; // 90% of height
    int iAppXOffset = (iScreenWidth - iAppWidth)/2;
    int iAppYOffset = (iScreenHeight - iAppHeight)/2;
  
    // Create the main window
    g_hInstance = hInstance;
    g_hMainWnd = CreateWindow(
         g_pszMainWndClass, 
         g_pszTitle, 
         WS_OVERLAPPEDWINDOW,
         iAppXOffset, 
         iAppYOffset, 
         iAppWidth, 
         iAppHeight, 
         NULL, 
         g_hMenu, 
         hInstance, NULL);
    if (g_hMainWnd == NULL) return FALSE;

    ShowWindow(g_hMainWnd, nCmdShow);
    UpdateWindow(g_hMainWnd);
    UpdateControls();
    return TRUE;
  }

//-----------------------------------------------------------------------------

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE /*hPrevInstance*/,
                   LPSTR /*lpCmdLine*/, int nCmdShow)
{
    if (LvLibrary::OpenLibrary() != LVSTATUS_OK)
    {
        MessageBox(g_hMainWnd, "Opening the library failed.", "Error", MB_OK | MB_ICONEXCLAMATION);
        return 0;
    }
    if (LvSystem::Open("", g_pSystem) != LVSTATUS_OK) 
    {
        MessageBox(g_hMainWnd, "Opening the system failed.", "Error", MB_OK | MB_ICONEXCLAMATION);
        LvLibrary::CloseLibrary();
        return 0;
    }
    // Create one global CCamera class instance
    g_pCamera = new CCamera();
  
    MSG msg;
    if (!InitApplication(hInstance)) 
    {
        MessageBox(g_hMainWnd, "Application init failed.", "Error", MB_OK | MB_ICONEXCLAMATION);
        delete g_pCamera;
        LvSystem::Close(g_pSystem);
        LvLibrary::CloseLibrary();
        return 0;
    }
  
    if (!InitInstance(hInstance, nCmdShow)) 
    {
        MessageBox(g_hMainWnd, "Instance init failed.", "Error", MB_OK | MB_ICONEXCLAMATION);
        delete g_pCamera;
        LvSystem::Close(g_pSystem);
        LvLibrary::CloseLibrary();
        return 0;
    }
  
    while (GetMessage(&msg, NULL, NULL, NULL)) 
    {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }
  
    delete g_pCamera;
    LvSystem::Close(g_pSystem);
    LvLibrary::CloseLibrary();
    return (int) msg.wParam;
}

//-----------------------------------------------------------------------------
