#ifndef _LV_INI_H
#ifndef _SV_SYNVIEW_INI_H
#define _SV_SYNVIEW_INI_H
// internal note: if synchronize changes with lv.ini.h

#ifdef _WIN32
    #define LVINI_PUBLIC
#else
    #if __GNUC__>=4
        #define LVINI_PUBLIC   __attribute__((visibility("default")))
    #endif
#endif

typedef void* LvHIniFile;
typedef void* LvHIniSection;
typedef void* LvHIniLine;

#define LVINI_TYPE_STRING        1
#define LVINI_TYPE_INTEGER       2
#define LVINI_TYPE_BOOLEAN       3
#define LVINI_TYPE_ENUMERATION   4
#define LVINI_TYPE_CATEGORY      5

#define LVINI_VISIBILITY_BEGINNER      1
#define LVINI_VISIBILITY_EXPERT        2
#define LVINI_VISIBILITY_GURU          3
#define LVINI_VISIBILITY_GENIUS        4
#define LVINI_VISIBILITY_INVISIBLE     5

#define LVINI_FLAG_HEXADECIMAL     0x00000001
#define LVINI_FLAG_FILE            0x00000002
#define LVINI_FLAG_DIR             0x00000004

#include <sv.synview.defs.h>

/** @defgroup GroupSynview_Ini SynView INI file API
 *  The sv.synview.ini is a helper library, enabling to read and write INI files
 *  in all supported operating systems.
 *  @ingroup GroupSynview
 */

//-------------------------------------------------------------------------
/** Creates the INI file underlying structures and returns a handle to it.
  * It does not read the INI file contents, this is done by LvIniLoad().
  * @param pCommentSeparator If specified, all lines where it appears at the beginning of the line
  *                          are considered to be a comment. It need not be placed at the very first
  *                          position, but to the left must be only whitespace characters, otherwise
  *                          the line is not considered to be a comment.
  *                          If a value is commented out and the new value of the same name 
  *                          is written, it is placed before the commented value (not to the end of 
  *                          the section).
  * @ingroup GroupSynview_Ini
  */

#ifdef __cplusplus
LV_EXTC LVINI_PUBLIC LvHIniFile LV_STDC LvIniOpen (const char* pCommentSeparator=";");
#else
LV_EXTC LVINI_PUBLIC LvHIniFile LV_STDC LvIniOpen (const char* pCommentSeparator);
#endif

//-------------------------------------------------------------------------
/** Closes the INI file underlying structures and all its parents.
  * It does nto write the contents to disk; if you want to save the INI content,
  * use first LvIniSave().
  * @param  hIniFile  Handle to the INI file.
  * @ingroup GroupSynview_Ini
  */
LV_EXTC LVINI_PUBLIC void LV_STDC LvIniClose (LvHIniFile hIniFile);

//-------------------------------------------------------------------------
/** Loads the ini file contents. 
  * All subsequent changes are done in memory, until the ini file is saved by the 
  * LvIniSave() function. If there is a parent already specified (see LvIniSetParent()), 
  * it is opened (recursively) as well.
  * @param hIniFile    Handle to the INI file.
  * @param pFileName The INI file name.
  * @return If the file does not exist, or file I/O fails, it returns 0, otherwise 1.
  * @ingroup GroupSynview_Ini
  */
LV_EXTC LVINI_PUBLIC uint32_t LV_STDC LvIniLoad (LvHIniFile hIniFile, 
                                                 const char* pFileName);

//-------------------------------------------------------------------------
/** Saves the INI file contents to a file. 
  * @param hIniFile      Handle to the INI file.
  * @param pFileName     The INI file name.
  * @param CreateBackup  If is set, the original file is preserved
  *                      with added ".bak" extension. 
  * @return The return value 1 indicates a success of the file I/O.
  * @ingroup GroupSynview_Ini
  */
#ifdef __cplusplus
LV_EXTC LVINI_PUBLIC uint32_t LV_STDC LvIniSave (LvHIniFile hIniFile, 
                                                 const char* pFileName, 
                                                 uint32_t CreateBackup=0);
#else
LV_EXTC LVINI_PUBLIC uint32_t LV_STDC LvIniSave (LvHIniFile hIniFile, 
                                                 const char* pFileName, 
                                                 uint32_t CreateBackup);
#endif

//-------------------------------------------------------------------------
/** Writes the modified items to an existing INI file.
  * This function enables to preserve the changes made in the INI file after it was
  * read by LvIniLoad(). It loads the current contents of the file,
  * sets modifies the items in it and saves it.
  * Note that this function writes ONLY the modified and new values.
  * It does not write any other lines (modified by LvIniSetSectionRawLine()) and
  * if you have deleted some values or sections, they will not be deleted in the file.
  *
  * @param hIniFile      Handle to the INI file.
  * @param pszFileName   The INI file name.
  * @param bCreateBackup If is set, the original file is preserved
  *                      with added ".bak" extension. 
  * @return The return value 1 indicates a success of the file I/O.
  */
#ifdef __cplusplus
LV_EXTC LVINI_PUBLIC uint32_t LV_STDC LvIniUpdateFile (LvHIniFile hIniFile, 
                                                       const char* pszFileName,
                                                       uint32_t bCreateBackup=0);
#else
LV_EXTC LVINI_PUBLIC uint32_t LV_STDC LvIniUpdateFile (LvHIniFile hIniFile, 
                                                       const char* pszFileName,
                                                       uint32_t bCreateBackup);
#endif

//-------------------------------------------------------------------------
/** Returns 1 if the file content was modified.
  * @return 1 if at least one item was modified or deleted or section deleted.
  * @ingroup GroupSynview_Ini
  */
LV_EXTC LVINI_PUBLIC uint32_t LV_STDC LvIniModified (LvHIniFile hIniFile);

//-------------------------------------------------------------------------
/** Checks if the item exists. 
  * @param hIniFile   Handle to the INI file.
  * @param pSection   Section in the INI file (without brackets).
  * @param pName      Name of the item in the section.
  * @param Order      Can be used to distiguish between multiple items of the same name in one section.
  * @return           The return value 1 indicates the existence of the item.
  * @ingroup GroupSynview_Ini
  */
#ifdef __cplusplus
LV_EXTC LVINI_PUBLIC uint32_t LV_STDC LvIniItemExists (LvHIniFile hIniFile, 
                                                       const char* pSection, 
                                                       const char* pName, 
                                                       uint32_t Order=1);
#else
LV_EXTC LVINI_PUBLIC uint32_t LV_STDC LvIniItemExists (LvHIniFile hIniFile, 
                                                       const char* pSection, 
                                                       const char* pName, 
                                                       uint32_t Order);
#endif

//-------------------------------------------------------------------------
/** Checks is the section exists. 
  * @param hIniFile   Handle to the INI file.
  * @param pSection Section in the INI file (without brackets).
  * @return           1 if the section exists.
  * @ingroup GroupSynview_Ini
  */
LV_EXTC LVINI_PUBLIC uint32_t LV_STDC LvIniSectionExists (LvHIniFile hIniFile, 
                                                          const char* pSection);

//-------------------------------------------------------------------------
/** Deletes the item. 
  * @param hIniFile   Handle to the INI file.
  * @param pSection   Section in the INI file (without brackets).
  * @param pName      Name of the item in the section.
  * @param Order      Can be used to distiguish between multiple items of the same name in one section.
  * @ingroup GroupSynview_Ini
  */
#ifdef __cplusplus
LV_EXTC LVINI_PUBLIC void LV_STDC LvIniDeleteItem (LvHIniFile hIniFile, 
                                                   const char* pSection, 
                                                   const char* pName, 
                                                   uint32_t Order=1);
#else
LV_EXTC LVINI_PUBLIC void LV_STDC LvIniDeleteItem (LvHIniFile hIniFile, 
                                                   const char* pSection, 
                                                   const char* pName, 
                                                   uint32_t Order);
#endif

//-------------------------------------------------------------------------
/** Deletes the section.
  * @param hIniFile   Handle to the INI file.
  * @param pSection Section in the INI file (without brackets).
  * @ingroup GroupSynview_Ini
  */
LV_EXTC LVINI_PUBLIC void LV_STDC LvIniDeleteSection (LvHIniFile hIniFile, 
                                                      const char* pSection);

//-------------------------------------------------------------------------
/** Reads an integer value. 
  * @param hIniFile   Handle to the INI file.
  * @param pSection   Section in the INI file (without brackets).
  * @param pName      Name of the item in the section.
  * @param Default    Default value to be used when the item is not found or is empty.
  * @param Order      Can be used to distiguish between multiple items of the same name in one section.
  * @return           Read value.
  * @ingroup GroupSynview_Ini
  */
#ifdef __cplusplus
LV_EXTC LVINI_PUBLIC int32_t LV_STDC LvIniGetInteger (LvHIniFile hIniFile, 
                                                      const char* pSection, 
                                                      const char* pName, 
                                                      int32_t Default, 
                                                      uint32_t Order=1);
#else
LV_EXTC LVINI_PUBLIC int32_t LV_STDC LvIniGetInteger (LvHIniFile hIniFile, 
                                                      const char* pSection, 
                                                      const char* pName, 
                                                      int32_t Default, 
                                                      uint32_t Order);
#endif

//-------------------------------------------------------------------------
/** Writes an integer value. 
  * @param hIniFile     Handle to the INI file.
  * @param pSection     Section in the INI file (without brackets).
  * @param pName        Name of the item in the section.
  * @param Value        Value of the item to be set.
  * @param Hexadecimal  If true, the value is written as hexa with the "0x" prefix.
  * @param Order        Can be used to distiguish between multiple items of the same name in one section.
  * @ingroup GroupSynview_Ini
  */
#ifdef __cplusplus
LV_EXTC LVINI_PUBLIC void LV_STDC LvIniSetInteger (LvHIniFile hIniFile, 
                                                   const char* pSection, 
                                                   const char* pName, 
                                                   int32_t Value, 
                                                   uint32_t Hexadecimal=0, 
                                                   uint32_t Order=1);
#else
LV_EXTC LVINI_PUBLIC void LV_STDC LvIniSetInteger (LvHIniFile hIniFile, 
                                                   const char* pSection, 
                                                   const char* pName, 
                                                   int32_t Value, 
                                                   uint32_t Hexadecimal, 
                                                   uint32_t Order);
#endif

//-------------------------------------------------------------------------
/** Reads a float value. 
  * @param hIniFile   Handle to the INI file.
  * @param pSection   Section in the INI file (without brackets).
  * @param pName      Name of the item in the section.
  * @param Default    Default value to be used when the item is not found or is empty.
  * @param Order      Can be used to distiguish between multiple items of the same name in one section.
  * @return           Read value.
  * @ingroup GroupSynview_Ini
  */
#ifdef __cplusplus
LV_EXTC LVINI_PUBLIC double LV_STDC LvIniGetFloat (LvHIniFile hIniFile, 
                                                   const char* pSection, 
                                                   const char* pName, 
                                                   double Default, 
                                                   uint32_t Order=1);
#else
LV_EXTC LVINI_PUBLIC double LV_STDC LvIniGetFloat (LvHIniFile hIniFile, 
                                                   const char* pSection, 
                                                   const char* pName, 
                                                   double Default, 
                                                   uint32_t Order);
#endif

//-------------------------------------------------------------------------
/** Writes a float value. 
  * @param hIniFile     Handle to the INI file.
  * @param pSection     Section in the INI file (without brackets).
  * @param pName        Name of the item in the section.
  * @param Value        Value of the item to be set.
  * @param Order        Can be used to distiguish between multiple items of the same name in one section.
  * @ingroup GroupSynview_Ini
  */
#ifdef __cplusplus
LV_EXTC LVINI_PUBLIC void LV_STDC LvIniSetFloat (LvHIniFile hIniFile, 
                                                 const char* pSection, 
                                                 const char* pName, 
                                                 double Value, 
                                                 uint32_t Order=1);
#else
LV_EXTC LVINI_PUBLIC void LV_STDC LvIniSetFloat (LvHIniFile hIniFile, 
                                                 const char* pSection, 
                                                 const char* pName, 
                                                 double Value, 
                                                 uint32_t Order);
#endif


//-------------------------------------------------------------------------
/** Reads a boolean value. 
  * @param hIniFile   Handle to the INI file.
  * @param pSection   Section in the INI file (without brackets).
  * @param pName      Name of the item in the section.
  * @param Default    Default value to be used when the item is not found or is empty.
  * @param Order      Can be used to distiguish between multiple items of the same name in one section.
  * @return           Read value (0 or 1).
  * @ingroup GroupSynview_Ini
  */
#ifdef __cplusplus
LV_EXTC LVINI_PUBLIC uint32_t LV_STDC LvIniGetBool (LvHIniFile hIniFile, 
                                                    const char* pSection, 
                                                    const char* pName, 
                                                    uint32_t Default, 
                                                    uint32_t Order=1);
#else
LV_EXTC LVINI_PUBLIC uint32_t LV_STDC LvIniGetBool (LvHIniFile hIniFile, 
                                                    const char* pSection, 
                                                    const char* pName, 
                                                    uint32_t Default, 
                                                    uint32_t Order);
#endif

//-------------------------------------------------------------------------
/** Writes a boolean value. 
  * @param hIniFile   Handle to the INI file.
  * @param pSection   Section in the INI file (without brackets).
  * @param pName      Name of the item in the section.
  * @param Value      Value of the item to be set.
  * @param Order      Can be used to distiguish between multiple items of the same name in one section.
  * @ingroup GroupSynview_Ini
  */
#ifdef __cplusplus
LV_EXTC LVINI_PUBLIC void LV_STDC LvIniSetBool (LvHIniFile hIniFile, 
                                                const char* pSection,
                                                const char* pName,
                                                uint32_t Value,
                                                uint32_t Order=1);
#else
LV_EXTC LVINI_PUBLIC void LV_STDC LvIniSetBool (LvHIniFile hIniFile, 
                                                const char* pSection,
                                                const char* pName,
                                                uint32_t Value,
                                                uint32_t Order);
#endif

//-------------------------------------------------------------------------
/** Reads a string value. 
  * @param hIniFile   Handle to the INI file.
  * @param pSection   Section in the INI file (without brackets).
  * @param pName      Name of the item in the section.
  * @param pDefault   Default value to be used when the item is not found or is empty.
  * @param pString    The string value is returned in this parameter.
  * @param Size       Size of the pString buffer.
  * @param Order      Can be used to distiguish between multiple items of the same name in one section.
  * @ingroup GroupSynview_Ini
  */
#ifdef __cplusplus
LV_EXTC LVINI_PUBLIC void LV_STDC LvIniGetString (LvHIniFile hIniFile, 
                                                  const char* pSection,
                                                  const char* pName,
                                                  const char* pDefault,
                                                  char* pString,
                                                  uint32_t Size,
                                                  uint32_t Order=1);
#else
LV_EXTC LVINI_PUBLIC void LV_STDC LvIniGetString (LvHIniFile hIniFile, 
                                                  const char* pSection,
                                                  const char* pName,
                                                  const char* pDefault,
                                                  char* pString,
                                                  uint32_t Size,
                                                  uint32_t Order);
#endif

//-------------------------------------------------------------------------
/** Returns a size of buffer needed to read the string.
  * @param hIniFile   Handle to the INI file.
  * @param pSection   Section in the INI file (without brackets).
  * @param pName      Name of the item in the section.
  * @param pDefault   Default value to be used when the item is not found or is empty.
  * @param Order      Can be used to distiguish between multiple items of the same name in one section.
  * @return           Size of the buffer (string length+1).
  * @ingroup GroupSynview_Ini
  */
#ifdef __cplusplus
LV_EXTC LVINI_PUBLIC uint32_t LV_STDC LvIniGetStringSize (LvHIniFile hIniFile, 
                                                          const char* pSection,
                                                          const char* pName,
                                                          const char* pDefault,
                                                          uint32_t Order=1);
#else
LV_EXTC LVINI_PUBLIC uint32_t LV_STDC LvIniGetStringSize (LvHIniFile hIniFile, 
                                                          const char* pSection,
                                                          const char* pName,
                                                          const char* pDefault,
                                                          uint32_t Order);
#endif

//-------------------------------------------------------------------------
/** Writes a string value. 
  * @param hIniFile   Handle to the INI file.
  * @param pSection   Section in the INI file (without brackets).
  * @param pName      Name of the item in the section.
  * @param pValue     Value of the item to be set.
  * @param Order      Can be used to distiguish between multiple items of the same name in one section.
  * @ingroup GroupSynview_Ini
  */
#ifdef __cplusplus
LV_EXTC LVINI_PUBLIC void LV_STDC LvIniSetString (LvHIniFile hIniFile, 
                                                  const char* pSection, 
                                                  const char* pName, 
                                                  const char* pValue, 
                                                  uint32_t Order=1);
#else
LV_EXTC LVINI_PUBLIC void LV_STDC LvIniSetString (LvHIniFile hIniFile, 
                                                  const char* pSection, 
                                                  const char* pName, 
                                                  const char* pValue, 
                                                  uint32_t Order);
#endif

//-------------------------------------------------------------------------
/** Gets the raw string in specified section at position, specified by Order. 
  * Commented out and empty lines are not counted. 
  * If the item is not found, empty string is returned.
  * @param hIniFile   Handle to the INI file.
  * @param pSection   Section in the INI file (without brackets).
  * @param pLine      Buffer for the line contents.
  * @param Size       Size of the pLine buffer.
  * @param Order      Order of the raw line. The first valid line has Order=1
  * @ingroup GroupSynview_Ini
  */
LV_EXTC LVINI_PUBLIC void LV_STDC LvIniGetSectionRawLine (LvHIniFile hIniFile, 
                                                          const char* pSection, 
                                                          char* pLine, 
                                                          uint32_t Size, 
                                                          uint32_t Order);

//-------------------------------------------------------------------------
/** Gets the raw size of buffer needed for the raw line in specified section at position, 
  * specified by Order. 
  * Commented out and empty lines are not counted. 
  * If the item is not found, empty string is returned.
  * @param hIniFile   Handle to the INI file.
  * @param pSection   Section in the INI file (without brackets).
  * @param Order      Order of the raw line. The first valid line has Order=1.
  * @return           Size of the buffer (string length+1).
  * @ingroup GroupSynview_Ini
  */
LV_EXTC LVINI_PUBLIC uint32_t LV_STDC LvIniGetSectionRawLineSize (LvHIniFile hIniFile, 
                                                                  const char* pSection, 
                                                                  uint32_t Order);

//-------------------------------------------------------------------------
/** Sets the raw string in specified section at position, specified by Order. 
  * Commented out and empty lines are not counted. The first valid line has Order=1
  * If the item with the Order is not found, a new line is created and added at the end of section.
  * @param hIniFile   Handle to the INI file.
  * @param pSection Section in the INI file (without brackets).
  * @param pLine    The raw line to be set.
  * @param Order     Order of the raw line.
  * @ingroup GroupSynview_Ini
  */
LV_EXTC LVINI_PUBLIC void LV_STDC LvIniSetSectionRawLine (LvHIniFile hIniFile, 
                                                          const char* pSection, 
                                                          const char* pLine, 
                                                          uint32_t Order);

//-------------------------------------------------------------------------
/** Sets the section and name, where the specification of the parent INI file should be read.
  * For example for the following INI the call would be SetParent("Linked", "Parent");
  * @code
  *   [Linked]
  *   Parent=main.ini
  * @endcode
  * Use empty strings to disable the parent.
  * 
  * When you specify a parent, you can automatically work with a hierarchy of INI 
  * files - when the file is open, it searches for a section and name, specified 
  * in the SetParent() function. If found and a valid file name is specified there, 
  * it creates a parent class instance and reads to it the contents of the file. 
  * Then every item, which is not explicitly specified in own INI file is searched in 
  * this parent INI file.
  * The opening is recusive - the parent INI can have its own parent etc., so be sure you 
  * do not make a circular reference. The maximum level of recursion is intentionally limited to 10.
  * 
  * @param hIniFile Handle to the INI file.
  * @param pSection Section in the INI file (without brackets).
  * @param pName    Name of the item in the section.
  * @ingroup GroupSynview_Ini
  */
LV_EXTC LVINI_PUBLIC void LV_STDC LvIniSetParent (LvHIniFile hIniFile, 
                                                  const char* pSection, 
                                                  const char* pName);


//-------------------------------------------------------------------------
/** Returns the section name at specified position. The positions are numbered from 1.
  * When the position is out of range, empty string is returned.
  * @param hIniFile   Handle to the INI file.
  * @param pszSection The buffer, to which the section name is stored.
  * @param Size       Size of the pszSection buffer.
  * @param iOrder     Order of the section in the file. Numbered from 1.
  */
LV_EXTC LVINI_PUBLIC void LV_STDC LvIniGetSection (LvHIniFile hIniFile, 
                                                     char* pszSection,
                                                     uint32_t Size,
                                                     uint32_t iOrder);

//-------------------------------------------------------------------------
/** Returns an item name at specified position. The positions are numbered from 1.
  * When the position is out of range, empty string is returned.
  * @param hIniFile   Handle to the INI file.
  * @param pszSection Section in the INI file (without brackets).
  * @param pszItem    The buffer, to which the item name is stored.
  * @param Size       Size of the pszItem buffer.
  * @param iOrder     Order of the item in the section. Numbered from 1.
  */
LV_EXTC LVINI_PUBLIC void LV_STDC LvIniGetItem (LvHIniFile hIniFile, 
                                                  const char* pszSection,
                                                  char* pszItem,
                                                  uint32_t Size,
                                                  uint32_t iOrder);

//  ==============================================================================
//   The following functions are NET internal and are not documented for public
//  ==============================================================================

LV_EXTC LVINI_PUBLIC LvHIniSection LV_STDC LvIniGetSectionHandle(LvHIniFile hIniFile,
                                                                 int32_t Order);

LV_EXTC LVINI_PUBLIC void LV_STDC LvIniGetSectionLayout(LvHIniFile hIniFile,
                                                        LvHIniSection hSection, 
                                                        int32_t* pVisibility,
                                                        int32_t* pTreeLevel,
                                                        uint32_t* pFlags,
                                                        char* pName, uint32_t NameSize,
                                                        char* pCaption, uint32_t CaptionSize,
                                                        char* pHelp, uint32_t HelpSize);

LV_EXTC LVINI_PUBLIC LvHIniLine LV_STDC LvIniGetLineHandle(LvHIniFile hIniFile,
                                                           LvHIniSection hSection, 
                                                           int32_t Order);

LV_EXTC LVINI_PUBLIC uint32_t LV_STDC LvIniIsValue(LvHIniFile hIniFile,
                                                   LvHIniLine hLine);

LV_EXTC LVINI_PUBLIC void LV_STDC LvIniGetLineLayout(LvHIniFile hIniFile,
                                                     LvHIniLine hLine, 
                                                     int32_t* pItemType, 
                                                     int32_t* pVisibility, 
                                                     int32_t* pTreeLevel, 
                                                     uint32_t* pFlags,
                                                     int32_t* pEnumItems,
                                                     char* pSection, uint32_t SectionSize,
                                                     char* pName, uint32_t NameSize,
                                                     char* pCaption, uint32_t CaptionSize,
                                                     char* pHelp, uint32_t HelpSize);

LV_EXTC LVINI_PUBLIC void LV_STDC LvIniGetLineLayoutInt(LvHIniFile hIniFile,
                                                        LvHIniLine hLine, 
                                                        int32_t* pMin, 
                                                        int32_t* pMax, 
                                                        int32_t* pInc);

LV_EXTC LVINI_PUBLIC void LV_STDC LvIniGetLineLayoutEnumEntry(LvHIniFile hIniFile,
                                                              LvHIniLine hLine, 
                                                              int32_t Index, 
                                                              char* pCaption, 
                                                              uint32_t CaptionSize,
                                                              char* pValue, 
                                                              uint32_t ValueSize);

#endif
#endif
