#ifndef SV_SYNVIEW_IMGPROC_H
#define SV_SYNVIEW_IMGPROC_H

#include <sv.synview.defs.h>
#include <sv.synview.enums.h>
#include <sv.synview.status.h>

/** @defgroup GroupImgproc_Functions Image Processing Library functions
 *  @ingroup GroupImgproc
 */

/** @defgroup GroupImgproc_FunctionsCommon Common functions
 *  @ingroup GroupImgproc_Functions
 */

/** @defgroup GroupImgproc_FunctionsInit Image initialization functions
 *  @ingroup GroupImgproc_Functions
 */

/** @defgroup GroupImgproc_FunctionsRoi Region of Interest (ROI) functions
 *  @ingroup GroupImgproc_Functions
 */

/** @defgroup GroupImgproc_FunctionsLut Lookup Table (LUT) functions
 *  @ingroup GroupImgproc_Functions
 */

/** @defgroup GroupImgproc_FunctionsBayer Bayer decoding/encoding functions
 *  @ingroup GroupImgproc_Functions
 */

/** @defgroup GroupImgproc_FunctionsRotate Rotation and line manipulation functions
 *  @ingroup GroupImgproc_Functions
 */

/** @defgroup GroupImgproc_FunctionsConversion Pixel format conversion functions
 *  @ingroup GroupImgproc_Functions
 */

/** @defgroup GroupImgproc_FunctionsSaveLoad Saving/loading functions
 *  @ingroup GroupImgproc_Functions
 */

/** @defgroup GroupImgproc_FunctionsOverlay Overlay functions
 *  @ingroup GroupImgproc_Functions
 */

/** @defgroup GroupImgproc_FunctionsMatrix RGB color correction and convolution functions
 *  @ingroup GroupImgproc_Functions
 */
 
/** @defgroup GroupImgproc_FunctionsShading Shading correction functions
 *  @ingroup GroupImgproc_Functions
 */

//------------------------------------------------------------------------------
/** Retrieves a text desctibing the status.
 *  @param TlStatus    Error status code.
 *  @param pMsg        Pointer to buffer for the error message.
 *  @param MsgBufSize  Size of the buffer.
 *  @ingroup GroupImgproc_FunctionsCommon
 */
LVIP_DLLENTRY void LV_STDC LvipGetStatusMsg  (LvStatus TlStatus,
                                              char* pMsg,
                                              size_t MsgBufSize);

//------------------------------------------------------------------------------
/** Initializes the @ref LvipImgInfo to specified values, calculates the line increment and sets pData to NULL
 *  (be sure to deallocate the image buffers if were allocated, before this function call).
 *  If pData of other owner is used, set the @ref LvipImgAttr_NotDataOwner flag
 *  so that the data are not deallocated when LvipDeallocateImageData() is applied to this ImgInfo.
 *
 *  @param pImgInfo    Pointer to @ref LvipImgInfo structure which is to be initialized
 *  @param Width       Width of image in pixels
 *  @param Height      Height of image in pixels
 *  @param PixelFormat Pixel format; one of the @ref LvPixelFormat.
 *  @param Attributes  Image attributes; OR-ed combination of the @ref LvipImgAttr.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsInit
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipInitImgInfo (LvipImgInfo* pImgInfo,
                                                uint32_t Width,
                                                uint32_t Height,
                                                uint32_t PixelFormat,
                                                uint32_t Attributes);

//------------------------------------------------------------------------------
/** Returns the data size required for the image.
 *  Expects the Height and LineIncrement are
 *  already calculated. In case of color planes returns the size of one plane
 *  @param pImgInfo pointer to the @ref LvipImgInfo of the image.
 *  @return The data size required for the image in bytes.
 *  @ingroup GroupImgproc_FunctionsInit
 */
LVIP_DLLENTRY uint32_t LV_STDC LvipGetImageDataSize (LvipImgInfo* pImgInfo);

//------------------------------------------------------------------------------
/** Allocates appropriate space to pData or color planes, according to the Height and LineIncrement.
 *  @param pImgInfo pointer to the @ref LvipImgInfo of the image.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsInit
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipAllocateImageData(LvipImgInfo* pImgInfo);

//------------------------------------------------------------------------------
/** Deallocates the image data buffer(s)
 *  If the flags is not containing @ref LvipImgAttr_NotDataOwner, deallocates pData or color
 *  planes and sets them to NULL.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsInit
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipDeallocateImageData(LvipImgInfo* pImgInfo);

#ifdef _WIN32
//------------------------------------------------------------------------------
    /** Converts the header from Windows BITMAPINFO to @ref LvipImgInfo.
     *  @param pBmpInfo pointer to BITMAPINFO
     *  @param pImgInfo pointer to Image info structure
     *  @return LVSTATUS_OK in case of success, or error status in case of failure
     *  @ingroup GroupImgproc_FunctionsInit
     */
    LVIP_DLLENTRY LvStatus LV_STDC LvipBmpInfoToImgInfo (BITMAPINFO* pBmpInfo,
                                                         LvipImgInfo* pImgInfo);

//------------------------------------------------------------------------------
    /** Converts @ref LvipImgInfo to BITMAPINFO Info.
     *  The typical usage of this function is when you need to use Windows API functions for image display;
     *  these functions usually require a pointer to BITMAPINFO and a pointer to image data.
     *
     *  Note that Windows Device Independent Bitmap format does cover all the possible image formats of @ref LvipImgInfo
     *  (for example 9- to 16-bit mono formats, incompatible line increments etc.); in such case the conversion of the header fails
     *  and you will have to convert the image using the LvipConvertToPixelFormat() function first.
     *  Expects the BitmapInfo to have appropriate size, i.e. for example
     *  @code
     *    sizeof(BITAMAPINFOHEADER) + 256*sizeof(RGBQUAD) for 8-bit image
     *  @endcode
     *  @param pImgInfo The source @ref LvipImgInfo which will be converted to BITMAPINFO
     *  @param pBmpInfo The address of the destination BITMAPINFO structure
     *  @return LVSTATUS_OK in case of success, or error status in case of failure.
     *  @ingroup GroupImgproc_FunctionsInit
     */
    LVIP_DLLENTRY LvStatus LV_STDC LvipImgInfoToBmpInfo (LvipImgInfo* pImgInfo,
                                                         BITMAPINFO* pBmpInfo);
#else
    //LVIP_DLLENTRY LvStatus LvipImgInfoToXImage (LvipImgInfo* pImgInfo, XImage *xImg);
#endif

//------------------------------------------------------------------------------
/** Fills image data with specified color.
 *  @param pImgInfo pointer to @ref LvipImgInfo structure, the data of which has to be filled with the selected color
 *  @param Red     8bit Red value
 *  @param Green   8bit Green value
 *  @param Blue    8bit Blue value
 *  @param Options Options reserved - should be set to 0.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsInit
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipFillWithColor (LvipImgInfo* pImgInfo,
                                                  uint8_t Red,
                                                  uint8_t Green,
                                                  uint8_t Blue,
                                                  uint32_t Options);

//------------------------------------------------------------------------------
/** Allocates the LUT.
 *  @param LutType type of LUT which has to be allocated. One of @ref LvipLutType, this value could be optionally OR-ed with
 *                   the LVIP_LUT_BAYER flag
 *  @return handle to the allocated LUT
 *  @note LUT has to be freed up before end using LvipFreeLut() function
 *  @ingroup GroupImgproc_FunctionsLut
 */
LVIP_DLLENTRY LvipHLut LV_STDC LvipAllocateLut (uint32_t LutType);

//------------------------------------------------------------------------------
/** Deallocates the LUT.
 *  @param hLut Handle to LUT (which had been allocated by the LvipAllocateLut() function)
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsLut
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipFreeLut (LvipHLut hLut);

//------------------------------------------------------------------------------
/** Resets the LUT data to the linear order.
 *  @param hLut Handle to LUT allocated using the LvipAllocateLut() function
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsLut
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipResetLut (LvipHLut hLut);

//------------------------------------------------------------------------------
/** Sets up the 8-bit LUT data.
 *  Sets the LUT from 3 arrays of 256 uint8_t values. For processing the monochrome images 
 *  only the green is used.
 *
 *  @param hLut Handle to LUT
 *  @param pRed pointer to an array of 256 uint8_t values with red LUT values
 *  @param pGreen pointer to an array of 256 uint8_t values with green LUT values
 *  @param pBlue pointer to an array of 256 uint8_t values with blue LUT values
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsLut
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipSet8BitLut (LvipHLut hLut,
                                               uint8_t* pRed,
                                               uint8_t* pGreen,
                                               uint8_t* pBlue);

//------------------------------------------------------------------------------
/** Gets 8-bit LUT data.
 *  This function fills up supplied arrays with the current LUT data. It is useful for example after calling LvipAddGammaToLut() or
 *  LvipAddWbToLut() to get the values of current LUT.
 *  @param hLut Handle to LUT
 *  @param pRed pointer to an array of 256 uint8_t values, which will be filled with the Red LUT values
 *  @param pGreen pointer to an array of 256 uint8_t values, which will be filled with the Green LUT values
 *  @param pBlue pointer to an array of 256 uint8_t values, will be filled with the LUT values
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsLut
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipGet8BitLut  (LvipHLut hLut,
                                                uint8_t* pRed,
                                                uint8_t* pGreen,
                                                uint8_t* pBlue);

//------------------------------------------------------------------------------
/** Sets up 10-bit LUT data.
 *  Sets the LUT from 3 arrays of 1024 uint16_t values with 10-bit values.
 *  For processing the monochrome images only the green is used.
 *
 *  @param hLut Handle to LUT
 *  @param pRed pointer to an array of 1024 uint16_t red LUT values
 *  @param pGreen pointer to an array of 1024 uint16_t green LUT values
 *  @param pBlue pointer to an array of 1024 uint16_t blue LUT values
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsLut
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipSet10BitLut (LvipHLut hLut,
                                                uint16_t* pRed,
                                                uint16_t* pGreen,
                                                uint16_t* pBlue);

//------------------------------------------------------------------------------
/** Gets 10-bit LUT data.
 *  This function fills up supplied arrays with the current LUT data. It is useful for example after calling LvipAddGammaToLut() or
 *  LvipAddWbToLut() to get the values of current LUT.
 *  @param hLut Handle to LUT
 *  @param pRed pointer to an array of 1024 uint16_t values, will be filled with red
 *  @param pGreen pointer to an array of 1024 uint16_t values, will be filled with green
 *  @param pBlue pointer to an array of 1024 uint16_t values, will be filled with blue
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsLut
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipGet10BitLut (LvipHLut hLut,
                                                uint16_t* pRed,
                                                uint16_t* pGreen,
                                                uint16_t* pBlue);

//------------------------------------------------------------------------------
/** Sets up 12-bit LUT data.
 *  Sets the LUT from 3 arrays of 4096 uint16_t values with 12-bit values.
 *  For processing the monochrome images only the green is used.
 *
 *  @param hLut Handle to LUT
 *  @param pRed pointer to an array of 4096 uint16_t red LUT values
 *  @param pGreen pointer to an array of 4096 uint16_t green LUT values
 *  @param pBlue pointer to an array of 4096 uint16_t blue LUT values
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsLut
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipSet12BitLut (LvipHLut hLut,
                                                uint16_t* pRed,
                                                uint16_t* pGreen,
                                                uint16_t* pBlue);

//------------------------------------------------------------------------------
/** Gets 12-bit LUT data.
 *  This function fills up supplied arrays with the current LUT data. It is useful for example after calling LvipAddGammaToLut() or
 *  LvipAddWbToLut() to get the values of current LUT.
 *  @param hLut Handle to LUT
 *  @param pRed pointer to an array of 4096 uint16_t values, will be filled with red
 *  @param pGreen pointer to an array of 4096 uint16_t values, will be filled with green
 *  @param pBlue pointer to an array of 4096 uint16_t values, will be filled with blue
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsLut
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipGet12BitLut (LvipHLut hLut,
                                                uint16_t* pRed,
                                                uint16_t* pGreen,
                                                uint16_t* pBlue);

//------------------------------------------------------------------------------
/** Sets one 8-bit LUT value.
 *  This function writes one value to the LUT. Note that for writing the whole LUT a more effective 
 *  function LvipSet8BitLut() is available.
 *  @param hLut Handle to LUT
 *  @param LutSelector  LUT selector (see LvLUTSelector). The Luminance LUT is actually stored in the Green one.
 *  @param Index        Value index in the LUT.
 *  @param Value        The value.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsLut
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipSet8BitLutValue (LvipHLut hLut, 
                                                    LvEnum LutSelector,
                                                    uint32_t Index,
                                                    uint8_t Value);
//------------------------------------------------------------------------------
/** Gets one 8-bit LUT value.
 *  This function reads one value from the LUT. Note that for reading the whole LUT a more effective 
 *  function LvipGet8BitLut() is available.
 *  @param hLut         Handle to LUT
 *  @param LutSelector  LUT selector (see LvLUTSelector). The Luminance LUT is actually stored in the Green one.
 *  @param Index        Value index in the LUT
 *  @param pValue       Pointer to the variable which receives the value
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsLut
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipGet8BitLutValue (LvipHLut hLut, 
                                                    LvEnum LutSelector,
                                                    uint32_t Index,
                                                    uint8_t* pValue);

//------------------------------------------------------------------------------
/** Sets one 10-bit LUT value.
 *  This function writes one value to the LUT. Note that for writing the whole LUT a more effective 
 *  function LvipSet10BitLut() is available.
 *  @param hLut Handle to LUT
 *  @param LutSelector  LUT selector (see LvLUTSelector). The Luminance LUT is actually stored in the Green one.
 *  @param Index        Value index in the LUT.
 *  @param Value        The value.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsLut
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipSet10BitLutValue (LvipHLut hLut, 
                                                     LvEnum LutSelector,
                                                     uint32_t Index,
                                                     uint16_t Value);

//------------------------------------------------------------------------------
/** Gets 10-bit LUT value.
 *  This function reads one value from the LUT. Note that for reading the whole LUT a more effective 
 *  function LvipGet10BitLut() is available.
 *  @param hLut         Handle to LUT
 *  @param LutSelector  LUT selector (see LvLUTSelector). The Luminance LUT is actually stored in the Green one.
 *  @param Index        Value index in the LUT
 *  @param pValue       Pointer to the variable which receives the value
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsLut
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipGet10BitLutValue (LvipHLut hLut, 
                                                     LvEnum LutSelector,
                                                     uint32_t Index,
                                                     uint16_t* pValue);

//------------------------------------------------------------------------------
/** Sets one 12-bit LUT value.
 *  This function writes one value to the LUT. Note that for writing the whole LUT a more effective 
 *  function LvipSet12BitLut() is available.
 *  @param hLut Handle to LUT
 *  @param LutSelector  LUT selector (see LvLUTSelector). The Luminance LUT is actually stored in the Green one.
 *  @param Index        Value index in the LUT.
 *  @param Value        The value.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsLut
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipSet12BitLutValue (LvipHLut hLut, 
                                                     LvEnum LutSelector,
                                                     uint32_t Index,
                                                     uint16_t Value);

//------------------------------------------------------------------------------
/** Gets 12-bit LUT value.
 *  This function reads one value from the LUT. Note that for reading the whole LUT a more effective 
 *  function LvipGet12BitLut() is available.
 *  @param hLut         Handle to LUT
 *  @param LutSelector  LUT selector (see LvLUTSelector). The Luminance LUT is actually stored in the Green one.
 *  @param Index        Value index in the LUT
 *  @param pValue       Pointer to the variable which receives the value
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsLut
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipGet12BitLutValue (LvipHLut hLut, 
                                                     LvEnum LutSelector,
                                                     uint32_t Index,
                                                     uint16_t* pValue);

//------------------------------------------------------------------------------
// Gamma is multiplies by 1000
/** Adds gamma to LUT.
 *  Recalculates each value in the LUT table by applying the Gamma curve.
 *  Gamma is supplied multiplied by 1000, i.e. for gamma = 1.0 the passed value will be 1000.
 *  There is a possibility to do image lighter/darker using different gamma value. This gamma will be added to LUT
 *  and when the image is being transformed using any of function, add this LUT to this function as the last parameter.
 *  @note There is a need to have LUT - see LvipAllocateLut() and its company.
 *  @param Gamma Minimal gamma value is 10 - it means that there is a need to enter gamma multiplies by 1000
 *  @param hLut Handle to LUT
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsLut
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipAddGammaToLut (uint32_t Gamma,
                                                  LvipHLut hLut);

//------------------------------------------------------------------------------
/** Adds white balance to LUT.
 *  Recalculates each value in the LUT table by applying the white balance factors.
 *  The factors are supplied multiplied by 1000, i.e. for the factor = 1.0 the passed value will be 1000.
 *  See LvipCalcWbFactors() for obtaining the WB factors from an image.
 *
 *  @param FactorRed Red factor of white balance, multiplied by 1000
 *  @param FactorGreen Green factor of white balance, multiplied by 1000
 *  @param FactorBlue Blue factor of white balance, multiplied by 1000
 *  @param hLut Handle to the LUT
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsLut
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipAddWbToLut (uint32_t FactorRed,
                                               uint32_t FactorGreen,
                                               uint32_t FactorBlue,
                                               LvipHLut hLut);

//------------------------------------------------------------------------------
/** Adds offset and gain to LUT.
 *  Recalculates each value in the LUT table by adding the offset and multiplying by gain.
 *  The offset is in range -1000 to +1000, where 0 means no change and 1000
 *  the maximum pixel value - adding 1000 will make the image fully white,
 *  adding -1000 will make it fully black.
 *  The offset is corresponding to Brightness - 1000, see LvipAddBrightnessAndContrastToLut().
 *
 *  The gain is the gain factor multiplied by 1000, i.e. 1000 means 1.0 = no change.
 *  Can be also negative - 1000 makes inversion. It is equivalent to contrast.
 *
 *  @param Offset The Offset to be added expressed in 1/1000 of the maximum pixel value. See the explanation above.
 *  @param Gain The Gain factor multiplied by 1000.
 *  @param hLut Handle to the LUT
 *  @return LVSTATUS_OK in case of success, or error status in case of failure.
 *  @ingroup GroupImgproc_FunctionsLut
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipAddOffsetAndGainToLut(int32_t Offset,
                                                         int32_t Gain,
                                                         LvipHLut hLut);

//------------------------------------------------------------------------------
/** Adds brightness and contrast to LUT.
 *  Recalculates each value in the LUT table by adding the brightness and multiplying by contrast.
 *  This function is similar to the LvipAddOffsetAndGainToLut() function, with the following 2 differences:
 *  - The Brightness middle value is 1000, meaning no change. The Brightness 0 means black image and 2000 means
 *  fully white image, because subtracting or adding the 1000 means subtracting or adding the maximum pixel value.
 *  - The Brightness factor is internally corrected in dependence on contrast.
 *  The Contrast is equivalent to Gain in the LvipAddOffsetAndGainToLut() function. It is a factor multiplied
 *  by 1000, i.e. 1000 means 1.0 = no change. Can be also negative - 1000 makes inversion.
 *
 *  @param Brightness The Brightness to be added expressed in 1/1000 of the maximum pixel value. See the explanation above.
 *  @param Contrast The Contrast factor multiplied by 1000.
 *  @param hLut Handle to the LUT.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure.
 *  @ingroup GroupImgproc_FunctionsLut
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipAddBrightnessAndContrastToLut(int32_t Brightness,
                                                                 int32_t Contrast,
                                                                 LvipHLut hLut);


//------------------------------------------------------------------------------
/** Apply LUT to source image and save it in the destination image.
 *  Applies the LUT to the image. Note that the LUT can be applied in other functions as well, which is faster than
 *  this separate processing.
 *
 *  Supported input pixel formats: 8-bit mono, 10-bit mono, 12-bit mono, 24-bit BGR, 32-bit BGR, 24-bit RGB, 32-bit RGB.
 *  Supported output pixel formats: equal to the input pixel format.
 *  Can be done in-place: Yes (DstImgInfo can be NULL).
 *  @param pSrcImgInfo Source image info
 *  @param pDstImgInfo Destination image info
 *  @param hLut        Handle to the LUT, which has to be applied
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsLut
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipApplyLut (LvipImgInfo* pSrcImgInfo,
                                             LvipImgInfo* pDstImgInfo,
                                             LvipHLut hLut,
                                             uint32_t Options);

//------------------------------------------------------------------------------
/** Calculates white balance factors. The image is expected to be obtained from camera
 *  pointed at a neutral grey area.
 *  The factor is a gain applied to each pixel component. The gain = 1.0 means no change. In order
 *  to avoid using float numbers, the factors are multiplied by 1000 and stored in uint32_t.
 *  If the image pixel format is MONO, the image is expected to be Bayer Array encoded.
 *
 *  The factors are normalized, so that all are >= 1.0. This assures the areas with saturated colors
 *  remain white.
 *  The obtained factors could be used in the LvipAddWbToLut() function.
 *
 *  @note If the LvipOption_WbCorrectFactors flag is used, it is assumed that the
 *        white balance is calculated from the image to which were applied white balancing factors
 *        passed as parameters. Thus only a correction is calculated and the existing factors are
 *        modified. This flag cannot be used on undecoded Bayer array image.
 *
 *  @param pSrcImgInfo Source image info from which the white balance has to be calculated
 *  @param pFactorRed   Pointer to uint32_t variable to which will be saved the Red factor, multiplied by 1000.
 *                        If the LvipOption_WbCorrectFactors flag is used, the variable should contain the
 *                        factor already used for WB of the current image.
 *  @param pFactorGreen Pointer to uint32_t variable to which will be saved the Green factor, multiplied by 1000.
 *                        If the LvipOption_WbCorrectFactors flag is used, the variable should contain the
 *                        factor already used for WB of the current image.
 *  @param pFactorBlue  Pointer to uint32_t variable to which will be saved the Blue factor, multiplied by 1000.
 *                        If the LvipOption_WbCorrectFactors flag is used, the variable should contain the
 *                        factor already used for WB of the current image.
 *  @param Options      Options, see @ref LvipOption_WbCorrectFactors
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsLut
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipCalcWbFactors (LvipImgInfo* pSrcImgInfo,
                                                  uint32_t* pFactorRed,
                                                  uint32_t* pFactorGreen,
                                                  uint32_t* pFactorBlue,
                                                  uint32_t Options);

//------------------------------------------------------------------------------
/** Deinterlacing.
 *  Deinterlaces by averaging the neighbour lines.
 *  Deinterlace function reduces the artefacts resulting from capturing a moving object by an interlaced camera.
 *  - Supported input pixel formats: 8-bit mono, 15-bit BGR, 16-bit BGR, 24-bit BGR, 32-bit BGR.
 *  - Supported output pixel formats: equal to the input pixel format.
 *  - Can be done in-place: Yes (DstImgInfo can be NULL).
 *
 *  @param pSrcImgInfo Source image info
 *  @param pDstImgInfo Destination image info
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsRotate
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipDeinterlace(LvipImgInfo* pSrcImgInfo,
                                               LvipImgInfo* pDstImgInfo,
                                               uint32_t Options);

//------------------------------------------------------------------------------
/** Rotates the image by 90 degrees clockwise or counterclockwise.
 *
 *  - Supported input pixel formats: 8-bit mono, 15-bit BGR, 16-bit BGR, 24-bit BGR, 32-bit BGR.
 *  - Supported output pixel formats: equal to the input pixel format.
 *  - Can be done in-place: No.
 *
 *  @note the LUT in this function is not yet implemented.
 *  @note For 180 degrees rotation use the LvipMirror() function and set mirroring along both axes.
 *  @param pSrcImgInfo Source image info
 *  @param pDstImgInfo Destination image info
 *  @param ClockWise \b 1 if the image has to be rotated clockwise, \b 0 if the image
 *                    has to be rotated by counterclockwise
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @param hLut Handle to LUT
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsRotate
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipRotate90(LvipImgInfo* pSrcImgInfo,
                                            LvipImgInfo* pDstImgInfo,
                                            int32_t ClockWise,
                                            uint32_t Options,
                                            LvipHLut hLut);

//------------------------------------------------------------------------------
/** Mirrors the image along the horizontal axis (TopBottomMirror) or vertical axis (LeftRightMirror).
 *  - Supported input pixel formats: 8-bit mono, 15-bit BGR, 16-bit BGR, 24-bit BGR, 32-bit BGR.
 *  - Supported output pixel formats: equal to the input pixel format.
 *  - Can be done in-place: Yes (DstImgInfo can be NULL).
 *
 *  @param pSrcImgInfo Source image info
 *  @param pDstImgInfo Destination image info.
 *  @param TopBottomMirror \b 1 for top-bottom mirror, \b 0 if not
 *  @param LeftRightMirror \b 1 for left-right mirror, \b 0 if not
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @param hLut Handle to LUT
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @note the LUT in this function is not yet implemented.
 *  @ingroup GroupImgproc_FunctionsRotate
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipMirror(LvipImgInfo* pSrcImgInfo,
                                          LvipImgInfo* pDstImgInfo,
                                          int32_t TopBottomMirror,
                                          int32_t LeftRightMirror,
                                          uint32_t Options,
                                          LvipHLut hLut);

//------------------------------------------------------------------------------
/** It does the rotation and mirroring in the same step.
 *  If the Options contain LvipOption_ReallocateDst and the pDstImgInfo contains different image
 *  width or height or the pData is NULL, the pData is reallocated and the image parameters
 *  are adjusted. The Options in such case can contain also LvipImgAttr flags for new image descriptor
 *  creation.
 *  - Supported input pixel formats: 8-bit mono, 15-bit BGR, 16-bit BGR, 24-bit BGR, 32-bit BGR.
 *  - Supported output pixel formats: equal to the input pixel format.
 *  - Can be done in-place: No.
 *
 *  @param pSrcImgInfo Source image info
 *  @param pDstImgInfo Destination image info
 *  @param ClockWise \b 1 if image has to be rotated by 90 degrees clockwise, otherwise (counterclockwise) \b 0
 *  @param TopBottomMirror \b 1 if top-bottom mirror has to be used, otherwise \b 0
 *  @param LeftRightMirror \b 1 if left-right mirror has to be used, otherwise \b 0
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @param hLut Handle to LUT
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @note the LUT in this function is not yet implemented.
 *  @ingroup GroupImgproc_FunctionsRotate
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipRotate90AndMirror(LvipImgInfo* pSrcImgInfo,
                                                     LvipImgInfo* pDstImgInfo,
                                                     int32_t ClockWise,
                                                     int32_t TopBottomMirror,
                                                     int32_t LeftRightMirror,
                                                     uint32_t Options,
                                                     LvipHLut hLut);

//------------------------------------------------------------------------------
/** Reversed lines for switching between the top-down and bottom-up formats.
 *  Performs the same action as TopBottom mirror, but updates also the @ref LvipImgInfo with a flag
 *  indicating the orientation (this has a meaning when switching between top-down and bottom-up formats).
 *  - Supported input pixel formats: 8-bit mono, 15-bit BGR, 16-bit BGR, 24-bit BGR, 32-bit BGR.
 *  - Supported output pixel formats: equal to the input pixel format.
 *  - Can be done in-place: Yes (DstImgInfo can be NULL).
 *
 *  @param pSrcImgInfo Source image info
 *  @param pDstImgInfo Destination image info
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsRotate
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipReverseLines(LvipImgInfo* pSrcImgInfo,
                                                LvipImgInfo* pDstImgInfo,
                                                uint32_t Options);

//------------------------------------------------------------------------------
/** Fastly reverses lines (copying whole lines).
 *  The pDstImgInfo can be NULL (in-place reversion). In such case a temporary buffer for a line is needed.
 *
 *  The buffer can be supplied in pLineBuffer (must have sufficient size to hold the whole line in its
 *  pixel format, that means >= ImgInfo.dwLineIncrement).
 *
 *  If the pLineBuffer is NULL, the buffer is temporarily allocated and deallocated,
 *  which might require additional CPU time, so for the repeated call of this function it is
 *  better to allocate the buffer outside the function and pass it as pLineBuffer parameter.
 *  - Supported input pixel formats: 8-bit mono, 15-bit BGR, 16-bit BGR, 24-bit BGR, 32-bit BGR.
 *  - Supported output pixel formats: equal to the input pixel format.
 *  - Can be done in-place: Yes (DstImgInfo can be NULL).
 *
 *  @param pSrcImgInfo Source image info
 *  @param pDstImgInfo Destination image info; can be NULL - then in-place reversion will be done (but in this case a temporary buffer for
 *                     line is needed)
 *  @param pLineBuffer Pointer to temporary line buffer. Can be NULL.
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsRotate
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipReverseLinesFast(LvipImgInfo* pSrcImgInfo,
                                                    LvipImgInfo* pDstImgInfo,
                                                    void* pLineBuffer,
                                                    uint32_t Options);

//------------------------------------------------------------------------------
/** Extracts from the source bitmap a rectangle as destination bitmap.
 *  If the rectangle goes outside of the source image, the intersection rectangle is taken,
 *  that means the result width and/or height can be smaller than required.
 *
 *  If the intersection is zero, the function returns LVSTATUS_LVIP_DST_RECT_OUTSIDE_SRC.
 *  - Supported input pixel formats: 8-bit mono, 15-bit BGR, 16-bit BGR, 24-bit BGR, 32-bit BGR.
 *  - Supported output pixel formats: equal to the input pixel format.
 *  - Can be done in-place: No.
 *
 *  @param pSrcImgInfo Source image info
 *  @param pDstImgInfo Destination image info
 *  @param DstXOffset Left offset of the rectangle
 *  @param DstYOffset Upper offset of the rectangle
 *  @param DstWidth Width of area which has to be copied
 *  @param DstHeight Height of area which has to be copied
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsRoi
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipCopyArea(LvipImgInfo* pSrcImgInfo,
                                            LvipImgInfo* pDstImgInfo,
                                            int32_t DstXOffset,
                                            int32_t DstYOffset,
                                            uint32_t DstWidth,
                                            uint32_t DstHeight,
                                            uint32_t Options);

//------------------------------------------------------------------------------
/** Saves the image to the TIFF file.
 *  In contrast to the BMP format, it enables to store also 9- to 16-bit mono formats.
 *
 *  The flag LvipOption_TiffConvertTo16Bit can be used to force the conversion to 16bit format, which is
 *  supported by wider range of applications.
 *  -  Supported pixel formats: 8-bit to 16-bit mono, 15-bit BGR, 16-bit BGR, 24-bit BGR, 32-bit BGR.
 *
 *  @param pFileName File name
 *  @param pImgInfo Image info of an image to be saved
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *  @return LVSTATUS_OK in case of success, or error status in case of failure.
 *  @ingroup GroupImgproc_FunctionsSaveLoad
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipSaveToTiff(const char* pFileName,
                                              LvipImgInfo* pImgInfo,
                                              uint32_t Options);

//------------------------------------------------------------------------------
/** Loads the image from TIFF file.
 *  Is preferred to load the image from TIFF file which had been previously saved by LvipSaveToTiff() function -
 *  this library supports only a base TIFF format and there it is not assured that the TIFF image created by another
 *  application could be loaded without error.
 *
 *  @param pFileName File name
 *  @param pImgInfo Image info for the loaded image.
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @note The pData are always reallocated.
 *  @ingroup GroupImgproc_FunctionsSaveLoad
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipLoadFromTiff(const char* pFileName,
                                                LvipImgInfo* pImgInfo,
                                                uint32_t Options);

//------------------------------------------------------------------------------
/** Saves the image to a BMP file if the pixel format is compatible with BMP.
 *  \a Compatible \a with \a BMP means that @ref LvPixelFormat is one of 8-bit mono, 24- or 32-bit BGR.
 *  - Supported pixel formats: 8-bit mono, 15-bit BGR, 16-bit BGR, 24-bit BGR, 32-bit BGR.
 *
 *  @param pFileName File name
 *  @param pImgInfo Image info of an image to be saved
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsSaveLoad
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipSaveToBmp  (const char* pFileName,
                                               LvipImgInfo* pImgInfo,
                                               uint32_t Options);

//------------------------------------------------------------------------------
/** Loads image from BMP file.
 *  Formats with less 8 bits per pixel are not supported. The color palette by
 *  8-bit pixel format is expected to be greyscale.
 *  @param pFileName File name
 *  @param pImgInfo Image info for the loaded image
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsSaveLoad
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipLoadFromBmp(const char* pFileName,
                                               LvipImgInfo* pImgInfo,
                                               uint32_t Options);

//------------------------------------------------------------------------------
/** Saves the image to the JPEG file.
 *  In contrast to the BMP format, it enables to store also 9- to 16-bit mono formats.
 *
 *  -  Supported pixel formats: 8-bit to 16-bit mono, all RGB and BGR formats.
 *  For JPEG the native pixel format is either 8-bit mono or 24-bit BGR. If the image is in
 *  different pixel format, it is automatically converted to one of these 2 formats.
 *
 *  @param pFileName     File name.
 *  @param pImgInfo      Image info of an image to be saved.
 *  @param QualityFactor The quality factor in range from 0 to 100. The higher the quality, the lower is the compression. 
 *                       The default quality is 75.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure.
 *  @ingroup GroupImgproc_FunctionsSaveLoad
 */
#ifdef __cplusplus
LVIP_DLLENTRY LvStatus LV_STDC LvipSaveToJpeg(const char* pFileName,
                                              LvipImgInfo* pImgInfo,
                                              uint32_t QualityFactor = 75);
#else
LVIP_DLLENTRY LvStatus LV_STDC LvipSaveToJpeg(const char* pFileName,
                                              LvipImgInfo* pImgInfo,
                                              uint32_t QualityFactor);
#endif

//------------------------------------------------------------------------------
/** Loads the image from JPEG file.
 *  @param pFileName    File name.
 *  @param pImgInfo     Image info for the loaded image.
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *                     See @ref LvipOption_JpegConvertToBgr, @ref LvipOption_JpegReadHeaderOnly and @ref LvipOption_ReallocateDst.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @note    You can either supply the pImgInfo with already allocated buffer or use empty ImgInfo and specify 
 *           the @ref LvipOption_ReallocateDst flag. In the first case you can utilize the @ref LvipOption_JpegReadHeaderOnly flag
 *           to obtain the image attributes.
 *  @ingroup GroupImgproc_FunctionsSaveLoad
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipLoadFromJpeg(const char* pFileName,
                                                LvipImgInfo* pImgInfo,
                                                uint32_t Options);

//------------------------------------------------------------------------------
/** Converts the image from one pixel format to another one.
 *  - Supported input pixel formats: 8-bit to 16-bit mono, 15-bit BGR, 16-bit BGR, 24-bit BGR, 32-bit BGR.
 *  - Supported output pixel formats: 8-bit mono, 24-bit BGR, 32-bit BGR.
 *  - Can be done in-place: No.
 *
 *  @param pSrcImgInfo Source Image Info
 *  @param pDstImgInfo Destination Image Info
 *  @param DstPixelFormat Destination @ref LvPixelFormat
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsConversion
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipConvertToPixelFormat(LvipImgInfo* pSrcImgInfo,
                                                        LvipImgInfo* pDstImgInfo,
                                                        uint32_t DstPixelFormat,
                                                        uint32_t Options);


//------------------------------------------------------------------------------
/** Returns 1 if the source pixel format can be converted to destination pixel format by
 *  the LvipConvertToPixelFormat() function.
 *
 *  @param dwSrcPixelFormat Source @ref LvPixelFormat
 *  @param dwDstPixelFormat Destination @ref LvPixelFormat
 *  @return 1 in case of the conversion is available, otherwise 0.
 *  @ingroup GroupImgproc_FunctionsConversion
 */
LVIP_DLLENTRY uint32_t LV_STDC LvipCanConvertToPixelFormat(uint32_t dwSrcPixelFormat,
                                                           uint32_t dwDstPixelFormat);

//------------------------------------------------------------------------------
/** RGB color correction.
 *  A color correction 3x3 matrix is applied to RGB components of each pixel.
 *  -  Supported input pixel formats: 24-bit BGR, 32-bit BGR, 24-bit BGR, 32-bit BGR.
 *  -  Supported output pixel formats: equal to the input pixel format.
 *  -  Can be done in-place: Yes (DstImgInfo can be NULL).
 *
 *  @param pSrcImgInfo Image which needs to correct colors
 *  @param pDstImgInfo Where to save image with correct colors
 *  @param piMatrix    3x3 matrix used to correct colors. It could be filled up using LvipSetSaturationMatrix().
 *                     The factors in the matrix are expressed as multiplied by 1000, that means 1000 = factor 1.0.
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @param hLut Handle to LUT (could be NULL)
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @note the LUT in this function is not yet implemented.
 *  @ingroup GroupImgproc_FunctionsMatrix
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipApplyRgbColorCorrection (LvipImgInfo* pSrcImgInfo,
                                                            LvipImgInfo* pDstImgInfo,
                                                            int32_t* piMatrix,
                                                            uint32_t Options,
                                                            LvipHLut hLut);

//------------------------------------------------------------------------------
/** Sets up the color saturation 3x3 matrix.
 *  The saturation factor is in percents, eg. 100 = 1.0 = unchanged image.
 *  The matrix can be used as parameter in the LvipApplyRgbColorCorrection() function.
 *  @param SaturationFactor the saturation factor in percents
 *  @param piMatrix  3x3 matrix of int32_t values which will obtain the calculated factors
 *  @param Options   Options, reserved for future use, must be 0.  
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsMatrix
 */
#ifdef __cplusplus
LVIP_DLLENTRY LvStatus LV_STDC LvipSetSaturationMatrix (uint32_t SaturationFactor,
                                                        int32_t* piMatrix,
                                                        uint32_t Options = 0);
#else
LVIP_DLLENTRY LvStatus LV_STDC LvipSetSaturationMatrix (uint32_t SaturationFactor,
                                                        int32_t* piMatrix,
                                                        uint32_t Options);
#endif

//------------------------------------------------------------------------------
/** Sets up sharpening matrix.
 *  Fills the matrix with weighted values for 3x3 sharpening.
 *
 *  The factor is 0 for no-change matrix, +100 for maximum sharpening, -100 for blurring
 *  @param Factor Factor of sharpening
 *  @param piMatrix 3x3 matrix of int32_t values which will obtain the calculated values
 *  @param Options     Options: 0 for faster sharpening from 4 neighboring pixels, 
 *                              1 for full sharpening from 8 neighboring pixels. 
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsMatrix
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipSet3x3MatrixSharpening (int32_t Factor,
                                                           int32_t* piMatrix,
                                                           uint32_t Options);

//------------------------------------------------------------------------------
/** Does 3x3 convolution.
 *  Applies the 3x3 matrix convolution operation. Typically, if the matrix is set for sharpening, sharpens the image.
 *
 *  @sa LvipSet3x3MatrixSharpening() for creation of the sharpening matrix.
 *  - Supported input pixel formats: 8-bit to 16-bit mono, 24-bit BGR, 32-bit BGR.
 *  - Supported output pixel formats: equal to the input pixel format.
 *  - Can be done in-place: No.
 *
 *  @param pSrcImgInfo Source image
 *  @param pDstImgInfo Destination image info
 *  @param piMatrix Matrix for the convolution operation.
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @param hLut Handle to LUT
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @note the LUT in this function is not yet implemented.
 *  @ingroup GroupImgproc_FunctionsMatrix
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipApply3x3Convolution (LvipImgInfo* pSrcImgInfo,
                                                        LvipImgInfo* pDstImgInfo,
                                                        int32_t* piMatrix,
                                                        uint32_t Options,
                                                        LvipHLut hLut);

//------------------------------------------------------------------------------
/** Applies the shading correction.
 *  The pBlackRefImgInfo and pWhiteRefImgInfo must be either NULL or must point to a
 *  valid image of the same pixel format as the pSrcImgInfo.
 *  - Supported input pixel formats: 8-bit to 16-bit mono, 24-bit BGR, 32-bit BGR.
 *  - Supported output pixel formats: equal to the input pixel format.
 *  - Can be done in-place: Yes (DstImgInfo can be NULL).
 *
 *  @param pSrcImgInfo Source image info
 *  @param pDstImgInfo Destination image info
 *  @param pBlackRefImgInfo Black reference image
 *  @param pWhiteRefImgInfo White reference image
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @param hLut Handle to LUT
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsShading
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipApplyShadingCorrection (LvipImgInfo* pSrcImgInfo,
                                                           LvipImgInfo* pDstImgInfo,
                                                           LvipImgInfo* pBlackRefImgInfo,
                                                           LvipImgInfo* pWhiteRefImgInfo,
                                                           uint32_t Options,
                                                           LvipHLut hLut);

//------------------------------------------------------------------------------
/** Bayer Decoding: Show mosaic.
 *  This function converts the Bayer encoded image to RGB format, without decoding the color information, only
 *  showing in the color how the image is seen by the chip after the light goes through the color lenses. The
 *  purpose of this function is only to help in Bayer decoding algorithms investigations.
 *  -  Supported input pixel formats: 8-bit mono.
 *  -  Supported output pixel formats: 24-bit BGR, 32-bit BGR.
 *  -  Can be done in-place: No.
 *
 *  @param pSrcImgInfo Bayer encoded image
 *  @param pDstImgInfo Where to save displayable image
 *  @param DstPixelFormat To which @ref LvPixelFormat convert bayer encoded image
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsBayer
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipBdShowMosaic (LvipImgInfo* pSrcImgInfo,
                                                 LvipImgInfo* pDstImgInfo,
                                                 uint32_t DstPixelFormat,
                                                 uint32_t Options);

//------------------------------------------------------------------------------
/** Bayer Decoding: Convert green to greyscale
 *  Converts fast but roughly the Bayer encoded image to a greyscale
 *  by using only the green pixels, which cover the half of all pixels.
 *  The other half is calculated by bilinear interpolation.
 *  - Supported input pixel formats: 8-bit mono.
 *  - Supported output pixel formats: 8-bit mono.
 *  - Can be done in-place: No.
 *
 *  @param pSrcImgInfo source image info
 *  @param pDstImgInfo destination image info
 *  @param DstPixelFormat destination @ref LvPixelFormat
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsBayer
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipBdGreenToGreyscale (LvipImgInfo* pSrcImgInfo,
                                                       LvipImgInfo* pDstImgInfo,
                                                       uint32_t DstPixelFormat,
                                                       uint32_t Options);

//------------------------------------------------------------------------------
/** Bayer Decoding: The Nearest Neigbour method
 *  The fastest method for Bayer array decoding.
 *  It uses the nearest pixel with the required lense color to get the
 *  pixel value. Gives rough results.
 *  - Supported input pixel formats: 8-bit mono.
 *  - Supported output pixel formats: 8-bit mono, 24-bit BGR, 32-bit BGR.
 *  - Can be done in-place: No.
 *
 *  @param pSrcImgInfo Bayer encoded source image info
 *  @param pDstImgInfo Destination image info
 *  @param DstPixelFormat To which @ref LvPixelFormat
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @param hLut Handle to LUT
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsBayer
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipBdNearestNeighbour (LvipImgInfo* pSrcImgInfo,
                                                       LvipImgInfo* pDstImgInfo,
                                                       uint32_t DstPixelFormat,
                                                       uint32_t Options,
                                                       LvipHLut hLut);

//------------------------------------------------------------------------------
/** Bayer Decoding: The Bilinear Interpolation method
 *  The most commonly used method for fast Bayer decoding.
 *  For the color not directly available for the given pixel makes the
 *  linear interpolation between the 2 or 4 neigbouring pixels to get it.
 *  Gives good results with a high speed.
 *  - Supported input pixel formats: 8-bit mono.
 *  - Supported output pixel formats: 8-bit mono, 24-bit BGR, 32-bit BGR.
 *  - Can be done in-place: No.
 *
 *  @param pSrcImgInfo Bayer encoded source image info
 *  @param pDstImgInfo Destination image info
 *  @param DstPixelFormat To which @ref LvPixelFormat
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @param hLut Handle to LUT
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsBayer
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipBdBilinearInterpolation (LvipImgInfo* pSrcImgInfo,
                                                            LvipImgInfo* pDstImgInfo,
                                                            uint32_t DstPixelFormat,
                                                            uint32_t Options,
                                                            LvipHLut hLut);

//------------------------------------------------------------------------------
/** Bayer Decoding: The Bilinear interpolation with Linear Color Correction method
 *  The interpolation with Linear Color Correction (LCC) is another adaptive algorithm and optimized
 *  for images with edges in horizontal and vertical direction.
 *  @note This function does not support LUT due to the 2-pass algorithm
 *  @param pSrcImgInfo Pointer to source image info
 *  @param pDstImgInfo Pointer to destination image info
 *  @param DstPixelFormat to which @ref LvPixelFormat has to be image converted
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsBayer
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipBdBilinearColorCorrection (LvipImgInfo* pSrcImgInfo,
                                                              LvipImgInfo* pDstImgInfo,
                                                              uint32_t DstPixelFormat,
                                                              uint32_t Options);

//------------------------------------------------------------------------------
/** Bayer Decoding: Variable gradients method
 *  One of the best known methods for Bayer decoding, but significantly slower than the bilinear interpolation.
 *  It is based on evaluation the color gradients in 8 directions around the pixel and selecting the
 *  set of best set for the color interpolation.
 *  - Supported input pixel formats: 8-bit mono.
 *  - Supported output pixel formats: 24-bit BGR, 32-bit BGR.
 *  - Can be done in-place: No.
 *
 *  @param pSrcImgInfo Bayer encoded source image info
 *  @param pDstImgInfo Destination image info
 *  @param DstPixelFormat To which @ref LvPixelFormat convert
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @param hLut Handle to LUT
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsBayer
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipBdVariableGradients (LvipImgInfo* pSrcImgInfo,
                                                        LvipImgInfo* pDstImgInfo,
                                                        uint32_t DstPixelFormat,
                                                        uint32_t Options,
                                                        LvipHLut hLut);

//------------------------------------------------------------------------------
/** Bayer Decoding: The Pixel Grouping method.
 *  A method similar to the LvipBdVariableGradients(), but simplified and thus faster, still giving very good results.
 *  - Supported input pixel formats: 8-bit mono.
 *  - Supported output pixel formats: 24-bit BGR, 32-bit BGR.
 *  - Can be done in-place: No.
 *
 *  @note This function does not support LUT operations because of too high CPU load
 *  @param pSrcImgInfo Bayer encoded source image info
 *  @param pDstImgInfo Destination image info
 *  @param DstPixelFormat To which @ref LvPixelFormat convert
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsBayer
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipBdPixelGrouping (LvipImgInfo* pSrcImgInfo,
                                                    LvipImgInfo* pDstImgInfo,
                                                    uint32_t DstPixelFormat,
                                                    uint32_t Options);

//------------------------------------------------------------------------------
/** This function encode an RGB image back to a Bayer encoded image.
 *  This function is generally for testing purposes.
 *  @param pSrcImgInfo RGB source image info
 *  @param pDstImgInfo It will contain bayer encoded image
 *  @param DstPixelFormat To which @ref LvPixelFormat convert
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @return LVSTATUS_OK in case of success, or error status in case of failure
 *  @ingroup GroupImgproc_FunctionsBayer
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipBdEncodeToBayer (LvipImgInfo* pSrcImgInfo,
                                                    LvipImgInfo* pDstImgInfo,
                                                    uint32_t DstPixelFormat,
                                                    uint32_t Options);

//------------------------------------------------------------------------------
/** Converts the image to format compatible with Windows Device Independent Bitmap (DIB/BMP), if necessary.
 *  If the pixel format is not compatible
 *  for display, it uses pConvImgInfo for converting the image - mono images to 8-bit
 *  color to 24-bit BGR.
 *
 *  If pConvImgInfo is NULL, and the image has non-compatible format, the image is not converted.
 *  If the pConvImgInfo is not NULL, it is used for conversion. It is the user responsibility
 *  to deallocate this image when not needed anymore.
 *
 *  Recommended: Create empty @ref LvipImgInfo for pConvImgInfo and in the Options specify the LvipOption_ReallocateDst
 *  flag. The function will allocate the appropriate image buffer for the conversion.
 *
 *  Next time this function is called it only checks if the buffer is sufficient, if so, it does
 *  not reallocate it. Call LvipDeallocateImageData() to deallocate this helper image at the end of your application.
 *  - Supported pixel formats: 8-bit to 16-bit mono, 15-bit BGR, 16-bit BGR, 24-bit BGR, 32-bit BGR.
 *
 *  @param pImgInfo pointer to Image info of the image to be converted
 *  @param pConvImgInfo If conversion to compatible format has to be done, there is a need to give as param to initialized
 *                      conversion image info structure which will be used for the conversion
 *  @param Options     Options - OR-ed combination of @ref LvipOption. 
 *                     If the LvipOption_ReallocateDst is used, then also 
 *                     can contain attributes from @ref LvipImgAttr for (re)allocated image.
 *  @param pConversionNeeded Indicates if image format needs conversion
 *  @return \b 0 in case of failure (use LvipGetLastStatus() for details), 1 in case the image was converted, 2 in case no conversion was needed.
 *  @ingroup GroupImgproc_FunctionsDisplay
 */
LVIP_DLLENTRY LvStatus LV_STDC LvipConvertToWinBmpCompatibleFormat (LvipImgInfo* pImgInfo,
                                                                    LvipImgInfo* pConvImgInfo,
                                                                    uint32_t Options,
                                                                    int32_t* pConversionNeeded);

//------------------------------------------------------------------------------
// Specific functions

LVIP_DLLENTRY LvStatus LV_STDC Lvip3x3ThresholdSharpening (LvipImgInfo* pSrcImgInfo,
                                                           LvipImgInfo* pDstImgInfo,
                                                           uint32_t Factor0,
                                                           uint32_t Threshold1,
                                                           uint32_t Factor1,
                                                           uint32_t Threshold2,
                                                           uint32_t Factor2,
                                                           uint32_t Options,
                                                           LvipHLut hLut);

#ifdef _WIN32

//------------------------------------------------------------------------------
    /** Creates an overlay object.
     *  Creates an overlay object, which can be used by the following functions:
     *  - the overlay can be filled with a color using the LvipWipeOverlay() function,
     *  - text can be written to the overlay using the LvipWriteTextToOverlay() function,
     *  - a bitmap can placed to the overlay using the LvipPutBitmapToOverlay() and LvipPutBitmapToOverlayFromBmpFile() functions
     *  - Windows GDI functions can be used to paint to the overlay using the LvipGetOverlayDc() and LvipReleaseOverlayDc() functions,
     *  - transparent color can be specified using the LvipSetOverlayTransparentColor() function.
     *  The overlay can be placed on an image using the LvipPaintOverlay() function. The overlay functionality is based
     *  on Windows GDI, thus the overlay functions are available only in the Windows version of the library and only
     *  Windows DIB compatible pixel formats are supported.
     *  @param Width Width of the overlay. The smaller is the overlay, the faster is the process of combining
     *     the overlay with the destination image, so use only necessary size. The position of the overlay is specified
     *     at the time of its painting using the LvipPaintOverlay() function.
     *  @param Height Height of the overlay.
     *  @param PixelFormat The pixel format of the overlay. Must be equal to the pixel format of the destination image
     *   with which the overlay is to be combined. Only Windows DIB compatible pixel formats are supported:
     *   @ref LVIP_PIXELFORMAT_MONO8, @ref LVIP_PIXELFORMAT_BGR555_PACKED,
     *   @ref LVIP_PIXELFORMAT_BGR565_PACKED, @ref LVIP_PIXELFORMAT_BGR8_PACKED, @ref LVIP_PIXELFORMAT_BGRA8_PACKED.
     *  @return Handle to the created Overlay object or NULL if the creation fails.
     *  @note Available in the Windows version of the library only.
     *  @ingroup GroupImgproc_FunctionsOverlay
     */
    LVIP_DLLENTRY LvipHOverlay LV_STDC LvipCreateOverlay (uint32_t Width,
                                                          uint32_t Height,
                                                          uint32_t PixelFormat);

//------------------------------------------------------------------------------
    /** Deletes existing object.
     *  Deletes existing overlay object identified by its handle.
     *  @param hOverlay handle to the overlay created by the LvipCreateOverlay() function.
     *  @return LVSTATUS_OK in case of success, or error status in case of failure
     *  @note Available in the Windows version of the library only.
     *  @ingroup GroupImgproc_FunctionsOverlay
     */
    LVIP_DLLENTRY LvStatus LV_STDC LvipDeleteOverlay (LvipHOverlay hOverlay);

//------------------------------------------------------------------------------
    /** Returns overlay width.
     *  Returns the overlay width specified in the LvipCreateOverlay() function, when the overlay was created.
     *  @param hOverlay handle to the overlay created by the LvipCreateOverlay() function.
     *  @return Width of the overlay.
     *  @note Available in the Windows version of the library only.
     *  @ingroup GroupImgproc_FunctionsOverlay
     */
    LVIP_DLLENTRY uint32_t LV_STDC LvipGetOverlayWidth (LvipHOverlay hOverlay);

//------------------------------------------------------------------------------
    /** Returns overlay height.
     *  Returns the overlay height specified in the LvipCreateOverlay() function, when the overlay was created.
     *  @param hOverlay handle to the overlay created by the LvipCreateOverlay() function.
     *  @return Height of the overlay.
     *  @note Available in the Windows version of the library only.
     *  @ingroup GroupImgproc_FunctionsOverlay
     */
    LVIP_DLLENTRY uint32_t LV_STDC LvipGetOverlayHeight (LvipHOverlay hOverlay);

//------------------------------------------------------------------------------
    /** Returns overlay pixel format.
     *  Returns the overlay pixel format specified in the LvipCreateOverlay() function, when the overlay was created.
     *  @param hOverlay handle to the overlay created by the LvipCreateOverlay() function.
     *  @return Pixel format of the overlay.
     *  @note Available in the Windows version of the library only.
     *  @ingroup GroupImgproc_FunctionsOverlay
     */
    LVIP_DLLENTRY uint32_t LV_STDC LvipGetOverlayPixelFormat (LvipHOverlay hOverlay);

//------------------------------------------------------------------------------
    /** Paints the overlay at specified position.
     *  Paints the overlay at specified position in the destination image.
     *  @param hOverlay Handle to the overlay created by the LvipCreateOverlay() function.
     *  @param pDstImgInfo Destination image info. Note that the overlay must have the same pixel format as the destination image.
     *  @param XPos X-axis position of the upper left corner of the overlay in the image.
     *  @param YPos Y-axis position of the upper left corner of the overlay in the image. Note that the upper left corner of the
     *         image has coordinates 0,0 and the X-axis goes from left to right and Y-axis from top to bottom.
     *  @return LVSTATUS_OK in case of success, or error status in case of failure.
     *  The overlay can be placed partially or fully outside of the destination image; in such case it is clipped.
     *  @note Available in the Windows version of the library only.
     *  @ingroup GroupImgproc_FunctionsOverlay
     */
    LVIP_DLLENTRY LvStatus LV_STDC LvipPaintOverlay (LvipHOverlay hOverlay,
                                                     LvipImgInfo* pDstImgInfo,
                                                     int32_t XPos,
                                                     int32_t YPos);

//------------------------------------------------------------------------------
    /** Fills the whole overlay with a specified color.
     *  Fills the whole overlay with a specified color. Initially the overalay is filled with a black color.
     *
     *  @param hOverlay Handle to the overlay created by the LvipCreateOverlay() function.
     *  @param Color Color defined as the COLORREF value, which has the 0x00bbggrr hexadecimal form.
     *         The low-order byte contains a value for the relative intensity of red; the second byte contains a
     *         value for green; and the third byte contains a value for blue. The high-order byte must be zero.
     *  @return LVSTATUS_OK in case of success, or error status in case of failure.
     *  @note Available in the Windows version of the library only.
     *  @ingroup GroupImgproc_FunctionsOverlay
     */
    LVIP_DLLENTRY LvStatus LV_STDC LvipWipeOverlay (LvipHOverlay hOverlay,
                                                    uint32_t Color);

//------------------------------------------------------------------------------
    /** Specifies the overlay transparent color.
     *  Specifies the transparent color. When combining the overlay with the destination image, the
     *  pixels in the overlay with the transparent color are not copied to the the destination image.
     *
     *  Initially, the transparent color is set to @ref LVIP_COLOR_NONE, that means no overlay transparency is used.
     *
     *  @param hOverlay Handle to the overlay created by the LvipCreateOverlay() function.
     *  @param Color Color defined as the COLORREF value, which has the 0x00bbggrr hexadecimal form.
     *         The low-order byte contains a value for the relative intensity of red; the second byte contains a
     *         value for green; and the third byte contains a value for blue. The high-order byte must be zero.
     *         Use @ref LVIP_COLOR_NONE for disabling the transparency.
     *  @return LVSTATUS_OK in case of success, or error status in case of failure.
     *  @note Available in the Windows version of the library only.
     *  @ingroup GroupImgproc_FunctionsOverlay
     */
    LVIP_DLLENTRY LvStatus LV_STDC LvipSetOverlayTransparentColor (LvipHOverlay hOverlay,
                                                                   uint32_t Color);

//------------------------------------------------------------------------------
    /** Returns the overlay transparent color.
     *  Returns the overlay transparent color, set by the LvipSetOverlayTransparentColor() function.
     *
     *  @param hOverlay Handle to the overlay created by the LvipCreateOverlay() function.
     *  @return The transparent color as the COLORREF value or the @ref LVIP_COLOR_NONE constant.
     *  @note Available in the Windows version of the library only.
     *  @ingroup GroupImgproc_FunctionsOverlay
     */
    LVIP_DLLENTRY uint32_t LV_STDC LvipGetOverlayTransparentColor (LvipHOverlay hOverlay);

//------------------------------------------------------------------------------
    /** Returns Windows device context for painting on the overlay.
     *  Returns Windows device context for painting on the overlay. Thus you can use Windows GDI functions, like
     *  MoveToEx(), LineTo(), DrawText() etc. for free painting on the overlay object. It may be useful to wipe the
     *  overlay object before painting by the LvipWipeOverlay() function.
     *
     *  Be sure you do not forget to free the device context by the LvipReleaseOverlayDc() function after the painting is done.
     *
     *  @param hOverlay Handle to the overlay created by the LvipCreateOverlay() function.
     *  @return The device context of the overlay object, or NULL in case of failure.
     *  @note Available in the Windows version of the library only.
     *  @ingroup GroupImgproc_FunctionsOverlay
     */
    LVIP_DLLENTRY HDC LV_STDC  LvipGetOverlayDc (LvipHOverlay hOverlay);

//------------------------------------------------------------------------------
    /** Releases the Windows device context of the overlay.
     *  Releases the Windows device context of the overlay obtained by LvipGetOverlayDc() function.
     *
     *  @param hOverlay Handle to the overlay created by the LvipCreateOverlay() function.
     *  @return LVSTATUS_OK in case of success, or error status in case of failure.
     *  @note Available in the Windows version of the library only.
     *  @ingroup GroupImgproc_FunctionsOverlay
     */
    LVIP_DLLENTRY LvStatus LV_STDC LvipReleaseOverlayDc (LvipHOverlay hOverlay);

//------------------------------------------------------------------------------
    /** Sets the parameters for the overlay text.
     *  Sets the parameters for the overlay text, which can be written to the overlay by the LvipWriteTextToOverlay() function.
     *
     *  @param hOverlay Handle to the overlay created by the LvipCreateOverlay() function.
     *  @param pFont Name of the font, for example "Arial".
     *  @param Size Size of the font.
     *  @param Color Color of the font, as the COLORREF value (see lvx_WipeOverlay() for details about the color values).
     *  @param OutlineColor Color for the font shadow or outline, as the COLORREF value.
     *  @param Attributes Font attributes. Use the @ref LvipTextAttr constants to specify attributes. You can use
     *         PLUS or bitwise OR operator to combine multiple attributes.
     *  @param CharSet The character set to be used. Use Windows constants: DEFAULT_CHARSET=1, ANSI_CHARSET=0, etc.
     *  @return LVSTATUS_OK in case of success, or error status in case of failure.
     *  @note Available in the Windows version of the library only.
     *  @ingroup GroupImgproc_FunctionsOverlay
     */
    LVIP_DLLENTRY LvStatus LV_STDC LvipSetOverlayTextParams (LvipHOverlay hOverlay,
                                                             const char* pFont,
                                                             int32_t Size,
                                                             uint32_t Color,
                                                             uint32_t OutlineColor,
                                                             uint32_t Attributes,
                                                             uint32_t CharSet);

//------------------------------------------------------------------------------
    /** Writes the text to the overlay at the specified position.
     *  Writes the text to the overlay at the specified position, using the text attributes set by the
     *  LvipSetOverlayTextParams() function.
     *
     *  @param hOverlay Handle to the overlay created by the LvipCreateOverlay() function.
     *  @param pText Text to be written.
     *  @param XOffset Horizontal distance of the text upper left corner from the overlay rectangle left side.
     *  @param YOffset Vertical distance of the text upper left corner from the overlay rectangle top side.
     *  @return LVSTATUS_OK in case of success, or error status in case of failure.
     *  @note Available in the Windows version of the library only.
     *  @note If you want to change the existing text in the overlay, you must first
     *  wipe the area of the existing text and then to place the new text.
     *  @ingroup GroupImgproc_FunctionsOverlay
     */
    LVIP_DLLENTRY LvStatus LV_STDC LvipWriteTextToOverlay (LvipHOverlay hOverlay,
                                                           const char* pText,
                                                           int32_t XOffset,
                                                           int32_t YOffset);

//------------------------------------------------------------------------------
    /** Places a bitmap to the overlay.
     *  Places a bitmap to the overlay. The bitmap must be in Windows Device Independent Bitmap (DIB) format.
     *
     *  @param hOverlay Handle to the overlay created by the LvipCreateOverlay() function.
     *  @param pBitmapInfo Pointer to the bitmap header (in DIB format).
     *  @param pData Pointer to the bitmap data.
     *  @param XOffset Horizontal distance of the bitmap upper left corner from the overlay rectangle left side.
     *  @param YOffset Vertical distance of the bitmap upper left corner from the overlay rectangle top side.
     *  @return LVSTATUS_OK in case of success, or error status in case of failure.
     *  @note Available in the Windows version of the library only.
     *  @ingroup GroupImgproc_FunctionsOverlay
     */
    LVIP_DLLENTRY LvStatus LV_STDC LvipPutBitmapToOverlay (LvipHOverlay hOverlay,
                                                           BITMAPINFO* pBitmapInfo,
                                                           uint8_t* pData,
                                                           int32_t XOffset,
                                                           int32_t YOffset);

//------------------------------------------------------------------------------
    /** Places a bitmap from BMP file to the overlay.
     *  Places a bitmap from BMP file to the overlay. The bitmap must be in Windows Device Independent Bitmap (DIB) format.
     *
     *  @param hOverlay Handle to the overlay created by the LvipCreateOverlay() function.
     *  @param pFileName Name of the BMP file, including the full path.
     *  @param XOffset Horizontal distance of the bitmap upper left corner from the overlay rectangle left side.
     *  @param YOffset Vertical distance of the bitmap upper left corner from the overlay rectangle top side.
     *  @return LVSTATUS_OK in case of success, or error status in case of failure.
     *  @note Available in the Windows version of the library only.
     *  @ingroup GroupImgproc_FunctionsOverlay
     */
    LVIP_DLLENTRY LvStatus LV_STDC LvipPutBitmapToOverlayFromBmpFile (LvipHOverlay hOverlay,
                                                                      const char* pFileName,
                                                                      int32_t XOffset,
                                                                      int32_t YOffset);

//------------------------------------------------------------------------------
#endif /* _WIN32 */

#endif
