#ifndef SV_SYNVIEW_DEFS_H
#define SV_SYNVIEW_DEFS_H

//------------------------------------------------------------------------------
// The following typedefs are normally included in stdint.h, but this file
// is not included in MS Visual Studio before version 2010. More info at:
//
//    http://en.wikipedia.org/wiki/Stdint.h
//    Google search for: visual studio stdint.h
//

#if defined(_WIN32)
    #ifndef _STDINT_H 
        #ifdef _MSC_VER // Microsoft compiler
            #if _MSC_VER < 1600
                typedef __int8            int8_t;
                typedef __int16           int16_t;
                typedef __int32           int32_t;
                typedef __int64           int64_t;
                typedef unsigned __int8   uint8_t;
                typedef unsigned __int16  uint16_t;
                typedef unsigned __int32  uint32_t;
                typedef unsigned __int64  uint64_t;
            #else
                // In Visual Studio 2010 is stdint.h already included
                #include <stdint.h>
            #endif
        #else
            // Not a Microsoft compiler
            #include <stdint.h>
        #endif
    #endif 
#else
    // Linux
    #include <stdint.h>
#endif

//------------------------------------------------------------------------------
//  OPERATING SYSTEM DEFINES
//------------------------------------------------------------------------------

#include "sv.synview.status.h"
#include <stddef.h>

#ifdef WIN32
    #ifndef _WIN32
        #define _WIN32
    #endif
#endif

#ifdef _WIN32
    #define _WINSOCKAPI_  /* Prevent inclusion of winsock.h in windows.h - From winsock2.h */
    #include <windows.h>

    #define LV_DLLIMPORT   __declspec(dllimport)
    #define LV_DLLEXPORT   __declspec(dllexport)

    /** Macro for stdcall calling convention of an API function.
     *  @ingroup GroupSynview_Defines */
    #define LV_STDC __stdcall
    /** Macro for cdecl calling convention of an API function.
     *  @ingroup GroupSynview_Defines */
    #define LV_CDEC __cdecl

    #if defined(__cplusplus)
        #define LV_EXTC extern "C"
    #else
        #define LV_EXTC
    #endif
#else
    // remove the None #define conflicting with GenApi
    #undef None

    #if __GNUC__>=4
        #define LV_DLLIMPORT   __attribute__((visibility("default")))
        #define LV_DLLEXPORT   __attribute__((visibility("default")))

        #if defined(__i386__)
            /** Macro for stdcall calling convention of an API function.
             *  @ingroup GroupSynview_Defines */
            #define LV_STDC __attribute__((stdcall))
            /** Macro for cdecl calling convention of an API function.
             *  @ingroup GroupSynview_Defines */
            #define LV_CDEC __attribute__((cdecl))
        #else
            #define LV_STDC 
            #define LV_CDEC 
        #endif

        #if defined(__cplusplus)
            #define LV_EXTC extern "C"
        #else
            #define LV_EXTC
        #endif
    #else
        #error Unknown compiler
    #endif
#endif


#ifdef LV_SIMPLON_DLL
    /** Typedef for the API functions calling convention and export/import type.
     *  @ingroup GroupSynview_Defines */
    #define LV_DLLENTRY LV_EXTC LV_DLLEXPORT
#else
    /** Typedef for the API functions calling convention and export/import type.
     *  @ingroup GroupSynview_Defines */
    #define LV_DLLENTRY LV_EXTC LV_DLLIMPORT
#endif

#ifdef LV_IMGPROC_DLL
    /** Typedef for the API functions calling convention and export/import type.
     *  @ingroup GroupSynview_Defines */
    #define LVIP_DLLENTRY LV_EXTC LV_DLLEXPORT
#else
    /** Typedef for the API functions calling convention and export/import type.
     *  @ingroup GroupSynview_Defines */
    #define LVIP_DLLENTRY LV_EXTC LV_DLLIMPORT
#endif

//------------------------------------------------------------------------------

/** @defgroup GroupSynview SynView
 */

/** @defgroup GroupSynview_Defines SynView defines and typedefs
 *  @ingroup GroupSynview
 */

/** @defgroup GroupSynview_Enums SynView enumerations
 *  @ingroup GroupSynview
 */

/** @defgroup GroupImgproc SynView Image Processing Library
 *  @ingroup GroupSynview
 */

/** @defgroup GroupImgproc_Defines Image Processing Library defines, typedefs and enums
 *  @ingroup GroupImgproc
 */

/** @mainpage SynView Reference Guide
 */

//------------------------------------------------------------------------------
//  DEFINES
//------------------------------------------------------------------------------
// the following defines are signs added to enum constants
// LOC - locally implemented features
#ifdef SIMPLON_INC_OBSOLETE
    #define LV_SYS_LOC    0x00100000ul
    #define LV_ITF_LOC    0x00200000ul
    #define LV_DEV_LOC    0x00300000ul
    #define LV_STM_LOC    0x00400000ul
    #define LV_EVT_LOC    0x00500000ul
    #define LV_PRC_LOC    0x00600000ul
    #define LV_RND_LOC    0x00700000ul
    #define LV_AVI_LOC    0x00800000ul
#endif
#define LV_BUF_LOC    0x00900000ul

// GTL - corresponds to GenTL Info
#define LV_SYS_GTL    0x01100000ul
#define LV_ITF_GTL    0x01200000ul
#define LV_DEV_GTL    0x01300000ul
#define LV_STM_GTL    0x01400000ul
#define LV_EVT_GTL    0x01500000ul
#define LV_PRC_GTL    0x01600000ul
#define LV_RND_GTL    0x01700000ul
#define LV_AVI_GTL    0x01800000ul
#define LV_BUF_GTL    0x01900000ul

// GFTR - local features, obtained from XML
#define LV_SYS_GFTR   0x02100000ul
#define LV_ITF_GFTR   0x02200000ul
#define LV_DEV_GFTR   0x02300000ul
#define LV_STM_GFTR   0x02400000ul
#define LV_PRC_GFTR   0x02600000ul
#define LV_RND_GFTR   0x02700000ul
#define LV_AVI_GFTR   0x02800000ul
#define LV_BUF_GFTR   0x02900000ul

// RFTR - remote features, reported by the Device
#define LV_DEV_RFTR   0x03300000ul

// LFTR - local SynView features, obtained from XML
#define LV_SYS_LFTR   0x04100000ul
#define LV_ITF_LFTR   0x04200000ul
#define LV_DEV_LFTR   0x04300000ul
#define LV_STM_LFTR   0x04400000ul
#define LV_PRC_LFTR   0x04600000ul
#define LV_RND_LFTR   0x04700000ul
#define LV_AVI_LFTR   0x04800000ul
#define LV_BUF_LFTR   0x04900000ul

// NFTR - non-feature constants, used for getting various info
#define LV_SYS_NFTR   0x0F100000ul
#define LV_ITF_NFTR   0x0F200000ul
#define LV_DEV_NFTR   0x0F300000ul
#define LV_STM_NFTR   0x0F400000ul
#define LV_PRC_NFTR   0x0F600000ul
#define LV_RND_NFTR   0x0F700000ul
#define LV_AVI_NFTR   0x0F800000ul
#define LV_BUF_NFTR   0x0F900000ul

// INFO - constants for LvFtrInfo_xxx

#define LV_SYS_INFO   0x0E100000ul
#define LV_ITF_INFO   0x0E200000ul
#define LV_DEV_INFO   0x0E300000ul
#define LV_STM_INFO   0x0E400000ul
#define LV_PRC_INFO   0x0E600000ul
#define LV_RND_INFO   0x0E700000ul
#define LV_AVI_INFO   0x0E800000ul
#define LV_BUF_INFO   0x0E900000ul

// signature for feature enumerations
#define LV_FTR_ENUM   0x50000000ul

// defines base number for non-GenTL events
#define LV_SIMPLON_EVENT_BASE 2000


/** @defgroup GroupImgproc_EnumEntryInfo Definitions for Enumeration Entry Info
 *  @ingroup GroupImgproc_Defines
 */
/*@{*/
/** If used as Param of the LvGetInfo(), LvGetInfoStr() and LvGetInfoStrSize()
 *  the returned value is for the current enum entry.
 */
#define LV_ENUMENTRY_CURRENT -1
/*@}*/

//--- Image processing library defines -----------------------------------------

/** Handle to the LUT.
 *  Application should not access LUT data directly, the LUT is to be always modified by
 *  LUT functions. Thus, instead of a pointer to a structure, a handle is used for
 *  manipulating with LUT
*/
typedef uint32_t* LvipHLut;


/** Handle to the overlay.
 *  Application should not access the overlay data directly, the overlay is to be always modified by
 *  overlay functions. Thus, instead of a pointer to a structure, a handle is used for
 *  manipulating with the overlay.
*/
typedef void* LvipHOverlay;


//------------------------------------------------------------------------------
//  TYPEDEFS
//------------------------------------------------------------------------------

/** @ingroup GroupSynview_Defines
 */
/*@{*/

//#define LVAVISAVER - the LvAviSaver module is not implemented

typedef uint32_t  LvHModule;        /**< Base typedef for a handle to a module. */

typedef LvHModule LvHSystem;        /**< Typedef for a handle to the System module. */
typedef LvHModule LvHInterface;     /**< Typedef for a handle to the Interface module. */
typedef LvHModule LvHDevice;        /**< Typedef for a handle to the Device module. */
typedef LvHModule LvHStream;        /**< Typedef for a handle to the Stream module. */
typedef LvHModule LvHEvent;         /**< Typedef for a handle to the Event module. */
typedef LvHModule LvHRenderer;      /**< Typedef for a handle to the Renderer module. */
#ifdef LVAVISAVER
    typedef LvHModule LvHAviSaver;      
#endif
typedef LvHModule LvHBuffer;        /**< Typedef for a handle to the Buffer module. */

typedef uint32_t  LvHOverlay;       /**< Typedef for a handle to the overlay. */

typedef uint32_t  LvFeature;        /**< Base typedef for the ID of the feature. */
typedef uint32_t  LvEnum;           /**< Base typedef for the entry of the enumeration item. */

/*@}*/


/** Prototype for the general callback function, which can be registered at the Event, using the
 *  @ref LvEventSetCallback() function or @ref LvEvent::SetCallback() class.
 *  IMPORTANT: the function must have the LV_STDC calling convention.
 *  @param pBuffer     Pointer to buffer, extracted from the output queue.
 *  @param Size        Buffer size.
 *  @param pUserParam  User parameter, supplied in the @ref LvEventSetCallback() function or
 *                     @ref LvEvent::SetCallback(). It enables the application to distingush
 *                     from which object the callback was called in case the same callback
 *                     function is shared by multiple Event modules.
 *  @ingroup GroupSynview_Defines
 */
typedef void (LV_STDC* LvEventCallbackFunct) (void* pBuffer,
                                              size_t Size,
                                              void* pUserParam);


/** Prototype for the new image callback function, which can be registered at the Event, using the
 *  @ref LvEventSetCallbackNewBuffer() function or @ref LvEvent::SetCallbackNewBuffer() class.
 *  IMPORTANT: the function must have the LV_STDC calling convention.
 *  @param hBuffer      Handle to LvBuffer, extracted from the output queue.
 *  @param pUserPointer The pUserPointer of the LvBuffer is passed here. In the C++ wrapper class this
 *                      is used to give direct pointer to the LvBuffer class instance.
 *  @param pUserParam   User parameter, supplied in the @ref LvEventSetCallbackNewBuffer() function or
 *                      @ref LvEvent::SetCallbackNewBuffer(). It enables the application to distinguish
 *                      from which object the callback was called in case the same callback
 *                      function is shared by multiple Event modules.
 *  @ingroup GroupSynview_Defines
 */
typedef void (LV_STDC* LvEventCallbackNewBufferFunct) (LvHBuffer hBuffer,
                                                       void* pUserPointer,
                                                       void* pUserParam);


/** Prototype for the feature updated callback function, which can be registered using the
 *  @ref LvRegisterFeatureCallback() function.
 *  IMPORTANT: the function must have the LV_STDC calling convention.
 *  @param pUserParam     User pUserParam, supplied in the @ref LvRegisterFeatureCallback() function.
 *                        It can be used to distinguish from which object the callback was called
 *                        in case the same callback function is shared by multiple Event modules.
 *  @param pFeatureParam  The pFeatureParam passed in the @ref LvRegisterFeatureCallback(). It is usually 
 *                        used to identify the feature, which has changed.
 *  @param pName          The string ID of the feature.
 *  @ingroup GroupSynview_Defines
 */
typedef void (LV_STDC* LvFeatureCallbackFunct) (void* pUserParam,
                                                void* pFeatureParam,
                                                const char* pName);


//--- Image processing library typedefs -----------------------------------------

#ifndef LVIP_INTERNAL

/** Image Info structure.
 *  Each image handled by the library must be described by the LvipImgInfo structure.
 *
 *  Although you can set the Image Info members directly, it is highly recommended
 *  to use the @ref LvipInitImgInfo() function for the structure initialization.
 *  @ingroup GroupImgproc_Defines
 */
typedef struct
{
    /** Size of image info structure.
     *  Should be set to the sizeof(LvipImgInfo). This member may be used in
     *  the future versions for the compatibility check.
     */
    uint32_t StructSize;
    /** Width of the image in pixels. */
    uint32_t Width;
    /** Height of the image in pixels. */
    uint32_t Height;
    /** Pixel format of the image which is saved in this structure.
     *  One of the @ref LvPixelFormat
     *
     *  In case of color planes, the pixel format applies to one plane, so use
     *  only the MONO formats for the planes. For example for 3x8-bit RGB use the @ref LvPixelFormat_Mono8 format.
     */
    uint32_t PixelFormat;
    /** Image attributes.
     *  OR-ed definitions from @ref LvipImgAttr definitions.
     */
    uint32_t Attributes;
    /** Size of one pixel in bytes.
     */
    uint32_t BytesPerPixel;
    /** Size of one line in bytes.
     *
     *  Example:
     *  @code
     *    8-bit mono image: LineIncrement = Width;
     *    24-bit RGB image: LineIncrement = Width * 3;
     *  @endcode
     *  However, when the @ref LvipImgAttr_DWordAligned attribute is used, the line increment must be rounded up
     *  to whole double-words, so the calculation would then look like this:
     *  @code
     *    8-bit mono image: LineIncrement = (Width+3)/4 * 4;
     *    24-bit RGB image: LineIncrement = ((Width*3)+3)/4 * 4;
     *  @endcode
     */
    uint32_t LinePitch;
    /** Pointer to image data.
     *  If color planes are not used, this member points to the data of the image.
     *  Use LvipAllocateImageData() to allocate the buffer for the image. If you set the pointer to an existing image,
     *  which is not owned by this @ref LvipImgInfo, use the @ref LvipImgAttr_NotDataOwner attribute.
     */
    uint8_t* pData;
    /** If color planes are used, this member points to the Red plane data of the image.
     *  Use LvipAllocateImageData() to allocate the buffer for the image. If you set the pointer to an existing image,
     *  which is not owned by this @ref LvipImgInfo, use the @ref LvipImgAttr_NotDataOwner attribute.
     */
    uint8_t* pDataR;
    /** If color planes are used, this member points to the Green plane data of the image.
     *  Use LvipAllocateImageData() to allocate the buffer for the image. If you set the pointer to an existing image,
     *  which is not owned by this @ref LvipImgInfo, use the @ref LvipImgAttr_NotDataOwner attribute.
     */
    uint8_t* pDataG;
    /** If color planes are used, this member points to the Blue plane data of the image.
     *  Use LvipAllocateImageData() to allocate the buffer for the image. If you set the pointer to an existing image,
     *  which is not owned by this @ref LvipImgInfo, use the @ref LvipImgAttr_NotDataOwner attribute.
     */
    uint8_t* pDataB;
} LvipImgInfo;

#else
    #include "lv.imgproc.intdef.h"
#endif




#endif
